/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess.papyrus;

import restringer.ess.AnalyzableElement;
import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.SortedSet;
import restringer.IString;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;
import restringer.Profile;
import restringer.ess.ESS;
import restringer.ess.Element;

/**
 * Describes a function message data in a Skyrim savegame.
 *
 * @author Mark Fairchild
 * @version 2016/06/21
 */
public class FunctionMessageData implements PapyrusElement, AnalyzableElement {

    /**
     * Creates a new <code>FunctionMessageData</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @param parent The parent of the message.
     * @param scripts The list of scripts.
     * @param ctx The PapyrusContext.
     * @throws IOException
     */
    public FunctionMessageData(LittleEndianInput input, PapyrusElement parent, ScriptMap scripts, PapyrusContext ctx) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(scripts);
        Objects.requireNonNull(ctx);

        this.UNKNOWN = input.readByte();
        this.SCRIPTNAME = ctx.STRINGS.read(input);
        this.SCRIPT = scripts.getOrDefault(this.SCRIPTNAME, null);
        this.EVENT = ctx.STRINGS.read(input);
        this.UNKNOWNVAR = Variable.read(input, ctx);

        int varCount = input.readInt();
        this.MEMBERS = new java.util.ArrayList<>(varCount);

        for (int i = 0; i < varCount; i++) {
            Variable var = Variable.read(input, ctx);
            this.MEMBERS.add(var);
        }
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        assert null != output;
        output.writeByte(this.UNKNOWN);
        this.SCRIPTNAME.write(output);
        this.EVENT.write(output);
        this.UNKNOWNVAR.write(output);
        output.writeInt(this.MEMBERS.size());

        for (Variable var : this.MEMBERS) {
            var.write(output);
        }
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        int sum = 1;
        sum += this.SCRIPTNAME.calculateSize();
        sum += this.EVENT.calculateSize();
        sum += this.UNKNOWNVAR.calculateSize();
        sum += 4;
        sum += this.MEMBERS.stream().mapToInt(var -> var.calculateSize()).sum();
        return sum;
    }

    /**
     * @return The script name field.
     */
    public TString getScriptName() {
        return this.SCRIPTNAME;
    }

    /**
     * @return The script field.
     */
    public Script getScript() {
        return this.SCRIPT;
    }

    /**
     * @return The event field.
     */
    public TString getEvent() {
        return this.EVENT;
    }

    /**
     * @return The variables field.
     */
    public List<Variable> getMembers() {
        return Collections.unmodifiableList(this.MEMBERS);
    }

    /**
     * @return The qualified name of the function being executed.
     */
    public IString getFName() {
        IString fname = IString.format("%s.%s", this.SCRIPTNAME, this.EVENT);
        return fname;
    }

    /**
     * @return String representation.
     */
    @Override
    public String toString() {
        if (this.isUndefined()) {
            return "#" + this.SCRIPTNAME + "#." + this.EVENT;
        } else {
            return this.SCRIPTNAME + "." + this.EVENT;
        }
    }

    /**
     * @see AnalyzableElement#getInfo(restringer.Profile.Analysis,
     * restringer.ess.ESS)
     * @param analysis
     * @param save
     * @return
     */
    @Override
    public String getInfo(Profile.Analysis analysis, ESS save) {
        final StringBuilder BUILDER = new StringBuilder();
        if (null != this.SCRIPT) {
            BUILDER.append(String.format("<html><h3>FUNCTIONMESSAGEDATA of %s</h3>", this.SCRIPT.toHTML()));
        } else {
            BUILDER.append(String.format("<html><h3>FUNCTIONMESSAGEDATA of %s</h3>", this.SCRIPTNAME));
        }

        if (null != analysis) {
            SortedSet<String> providers = analysis.SCRIPT_ORIGINS.get(this.SCRIPTNAME);
            if (null != providers) {
                String probablyProvider = providers.last();
                BUILDER.append(String.format("<p>This message probably came from \"%s\".</p>", probablyProvider));
                if (providers.size() > 1) {
                    BUILDER.append("<p>Full list of providers:</p><ul>");
                    providers.forEach(mod -> BUILDER.append(String.format("<li>%s", mod)));
                    BUILDER.append("</ul>");
                }
            }
        }

        BUILDER.append("<p>");

        if (null != this.SCRIPT) {
            BUILDER.append(String.format("Script: %s<br/>", this.SCRIPT.toHTML()));
        } else {
            BUILDER.append(String.format("Script: %s<br/>", this.SCRIPTNAME));
        }
        
        BUILDER.append(String.format("Event: %s<br/>", this.EVENT));
        BUILDER.append(String.format("Unknown: %02x<br/>", this.UNKNOWN));

        if (null != this.UNKNOWNVAR) {
            BUILDER.append(String.format("Unknown variable: %s<br/>", this.UNKNOWNVAR.toHTML()));
        } else {
            BUILDER.append(String.format("Unknown variable: %s<br/>", this.UNKNOWNVAR));
        }

        BUILDER.append(String.format("%d additional variables.<br/>", this.MEMBERS.size()));
        BUILDER.append("</p>");
        BUILDER.append("</html>");
        return BUILDER.toString();

    }

    /**
     * @see AnalyzableElement#matches(restringer.Profile.Analysis,
     * restringer.Mod)
     * @param analysis
     * @param mod
     * @return
     */
    @Override
    public boolean matches(Profile.Analysis analysis, String mod) {
        Objects.requireNonNull(analysis);
        Objects.requireNonNull(mod);

        final SortedSet<String> OWNERS = analysis.SCRIPT_ORIGINS.get(this.SCRIPTNAME);
        if (null == OWNERS) {
            return false;
        }
        return OWNERS.contains(mod);
    }

    /**
     * @see PapyrusElement#addNames(restringer.esp.ESPIDMap,
     * restringer.esp.StringTable)
     * @param names The map of IDs to names.
     * @param strings The stringtable.
     */
    @Override
    public void addNames(restringer.esp.ESPIDMap names, restringer.esp.StringTable strings) {
        this.UNKNOWNVAR.addNames(names, strings);
        this.MEMBERS.forEach(var -> var.addNames(names, strings));
    }

    /**
     * @see PapyrusElement#resolveRefs(ESS, Element)
     * @param ess The full savegame.
     * @param owner The owner of the element, or null if it is not owned.
     */
    @Override
    public void resolveRefs(ESS ess, Element owner) {
        this.UNKNOWNVAR.resolveRefs(ess, this);
        this.MEMBERS.forEach(var -> var.resolveRefs(ess, this));
        this.SCRIPTNAME.addRefHolder(owner);
        this.EVENT.addRefHolder(owner);
    }

    /**
     * @return A flag indicating if the <code>FunctionMessageData</code> is
     * undefined.
     *
     */
    public boolean isUndefined() {
        if (null == this.SCRIPT) {
            return !Script.IMPLICITSCRIPTS.contains(this.SCRIPTNAME);
        }

        return this.SCRIPT.isUndefined();
    }

    final private byte UNKNOWN;
    final private TString SCRIPTNAME;
    final private Script SCRIPT;
    final private TString EVENT;
    final private Variable UNKNOWNVAR;
    final private List<Variable> MEMBERS;
}
