/*
 * Copyright 2016 Mark.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess.papyrus;

import java.io.IOException;
import java.util.Objects;
import java.util.SortedSet;
import restringer.LittleEndianDataOutput;
import restringer.LittleEndianInput;
import restringer.Profile;
import restringer.ess.AnalyzableElement;
import restringer.ess.Element;
import restringer.ess.ESS;
import restringer.ess.Linkable;

/**
 * <code>GameElement</code> is a superclass of <code>ScriptInstance</code> and
 * <code>Reference</code>, for situations in which they are interchangeable.
 *
 * @author Mark Fairchild
 */
abstract public class GameElement implements AnalyzableElement, Linkable, PapyrusElement, HasID {

    /**
     * Creates a new <code>ScriptInstances</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @param scripts The list of scripts.
     * @param ctx The PapyrusContext.
     * @throws IOException
     */
    public GameElement(LittleEndianInput input, ScriptMap scripts, PapyrusContext ctx) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(scripts);
        Objects.requireNonNull(ctx);

        if (ctx.GAME.isSSE() || ctx.GAME.isFO4()) {
            this.ID = EID.read8byte(input);
        } else {
            this.ID = EID.read4byte(input);
        }

        this.SCRIPTNAME = ctx.STRINGS.read(input);
        this.SCRIPT = scripts.getOrDefault(this.SCRIPTNAME, null);
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        Objects.requireNonNull(output);
        output.writeESSElement(this.ID);
        output.writeESSElement(this.SCRIPTNAME);
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        int sum = 0;
        sum += this.ID.calculateSize();
        sum += this.SCRIPTNAME.calculateSize();
        return sum;
    }

    /**
     * @return The ID of the papyrus element.
     */
    @Override
    public EID getID() {
        return this.ID;
    }

    /**
     * @return The name of the corresponding <code>Script</code>.
     */
    public TString getScriptName() {
        return this.SCRIPTNAME;
    }

    /**
     * @return The corresponding <code>Script</code>.
     */
    public Script getScript() {
        return this.SCRIPT;
    }

    /**
     * @return A flag indicating if the <code>Reference</code> is undefined.
     *
     */
    public boolean isUndefined() {
        if (null == this.SCRIPT) {
            return !Script.IMPLICITSCRIPTS.contains(this.SCRIPTNAME);
        }

        return this.SCRIPT.isUndefined();
    }

    /**
     * @see PapyrusElement#resolveRefs(ESS, Element)
     * @param ess The full savegame.
     * @param owner The owner of the element, or null if it is not owned.
     */
    @Override
    public void resolveRefs(ESS ess, Element owner) {
        this.SCRIPTNAME.addRefHolder(this);
        if (null != this.SCRIPT) {
            this.SCRIPT.incrementInstanceCount();
        }
    }

    /**
     * @see AnalyzableElement#matches(restringer.Profile.Analysis,
     * restringer.Mod)
     * @param analysis
     * @param mod
     * @return
     */
    @Override
    public boolean matches(Profile.Analysis analysis, String mod) {
        Objects.requireNonNull(analysis);
        Objects.requireNonNull(mod);

        final SortedSet<String> OWNERS = analysis.SCRIPT_ORIGINS.get(this.SCRIPTNAME);
        if (null == OWNERS) {
            return false;
        }
        return OWNERS.contains(mod);
    }

    final private EID ID;
    final private TString SCRIPTNAME;
    final private Script SCRIPT;
}
