/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess.papyrus;

import java.io.IOException;
import java.util.ArrayList;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;
import restringer.ess.Element;
import restringer.ess.RefID;

/**
 * Describes the various small arrays in a skyrim savefile.
 *
 * @author Mark Fairchild
 * @version 2016/06/21
 */
abstract public class MiscArray implements Element {

    static public class Array_1_2 extends MiscArray {

        public Array_1_2(LittleEndianInput input, StringTable strings) throws IOException {
            CONTENTS.add(new PInt(input));
            CONTENTS.add(new PInt(input));
        }
    }

    static public class Array_3 extends MiscArray {

        public Array_3(LittleEndianInput input, StringTable strings) throws IOException {
            CONTENTS.add(new PByte(input));
            CONTENTS.add(strings.read(input));
            CONTENTS.add(new PShort(input));
            CONTENTS.add(strings.read(input));
            CONTENTS.add(new PInt(input));
            CONTENTS.add(new PShort(input));
            CONTENTS.add(new RefID(input));
        }
    }

    static public class Array_4 extends MiscArray {

        public Array_4(LittleEndianInput input, StringTable strings) throws IOException {
            CONTENTS.add(strings.read(input));
            CONTENTS.add(new PShort(input));
            CONTENTS.add(new PByte(input));
            CONTENTS.add(strings.read(input));
            CONTENTS.add(new PInt(input));
            CONTENTS.add(new PShort(input));
            CONTENTS.add(new RefID(input));
        }
    }

    static public class Array_4b extends MiscArray {

        public Array_4b(LittleEndianInput input, StringTable strings) throws IOException {
            CONTENTS.add(new PByte(input));
            CONTENTS.add(new PShort(input));
            CONTENTS.add(new PShort(input));
            CONTENTS.add(new RefID(input));
            CONTENTS.add(new RefID(input));
            CONTENTS.add(new RefID(input));
            CONTENTS.add(new RefID(input));
        }
    }

    static public class Array_4c extends MiscArray {

        public Array_4c(LittleEndianInput input, StringTable strings) throws IOException {
            byte flag = input.readByte();
            CONTENTS.add(new PByte(flag));
            CONTENTS.add(new PInt(input));
            CONTENTS.add(new RefID(input));

            if (0 <= flag && flag <= 6) {
                CONTENTS.add(new PInt(input));
                CONTENTS.add(new PInt(input));
                CONTENTS.add(new PInt(input));
            }

            if (flag == 0) {
                CONTENTS.add(new PInt(input));
                CONTENTS.add(new PInt(input));
                CONTENTS.add(new PInt(input));
                CONTENTS.add(new PInt(input));
            }
            
            if (0 <= flag && flag <= 3) {
                CONTENTS.add(new PByte(input));
            }
        }
    }

    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        for (Element e : this.CONTENTS) {
            e.write(output);
        }
    }

    @Override
    public int calculateSize() {
        return this.CONTENTS.stream().mapToInt(v -> v.calculateSize()).sum();
    }

    final protected ArrayList<Element> CONTENTS = new ArrayList<>();

    /**
     * Store an integer as a papyrus element.
     */
    static private class PInt implements Element {

        public PInt(LittleEndianInput input) throws IOException {
            this.VAL = input.readInt();
        }

        public PInt(int v) {
            this.VAL = v;
        }

        @Override
        public void write(LittleEndianDataOutput output) throws IOException {
            output.writeInt(this.VAL);
        }

        @Override
        public int calculateSize() {
            return 4;
        }

        final int VAL;
    }

    /**
     * Store an integer as a papyrus element.
     */
    static private class PShort implements Element {

        public PShort(LittleEndianInput input) throws IOException {
            this.VAL = input.readShort();
        }

        public PShort(short v) {
            this.VAL = v;
        }

        @Override
        public void write(LittleEndianDataOutput output) throws IOException {
            output.writeShort(this.VAL);
        }

        @Override
        public int calculateSize() {
            return 2;
        }

        final short VAL;
    }

    /**
     * Store an integer as a papyrus element.
     */
    static private class PByte implements Element {

        public PByte(LittleEndianInput input) throws IOException {
            this.VAL = input.readByte();
        }

        public PByte(byte b) {
            this.VAL = b;
        }

        @Override
        public void write(LittleEndianDataOutput output) throws IOException {
            output.writeByte(this.VAL);
        }

        @Override
        public int calculateSize() {
            return 1;
        }

        final byte VAL;
    }
}
