/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess.papyrus;

import java.io.IOException;
import java.util.*;
import restringer.LittleEndianDataOutput;
import restringer.LittleEndianInputStream;
import restringer.ess.GlobalDataBlock;
import restringer.ess.ESS;
import restringer.ess.ESSContext;
import restringer.ess.Element;
import java.util.stream.Collectors;

/**
 * Describes a the data for a <code>GlobalData</code> when it is the Papyrus
 * script section.
 *
 * @author Mark Fairchild
 * @version 2016/06/21
 */
final public class Papyrus implements PapyrusElement, GlobalDataBlock {

    /**
     * Creates a new <code>Papyrus</code> by reading from a byte buffer.
     *
     * @param buffer The data.
     * @param essCTX The ESSContext.
     * @param applySTBCorrection A flag indicating that <code>StringTable</code>
     * should ATTEMPT to correct for the string table bug.
     * @throws IOException
     */
    public Papyrus(byte[] buffer, ESSContext essCTX, boolean applySTBCorrection) throws IOException {
        Objects.requireNonNull(buffer);

        //System.out.println("DEBUGGING");
        try (LittleEndianInputStream input = LittleEndianInputStream.debug(buffer)) {
            int sum = 0;
            int i = 0;

            // Read the header.            
            this.HEADER = input.readShort();
            sum += 2;

            // Read the string table.
            this.STRINGS = new StringTable(input, essCTX.GAME, applySTBCorrection);
            sum += this.STRINGS.calculateSize();

            // Create the PapyrusContext.
            final PapyrusContext CTX = new PapyrusContext(essCTX, this.STRINGS);

            // Read the script table and struct table.
            if (CTX.GAME.isFO4()) {
                // Fallout4 has scripts and structs, so the data is a bit
                // different right here.

                int scriptCount = input.readInt();
                this.SCRIPTS = new ScriptMap(scriptCount);

                int structDefCount = input.readInt();
                this.STRUCTDEFS = new StructDefMap(structDefCount);

                // Read the scripts.
                try {
                    for (i = 0; i < scriptCount; i++) {
                        Script script = new Script(input, CTX);
                        this.SCRIPTS.put(script.getName(), script);
                    }
                    sum += 4 + this.SCRIPTS.values().stream().mapToInt(v -> v.calculateSize()).sum();
                } catch (IOException ex) {
                    throw new IOException(String.format("Error; read %d/%d scripts.", i, scriptCount), ex);
                }

                // Read the structs.
                try {
                    for (i = 0; i < structDefCount; i++) {
                        StructDef struct = new StructDef(input, CTX);
                        this.STRUCTDEFS.put(struct.getName(), struct);
                    }
                    sum += 4 + this.STRUCTDEFS.values().stream().mapToInt(v -> v.calculateSize()).sum();
                } catch (IOException ex) {
                    throw new IOException(String.format("Error; read %d/%d struct definitions.", i, structDefCount), ex);
                }

            } else {
                // Skyrim and SkyrimSE just have scripts.
                // No real differences between them here.
                int scriptCount = input.readInt();
                this.SCRIPTS = new ScriptMap(scriptCount);
                this.STRUCTDEFS = null;

                try {
                    for (i = 0; i < scriptCount; i++) {
                        Script script = new Script(input, CTX);
                        this.SCRIPTS.put(script.getName(), script);

                    }
                    sum += 4 + this.SCRIPTS.values().stream().mapToInt(v -> v.calculateSize()).sum();
                } catch (IOException | AssertionError ex) {
                    throw new IOException(String.format("Error; read %d/%d script definitions.", i, scriptCount), ex);
                }

            }

            // Read the script instance table.
            int instanceCount = input.readInt();
            this.INSTANCES = new InstanceMap(instanceCount);

            try {
                for (i = 0; i < instanceCount; i++) {
                    ScriptInstance instance = new ScriptInstance(input, this.SCRIPTS, CTX);
                    this.INSTANCES.put(instance.getID(), instance);
                }
                sum += 4 + this.INSTANCES.values().parallelStream().mapToInt(v -> v.calculateSize()).sum();
            } catch (IOException ex) {
                throw new IOException(String.format("Error; read %d/%d instance definitions.", i, instanceCount), ex);
            }

            // Read the reference table.
            int referenceCount = input.readInt();
            this.REFERENCES = new ReferenceMap(referenceCount);

            try {
                for (i = 0; i < referenceCount; i++) {
                    Reference reference = new Reference(input, this.SCRIPTS, CTX);
                    this.REFERENCES.put(reference.getID(), reference);
                }
                sum += 4 + this.REFERENCES.values().stream().mapToInt(v -> v.calculateSize()).sum();
            } catch (IOException ex) {
                throw new IOException(String.format("Error; read %d/%d reference definitions.", i, referenceCount), ex);
            }

            // Read the struct body table.
            if (CTX.GAME.isFO4()) {
                // Read the struct table.
                int structCount = input.readInt();
                this.STRUCTS = new StructMap(structCount);

                try {
                    for (i = 0; i < structCount; i++) {
                        Struct struct = new Struct(input, this.SCRIPTS, CTX);
                        this.STRUCTS.put(struct.getID(), struct);
                    }
                    sum += 4 + this.STRUCTS.values().stream().mapToInt(v -> v.calculateSize()).sum();
                } catch (IOException ex) {
                    throw new IOException(String.format("Error; read %d/%d struct instances.", i, structCount), ex);
                }
            } else {
                this.STRUCTS = null;
            }

            // Read the array table.
            int arrayCount = input.readInt();
            this.ARRAYS = new ArrayMap(arrayCount);

            try {
                for (i = 0; i < arrayCount; i++) {
                    ArrayInfo info = new ArrayInfo(input, CTX);
                    this.ARRAYS.put(info.getID(), info);
                }
                sum += 4 + this.ARRAYS.values().stream().mapToInt(v -> v.calculateSize()).sum();
            } catch (IOException ex) {
                throw new IOException(String.format("Error; read %d/%d array definitions.", i, arrayCount), ex);
            }

            this.PAPYRUS_RUNTIME = input.readInt();
            sum += 4;

            // Read the active script table.
            int activeScriptCount = input.readInt();
            this.ACTIVESCRIPTS = new ActiveScriptMap(activeScriptCount);

            try {
                for (i = 0; i < activeScriptCount; i++) {
                    ActiveScript active = new ActiveScript(input, CTX);
                    this.ACTIVESCRIPTS.put(active.getID(), active);
                }
                sum += 4 + this.ACTIVESCRIPTS.values().stream().mapToInt(v -> v.calculateSize()).sum();
            } catch (IOException ex) {
                throw new IOException(String.format("Error; read %d/%d activescript definitions.", i, activeScriptCount), ex);
            }

            // Read the script instance data table and associate the data
            // with the relevant script instances.
            try {
                for (i = 0; i < instanceCount; i++) {
                    ScriptData data = new ScriptData(input, CTX);
                    ScriptInstance instance = this.INSTANCES.get(data.getID());
                    instance.setData(data);
                }
                sum += this.INSTANCES.values().parallelStream().mapToInt(v -> v.getData().calculateSize()).sum();
            } catch (Exception | Error ex) {
                throw new IOException(String.format("Error; read %d/%d instance data blocks.", i, instanceCount), ex);
            }

            // Read the reference data table and associate the data
            // with the relevant references.
            try {
                for (i = 0; i < referenceCount; i++) {
                    ReferenceData data = new ReferenceData(input, CTX);
                    Reference ref = this.REFERENCES.get(data.getID());
                    ref.setData(data);
                }
                sum += this.REFERENCES.values().stream().mapToInt(v -> v.getData().calculateSize()).sum();
            } catch (IOException ex) {
                throw new IOException(String.format("Error; read %d/%d reference data blocks.", i, referenceCount), ex);
            }

            // Read the array data table and associate the data
            // with the relevant arrays.
            try {
                for (i = 0; i < arrayCount; i++) {
                    ArrayData data = new ArrayData(input, this.ARRAYS, CTX);
                    ArrayInfo array = this.ARRAYS.get(data.getID());
                    array.setData(data);
                }
                sum += this.ARRAYS.values().stream().mapToInt(v -> v.getData().calculateSize()).sum();
            } catch (IOException ex) {
                throw new IOException(String.format("Error; read %d/%d array data blocks.", i, arrayCount), ex);
            }

            // Read the active script data table and associate the data
            // with the relevant script instance.
            try {
                for (i = 0; i < activeScriptCount; i++) {
                    ActiveScriptData data = new ActiveScriptData(input, this.SCRIPTS, CTX);
                    ActiveScript script = this.ACTIVESCRIPTS.get(data.getID());
                    script.setData(data);
                }
                sum += this.ACTIVESCRIPTS.values().stream().mapToInt(v -> v.getData().calculateSize()).sum();
            } catch (IOException ex) {
                throw new IOException(String.format("Error; read %d/%d activescript data blocks.", i, activeScriptCount), ex);
            }

            // Read the function message table.
            int functionMessageCount = input.readInt();
            this.FUNCTIONMESSAGES = new ArrayList<>(functionMessageCount);

            try {
                for (i = 0; i < functionMessageCount; i++) {
                    FunctionMessage message = new FunctionMessage(input, this.SCRIPTS, CTX);
                    this.FUNCTIONMESSAGES.add(message);
                }
                sum += 4 + this.FUNCTIONMESSAGES.stream().mapToInt(v -> v.calculateSize()).sum();
            } catch (IOException ex) {
                throw new IOException(String.format("Error; read %d/%d functionmessages.", i, functionMessageCount), ex);
            }

            // Read the first SuspendedStack table.
            int stack1Count = input.readInt();
            this.SUSPENDEDSTACKS1 = new ArrayList<>(stack1Count);

            try {
                for (i = 0; i < stack1Count; i++) {
                    SuspendedStack stack = new SuspendedStack(input, this.SCRIPTS, CTX);
                    this.SUSPENDEDSTACKS1.add(stack);
                }
                sum += 4 + this.SUSPENDEDSTACKS1.stream().mapToInt(v -> v.calculateSize()).sum();
            } catch (IOException ex) {
                throw new IOException(String.format("Error; read %d/%d suspended stacks (1).", i, stack1Count), ex);
            }

            // Read the second SuspendedStack table.
            int stack2Count = input.readInt();
            this.SUSPENDEDSTACKS2 = new ArrayList<>(stack2Count);

            try {
                for (i = 0; i < stack2Count; i++) {
                    SuspendedStack stack = new SuspendedStack(input, this.SCRIPTS, CTX);
                    this.SUSPENDEDSTACKS2.add(stack);
                }
                sum += 4 + this.SUSPENDEDSTACKS2.stream().mapToInt(v -> v.calculateSize()).sum();
            } catch (IOException ex) {
                throw new IOException(String.format("Error; read %d/%d suspended stacks (2).", i, stack2Count), ex);
            }

            // Read the "unknown" fields.
            this.UNK1 = input.readInt();
            this.UNK2 = (this.UNK1 == 0 ? 0 : input.readInt());
            sum += (this.UNK1 == 0 ? 4 : 8);

            int unknownCount = input.readInt();
            this.UNKS = new ArrayList<>(unknownCount);

            if (CTX.GAME.isSSE() || CTX.GAME.isFO4()) {
                for (i = 0; i < unknownCount; i++) {
                    EID id = EID.read8byte(input);
                    this.UNKS.add(id);
                }
            } else {
                for (i = 0; i < unknownCount; i++) {
                    EID id = EID.read4byte(input);
                    this.UNKS.add(id);
                }
            }
            sum += 4 + this.UNKS.parallelStream().mapToInt(v -> v.calculateSize()).sum();

            // Read the queued unbinds... whatever those are.
            int queuedUnbindCount = input.readInt();
            this.UNBINDS = new ArrayList<>(queuedUnbindCount);

            for (i = 0; i < queuedUnbindCount; i++) {
                QueuedUnbind unbind = new QueuedUnbind(input, this.INSTANCES, CTX);
                this.UNBINDS.add(unbind);
            }
            sum += 4 + this.UNBINDS.stream().mapToInt(v -> v.calculateSize()).sum();

            // Read the save file version field.
            this.SAVE_FILE_VERSION = input.readShort();
            sum += 2;

            // Stuff the remaining data into a buffer.
            int remaining = input.available();
            this.OTHERDATA = new byte[remaining];
            int read = input.read(this.OTHERDATA);
            sum += this.OTHERDATA.length;
            assert read == remaining;
            assert sum == this.calculateSize();
            assert sum == buffer.length;
        }

        /*
        try (final java.io.ByteArrayOutputStream BAOS = new java.io.ByteArrayOutputStream(buffer.length);
                final LittleEndianDataOutput LEDO = LittleEndianDataOutput.wrap(BAOS)) {
            LEDO.writeESSElement(this);

            final byte[] BUF2 = BAOS.toByteArray();
            for (int i = 0; i < buffer.length; i++) {
                if (buffer[i] != BUF2[i]) {
                    assert false : "Difference at " + i;
                }
            }
        }
         */
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        output = new LittleEndianDataOutput(output);

        assert null != output;
        output.writeShort(this.HEADER);

        // Write the string table.
        this.STRINGS.write(output);

        // Write the script table.
        if (null != this.STRUCTS) {
            output.writeInt(this.SCRIPTS.size());
            output.writeInt(this.STRUCTS.size());

            for (Script script : this.SCRIPTS.values()) {
                script.write(output);
            }
            for (StructDef struct : this.STRUCTDEFS.values()) {
                struct.write(output);
            }

        } else {
            output.writeInt(this.SCRIPTS.size());
            for (Script script : this.SCRIPTS.values()) {
                script.write(output);
            }
        }

        // Write the script instance table.
        output.writeInt(this.INSTANCES.size());
        for (ScriptInstance instance : this.INSTANCES.values()) {
            instance.write(output);
        }

        // Write the reference table.
        output.writeInt(this.REFERENCES.size());
        for (Reference ref : this.REFERENCES.values()) {
            ref.write(output);
        }

        // Write the struct instance table.
        if (null != this.STRUCTS) {
            output.writeInt(this.STRUCTS.size());

            for (Struct struct : this.STRUCTS.values()) {
                struct.write(output);
            }
        }

        // Write the array table.
        output.writeInt(this.ARRAYS.size());
        for (ArrayInfo info : this.ARRAYS.values()) {
            info.write(output);
        }

        output.writeInt(this.PAPYRUS_RUNTIME);

        // Write the active script table.
        output.writeInt(this.ACTIVESCRIPTS.size());
        for (ActiveScript script : this.ACTIVESCRIPTS.values()) {
            script.write(output);
        }

        // Write the script instance data table.
        for (ScriptInstance instance : this.INSTANCES.values()) {
            ScriptData data = instance.getData();
            data.write(output);
        }

        // Write the reference data table.
        for (Reference ref : this.REFERENCES.values()) {
            ref.getData().write(output);
        }

        // Write the array data table.
        for (ArrayInfo info : this.ARRAYS.values()) {
            info.getData().write(output);
        }

        // Write the active script data table.
        for (ActiveScript script : this.ACTIVESCRIPTS.values()) {
            script.getData().write(output);
        }

        // Write the function message table.
        output.writeInt(this.FUNCTIONMESSAGES.size());
        for (FunctionMessage message : this.FUNCTIONMESSAGES) {
            message.write(output);
        }

        System.out.println(output.getPosition());

        // Write the first suspended stack table.
        output.writeInt(this.SUSPENDEDSTACKS1.size());
        for (SuspendedStack stack : this.SUSPENDEDSTACKS1) {
            stack.write(output);
        }

        // Write the first suspended stack table.
        output.writeInt(this.SUSPENDEDSTACKS2.size());
        for (SuspendedStack stack : this.SUSPENDEDSTACKS2) {
            stack.write(output);
        }

        // Write the "unknown" fields.
        output.writeInt(this.UNK1);
        if (this.UNK1 != 0) {
            output.writeInt(this.UNK2);
        }

        output.writeInt(this.UNKS.size());
        for (EID id : this.UNKS) {
            output.writeESSElement(id);
        }

        // Write the queued unbind table.
        output.writeInt(this.UNBINDS.size());
        for (QueuedUnbind unbind : this.UNBINDS) {
            unbind.write(output);
        }

        // Write the save file version field.
        output.writeShort(this.SAVE_FILE_VERSION);

        // Write the remaining data.
        output.write(this.OTHERDATA);
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        int sum = 2;

        sum += this.STRINGS.calculateSize();

        sum += 4;
        sum += this.SCRIPTS.values().parallelStream().mapToInt(v -> v.calculateSize()).sum();

        if (null != this.STRUCTS) {
            sum += 4;
            sum += this.STRUCTS.values().parallelStream().mapToInt(v -> v.calculateSize()).sum();
        }

        sum += 4;
        sum += this.INSTANCES.values().parallelStream().mapToInt(v -> v.calculateSize()).sum();

        sum += 4;
        sum += this.REFERENCES.values().parallelStream().mapToInt(v -> v.calculateSize()).sum();

        sum += 4;
        sum += this.ARRAYS.values().parallelStream().mapToInt(v -> v.calculateSize()).sum();

        sum += 4;

        sum += 4;
        sum += this.ACTIVESCRIPTS.values().parallelStream().mapToInt(v -> v.calculateSize()).sum();

        sum += this.INSTANCES.values().parallelStream().mapToInt(v -> v.getData().calculateSize()).sum();
        sum += this.REFERENCES.values().parallelStream().mapToInt(v -> v.getData().calculateSize()).sum();
        sum += this.ARRAYS.values().parallelStream().mapToInt(v -> v.getData().calculateSize()).sum();
        sum += this.ACTIVESCRIPTS.values().parallelStream().mapToInt(v -> v.getData().calculateSize()).sum();

        sum += 4;
        sum += this.FUNCTIONMESSAGES.parallelStream().mapToInt(v -> v.calculateSize()).sum();

        sum += 4;
        sum += this.SUSPENDEDSTACKS1.parallelStream().mapToInt(v -> v.calculateSize()).sum();

        sum += 4;
        sum += this.SUSPENDEDSTACKS2.parallelStream().mapToInt(v -> v.calculateSize()).sum();

        sum += (this.UNK1 == 0 ? 4 : 8);
        sum += 4 + this.UNKS.parallelStream().mapToInt(v -> v.calculateSize()).sum();

        sum += 4;
        sum += this.UNBINDS.parallelStream().mapToInt(v -> v.calculateSize()).sum();

        sum += 2; // save file version.

        sum += this.OTHERDATA.length;

        return sum;
    }

    /**
     * @return Accessor for the string table.
     */
    public StringTable getStringTable() {
        return this.STRINGS;
    }

    /**
     * @return Accessor for the list of scripts.
     */
    public ScriptMap getScripts() {
        return this.SCRIPTS;
    }

    /**
     * @return Accessor for the list of function messages.
     */
    public List<FunctionMessage> getFunctionMessages() {
        return this.FUNCTIONMESSAGES;
    }

    /**
     * @return Accessor for the list of script instances.
     */
    public InstanceMap getInstances() {
        return this.INSTANCES;
    }

    /**
     * @return Accessor for the list of references.
     */
    public ReferenceMap getReferences() {
        return this.REFERENCES;
    }

    /**
     * @return Accessor for the list of arrays.
     */
    public ArrayMap getArrays() {
        return this.ARRAYS;
    }

    /**
     * @return Accessor for the list of active scripts.
     */
    public ActiveScriptMap getActiveScripts() {
        return this.ACTIVESCRIPTS;
    }

    /**
     * @return Accessor for the first list of suspended stacks.
     */
    public List<SuspendedStack> getSuspendedStacks1() {
        return this.SUSPENDEDSTACKS1;
    }

    /**
     * @return Accessor for the second list of suspended stacks.
     */
    public List<SuspendedStack> getSuspendedStacks2() {
        return this.SUSPENDEDSTACKS2;
    }

    /**
     * @return Accessor for the queued unbinds list.
     */
    public List<QueuedUnbind> getUnbinds() {
        return this.UNBINDS;
    }

    /**
     * Removes all <code>ScriptInstance</code> objects whose refID is 0.
     *
     * @return The number of instances removed.
     */
    public int cleanUnattachedInstances() {
        final List<ScriptInstance> UNATTACHED = this.INSTANCES.values()
                .stream()
                .filter(v -> v.isUnattached())
                .collect(Collectors.toList());

        this.INSTANCES.values().removeAll(UNATTACHED);
        return UNATTACHED.size();
    }

    /**
     * Removes all <code>ScriptInstance</code> objects whose script is null.
     * Also checks <code>ActiveScript</code>, <code>FunctionMessage</code>, and
     * <code>SuspendedStack</code>.
     *
     * @return The number of instances removed.
     */
    public int cleanUndefined() {
        int count = 0;
        count += this.SCRIPTS.values().stream().filter(v -> v.isUndefined()).count();
        count += this.INSTANCES.values().parallelStream().filter(v -> v.isUndefined()).count();
        count += this.ACTIVESCRIPTS.values().stream().filter(v -> v.isUndefined()).count();
        count += this.REFERENCES.values().stream().filter(v -> v.isUndefined()).count();

        this.SCRIPTS.values().removeIf(v -> v.isUndefined());
        this.INSTANCES.values().removeIf(v -> v.isUndefined());
        this.REFERENCES.values().removeIf(v -> v.isUndefined());
        this.ACTIVESCRIPTS.values().stream().filter(v -> v.isUndefined()).forEach(v -> v.zero());

        return count;
    }

    /**
     * Removes a script.
     *
     * @param script The script to remove.
     * @param removeAll A flag indicating whether to remove all elements that
     * contain the script.
     * @return A flag indicating that the script was found and removed.
     */
    private boolean removeScript(Script script, boolean removeAll) {
        Objects.requireNonNull(script);
        if (!this.SCRIPTS.containsKey(script.getName())) {
            return false;
        }

        this.SCRIPTS.remove(script.getName());

        if (removeAll) {
            this.INSTANCES.values().removeIf(v -> v.getScript() == script);
            this.REFERENCES.values().removeIf(v -> v.getScript() == script);
            this.ACTIVESCRIPTS.values().stream().filter(v -> v.hasScript(script)).forEach(v -> v.zero());
        }

        return true;
    }

    /**
     * Removes a <code>PapyrusElement</code>.
     *
     * @param element The element to remove.
     * @return A flag indicating that the element was found and removed.
     */
    public boolean removeElement(PapyrusElement element) {
        Objects.requireNonNull(element);

        if (element instanceof Script) {
            return this.removeScript((Script) element, true);

        } else if (element instanceof ScriptInstance) {
            ScriptInstance instance = (ScriptInstance) element;
            return null != this.INSTANCES.remove(instance.getID());

        } else if (element instanceof Reference) {
            Reference ref = (Reference) element;
            return null != this.REFERENCES.remove(ref.getID());

        } else if (element instanceof ArrayInfo) {
            ArrayInfo array = (ArrayInfo) element;
            return null != this.ARRAYS.remove(array.getID());

        } else if (element instanceof ActiveScript) {
            ActiveScript active = (ActiveScript) element;
            return null != this.ACTIVESCRIPTS.remove(active.getID());

        } else {
            return false;
        }

    }

    /**
     * @return String representation.
     */
    @Override
    public String toString() {
        return "Papyrus-" + super.toString();
    }

    /**
     * @see PapyrusElement#addNames(restringer.esp.ESPIDMap,
     * restringer.esp.StringTable)
     * @param names The map of IDs to names.
     * @param strings The stringtable.
     */
    @Override
    public void addNames(restringer.esp.ESPIDMap names, restringer.esp.StringTable strings) {
        this.INSTANCES.values().forEach(v -> v.addNames(names, strings));
        this.REFERENCES.values().forEach(v -> v.addNames(names, strings));
        this.ACTIVESCRIPTS.values().forEach(v -> v.addNames(names, strings));
        this.ARRAYS.values().forEach(v -> v.addNames(names, strings));
        this.SUSPENDEDSTACKS1.forEach(v -> v.addNames(names, strings));
        this.SUSPENDEDSTACKS2.forEach(v -> v.addNames(names, strings));
        this.FUNCTIONMESSAGES.forEach(v -> v.addNames(names, strings));
    }

    /**
     * @see PapyrusElement#resolveRefs(ESS, Element)
     * @param ess The full savegame.
     * @param owner The owner of the element, or null if it is not owned.
     */
    @Override
    public void resolveRefs(ESS ess, Element owner) {
        this.ARRAYS.values().forEach(array -> array.resolveRefs(ess, null));
        this.SCRIPTS.values().forEach(script -> script.resolveRefs(ess, null));
        this.INSTANCES.values().forEach(instance -> instance.resolveRefs(ess, null));
        this.REFERENCES.values().forEach(ref -> ref.resolveRefs(ess, null));
        this.ACTIVESCRIPTS.values().forEach(script -> script.resolveRefs(ess, null));
        this.SUSPENDEDSTACKS1.forEach(stack -> stack.resolveRefs(ess, null));
        this.SUSPENDEDSTACKS2.forEach(stack -> stack.resolveRefs(ess, null));
        this.FUNCTIONMESSAGES.forEach(msg -> msg.resolveRefs(ess, null));
    }

    /**
     * Does a very general search for an ID.
     *
     * @param id The ID to search for.
     * @return Any match of any kind.
     */
    public PapyrusElement broadSpectrumMatch(EID id) {
        if (this.INSTANCES.containsKey(id)) {
            return this.INSTANCES.get(id);
        }
        if (this.REFERENCES.containsKey(id)) {
            return this.REFERENCES.get(id);
        }
        if (this.ARRAYS.containsKey(id)) {
            return this.ARRAYS.get(id);
        }
        if (this.ACTIVESCRIPTS.containsKey(id)) {
            return this.ACTIVESCRIPTS.get(id);
        }

        Optional<FunctionMessage> msg = this.FUNCTIONMESSAGES.stream().filter(v -> v.getID().equals(id)).findAny();
        if (msg.isPresent()) {
            return msg.get();
        }

        Optional<SuspendedStack> susp1 = this.SUSPENDEDSTACKS1.stream().filter(v -> v.getID().equals(id)).findAny();
        if (susp1.isPresent()) {
            return susp1.get();
        }

        Optional<SuspendedStack> susp2 = this.SUSPENDEDSTACKS1.stream().filter(v -> v.getID().equals(id)).findAny();
        if (susp2.isPresent()) {
            return susp2.get();
        }
        Optional<QueuedUnbind> qu = this.UNBINDS.stream().filter(v -> v.getID().equals(id)).findAny();
        if (qu.isPresent()) {
            return qu.get();
        }

        return null;
    }

    final private short HEADER;
    final private int PAPYRUS_RUNTIME;
    final private short SAVE_FILE_VERSION;
    final private int UNK1;
    final private int UNK2;

    final private StringTable STRINGS;
    final private ScriptMap SCRIPTS;
    final private StructDefMap STRUCTDEFS;
    final private InstanceMap INSTANCES;
    final private ReferenceMap REFERENCES;
    final private StructMap STRUCTS;
    final private ArrayMap ARRAYS;
    final private ActiveScriptMap ACTIVESCRIPTS;
    final private List<FunctionMessage> FUNCTIONMESSAGES;
    final private List<SuspendedStack> SUSPENDEDSTACKS1;
    final private List<SuspendedStack> SUSPENDEDSTACKS2;
    final private List<EID> UNKS;
    final private List<QueuedUnbind> UNBINDS;

    final private byte[] OTHERDATA;

}
