/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess.papyrus;

import restringer.ess.AnalyzableElement;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.SortedSet;
import java.util.stream.Collectors;
import restringer.IString;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;
import restringer.Profile;
import restringer.ess.Element;
import restringer.ess.ESS;
import restringer.ess.Linkable;

/**
 * Describes a script in a Skyrim savegame.
 *
 * @author Mark Fairchild
 * @version 2016/06/19
 */
final public class Script implements PapyrusElement, AnalyzableElement, Linkable {

    /**
     * Creates a new <code>Script</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @param ctx The PapyrusContext.
     * @throws IOException
     */
    public Script(LittleEndianInput input, PapyrusContext ctx) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(ctx);

        this.NAME = ctx.STRINGS.read(input);
        this.TYPE = ctx.STRINGS.read(input);

        int memberCount = input.readInt();
        this.MEMBERS = new java.util.ArrayList<>(memberCount);
        for (int i = 0; i < memberCount; i++) {
            MemberData member = new MemberData(input, ctx);
            this.MEMBERS.add(member);
        }
        this.instanceCount = 0;
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        Objects.requireNonNull(output);
        
        this.NAME.write(output);
        this.TYPE.write(output);
        output.writeInt(this.MEMBERS.size());

        for (MemberData member : this.MEMBERS) {
            member.write(output);
        }
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        int sum = 4;
        sum += this.NAME.calculateSize();
        sum += this.TYPE.calculateSize();
        sum += this.MEMBERS.stream().mapToInt(member -> member.calculateSize()).sum();
        return sum;
    }

    /**
     * @return The ID of the papyrus element.
     */
    public TString getName() {
        return this.NAME;
    }

    /**
     * @return The type of the array.
     */
    public TString getType() {
        return this.TYPE;
    }

    /**
     * @return The list of <code>MemberData</code>.
     */
    public List<MemberData> getMembers() {
        return java.util.Collections.unmodifiableList(this.MEMBERS);
    }

    /**
     * @return The list of <code>MemberData</code> prepended by the
     * <code>MemberData</code> objects of all superscripts.
     */
    public List<MemberData> getExtendedMembers() {
        if (null != this.parent) {
            final List<MemberData> EXTENDED = this.parent.getExtendedMembers();
            EXTENDED.addAll(this.MEMBERS);
            return EXTENDED;
        } else {
            final List<MemberData> EXTENDED = new ArrayList<>(this.MEMBERS);
            return EXTENDED;
        }
    }

    /**
     * @see PapyrusElement#addNames(restringer.esp.ESPIDMap,
     * restringer.esp.StringTable)
     * @param names The map of IDs to names.
     * @param strings The stringtable.
     */
    @Override
    public void addNames(restringer.esp.ESPIDMap names, restringer.esp.StringTable strings) {
    }

    /**
     * @see PapyrusElement#resolveRefs(ESS, Element)
     * @param ess The full savegame.
     * @param owner The owner of the element, or null if it is not owned.
     */
    @Override
    public void resolveRefs(ESS ess, Element owner) {
        this.parent = ess.getPapyrus().getScripts().get(this.TYPE);
        this.MEMBERS.forEach(member -> member.resolveRefs(ess, this));
        this.NAME.addRefHolder(this);
        this.TYPE.addRefHolder(this);
    }

    /**
     * @see restringer.ess.Linkable#toHTML()
     * @return
     */
    @Override
    public String toHTML() {
        return String.format("<a href=\"script://%s\">%s</a>", this.NAME, this.NAME);
    }

    /**
     * @return String representation.
     */
    @Override
    public String toString() {
        if (this.isUndefined()) {
            return "#" + this.NAME + " (" + this.instanceCount + ")";
        }

        return this.NAME + " (" + this.instanceCount + ")";

    }

    /**
     * @see AnalyzableElement#getInfo(restringer.Profile.Analysis,
     * restringer.ess.ESS)
     * @param analysis
     * @param save
     * @return
     */
    @Override
    public String getInfo(Profile.Analysis analysis, ESS save) {
        final StringBuilder BUILDER = new StringBuilder();
        BUILDER.append("<html>");

        if (this.TYPE.isEmpty()) {
            BUILDER.append(String.format("<h3>SCRIPT %s</h3>", this.NAME));
        } else if (null != this.parent) {
            BUILDER.append(String.format("<h3>SCRIPT %s extends %s</h3>", this.NAME, this.parent.toHTML()));
        } else {
            BUILDER.append(String.format("<h3>SCRIPT %s extends %s</h3>", this.NAME, this.TYPE));
        }

        if (this.isUndefined()) {
            BUILDER.append("<p>WARNING: SCRIPT MISSING!<br />Selecting \"Remove Undefined Instances\" will delete this.</p>");
        }

        if (null != analysis) {
            SortedSet<String> mods = analysis.SCRIPT_ORIGINS.get(this.NAME);

            if (null != mods) {
                if (mods.size() > 1) {
                    BUILDER.append("<p>WARNING: MORE THAN ONE MOD PROVIDES THIS SCRIPT!<br />Exercise caution when editing or deleting this script!</p>");
                }

                String probablyProvider = mods.last();
                BUILDER.append(String.format("<p>This script probably came from \"%s\".</p>", probablyProvider));
                BUILDER.append("<p>Full list of providers:</p>");
                BUILDER.append("<ul>");
                mods.forEach(mod -> BUILDER.append(String.format("<li>%s", mod)));
                BUILDER.append("</ul>");
            }
        }

        int inheritCount = 0;
        for (Script p = this.parent; p != null; p = p.parent) {
            inheritCount += p.MEMBERS.size();
        }
        BUILDER.append(String.format("<p>Contains %d member variables, %d were inherited.</p>", this.MEMBERS.size() + inheritCount, inheritCount));

        final List<ScriptInstance> INSTANCES = save.getPapyrus()
                .getInstances()
                .values()
                .stream()
                .filter(instance -> instance.getScript() == this)
                .collect(Collectors.toList());

        BUILDER.append(String.format("<p>There are %d instances of this script.</p>", INSTANCES.size()));
        if (INSTANCES.size() < 20) {
            BUILDER.append("<ul>");
            INSTANCES.forEach(i -> {
                String s = String.format("<li>%s</a>", i.toHTML());
                BUILDER.append(s);
            });
            BUILDER.append("</ul>");
        }

        if (null != analysis && analysis.SCRIPTS.containsKey(this.NAME)) {
            final java.io.File PEXFILE = analysis.SCRIPTS.get(this.NAME);
            final java.io.File PARENT = PEXFILE.getParentFile();

            BUILDER.append("");
            BUILDER.append(String.format("<hr /><p>Disassembled source code:<br />(from %s)</p>", PEXFILE.getPath()));

            if (PEXFILE.exists() && PEXFILE.canRead()) {
                try {
                    final restringer.pex.Pex SCRIPT = restringer.pex.Pex.readScript(PEXFILE);

                    java.io.StringWriter code = new java.io.StringWriter();
                    SCRIPT.disassemble(code, restringer.pex.AssemblyLevel.STRIPPED);

                    BUILDER.append("<p<code><pre>");
                    BUILDER.append(code.getBuffer());
                    BUILDER.append("</pre></code></p>");

                } catch (RuntimeException ex) {
                    BUILDER.append("<p><em>Error: disassembly failed.</em></p>");
                } catch (java.io.IOException ex) {
                    BUILDER.append("<p><em>Error: couldn't read the script file.</em></p>");
                } catch (Error ex) {
                    BUILDER.append("<p><em>Error: unexpected error while reading script file.</em></p>");
                }

            } else if (PARENT.exists() && PARENT.isFile()) {
                try (restringer.LittleEndianRAF input = new restringer.LittleEndianRAF(PARENT, "r")) {
                    restringer.bsa.BSAParser BSA = new restringer.bsa.BSAParser(PARENT.getName(), input);
                    final restringer.pex.Pex SCRIPT = BSA.getScript(PEXFILE.getName());

                    java.io.StringWriter code = new java.io.StringWriter();
                    SCRIPT.disassemble(code, restringer.pex.AssemblyLevel.STRIPPED);

                    BUILDER.append("<p<code><pre>");
                    BUILDER.append(code.getBuffer());
                    BUILDER.append("</pre></code></p>");

                } catch (RuntimeException ex) {
                    BUILDER.append("<p><em>Error: disassembly failed.</em></p>");
                } catch (java.io.IOException ex) {
                    BUILDER.append("<p><em>Error: couldn't read the script file.</em></p>");
                } catch (Error ex) {
                    BUILDER.append("<p><em>Error: unexpected error while reading script file.</em></p>");
                }
            }
        }
        
        BUILDER.append("</html>");
        return BUILDER.toString();
    }

    /**
     * @see AnalyzableElement#matches(restringer.Profile.Analysis,
     * restringer.Mod)
     * @param analysis
     * @param mod
     * @return
     */
    @Override
    public boolean matches(Profile.Analysis analysis, String mod) {
        Objects.requireNonNull(analysis);
        Objects.requireNonNull(mod);

        final SortedSet<String> OWNERS = analysis.SCRIPT_ORIGINS.get(this.NAME);
        return null != OWNERS && OWNERS.contains(mod);
    }

    /**
     * @return A flag indicating if the <code>Script</code> is undefined.
     *
     */
    public boolean isUndefined() {
        if (null == this.TYPE || this.TYPE.isEmpty()) {
            return !Script.IMPLICITSCRIPTS.contains(this.NAME);
        }

        return false;
    }

    /**
     * Increments the instance count.
     */
    void incrementInstanceCount() {
        this.instanceCount++;
    }

    /**
     * A list of scripts that only exist implicitly.
     */
    static final private IString[] IMPLICITSCRIPTS_ARR = new IString[]{
        IString.get("Alias"),
        IString.get("Form"),
        IString.get("ActiveMagicEffect"),
        IString.get("Utility"),
        IString.get("Debug"),
        IString.get("Game"),
        IString.get("Math"),
        IString.get("Input"),
        IString.get("SKSE"),
        IString.get("StringUtil"),
        IString.get("UI"),
        IString.get("ModEvent")
    };

    static final java.util.List<IString> IMPLICITSCRIPTS = java.util.Arrays.asList(IMPLICITSCRIPTS_ARR);

    final private TString NAME;
    final private TString TYPE;
    final private List<MemberData> MEMBERS;
    private Script parent;
    private int instanceCount;

}
