/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess.papyrus;

import restringer.ess.AnalyzableElement;
import java.io.IOException;
import java.util.SortedSet;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;
import restringer.Profile;
import restringer.ess.Element;
import restringer.ess.ESS;
import restringer.ess.Plugin;
import restringer.ess.RefID;

/**
 * Describes a script instance in a Skyrim savegame.
 *
 * @author Mark Fairchild
 * @version 2016/06/21
 */
final public class ScriptInstance extends GameElement {

    /**
     * Creates a new <code>ScriptInstances</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @param scripts The list of scripts.
     * @param ctx The PapyrusContext.
     * @throws IOException
     */
    public ScriptInstance(LittleEndianInput input, ScriptMap scripts, PapyrusContext ctx) throws IOException {
        super(input, scripts, ctx);
        this.UNKNOWN2BITS = input.readShort();
        this.UNKNOWN = input.readShort();
        this.REFID = new RefID(input);
        this.UNKNOWN_BYTE = input.readByte();
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        super.write(output);
        output.writeShort(this.UNKNOWN2BITS);
        output.writeShort(this.UNKNOWN);
        output.writeESSElement(this.REFID);
        output.writeByte(this.UNKNOWN_BYTE);
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        int sum = 5;
        sum += super.calculateSize();
        sum += this.REFID.calculateSize();
        return sum;
    }

    /**
     * @return The RefID of the papyrus element.
     */
    public RefID getRefID() {
        return this.REFID;
    }

    /**
     * @return The <code>ScriptData</code> for the instance.
     */
    public ScriptData getData() {
        return this.data;
    }

    /**
     * Sets the data field.
     *
     * @param newData The new value for the data field.
     */
    public void setData(ScriptData newData) {
        this.data = newData;
    }

    /**
     * @see restringer.ess.Linkable#toHTML() 
     * @return 
     */
    @Override
    public String toHTML() {
        return String.format("<a href=\"instance://%s\">%s</a>", this.getID(), this.toString());
    }

    /**
     * @return String representation.
     */
    @Override
    public String toString() {
        if (this.isUndefined() && this.UNKNOWN != -1) {
            return "#" + this.getScriptName() + "#: refid=" + this.REFID + " (" + this.getID() + ")";

        } else if (this.UNKNOWN != -1) {
            return this.getScriptName() + ": refid=" + this.REFID + " (" + this.getID() + ")";

        } else if (this.isUndefined()) {
            return "#" + this.getScriptName() + "#: **refid=" + this.REFID + " (" + this.getID() + ")";

        } else {
            return this.getScriptName() + ": **refid=" + this.REFID + " (" + this.getID() + ")";
        }
    }

    /**
     * @see AnalyzableElement#getInfo(restringer.Profile.Analysis,
     * restringer.ess.ESS)
     * @param analysis
     * @param save
     * @return
     */
    @Override
    public String getInfo(Profile.Analysis analysis, ESS save) {
        final StringBuilder BUILDER = new StringBuilder();
        if (null != this.getScript()) {
            BUILDER.append(String.format("<html><h3>INSTANCE of %s</h3>", this.getScript().toHTML()));
        } else {
            BUILDER.append(String.format("<html><h3>INSTANCE of %s</h3>", this.getScriptName()));
        }

        final Plugin PLUGIN = this.REFID.getPlugin();
        if (PLUGIN != null) {
            BUILDER.append(String.format("<p>This instance is attached to an object from %s.</p>", PLUGIN.toHTML()));
        } else if (this.REFID.getType() == RefID.Type.CREATED) {
            BUILDER.append("<p>This instance was created in-game.</p>");
        }

        if (this.isUndefined()) {
            BUILDER.append("<p><em>WARNING: SCRIPT MISSING!</em><br/>Remove Undefined Instances\" will delete this.</p>");
        }

        if (this.isUnattached()) {
            BUILDER.append("<p><em>WARNING: OBJECT MISSING!</em><br/>Selecting \"Remove Unattached Instances\" will delete this.</p>");
        }

        if (this.REFID.getType() == RefID.Type.CREATED && this.REFID.getForm() == null) {
            BUILDER.append("<p><em>WARNING: REFID POINTS TO NONEXISTENT CREATED FORM!</em><br/>Remove non-existent form instances\" will delete this.</p>");
        }

        if (null != analysis) {
            final SortedSet<String> PROVIDERS = analysis.SCRIPT_ORIGINS.get(this.getScriptName());
            if (null != PROVIDERS) {
                String probableProvider = PROVIDERS.last();
                BUILDER.append(String.format("<p>The script probably came from mod \"%s\".</p>", probableProvider));

                if (PROVIDERS.size() > 1) {
                    BUILDER.append("<p>Full list of providers:</p><ul>");
                    PROVIDERS.forEach(mod -> BUILDER.append(String.format("<li>%s", mod)));
                    BUILDER.append("</ul>");
                }
            }
        }

        BUILDER.append(String.format("<p>ID: %s</p>", this.getID()));
        BUILDER.append(String.format("<p>Type: %s</p>", this.data.getType()));

        boolean mysteryFlag = this.UNKNOWN == -1;

        if (save.getChangeForms().containsKey(this.REFID)) {
            BUILDER.append(String.format("<p>RefID%s: %s</p>", (mysteryFlag ? "@" : ""), this.REFID.toHTML()));
        } else {
            BUILDER.append(String.format("<p>RefID%s: %s</p>", (mysteryFlag ? "@" : ""), this.REFID.toString()));
        }

        BUILDER.append(String.format("<p>Unknown2bits: %01X</p>", this.UNKNOWN2BITS));
        BUILDER.append(String.format("<p>UnknownShort: %04X</p>", this.UNKNOWN));
        BUILDER.append(String.format("<p>UnknownByte: %02x</p>", this.UNKNOWN_BYTE));

        BUILDER.append("</html>");
        return BUILDER.toString();
    }

    /**
     * @see PapyrusElement#addNames(restringer.esp.ESPIDMap,
     * restringer.esp.StringTable)
     * @param names The map of IDs to names.
     * @param strings The stringtable.
     */
    @Override
    public void addNames(restringer.esp.ESPIDMap names, restringer.esp.StringTable strings) {
        assert null != this.data;
        this.data.addNames(names, strings);
        this.REFID.addName(names, strings);
    }

    /**
     * @see PapyrusElement#resolveRefs(ESS, Element)
     * @param ess The full savegame.
     * @param owner The owner of the element, or null if it is not owned.
     */
    @Override
    public void resolveRefs(ESS ess, Element owner) {
        super.resolveRefs(ess, owner);
        this.data.resolveRefs(ess, this);
        this.REFID.resolveRefs(ess, this);
    }

    /**
     * @return A flag indicating if the <code>ScriptInstance</code> is
     * unattached.
     *
     */
    public boolean isUnattached() {
        return this.REFID.equals(RefID.NONE);
    }

    final private short UNKNOWN2BITS;
    final private short UNKNOWN;
    final private RefID REFID;
    final private byte UNKNOWN_BYTE;
    private ScriptData data;

}
