/*
 * Copyright 2016 Mark.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess.papyrus;

import java.io.IOException;
import java.util.SortedSet;
import restringer.LittleEndianDataOutput;
import restringer.LittleEndianInput;
import restringer.Profile;
import restringer.ess.ESS;
import restringer.ess.Element;
import restringer.ess.Linkable;

/**
 *
 * @author Mark Fairchild
 * @version 2016/10/06
 */
public class Struct extends GameElement {

    /**
     * Creates a new <code>Struct</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @param scripts The list of scripts.
     * @param ctx The PapyrusContext.
     * @throws IOException
     */
    public Struct(LittleEndianInput input, ScriptMap scripts, PapyrusContext ctx) throws IOException {
        super(input, scripts, ctx);
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        super.write(output);
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        return super.calculateSize();
    }

    /**
     * @return The <code>ReferenceData</code> for the instance.
     */
    public ReferenceData getData() {
        return this.data;
    }

    /**
     * Sets the data field.
     *
     * @param newData The new value for the data field.
     */
    public void setData(ReferenceData newData) {
        this.data = newData;
    }

    /**
     * @see restringer.ess.Linkable#toHTML() 
     * @return 
     */
    @Override
    public String toHTML() {
        return String.format("<a href=\"struct://%08x\">%s</a>", this.getID(), this.toString());
    }
    
    /**
     * @return String representation.
     */
    @Override
    public String toString() {
        if (null != cachedToString) {
            return cachedToString;
        }

        if (this.isUndefined()) {
            cachedToString = "#" + this.getScript().getName() + "# (" + this.getID() + ")";
        } else {
            cachedToString = this.getScript().getName() + " (" + this.getID() + ")";
        }

        return cachedToString;
    }

    /**
     * @see AnalyzableElement#getInfo(restringer.Profile.Analysis,
     * restringer.ess.ESS)
     * @param analysis
     * @param save
     * @return
     */
    @Override
    public String getInfo(Profile.Analysis analysis, ESS save) {
        final StringBuilder BUILDER = new StringBuilder();
        if (null != this.getScript()) {
            BUILDER.append(String.format("<html><h3>STRUCT-INSTANCE of %s</h3>", this.getScript().toHTML()));
        } else {
            BUILDER.append(String.format("<html><h3>STRUCT-INSTANCE of %s</h3>", this.getScriptName()));
        }

        if (null != analysis) {
            SortedSet<String> providers = analysis.SCRIPT_ORIGINS.get(this.getScriptName());
            if (null != providers) {
                String probablyProvider = providers.last();
                BUILDER.append(String.format("<p>This struct probably came from \"%s\".</p>", probablyProvider));

                if (providers.size() > 1) {
                    BUILDER.append("<p>Full list of providers:</p><ul>");
                    providers.forEach(mod -> BUILDER.append(String.format("<li>%s", mod)));
                    BUILDER.append("</ul>");
                }
            }
        }

        BUILDER.append(String.format("<p>ID: %s</p>", this.getID()));
        BUILDER.append(String.format("<p>Type2: %s</p>", this.data.getType()));
        BUILDER.append(String.format("<p>Unknown1: %08x</p>", this.data.getUnknown1()));
        BUILDER.append(String.format("<p>Unknown2: %08x</p>", this.data.getUnknown2()));

        Element UNKNOWN1 = save.broadSpectrumMatch(this.getData().getUnknown1());
        if (null != UNKNOWN1) {
            BUILDER.append("<p>Potential match for unknown1 found using general search:<br/>");
            if (UNKNOWN1 instanceof Linkable) {
                BUILDER.append(((Linkable)UNKNOWN1).toHTML());
            } else {
                BUILDER.append(UNKNOWN1.toString());
            }
            BUILDER.append("</p>");
        }
        
        Element UNKNOWN2 = save.broadSpectrumMatch(this.getData().getUnknown2());
        if (null != UNKNOWN2) {
            BUILDER.append("<p>Potential match for unknown2 found using general search:<br/>");
            if (UNKNOWN2 instanceof Linkable) {
                BUILDER.append(((Linkable)UNKNOWN2).toHTML());
            } else {
                BUILDER.append(UNKNOWN2.toString());
            }
            BUILDER.append("</p>");
        }
        
        BUILDER.append("</html>");
        return BUILDER.toString();
    }

    /**
     * @see PapyrusElement#addNames(restringer.esp.ESPIDMap,
     * restringer.esp.StringTable)
     * @param names The map of IDs to names.
     * @param strings The stringtable.
     */
    @Override
    public void addNames(restringer.esp.ESPIDMap names, restringer.esp.StringTable strings) {
        assert null != this.data;
        this.data.addNames(names, strings);
    }

    /**
     * @see PapyrusElement#resolveRefs(ESS, Element)
     * @param ess The full savegame.
     * @param owner The owner of the element, or null if it is not owned.
     */
    @Override
    public void resolveRefs(ESS ess, Element owner) {
        super.resolveRefs(ess, owner);
        this.data.resolveRefs(ess, this);
    }

    private ReferenceData data;
    private String cachedToString = null;

}
