/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess.papyrus;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Map;
import java.util.Objects;
import restringer.LittleEndianDataOutput;
import restringer.Profile;
import restringer.ess.AnalyzableElement;
import restringer.ess.ChangeForm;
import restringer.ess.ESS;
import restringer.ess.Element;
import restringer.ess.Plugin;
import restringer.ess.RefID;
import restringer.ess.WString;

/**
 * A case-insensitive string with value semantics that reads and writes as an
 * index into a string table.
 *
 * @author Mark Fairchild
 * @version 2016/06/21
 */
abstract public class TString extends WString implements PapyrusElement, AnalyzableElement {

    /**
     * Creates a new <code>TString</code> from a <code>WString</code> and an
     * index.
     *
     * @param wstr The <code>WString</code>.
     * @param index The index of the <code>TString</code>.
     */
    protected TString(WString wstr, int index) {
        super(wstr);
        this.INDEX = index;
        this.HOLDERS = new ArrayList<>(1);
    }

    /**
     * Creates a new <code>TString</code> from a character sequence and an
     * index.
     *
     * @param cs The <code>CharSequence</code>.
     * @param index The index of the <code>TString</code>.
     */
    protected TString(CharSequence cs, int index) {
        super(cs);
        this.INDEX = index;
        this.HOLDERS = new ArrayList<>(1);
    }

    /**
     * @see WString#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    public void writeFull(LittleEndianDataOutput output) throws IOException {
        super.write(output);
    }

    /**
     * @see WString#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    public int calculateFullSize() {
        return super.calculateSize();
    }

    /**
     * @see PapyrusElement#addNames(restringer.esp.ESPIDMap,
     * restringer.esp.StringTable)
     * @param names The map of IDs to names.
     * @param strings The stringtable.
     */
    @Override
    public void addNames(restringer.esp.ESPIDMap names, restringer.esp.StringTable strings) {
    }

    /**
     * @see PapyrusElement#resolveRefs(ESS, Element)
     * @param ess The full savegame.
     * @param owner The owner of the element, or null if it is not owned.
     */
    @Override
    public void resolveRefs(ESS ess, Element owner) {
    }

    /**
     * @see IString#hashCode()
     * @return
     */
    @Override
    public int hashCode() {
        return super.hashCode();
    }

    /**
     * Tests for case-insensitive value-equality with another
     * <code>TString</code>, <code>IString</code>, or <code>String</code>.
     *
     * @param obj The object to which to compare.
     * @see java.lang.String#equalsIgnoreCase(java.lang.String)
     */
    @Override
    public boolean equals(Object obj) {
        if (obj instanceof TString) {
            return this.equals((TString) obj);
        } else {
            return super.equals(obj);
        }
    }

    /**
     * Tests for case-insensitive value-equality with another
     * <code>TString</code>.
     *
     * @param other The <code>TString</code> to which to compare.
     * @return True if the strings have the same index, false otherwise.
     * @see java.lang.String#equalsIgnoreCase(java.lang.String)
     */
    public boolean equals(TString other) {
        Objects.requireNonNull(other);
        return this.INDEX == other.INDEX;
    }

    /**
     * Adds an element as a reference holder.
     *
     * @param newHolder The new reference holder.
     */
    public void addRefHolder(Element newHolder) {
        Objects.requireNonNull(newHolder);
        this.HOLDERS.add(newHolder);
    }

    /**
     * Getter for the index field.
     *
     * @return
     */
    public int getINDEX() {
        assert this.INDEX >= 0;
        return INDEX;
    }

    /**
     * @see AnalyzableElement#getInfo(restringer.Profile.Analysis,
     * restringer.ess.ESS)
     * @param analysis
     * @param save
     * @return
     */
    @Override
    public String getInfo(Profile.Analysis analysis, ESS save) {
        Objects.requireNonNull(save);

        // Filter the string for duplicates.
        java.util.Set<Element> temp = new java.util.LinkedHashSet<>(this.HOLDERS.size());
        temp.addAll(this.HOLDERS);
        this.HOLDERS.clear();
        this.HOLDERS.addAll(temp);
        temp.clear();

        final StringBuilder BUILDER = new StringBuilder();

        BUILDER.append("<html><h3>STRING</h3>");
        BUILDER.append(String.format("<p>Value: \"%s\".</p>", this));
        BUILDER.append(String.format("<p>Length: %d</p>", this.length()));

        if (null != analysis) {
            final Map<String, Integer> OWNERS = analysis.STRING_ORIGINS.get(this);

            if (null != OWNERS) {
                int total = OWNERS.values().stream().mapToInt(k -> k).sum();

                BUILDER.append(String.format("<p>This string appears %d times in the script files of %d mods.</p><ul>", total, OWNERS.size()));
                OWNERS.forEach((mod, count) -> BUILDER.append(String.format("<li>String appears %d times in the scripts of mod \"%s\".", count, mod)));
                BUILDER.append("</ul>");

            } else {
                BUILDER.append("<p>String origin could not be determined.</p>");
            }
        }

        if (this.HOLDERS.isEmpty()) {
            BUILDER.append("<p>The string could not be found in the savefile. This may mean that something has gone VERY WRONG.</p>");

        } else {
            BUILDER.append(String.format("<p>This string occurs %d times in this save.</p>", this.HOLDERS.size()));

            final Map<Plugin, Integer> PLUGINS = new java.util.HashMap<>();

            this.HOLDERS.forEach(holder -> {
                if (holder instanceof ScriptInstance) {
                    ScriptInstance instance = (ScriptInstance) holder;
                    RefID ref = instance.getRefID();
                    if (null != ref && null != ref.getPlugin()) {
                        PLUGINS.merge(ref.getPlugin(), 1, Integer::sum);
                    }
                } else if (holder instanceof ChangeForm) {
                    ChangeForm form = (ChangeForm) holder;
                    RefID ref = form.getRefID();
                    if (null != ref && null != ref.getPlugin()) {
                        PLUGINS.merge(ref.getPlugin(), 1, Integer::sum);
                    }
                }
            });

            BUILDER.append('\n');

            if (PLUGINS.isEmpty()) {
                BUILDER.append(String.format("<p>This string could not be backtraced to any plugins.</p>", PLUGINS.size()));
            } else {
                BUILDER.append(String.format("<p>This string was backtraced to %d plugins.</p><ul>", PLUGINS.size()));
                PLUGINS.forEach((p, c) -> BUILDER.append(String.format("<li>String was backtraced %d times to plugin %s.", c, p.toHTML())));
                BUILDER.append("</ul>");
            }
        }

        BUILDER.append("</html>");
        return BUILDER.toString();
    }

    @Override
    public String toString() {
        return super.toString() + "(" + this.INDEX + ")";
    }

    /**
     * @see AnalyzableElement#matches(restringer.Profile.Analysis,
     * restringer.Mod)
     * @param analysis
     * @param mod
     * @return
     */
    @Override
    public boolean matches(Profile.Analysis analysis, String mod) {
        Objects.requireNonNull(analysis);
        Objects.requireNonNull(mod);
        final Map<String, Integer> OWNERS = analysis.STRING_ORIGINS.get(this);
        return null != OWNERS
                && OWNERS.keySet().size() == 1
                && OWNERS.keySet().contains(mod);
    }

    final private int INDEX;
    final private Collection<Element> HOLDERS;

}
