/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess.papyrus;

import java.io.IOException;
import java.util.Objects;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;
import restringer.ess.ESS;
import restringer.ess.Element;

/**
 * Describes the eleven types of variable data.
 *
 * @author Mark Fairchild
 * @version 2016/06/19
 */
public enum Type implements PapyrusElement {
    NULL(0),
    REF(1),
    STRING(2),
    INTEGER(3),
    FLOAT(4),
    BOOLEAN(5),
    REF_ARRAY(11),
    STRING_ARRAY(12),
    INTEGER_ARRAY(13),
    FLOAT_ARRAY(14),
    BOOLEAN_ARRAY(15);

    /**
     * Read a <code>Type</code> from an input stream.
     *
     * @param input The input stream.
     * @return The <code>Type</code>.
     */
    static Type read(LittleEndianInput input) throws IOException {
        Objects.requireNonNull(input);

        int val = input.readUnsignedByte();
        Type t = match(val);
        if (null != t) {
            return t;
        }

        throw new IOException("Invalid type value: " + val);
    }

    /**
     * Returns the <code>Type</code> that matches a value.
     *
     * @param val The <code>Type</code> value.
     * @return The <code>Type</code>, or null if there was no match.
     */
    static Type match(int val) throws IOException {
        for (Type t : VALUES) {
            if (t.CODE == val) {
                return t;
            }
        }
        return null;
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        Objects.requireNonNull(output);
        output.writeByte(this.CODE);
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        return 1;
    }

    /**
     * @return True if the <code>Type</code> is an array type, false otherwise.
     */
    public boolean isArray() {
        return (this.CODE >= 10);
    }

    /**
     * Create a new <code>Type</code>.
     *
     * @param val
     */
    private Type(int code) {
        this.CODE = code;
    }

    /**
     * @see PapyrusElement#addNames(restringer.esp.ESPIDMap, restringer.esp.StringTable)
     * @param names The map of IDs to names.
     * @param strings The stringtable.
     */
    @Override
    public void addNames(restringer.esp.ESPIDMap names, restringer.esp.StringTable strings) {
    }

    /**
     * @see PapyrusElement#resolveRefs(ESS, Element)
     * @param ess The full savegame.
     * @param owner The owner of the element, or null if it is not owned.
     */
    @Override
    public void resolveRefs(ESS ess, Element owner) {
    }

    final public int CODE;
    static final private Type[] VALUES = values();
}
