/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess.papyrus;

import java.io.IOException;
import java.util.Objects;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;
import restringer.ess.RefID;
import restringer.ess.ESS;
import restringer.ess.Element;
import restringer.ess.Linkable;

/**
 * Describes the Unknown4 field of an <code>ActiveScriptData</code>.
 *
 * @author Mark Fairchild
 * @version 2016/06/29
 */
final public class Unknown4 implements PapyrusElement, Linkable {

    /**
     * Creates a new <code>Unknown4</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @param unknown3 The value of the unknown3 field.
     * @param ctx The PapyrusContext.
     * @throws IOException
     */
    public Unknown4(LittleEndianInput input, byte unknown3, PapyrusContext ctx) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(ctx);
        assert 1 <= unknown3 && unknown3 <= 3;

        if (unknown3 == 2) {
            this.VARIABLE = Variable.read(input, ctx);
            this.STRING = null;
            this.REFID = null;
            this.TSTRING = null;
            this.UNKNOWN_SHORT = 0;
            this.BYTE = 0;
            this.INT = 0;            
            return;
        }

        this.STRING = input.readLString();

        switch (this.STRING) {
            case "QuestStage":
                this.REFID = new RefID(input);
                //long b = input.readShort();
                //this.TSTRING = ctx.STRINGS.read(input);
                this.TSTRING = null;
                this.UNKNOWN_SHORT = input.readShort();
                this.BYTE = input.readByte();
                this.INT = 0;
                break;
            case "ScenePhaseResults":
                this.REFID = new RefID(input);
                this.TSTRING = null;
                this.UNKNOWN_SHORT = 0;
                this.BYTE = 0;
                this.INT = input.readInt();
                break;
            case "SceneActionResults":
                this.REFID = new RefID(input);
                this.TSTRING = null;
                this.UNKNOWN_SHORT = 0;
                this.BYTE = 0;
                this.INT = input.readInt();
                break;
            case "SceneResults":
                this.REFID = new RefID(input);
                this.TSTRING = null;
                this.UNKNOWN_SHORT = 0;
                this.BYTE = 0;
                this.INT = 0;
                break;
            default:
                this.REFID = null;
                this.TSTRING = null;
                this.UNKNOWN_SHORT = 0;
                this.BYTE = 0;
                this.INT = 0;
                break;
        }

        if (unknown3 == 3) {
            this.VARIABLE = Variable.read(input, ctx);
        } else {
            this.VARIABLE = null;
        }
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        assert null != output;
        assert null != this.STRING || null != this.VARIABLE;

        // Corresponds to the unknown3 == 2 case.
        if (null == this.STRING) {
            this.VARIABLE.write(output);
            return;
        }

        output.writeLString(this.STRING);

        switch (this.STRING) {
            case "QuestStage":
                this.REFID.write(output);
                //this.TSTRING.write(output);
                output.writeShort(this.UNKNOWN_SHORT);
                output.writeByte(this.BYTE);
                break;
            case "ScenePhaseResults":
                this.REFID.write(output);
                output.writeInt(this.INT);
                break;
            case "SceneActionResults":
                this.REFID.write(output);
                output.writeInt(this.INT);
                break;
            case "SceneResults":
                this.REFID.write(output);
                break;
            default:
                break;
        }

        // Corresponds to the unknown3 == 3 case.
        if (null != this.VARIABLE) {
            this.VARIABLE.write(output);
        }
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        assert null != this.STRING || null != this.VARIABLE;

        if (null == this.STRING) {
            return this.VARIABLE.calculateSize();
        }

        int sum = 4 + this.STRING.length();

        switch (this.STRING) {
            case "QuestStage":
                sum += this.REFID.calculateSize();
                //sum += this.TSTRING.calculateSize();
                sum += 2;
                sum += 1;
                break;
            case "ScenePhaseResults":
                sum += this.REFID.calculateSize();
                sum += 4;
                break;
            case "SceneActionResults":
                sum += this.REFID.calculateSize();
                sum += 4;
                break;
            case "SceneResults":
                sum += this.REFID.calculateSize();
                break;
            default:
                break;
        }

        if (null != this.VARIABLE) {
            sum += this.VARIABLE.calculateSize();
        }

        return sum;
    }

    /**
     * @see PapyrusElement#addNames(restringer.esp.ESPIDMap,
     * restringer.esp.StringTable)
     * @param names The map of IDs to names.
     * @param strings The stringtable.
     */
    @Override
    public void addNames(restringer.esp.ESPIDMap names, restringer.esp.StringTable strings) {
        if (null != this.VARIABLE) {
            this.VARIABLE.addNames(names, strings);
        }

        if (null != this.REFID) {
            this.REFID.addName(names, strings);
        }
    }

    /**
     * @see PapyrusElement#resolveRefs(ESS, Element)
     * @param ess The full savegame.
     * @param owner The owner of the element, or null if it is not owned.
     */
    @Override
    public void resolveRefs(ESS ess, Element owner) {
        if (null != this.VARIABLE) {
            this.VARIABLE.resolveRefs(ess, owner);
        }

        if (null != this.REFID) {
            this.REFID.resolveRefs(ess, owner);
        }

        if (null != this.TSTRING) {
            this.TSTRING.addRefHolder(owner);
        }
    }

    /**
     * @see restringer.ess.Linkable#toHTML() 
     * @return 
     */
    @Override
    public String toHTML() {
        assert null != this.STRING || null != this.VARIABLE;

        final StringBuilder BUILDER = new StringBuilder();
        BUILDER.append("UNK4");

        if (null == this.STRING) {
            BUILDER.append(": ").append(this.VARIABLE);
        }

        BUILDER.append("(").append(this.STRING).append("): ");

        switch (this.STRING) {
            case "QuestStage":
                BUILDER.append(this.REFID.toHTML()).append(" ");
                BUILDER.append(this.TSTRING).append(" ");
                break;
            case "ScenePhaseResults":
                BUILDER.append(this.REFID.toHTML());
                break;
            case "SceneActionResults":
                BUILDER.append(this.REFID.toHTML());
                break;
            case "SceneResults":
                BUILDER.append(this.REFID.toHTML());
                break;
            default:
                break;
        }

        if (null != this.VARIABLE) {
            BUILDER.append("(").append(this.VARIABLE.toHTML()).append(")");
        }

        return BUILDER.toString();
    }

    /**
     * @return String representation.
     */
    @Override
    public String toString() {
        assert null != this.STRING || null != this.VARIABLE;

        final StringBuilder BUILDER = new StringBuilder();
        BUILDER.append("UNK4");

        if (null == this.STRING) {
            BUILDER.append(": ").append(this.VARIABLE);
        }

        BUILDER.append("(").append(this.STRING).append("): ");

        switch (this.STRING) {
            case "QuestStage":
                BUILDER.append(this.REFID).append(" ");
                BUILDER.append(this.TSTRING).append(" ");
                break;
            case "ScenePhaseResults":
                BUILDER.append(this.REFID);
                break;
            case "SceneActionResults":
                BUILDER.append(this.REFID);
                break;
            case "SceneResults":
                BUILDER.append(this.REFID);
                break;
            default:
                break;
        }

        if (null != this.VARIABLE) {
            BUILDER.append("(").append(this.VARIABLE).append(")");
        }

        return BUILDER.toString();
    }

    final private String STRING;
    final private Variable VARIABLE;

    final private RefID REFID;
    final private byte BYTE;
    final private int INT;
    final private TString TSTRING;
    final private short UNKNOWN_SHORT;
}
