/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.gui;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.LinkedBlockingQueue;
import javax.swing.SwingWorker;
import javax.swing.table.AbstractTableModel;
import restringer.Mod;
import restringer.Profile;
import restringer.Settings;

/**
 * Describes the model for a table of mods.
 *
 * @author Mark Fairchild
 * @version 2016/06/16
 */
final class ModTableModel extends AbstractTableModel {

    /**
     * Creates a new <code>ModTableModel</code>. It will model a specified
     * <code>Profile</code>.
     *
     * @param settings The <code>Settings</code> storing the profiles.
     * @param profile The <code>Profile</code> to model.
     */
    public ModTableModel(Settings settings, Profile profile) {
        Objects.requireNonNull(settings);
        Objects.requireNonNull(profile);

        this.SETTINGS = settings;
        this.ANALYSES = new ConcurrentHashMap<>();
        this.QUEUE = new LinkedBlockingQueue<>();
        this.worker = new ModAnalyzer();
        this.setProfile(profile);
        this.worker.execute();
    }

    /**
     * Pauses the analysis thread.
     */
    public void pauseAnalysis() {
        synchronized (this) {
            this.worker.cancel(true);
            this.worker = null;
        }
    }

    /**
     * Resumes the analysis thread.
     */
    public void resumeAnalysis() {
        synchronized (this) {
            if (null == this.worker || this.worker.isCancelled()) {
                this.worker = new ModAnalyzer();
                this.worker.execute();
            }
        }
    }

    /**
     * Sets the <code>Profile</code> to model.
     *
     * @param profile The new <code>Profile</code> to model.
     */
    public void setProfile(Profile newProfile) {
        Objects.requireNonNull(newProfile);
        this.profile = newProfile;
        this.ANALYSES.clear();
        this.QUEUE.addAll(this.profile.getMods());
        this.fireTableDataChanged();
    }

    /**
     * Allows a <code>Mod</code> to the <code>ModTableModel</code> at a
     * specified index.
     *
     * @see ModTableModel#addMod(restringer.Mod, boolean)
     * @param index The index at which to add the <code>Mod</code>.
     * @param mod The <code>Mod</code> to add to the model.
     * @param checked A flag indicating if the mod should initially be
     * checkmarked or not.
     */
    public void addMod(int index, Mod mod, boolean checked) {
        Objects.requireNonNull(mod);

        synchronized (this) {
            int actualIndex = this.profile.addMod(index, mod);
            mod.setStatus(checked ? Mod.Status.CHECKED : Mod.Status.UNCHECKED);
            this.QUEUE.add(mod);
            super.fireTableRowsInserted(actualIndex, actualIndex);
        }

        this.saveSettings();
    }

    /**
     * Adds a <code>Mod</code> to the <code>ModTableModel</code>. If the
     * <code>Mod</code> is already contained in the model, this method will have
     * no effect.
     *
     * @param mod The <code>Mod</code> to add to the model.
     * @param checked A flag indicating if the mod should initially be
     * checkmarked or not.
     *
     */
    public void addMod(Mod mod, boolean checked) {
        this.addMod(this.profile.size(), mod, checked);
    }

    /**
     * Moves a <code>Mod</code> within the table.
     *
     * @param mod The <code>Mod</code> to move.
     * @param index The new index for the <code>Mod</code>.
     */
    public void moveMod(Mod mod, int index) {
        if (!this.profile.getMods().contains(mod)) {
            return;
        }

        synchronized (this) {
            int oldIndex = this.profile.getMods().indexOf(mod);
            if (index > oldIndex) {
                index--;
            }

            assert 0 <= oldIndex && oldIndex < this.profile.size();
            assert 0 <= index && index < this.profile.size();

            this.profile.addMod(index, mod);
            this.fireTableRowsDeleted(oldIndex, oldIndex);
            this.fireTableRowsInserted(index, index);
        }

        this.saveSettings();
    }

    /**
     * Adds a <code>Mod</code> <code>Collection</code> to the
     * <code>ModTableModel</code>. If the <code>Mod</code> is already contained
     * in the model, this method will have no effect.
     *
     * @param mods The <code>Collection</code> of <code>Mod</code> objects to
     * add to the model.
     *
     */
    public void addAllMods(Collection<Mod> mods) {
        Objects.requireNonNull(mods);

        if (mods.contains(null)) {
            throw new NullPointerException("Mods must not be null.");
        }

        mods.forEach(mod -> {
            synchronized (this) {
                int index = this.profile.addMod(mod);
                mod.setStatus(Mod.Status.UNCHECKED);
                this.QUEUE.add(mod);
                this.fireTableRowsInserted(index, index);
            }
        });

        this.saveSettings();
    }

    /**
     * Removes a <code>Mod</code> <code>Collection</code> from the
     * <code>ModTableModel</code>.
     *
     * @param mods The <code>Collection</code> of <code>Mod</code> objects to be
     * removed from the model.
     *
     */
    public void removeMods(Collection<Mod> mods) {
        Objects.requireNonNull(mods);

        mods.forEach(mod -> {
            synchronized (this) {
                if (this.profile.contains(mod)) {
                    int index = this.profile.removeMod(mod);
                    this.ANALYSES.remove(mod);
                    this.QUEUE.remove(mod);
                    this.fireTableRowsDeleted(index, index);
                }
            }
        });

        this.saveSettings();
    }

    /**
     * Returns the mod at the specified index.
     *
     * @param index The index of the mod.
     * @return The mod at the specified index.
     */
    public Mod getMod(int index) {
        if (index < 0 || index >= this.profile.size()) {
            throw new IllegalArgumentException();
        }

        return this.profile.getMods().get(index);
    }

    /**
     * Removes all <code>Mod</code> objects from the <code>ModTableModel</code>.
     */
    public void removeAll() {
        synchronized (this) {
            this.profile.clear();
            this.ANALYSES.clear();
            this.QUEUE.clear();
            this.fireTableDataChanged();
        }
        this.saveSettings();
    }

    /**
     * Selects all of the mods.
     */
    final void toggleRows(int[] rows) {
        Mod.Status newStatus = Mod.Status.CHECKED;

        synchronized (this) {
            for (int row : rows) {
                if (0 <= row && row < this.getRowCount()) {
                    Mod mod = this.profile.getMods().get(row);
                    Mod.Status status = mod.getStatus();

                    if (status == Mod.Status.CHECKED) {
                        newStatus = Mod.Status.UNCHECKED;
                    }
                }
            }

            for (int row : rows) {
                if (0 <= row && row < this.getRowCount()) {
                    Mod mod = this.profile.getMods().get(row);
                    Mod.Status status = mod.getStatus();
                    if (status == Mod.Status.DISABLED) {
                        continue;
                    }

                    mod.setStatus(newStatus);
                    this.fireTableCellUpdated(row, 0);
                }
            }
        }

        this.saveSettings();
    }

    /**
     * Selects all of the mods.
     */
    final void checkAll() {
        synchronized (this) {
            this.profile.getMods().forEach(mod -> {
                if (mod.getStatus() != Mod.Status.DISABLED) {
                    mod.setStatus(Mod.Status.CHECKED);
                }
            });
            super.fireTableDataChanged();
        }

        this.saveSettings();
    }

    /**
     * Deselects all of the mods.
     */
    final void checkNone() {
        synchronized (this) {
            this.profile.getMods().forEach(mod -> {
                if (mod.getStatus() != Mod.Status.DISABLED) {
                    mod.setStatus(Mod.Status.UNCHECKED);
                }
            });
            super.fireTableDataChanged();
        }

        this.saveSettings();
    }

    /**
     * @return The number of mods that are checkmarked.
     */
    final int getNumChecked() {
        return this.profile.getCheckmarkedMods().size();
    }

    /**
     * Writes the settings and profiles to a file.
     */
    public void saveSettings() {
        try {
            Settings.writeSettings(this.SETTINGS);
        } catch (java.io.IOException ex) {
        }
    }

    /**
     * @see AbstractTableModel#getRowCount()
     * @return
     */
    @Override
    public int getRowCount() {
        return this.profile.size();
    }

    /**
     * @see AbstractTableModel#getColumnCount()
     * @return
     */
    @Override
    public int getColumnCount() {
        return ModTableModel.COLUMN_NAMES.length;
    }

    /**
     * @see AbstractTableModel#getRowCount()
     * @return
     */
    @Override
    public Object getValueAt(int rowIndex, int columnIndex
    ) {
        final Mod MOD = this.profile.getMods().get(rowIndex);
        final Mod.Analysis ANALYSIS = this.ANALYSES.get(MOD);

        switch (columnIndex) {
            case 0:
                switch (MOD.getStatus()) {
                    case CHECKED:
                        return true;
                    case UNCHECKED:
                        return false;
                    default:
                        return null;
                }
            case 1:
                return MOD.getName();
            case 2:
                return rowIndex;
            case 3:
                return MOD.getNumESPs();
            case 4:
                return MOD.getNumBSAs();
            case 5:
                return MOD.getNumLooseScripts();
            case 6:
                if (null == ANALYSIS) {
                    return null;
                } else if (ANALYSIS == Mod.Analysis.INPROGRESS) {
                    return null;
                } else {
                    return ANALYSIS.NUMSCRIPTS;
                }
            case 7:
                if (null == ANALYSIS) {
                    return null;
                } else if (ANALYSIS == Mod.Analysis.INPROGRESS) {
                    return null;
                } else {
                    return ANALYSIS.NUMSTRINGS;
                }
            default:
                return null;
        }
    }

    @Override
    public String getColumnName(int columnIndex
    ) {
        return ModTableModel.COLUMN_NAMES[columnIndex];
    }

    @Override
    public Class<?> getColumnClass(int columnIndex
    ) {
        return ModTableModel.COLUMN_CLASSES[columnIndex];
    }

    @Override
    public boolean isCellEditable(int rowIndex, int columnIndex
    ) {
        return (columnIndex == 0);
    }

    @Override
    public void setValueAt(Object aValue, int rowIndex, int columnIndex) {
        assert columnIndex == 0;

        synchronized (this) {
            final Mod MOD = this.profile.getMods().get(rowIndex);

            switch (MOD.getStatus()) {
                case CHECKED:
                    MOD.setStatus(Mod.Status.UNCHECKED);
                    this.fireTableCellUpdated(rowIndex, columnIndex);
                    this.saveSettings();
                    break;
                case UNCHECKED:
                    MOD.setStatus(Mod.Status.CHECKED);
                    this.fireTableCellUpdated(rowIndex, columnIndex);
                    this.saveSettings();
                    break;
                default:
            }
        }
    }

    /**
     * The <code>Profile</code> to model.
     */
    private Profile profile;

    /**
     * The settings.
     */
    final private Settings SETTINGS;

    /**
     * Stores the string estimates for Mods.
     */
    final private Map<Mod, Mod.Analysis> ANALYSES;

    /**
     * Stores the list of mods to be analysed.
     */
    final private LinkedBlockingQueue<Mod> QUEUE;

    /**
     * Column names.
     */
    static final private ResourceBundle RES = ResourceBundle.getBundle("restringer/gui/General");
    static final private String[] COLUMN_NAMES = new String[]{
        "", 
        RES.getString("CHOOSER MODNAME"), 
        RES.getString("CHOOSER #"), 
        RES.getString("CHOOSER ESPS"), 
        RES.getString("CHOOSER BSAS"), 
        RES.getString("CHOOSER PEXS"), 
        RES.getString("CHOOSER SCRIPTS"), 
        RES.getString("CHOOSER STRINGS"),};

    /**
     * Column classes.
     */
    static final private Class[] COLUMN_CLASSES = new Class[]{
        Boolean.class, String.class, Integer.class, Integer.class, Integer.class, Integer.class, Integer.class, Integer.class,};

    /**
     * Thread that handles the computation-intensive job of analysing mods.
     */
    private SwingWorker worker;

    /**
     * A swingworker class that handles analyzing mods.
     */
    private class ModAnalyzer extends SwingWorker {

        @Override
        protected Object doInBackground() throws Exception {
            Mod mod = null;

            try {
                for (;;) {
                    Thread.yield();
                    mod = ModTableModel.this.QUEUE.take();
                    Thread.yield();

                    synchronized (ModTableModel.this) {
                        int index = profile.indexOf(mod);
                        if (index < 0) {
                            continue;
                        }

                        ModTableModel.this.ANALYSES.put(mod, Mod.Analysis.INPROGRESS);
                        try {
                            ModTableModel.this.fireTableCellUpdated(index, 6);
                            ModTableModel.this.fireTableCellUpdated(index, 7);
                        } catch (Exception | Error e) {
                            e.printStackTrace(System.out);
                        }
                    }

                    Mod.Analysis analysis = mod.analysis();
                    Thread.yield();

                    synchronized (ModTableModel.this) {
                        int index = profile.indexOf(mod);
                        if (index < 0) {
                            continue;
                        }

                        ModTableModel.this.ANALYSES.put(mod, analysis);
                        try {
                            ModTableModel.this.fireTableCellUpdated(index, 6);
                            ModTableModel.this.fireTableCellUpdated(index, 7);
                        } catch (Exception | Error e) {
                            e.printStackTrace(System.out);
                        }
                    }
                }
            } catch (InterruptedException ex) {
                ModTableModel.this.QUEUE.offer(mod);
                return this;
            }
        }
    }
}
