/*
 * Copyright 2016 Mark.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.gui;

import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Objects;
import javax.swing.JComponent;
import javax.swing.JTable;
import javax.swing.TransferHandler;
import restringer.Mod;

/**
 * Handles drag and drop for the Mod table.
 *
 * @author Mark Fairchild
 * @version 2016/06/09
 */
class ModTableTransferHandler extends TransferHandler {

    ModTableTransferHandler(ModTableModel model) {
        Objects.requireNonNull(model);
        this.MODEL = model;
    }

    @Override
    public boolean canImport(TransferSupport support) {
        if (!(support.isDataFlavorSupported(INTARRAYFLAVOUR)
                || support.isDataFlavorSupported(DataFlavor.javaFileListFlavor))) {
            return false;
        }

        JTable.DropLocation location = (JTable.DropLocation) support.getDropLocation();
        return location.isInsertRow() && 0 <= location.getRow();
    }

    @Override
    protected Transferable createTransferable(JComponent c) {
        final JTable TABLE = (JTable) c;
        final int[] ROWS = TABLE.getSelectedRows();

        return new Transferable() {
            @Override
            public DataFlavor[] getTransferDataFlavors() {
                return new DataFlavor[]{INTARRAYFLAVOUR};
            }

            @Override
            public boolean isDataFlavorSupported(DataFlavor flavor) {
                return flavor.match(INTARRAYFLAVOUR);
            }

            @Override
            public Object getTransferData(DataFlavor flavor) throws UnsupportedFlavorException, IOException {
                return ROWS;
            }
        };
    }

    @Override
    public int getSourceActions(JComponent c) {
        return TransferHandler.MOVE;
    }

    @Override
    public boolean importData(TransferSupport support) {
        assert this.canImport(support);
        final Transferable TRANSFERABLE = support.getTransferable();

        try {
            if (TRANSFERABLE.isDataFlavorSupported(INTARRAYFLAVOUR)) {
                final int[] ROWS = (int[]) TRANSFERABLE.getTransferData(INTARRAYFLAVOUR);
                if (ROWS.length == 0) {
                    return true;
                }

                final JTable.DropLocation LOCATION = (JTable.DropLocation) support.getDropLocation();
                final int DROPROW = LOCATION.getRow();
                final int OFFSET = ROWS[0];

                for (int i = 0; i < ROWS.length; i++) {
                    int row = ROWS[i];
                    int dest = row + DROPROW - OFFSET;
                    Mod mod = this.MODEL.getMod(row);
                    this.MODEL.moveMod(mod, dest);
                }

                return true;
            } else if (TRANSFERABLE.isDataFlavorSupported(DataFlavor.javaFileListFlavor)) {
                final List<File> FILES = (List<File>) TRANSFERABLE.getTransferData(DataFlavor.javaFileListFlavor);
                
                FILES.forEach(file -> {
                    try {
                        Mod mod = new Mod(file);
                        this.MODEL.addMod(mod, false);
                    } catch (IllegalArgumentException ex) {                        
                    }
                });
                
                return true;
            } else {
                return false;
            }
        } catch (UnsupportedFlavorException | IOException ex) {
            return false;
        }
    }

    final private ModTableModel MODEL;
    static final private DataFlavor INTARRAYFLAVOUR = new DataFlavor(int[].class, java.util.ResourceBundle.getBundle("restringer/gui/General").getString("INTARRAY"));

}
