/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.gui;

import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.io.IOException;
import java.util.Objects;
import java.util.logging.Logger;
import javax.swing.JOptionPane;
import javax.swing.SwingWorker;
import restringer.ess.ESS;

/**
 *
 * @author Mark Fairchild
 */
public class Opener extends SwingWorker<ESS, Double> {

    /**
     *
     * @param window
     * @param saveFile
     */
    public Opener(SaveWindow window, File saveFile) {
        this.WINDOW = Objects.requireNonNull(window, "The window field must not be null.");
        this.SAVEFILE = Objects.requireNonNull(saveFile, "The saveFile field must not be null.");
    }

    /**
     * 
     * @return
     * @throws Exception 
     */
    @Override
    protected ESS doInBackground() throws Exception {
        if (!Configurator.validateSavegame(this.SAVEFILE)) {
            return null;
        }

        final ProgressModel MODEL = new ProgressModel();

        ProgressIndicator.startWaitCursor(this.WINDOW.getRootPane());        
        this.WINDOW.addWindowListener(this.LISTENER);
        this.WINDOW.startProgressBar(MODEL);
        this.WINDOW.clearESS();

        try {
            LOG.info("================");
            LOG.info(String.format("Reading from savegame file \"%s\".", this.SAVEFILE));

            ESS newSave = ESS.readESS(this.SAVEFILE, MODEL);
            this.WINDOW.setESS(this.SAVEFILE, newSave);

            if (newSave.getPapyrus().getStringTable().isSTBCorrection()) {
                JOptionPane.showMessageDialog(this.WINDOW, "THIS FILE HAS THE STRING-TABLE-BUG.\nIt is corrupted and can never be recovered, not even with lasers or cheetah blood.\nSaving is disabled.", "Save Read", JOptionPane.ERROR_MESSAGE);
            } else {
                JOptionPane.showMessageDialog(this.WINDOW, "The savefile was successfully read.", "Save Read", JOptionPane.INFORMATION_MESSAGE);
            }
            return newSave;

        } catch (IOException ex) {
            final String MSG = String.format("Error while reading file \"%s\".\n%s", this.SAVEFILE.getName(), ex.getMessage());
            LOG.severe(MSG);
            LOG.severe(ex.toString());
            System.err.println(ex.getMessage());
            ex.printStackTrace(System.err);
            JOptionPane.showMessageDialog(WINDOW, MSG, "Read Error", JOptionPane.ERROR_MESSAGE);
            return null;

        } catch (Exception | Error ex) {
            final String MSG = String.format("Error while reading file \"%s\".\n%s", this.SAVEFILE.getName(), ex.getMessage());
            LOG.severe(MSG);
            LOG.severe(ex.toString());
            System.err.println(ex.getMessage());
            ex.printStackTrace(System.err);
            JOptionPane.showMessageDialog(this.WINDOW, MSG, "Read Error", JOptionPane.ERROR_MESSAGE);
            return null;

        } finally {
            this.WINDOW.removeWindowListener(this.LISTENER);
            ProgressIndicator.stopWaitCursor(this.WINDOW.getRootPane());
            this.WINDOW.clearProgressBar();
        }
    }

    final private File SAVEFILE;
    final private SaveWindow WINDOW;
    static final private Logger LOG = Logger.getLogger(Opener.class.getCanonicalName());

    final private WindowAdapter LISTENER = new WindowAdapter() {
        @Override
        public void windowClosing(WindowEvent e) {
            if (!isDone()) {
                cancel(true);
            }
        }
    };
}
