/*
 * Copyright 2016 Mark.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.gui;

import java.util.Collection;
import java.util.function.Predicate;

/**
 * A simple extension of the <code>CopyOnWriteArrayList</code> to add a few
 * useful performance features.
 *
 * @author Mark Fairchild
 * @version 2016/07/03
 */
final public class ThreadsafeArray<E> extends java.util.concurrent.CopyOnWriteArrayList<E> {

    /**
     * Creates a new <code>ThreadSafeArray</code>.
     */
    public ThreadsafeArray() {
    }

    /**
     * Filters a <code>Collection</code> using a <code>Predicate</code> and
     * appends every member of the <code>Collection</code> that tests <code>true</code>.
     *
     * @param c The <code>Collection</code>.
     * @param p The <code>Predicate</code>.
     * @return The number of elements that were added.
     */
    public int addAllIf(Collection<? extends E> c, Predicate<E> p) {
        Predicate<E> notP = element -> !p.test(element);
        return addAllUnless(c, notP);
    }

    /**
     * Filters a <code>Collection</code> using a <code>Predicate</code> and
     * appends every member of the <code>Collection</code> that tests <code>false</code>.
     *
     * @param c The <code>Collection</code>.
     * @param p The <code>Predicate</code>.
     * @return The number of elements that were added.
     */
    public int addAllUnless(Collection<? extends E> c, Predicate<E> p) {
        java.util.List<E> temp = new java.util.LinkedList<>(c);
        temp.removeIf(p);
        super.addAll(temp);
        return temp.size();
    }

}
