/*
 * Copyright 2016 Mark.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.gui;

import java.awt.Component;
import java.text.NumberFormat;
import javax.swing.AbstractCellEditor;
import javax.swing.JComboBox;
import javax.swing.JFormattedTextField;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.table.TableCellEditor;
import restringer.ess.papyrus.Variable;

/**
 * A <code>TableCellEditor</code> implementation for table cells that contain
 * <code>Variable</code> objects.
 *
 * @author Mark Fairchild
 * @version 2016/06/15
 */
abstract public class VariableCellEditor extends AbstractCellEditor implements TableCellEditor {

    static final public class Generic extends VariableCellEditor {

        @Override
        public Object getCellEditorValue() {
            return this.subeditor.getCellEditorValue();
        }

        @Override
        public Component getTableCellEditorComponent(JTable table, Object value, boolean isSelected, int row, int column) {
            if (value instanceof Variable.Str) {
                this.subeditor = this.STR;
            } else if (value instanceof Variable.Int) {
                this.subeditor = this.INT;
            } else if (value instanceof Variable.Flt) {
                this.subeditor = this.FLT;
            } else if (value instanceof Variable.Bool) {
                this.subeditor = this.BOOL;
            } else {
                throw new IllegalStateException();
            }

            return this.subeditor.getTableCellEditorComponent(table, value, isSelected, row, column);
        }

        final private Str STR = new Str();
        final private Int INT = new Int();
        final private Flt FLT = new Flt();
        final private Bool BOOL = new Bool();
        private VariableCellEditor subeditor = STR;

    }

    /**
     * Subclass that handles strings.
     */
    static final private class Str extends VariableCellEditor {

        public Str() {
            this.EDITER = new JTextField(10);
        }

        @Override
        public Object getCellEditorValue() {
            String text = this.EDITER.getText();
            return new Variable.Str(this.var, text);
        }

        @Override
        public Component getTableCellEditorComponent(JTable table, Object value, boolean isSelected, int row, int column) {
            if (!(value instanceof Variable.Str)) {
                return null;
            }

            this.var = (Variable.Str) value;
            this.EDITER.setText(var.getValue().toString());
            return this.EDITER;
        }

        private Variable.Str var;
        final private JTextField EDITER;
    }

    /**
     * Subclass that handles integers.
     */
    static final private class Int extends VariableCellEditor {

        public Int() {
            this.EDITER = new JFormattedTextField(NumberFormat.getIntegerInstance());
            this.EDITER.setColumns(5);
        }

        @Override
        public Object getCellEditorValue() {
            Integer value = (Integer) this.EDITER.getValue();
            if (null != value) {
                return new Variable.Int(value);
            } else {
                return null;
            }
        }

        @Override
        public Component getTableCellEditorComponent(JTable table, Object value, boolean isSelected, int row, int column) {
            if (!(value instanceof Variable.Int)) {
                return null;
            }

            Variable.Int var = (Variable.Int) value;
            this.EDITER.setValue(var.getValue());
            return this.EDITER;
        }

        final private JFormattedTextField EDITER;
    }

    /**
     * Subclass that handles floats.
     */
    static final private class Flt extends VariableCellEditor {

        public Flt() {
            this.EDITER = new JFormattedTextField(NumberFormat.getNumberInstance());
            this.EDITER.setColumns(5);
        }

        @Override
        public Object getCellEditorValue() {
            Number value = (Number) this.EDITER.getValue();
            if (null != value) {
                return new Variable.Flt(value.floatValue());
            } else {
                return null;
            }
        }

        @Override
        public Component getTableCellEditorComponent(JTable table, Object value, boolean isSelected, int row, int column) {
            if (!(value instanceof Variable.Flt)) {
                return null;
            }

            Variable.Flt var = (Variable.Flt) value;
            this.EDITER.setValue(var.getValue());
            return this.EDITER;
        }

        final private JFormattedTextField EDITER;
    }

    /**
     * Subclass that handles booleans.
     */
    static final private class Bool extends VariableCellEditor {

        public Bool() {
            this.EDITER = new JComboBox<>(new Boolean[]{Boolean.TRUE, Boolean.FALSE});
            this.EDITER.setPrototypeDisplayValue(Boolean.FALSE);
        }

        @Override
        public Object getCellEditorValue() {
            Boolean value = (Boolean) this.EDITER.getSelectedItem();
            if (null != value) {
                return new Variable.Bool(value);
            } else {
                return null;
            }
        }

        @Override
        public Component getTableCellEditorComponent(JTable table, Object value, boolean isSelected, int row, int column) {
            if (!(value instanceof Variable.Bool)) {
                return null;
            }

            Variable.Bool var = (Variable.Bool) value;
            this.EDITER.setSelectedItem(var.getValue());
            return this.EDITER;
        }

        final private JComboBox<Boolean> EDITER;
    }
}
