/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.gui;

import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ResourceBundle;
import java.util.function.Consumer;
import javax.swing.*;
import javax.swing.table.*;
import restringer.ess.papyrus.*;

/**
 * Describes a JTable specialized for displaying variable tables.
 *
 * @author Mark Fairchild
 * @version 2016-06-23
 */
public class VariableTable extends JTable {

    /**
     * Creates a new <code>VariableTable</code>.
     */
    public VariableTable() {
        this.queryHandler = null;
        this.MI_FIND = new JMenuItem(RES.getString("VARIABLETABLE FIND"), KeyEvent.VK_F);
        this.TABLE_POPUP_MENU = new JPopupMenu(RES.getString("VARIABLETABLE TABLE"));        
        this.initComponent();
    }

    /**
     * Initializes the table's components.
     */
    private void initComponent() {
        this.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        this.setDefaultRenderer(Variable.class, new VariableCellRenderer());
        this.setDefaultEditor(Variable.class, new VariableCellEditor.Generic());

        this.TABLE_POPUP_MENU.add(this.MI_FIND);
        
        this.MI_FIND.addActionListener(e -> {
            if (null != queryHandler) {
                int viewRow = getSelectedRow();
                int modelRow = convertRowIndexToModel(viewRow);
                int column = getModel().getColumnCount() - 1;
                Object o = getModel().getValueAt(modelRow, column);

                if (o instanceof Variable.RefID) {
                    Variable.RefID var = (Variable.RefID) o;
                    if (!var.getRefID().isZero()) {
                        queryHandler.accept(var);
                    }
                } else if (o instanceof Variable.Array) {
                    Variable.Array var = (Variable.Array) o;
                    if (!var.getArrayID().isZero()) {
                        queryHandler.accept(var);
                    }
                }
            }
        });

        this.addMouseListener(new MouseAdapter() {
            @Override
            public void mouseReleased(MouseEvent e) {
                if (e.isPopupTrigger()) {
                    int row = rowAtPoint(e.getPoint());
                    int col = columnAtPoint(e.getPoint());
                    setRowSelectionInterval(row, row);

                    int modelRow = convertRowIndexToModel(row);
                    int column = getModel().getColumnCount() - 1;
                    Object o = getModel().getValueAt(modelRow, column);

                    if (o instanceof Variable.RefID) {
                        Variable.RefID var = (Variable.RefID) o;
                        if (!var.getRefID().isZero()) {
                            TABLE_POPUP_MENU.show(e.getComponent(), e.getX(), e.getY());
                        }
                    } else if (o instanceof Variable.Array) {
                        Variable.Array var = (Variable.Array) o;
                        if (!var.getArrayID().isZero()) {
                            TABLE_POPUP_MENU.show(e.getComponent(), e.getX(), e.getY());
                        }
                    }
                }
            }
        });
    }

    /**
     * Sets the query handler.
     *
     * @param newQueryHandler The new query handler.
     */
    public void setQueryHandler(Consumer<Variable> newQueryHandler) {
        this.queryHandler = newQueryHandler;
    }

    /**
     * Clears the table.
     */
    public void clearTable() {
        this.setModel(new DefaultTableModel());
    }

    /**
     * Displays a <code>Script</code> using an appropriate model.
     *
     * @param script The element to display.
     */
    public void displayScript(Script script) {
        this.setModel(new ScriptTableModel(script));
        this.getColumn(this.getColumnName(0)).setMinWidth(25);
        this.getColumn(this.getColumnName(0)).setMaxWidth(25);
    }

    /**
     * Displays a <code>ScriptInstance</code> using an appropriate model.
     *
     * @param instance The element to display.
     */
    public void displayScriptInstance(ScriptInstance instance) {
        this.setModel(new ScriptInstanceTableModel(instance));
        this.getColumn(this.getColumnName(0)).setMinWidth(25);
        this.getColumn(this.getColumnName(0)).setMaxWidth(25);
        this.getColumn(this.getColumnName(1)).setMinWidth(120);
        this.getColumn(this.getColumnName(1)).setMaxWidth(120);
    }

    /**
     * Displays an <code>ArrayInfo</code> using an appropriate model.
     *
     * @param array The element to display.
     */
    public void displayArray(ArrayInfo array) {
        this.setModel(new ArrayTableModel(array));
        this.getColumn(this.getColumnName(0)).setMinWidth(25);
        this.getColumn(this.getColumnName(0)).setMaxWidth(25);
        this.getColumn(this.getColumnName(1)).setMinWidth(120);
        this.getColumn(this.getColumnName(1)).setMaxWidth(120);
    }

    /**
     * Displays a <code>StackFrame</code> using an appropriate model.
     *
     * @param frame The element to display.
     */
    public void displayStackFrame(StackFrame frame) {
        this.setModel(new StackFrameTableModel(frame));
        this.getColumn(this.getColumnName(0)).setMinWidth(25);
        this.getColumn(this.getColumnName(0)).setMaxWidth(25);
        this.getColumn(this.getColumnName(1)).setMinWidth(120);
        this.getColumn(this.getColumnName(1)).setMaxWidth(120);
    }

    /**
     * Displays a <code>Reference</code> using an appropriate model.
     *
     * @param ref The element to display.
     */
    public void displayReference(Reference ref) {
        this.setModel(new ReferenceTableModel(ref));
        this.getColumn(this.getColumnName(0)).setMinWidth(25);
        this.getColumn(this.getColumnName(0)).setMaxWidth(25);
        this.getColumn(this.getColumnName(1)).setMinWidth(120);
        this.getColumn(this.getColumnName(1)).setMaxWidth(120);
    }

    /**
     * Displays a <code>FunctionMessageData</code> using an appropriate model.
     *
     * @param msg The element to display.
     */
    public void displayFunctionMessageData(FunctionMessageData msg) {
        this.setModel(new FunctionMessageDataTableModel(msg));
        this.getColumn(this.getColumnName(0)).setMinWidth(25);
        this.getColumn(this.getColumnName(0)).setMaxWidth(25);
        this.getColumn(this.getColumnName(1)).setMinWidth(120);
        this.getColumn(this.getColumnName(1)).setMaxWidth(120);
    }

    final private JPopupMenu TABLE_POPUP_MENU;
    final private JMenuItem MI_FIND;
    private Consumer<Variable> queryHandler;
    static final private ResourceBundle RES = ResourceBundle.getBundle("restringer/gui/General");

}
