/*
 * Copyright 2016 Mark.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.pex;

import java.io.PrintWriter;
import java.util.LinkedList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import restringer.IString;
import restringer.pex.PexObject.Function.Instruction;

/**
 *
 * @author Mark
 */
public class Disassembler {

    /**
     *
     * @param block
     * @param locals
     * @param types
     * @param terms
     */
    static void preMap(List<Instruction> block, List<VariableType> locals, List<VariableType> types, TermMap terms) {
        try {
            for (int i = 0; i < block.size(); i++) {
                Instruction inst = block.get(i);
                if (null == inst) {
                    continue;
                }

                boolean del = makeTerm(inst.OPCODE, inst.ARGS, types, terms);
                if (del) {
                    block.set(i, null);
                }
            }

        } catch (Exception | Error ex) {
            ex.printStackTrace(System.err);
            throw ex;
        }
    }

    /**
     *
     * @param out
     * @param block
     * @param locals
     * @param types
     * @param terms
     * @param indent
     */
    static void disassemble(PrintWriter out, List<Instruction> block, List<VariableType> locals, List<VariableType> types, TermMap terms, int indent) {
        preMap(block, locals, types, terms);
        int ptr = 0;

        try {
            while (ptr < block.size()) {
                Instruction inst = block.get(ptr);
                if (null == inst) {
                    ptr++;
                    continue;
                }

                Opcode op = inst.OPCODE;

                if (op.isConditional()) {
                    int[] CONDITIONAL = detectConditional(block, ptr);
                    if (null == CONDITIONAL) {
                        int k = 0;
                        detectConditional(block, ptr);
                    }
                    assert null != CONDITIONAL : "Incorrect conditional block.";
                    int ending = ptr + CONDITIONAL[0];
                    List<Instruction> subBlock = block.subList(ptr, ending);
                    disassembleConditional(out, subBlock, locals, types, terms, indent, false);
                    ptr += CONDITIONAL[0];

                } else {
                    disassemble(out, inst, locals, types, terms, indent);
                    out.flush();
                    ptr++;
                }
            }

        } catch (Exception | Error ex) {
            for (int i = ptr; i < block.size(); i++) {
                Instruction inst = block.get(i);
                if (null != inst) {
                    out.print(Disassembler.tab(indent + 1));
                    out.println(inst);
                } else {
                    out.print(Disassembler.tab(indent + 1));
                    out.println("DELETED");
                }
            }

            ex.printStackTrace(System.err);
            throw ex;
        }
    }

    static void disassembleConditional(PrintWriter out, List<Instruction> block, List<VariableType> locals, List<VariableType> types, TermMap terms, int indent, boolean elseif) {
        int[] CONDITIONAL1 = detectConditional(block, 0);
        if (null == CONDITIONAL1) {
            throw new IllegalArgumentException("Not a conditional block.");
        }

        boolean rhs = false;
        boolean end = false;
        String lhs = "";
        LinkedList<Boolean> stack1 = new LinkedList<>();
        LinkedList<String> stack2 = new LinkedList<>();

        for (int ptr = 0; ptr < block.size() && !end; ptr++) {
            // Get the next instruction. If it's null, then it's been stripped
            // and we can skip it.
            Instruction inst = block.get(ptr);
            if (null == inst) {
                continue;
            }

            // Get these, we'll need them later.
            final int[] IF = detectIF(block, ptr);
            final int[] WHILE = detectWHILE(block, ptr);
            end = (null != IF) || (null != WHILE);

            // Get the opcode. Make sure it's a JMPF or a JMPT. Anything else
            // means that something went wrong earlier.
            Opcode OP = inst.OPCODE;
            if (!OP.isConditional()) {
                throw new IllegalArgumentException("Not a condition block, or not properly stripped.");
            }

            // Get the term and offset.
            String operator = (OP == Opcode.JMPF ? "&&" : "||");
            String term = inst.ARGS.get(0).paren();
            int offset = ((VData.Int) inst.ARGS.get(1)).getValue();

            if (!rhs && !end) {
                lhs += term + " " + operator + " ";
                rhs = true;
            } else if (!rhs && end) {
                lhs += term;
            } else if (rhs && !end) {
                lhs += term;
                while (!stack1.isEmpty() && rhs) {
                    rhs = stack1.pop();
                    lhs = stack2.pop() + "(" + lhs + ")";
                }
                lhs = "(" + lhs + ") " + operator + " ";
            } else if (rhs && end) {
                lhs += term;
                while (!stack1.isEmpty() && rhs) {
                    rhs = stack1.pop();
                    lhs = stack2.pop() + "(" + lhs + ")";
                }
            }

            // Look for a subclause.
            boolean subclause = !end && block.subList(ptr + 1, ptr + offset)
                    .stream()
                    .filter(v -> null != v)
                    .anyMatch(v -> v.OPCODE.isConditional());

            if (subclause) {
                stack1.push(rhs);
                stack2.push(lhs);
                rhs = false;
                lhs = "";
            }

            if (end) {
                if (null != IF) {
                    List<Instruction> subBlock = block.subList(ptr, block.size());
                    disassembleIfElseBlock(out, lhs, subBlock, locals, types, terms, indent, elseif);
                    return;
                } else if (null != WHILE) {
                    List<Instruction> subBlock = block.subList(ptr, block.size());
                    disassembleLoop(out, lhs, subBlock, locals, types, terms, indent);
                    return;
                } else {
                    throw new IllegalStateException();
                }
            }
        }

        throw new IllegalStateException();
    }

    /**
     *
     * @param out
     * @param condition
     * @param block
     * @param locals
     * @param types
     * @param terms
     * @param indent
     * @param elseif
     */
    static void disassembleIfElseBlock(PrintWriter out, String condition, List<Instruction> block, List<VariableType> locals, List<VariableType> types, TermMap terms, int indent, boolean elseif) {
        final int[] IF = detectIF(block, 0);
        assert null != IF;

        int offs1 = IF[0];
        int offs2 = IF[1];
        Instruction begin = block.get(0);
        Instruction end = block.get(offs1 - 1);
        List<Instruction> block1 = block.subList(1, offs1 - 1);
        List<Instruction> block2 = block.subList(offs1, offs1 + offs2 - 1);

        // Disassemble the IF block, which is relatively easy.
        if (elseif) {
            out.printf("%sELSEIF %s\n", tab(indent), condition);
        } else {
            out.printf("%sIF %s\n", tab(indent), condition);
        }
        disassemble(out, block1, locals, types, terms, indent + 1);

        // Try to find an ELSEIF block.
        int ptr = 0;
        while (ptr < block2.size()) {
            if (block2.get(ptr) == null) {
                ptr++;
                continue;
            }

            final int[] CONDITIONAL = detectConditional(block2, ptr);
            if (null == CONDITIONAL) {
                break;
            }

            // Found an ELSEIF.
            int ending = CONDITIONAL[0];
            List<Instruction> subBlock = block2.subList(ptr, ptr + ending);
            disassembleConditional(out, subBlock, types, locals, terms, indent, true);
            return;
        }

        // If there's no ELSEIF block, output the ELSE block.  
        if (offs2 > 1) {
            out.printf("%sELSE\n", tab(indent));
            disassemble(out, block2, locals, types, terms, indent + 1);
            out.printf("%sENDIF\n", tab(indent));

        } // If there's no ELSE, still need an ENDIF.
        else {
            out.printf("%sENDIF\n", tab(indent));
        }
    }

    /**
     *
     * @param out
     * @param block
     * @param locals
     * @param types
     * @param terms
     * @param indent
     */
    static void disassembleLoop(PrintWriter out, String condition, List<Instruction> block, List<VariableType> locals, List<VariableType> types, TermMap terms, int indent) {
        final int[] WHILE = detectWHILE(block, 0);
        int offset = WHILE[0];

        out.print(tab(indent));
        out.printf("WHILE %s\n", condition);

        disassemble(out, block.subList(1, offset - 1), locals, types, terms, indent + 1);

        out.print(tab(indent));
        out.println("ENDWHILE");
        out.flush();
    }

    /**
     *
     * @param block
     * @param ptr
     * @return
     */
    static int[] detectConditional(List<Instruction> block, int ptr) {
        if (block.isEmpty()) {
            return null;
        }

        Instruction begin = block.get(ptr);
        if (null == begin || !begin.OPCODE.isConditional()) {
            return null;
        }

        int subptr = ptr;
        while (subptr < block.size()) {
            Instruction next = block.get(subptr);
            if (null == next) {
                subptr++;
                continue;
            }

            final int[] IF = detectIF(block, subptr);
            final int[] WHILE = detectWHILE(block, subptr);
            if (null != IF) {
                return new int[]{subptr - ptr + IF[0] + IF[1] - 1};
            } else if (null != WHILE) {
                return new int[]{subptr - ptr + WHILE[0]};
            }

            if (!next.OPCODE.isConditional()) {
                //return null;
            }

            subptr++;
        }

        return null;
    }

    /**
     *
     * @param instructions
     * @param ptr
     * @return
     */
    static int[] detectIF(List<Instruction> instructions, int ptr) {
        if (instructions.isEmpty() || ptr >= instructions.size() || ptr < 0) {
            return null;
        }

        Instruction begin = instructions.get(ptr);

        if (null == begin || begin.OPCODE != Opcode.JMPF) {
            return null;
        }

        int offset1 = ((VData.Int) begin.ARGS.get(1)).getValue();
        Instruction end = instructions.get(ptr + offset1 - 1);

        if (null == end || end.OPCODE != Opcode.JMP) {
            return null;
        }

        int offset2 = ((VData.Int) end.ARGS.get(0)).getValue();
        if (offset2 <= 0) {
            return null;
        }

        return new int[]{offset1, offset2};
    }

    /**
     *
     * @param instructions
     * @param ptr
     * @return
     */
    static int[] detectWHILE(List<Instruction> instructions, int ptr) {
        Instruction begin = instructions.get(ptr);

        if (null == begin || begin.OPCODE != Opcode.JMPF) {
            return null;
        }

        int offset1 = ((VData.Int) begin.ARGS.get(1)).getValue();
        if (ptr + offset1 - 1 >= instructions.size()) {
            int k = 0;
        }
        Instruction end = instructions.get(ptr + offset1 - 1);

        if (null == end || end.OPCODE != Opcode.JMP) {
            return null;
        }

        int offset2 = ((VData.Int) end.ARGS.get(0)).getValue();
        if (offset2 > 0) {
            return null;
        }

        assert offset1 <= -offset2 : String.format("Offsets differ: while(%d), endwhile(%d)", offset1, offset2);
        return new int[]{offset1};
    }

    /**
     *
     * @param op
     * @param args
     * @param types
     * @param terms
     * @param indent
     * @return
     */
    static void disassemble(PrintWriter out, Instruction inst, List<VariableType> locals, List<VariableType> types, TermMap terms, int indent) {
        final String RHS = makeRHS(inst, types, terms);

        switch (inst.OPCODE) {
            case NOP:
                break;

            case IADD:
            case FADD:
            case ISUB:
            case FSUB:
            case IMUL:
            case FMUL:
            case IDIV:
            case FDIV:
            case IMOD:
            case STRCAT:
            case NOT:
            case INEG:
            case FNEG:
            case ASSIGN:
            case CAST:
            case CMP_EQ:
            case CMP_LT:
            case CMP_LE:
            case CMP_GT:
            case CMP_GE:
            case ARR_CREATE:
            case ARR_LENGTH:
            case ARR_GET:
                disassembleSimple(out, 0, RHS, inst.ARGS, locals, indent);
                break;

            case CALLMETHOD:
            case CALLSTATIC:
            case PROPGET:
                disassembleSimple(out, 2, RHS, inst.ARGS, locals, indent);
                break;

            case CALLPARENT:
            case ARR_FIND:
            case ARR_RFIND:
                disassembleSimple(out, 1, RHS, inst.ARGS, locals, indent);
                break;

            case RETURN:
                if (null == RHS) {
                    out.printf("%sRETURN\n", tab(indent));
                } else {
                    out.printf("%sRETURN %s\n", tab(indent), RHS);
                }
                break;

            case PROPSET:
                VData obj = inst.ARGS.get(1);
                VData.ID prop = (VData.ID) inst.ARGS.get(0);
                out.printf("%s%s.%s = %s\n", tab(indent), obj, prop, RHS);
                break;

            case ARR_SET: {
                VData.ID arr = (VData.ID) inst.ARGS.get(0);
                VData idx = inst.ARGS.get(1);
                out.printf("%s%s[%s] = %s\n", tab(indent), arr, idx, RHS);
                break;
            }

            case JMPT:
            case JMPF:
            case JMP:
                int k = 0;
                throw new IllegalArgumentException("No code for handling this command: " + inst);

        }
    }

    /**
     *
     * @param out
     * @param dest
     * @param term
     * @param args
     * @param locals
     * @param indent
     */
    static void disassembleSimple(PrintWriter out, int lhsPos, String rhs, List<VData> args, List<VariableType> locals, int indent) {
        if (lhsPos < 0 || lhsPos >= args.size()) {
            throw new IllegalArgumentException();
        }

        VData lhs = args.get(lhsPos);

        if (null == lhs || lhs instanceof VData.None) {
            out.print(tab(indent));
            out.println(rhs);
            return;

        } else if (!(lhs instanceof VData.ID)) {
            out.print(tab(indent));
            out.printf("%s = %s\n", lhs, rhs);
            return;
        }

        VData.ID var = (VData.ID) lhs;

        if (var.isTemp()) {
            assert false;

        } else if (var.isNonevar()) {
            out.print(tab(indent));
            out.println(rhs);

        } else {
            out.print(tab(indent));

            // Insert variable declaration.
            locals.stream()
                    .filter(v -> v.name.equals(var.getValue()))
                    .findAny()
                    .ifPresent(v -> {
                        locals.remove(v);
                        out.print(v.TYPE + " ");
                    });

            out.printf("%s = %s\n", lhs, rhs);
        }

        out.flush();
    }

    /**
     *
     * @param op
     * @param args
     * @param types
     * @param terms
     * @return
     */
    static boolean makeTerm(Opcode op, List<VData> args, List<VariableType> types, TermMap terms) {
        String term;
        VData operand1, operand2, obj, prop, arr, search, idx;
        VData.ID method;
        List<String> subArgs;

        switch (op) {
            case IADD:
            case FADD:
            case STRCAT:
                replaceVariables(args, terms, 0);
                operand1 = args.get(1);
                operand2 = args.get(2);
                term = String.format("%s + %s", operand1.paren(), operand2.paren());
                return processTerm(args, terms, 0, term);

            case ISUB:
            case FSUB:
                replaceVariables(args, terms, 0);
                operand1 = args.get(1);
                operand2 = args.get(2);
                term = String.format("%s - %s", operand1.paren(), operand2.paren());
                return processTerm(args, terms, 0, term);

            case IMUL:
            case FMUL:
                replaceVariables(args, terms, 0);
                operand1 = args.get(1);
                operand2 = args.get(2);
                term = String.format("%s * %s", operand1.paren(), operand2.paren());
                return processTerm(args, terms, 0, term);

            case IDIV:
            case FDIV:
                replaceVariables(args, terms, 0);
                operand1 = args.get(1);
                operand2 = args.get(2);
                term = String.format("%s / %s", operand1.paren(), operand2.paren());
                return processTerm(args, terms, 0, term);

            case IMOD:
                replaceVariables(args, terms, 0);
                operand1 = args.get(1);
                operand2 = args.get(2);
                term = String.format("%s %% %s", operand1.paren(), operand2.paren());
                return processTerm(args, terms, 0, term);

            case RETURN:
                replaceVariables(args, terms, -1);
                return false;

            case CALLMETHOD:
                replaceVariables(args, terms, 2);
                method = (VData.ID) args.get(0);
                obj = args.get(1);
                subArgs = args
                        .subList(4, args.size())
                        .stream()
                        .map(v -> v.paren())
                        .collect(Collectors.toList());
                term = String.format("%s.%s%s", obj, method, paramList(subArgs));
                return processTerm(args, terms, 2, term);

            case CALLPARENT:
                replaceVariables(args, terms, 1);
                method = (VData.ID) args.get(0);
                subArgs = args
                        .subList(3, args.size())
                        .stream()
                        .map(v -> v.paren())
                        .collect(Collectors.toList());
                term = String.format("parent.%s%s", method, paramList(subArgs));
                return processTerm(args, terms, 1, term);

            case CALLSTATIC:
                replaceVariables(args, terms, 2);
                obj = args.get(0);
                method = (VData.ID) args.get(1);
                subArgs = args
                        .subList(4, args.size())
                        .stream()
                        .map(v -> v.toString())
                        .collect(Collectors.toList());
                term = String.format("%s.%s%s", obj, method, paramList(subArgs));
                return processTerm(args, terms, 2, term);

            case NOT:
                replaceVariables(args, terms, 0);
                term = String.format("!%s", args.get(1).paren());
                return processTerm(args, terms, 0, term);

            case INEG:
            case FNEG:
                replaceVariables(args, terms, 0);
                term = String.format("-%s", args.get(1).paren());
                return processTerm(args, terms, 0, term);

            case ASSIGN:
                replaceVariables(args, terms, 0);
                term = String.format("%s", args.get(1));
                return processTerm(args, terms, 0, term);

            case CAST: {
                replaceVariables(args, terms, 0);
                VData.ID dest = (VData.ID) args.get(0);
                VData arg = args.get(1);
                IString name = dest.getValue();
                IString type = types.stream().filter(t -> t.name.equals(name)).findFirst().get().TYPE;

                if (type.equals(IString.get("bool"))) {
                    term = arg.toString();
                } else if (type.equals(IString.get("string"))) {
                    term = arg.toString();
                } else {
                    term = String.format("(%s)%s", type, arg.paren());
                }
                return processTerm(args, terms, 0, term);
            }

            case PROPGET:
                replaceVariables(args, terms, 2);
                obj = args.get(1);
                prop = args.get(0);
                term = String.format("%s.%s", obj, prop);
                return processTerm(args, terms, 2, term);

            case PROPSET:
                replaceVariables(args, terms, -1);
                return false;

            case CMP_EQ:
                replaceVariables(args, terms, 0);
                operand1 = args.get(1);
                operand2 = args.get(2);
                term = String.format("%s == %s", operand1.paren(), operand2.paren());
                return processTerm(args, terms, 0, term);

            case CMP_LT:
                replaceVariables(args, terms, 0);
                operand1 = args.get(1);
                operand2 = args.get(2);
                term = String.format("%s < %s", operand1.paren(), operand2.paren());
                return processTerm(args, terms, 0, term);

            case CMP_LE:
                replaceVariables(args, terms, 0);
                operand1 = args.get(1);
                operand2 = args.get(2);
                term = String.format("%s <= %s", operand1.paren(), operand2.paren());
                return processTerm(args, terms, 0, term);

            case CMP_GT:
                replaceVariables(args, terms, 0);
                operand1 = args.get(1);
                operand2 = args.get(2);
                term = String.format("%s > %s", operand1.paren(), operand2.paren());
                return processTerm(args, terms, 0, term);

            case CMP_GE:
                replaceVariables(args, terms, 0);
                operand1 = args.get(1);
                operand2 = args.get(2);
                term = String.format("%s >= %s", operand1.paren(), operand2.paren());
                return processTerm(args, terms, 0, term);

            case ARR_CREATE:
                VData size = args.get(1);
                VData.ID dest = (VData.ID) args.get(0);
                IString name = dest.getValue();
                IString type = types.stream().filter(t -> t.name.equals(name)).findFirst().get().TYPE;
                String subtype = type.toString().substring(0, type.length() - 2);
                term = String.format("new %s[%s]", subtype, size);
                return processTerm(args, terms, 0, term);

            case ARR_LENGTH:
                replaceVariables(args, terms, 0);
                term = String.format("%s.length", args.get(1));
                return processTerm(args, terms, 0, term);

            case ARR_GET:
                replaceVariables(args, terms, 0);
                idx = (VData) args.get(2);
                arr = args.get(1);
                term = String.format("%s[%s]", arr, idx);
                return processTerm(args, terms, 0, term);

            case ARR_SET:
                replaceVariables(args, terms, -1);
                return false;

            case JMPT:
            case JMPF:
                replaceVariables(args, terms, -1);
                return false;

            case ARR_FIND:
                replaceVariables(args, terms, 1);
                arr = args.get(0);
                search = args.get(2);
                idx = (VData.Int) args.get(3);
                term = String.format("%s.find(%s, %s)", arr, search, idx);
                return processTerm(args, terms, 1, term);

            case ARR_RFIND:
                replaceVariables(args, terms, 1);
                arr = args.get(0);
                search = args.get(2);
                idx = (VData.Int) args.get(3);
                term = String.format("%s.rfind(%s, %s)", arr, search, idx);
                return processTerm(args, terms, 1, term);

            case JMP:
            default:
                return false;
        }
    }

    static String makeRHS(Instruction inst, List<VariableType> types, TermMap terms) {
        switch (inst.OPCODE) {
            case IADD:
            case FADD:
            case STRCAT: {
                VData operand1 = inst.ARGS.get(1);
                VData operand2 = inst.ARGS.get(2);
                return String.format("%s + %s", operand1.paren(), operand2.paren());
            }
            case ISUB:
            case FSUB: {
                VData operand1 = inst.ARGS.get(1);
                VData operand2 = inst.ARGS.get(2);
                return String.format("%s - %s", operand1.paren(), operand2.paren());
            }
            case IMUL:
            case FMUL: {
                VData operand1 = inst.ARGS.get(1);
                VData operand2 = inst.ARGS.get(2);
                return String.format("%s * %s", operand1.paren(), operand2.paren());
            }
            case IDIV:
            case FDIV: {
                VData operand1 = inst.ARGS.get(1);
                VData operand2 = inst.ARGS.get(2);
                return String.format("%s / %s", operand1.paren(), operand2.paren());
            }
            case IMOD: {
                VData operand1 = inst.ARGS.get(1);
                VData operand2 = inst.ARGS.get(2);
                return String.format("%s %% %s", operand1.paren(), operand2.paren());
            }

            case RETURN: {
                return inst.ARGS.get(0).toString();
            }

            case CALLMETHOD: {
                VData.ID method = (VData.ID) inst.ARGS.get(0);
                VData obj = inst.ARGS.get(1);
                List<String> subArgs = inst.ARGS
                        .subList(4, inst.ARGS.size())
                        .stream()
                        .map(v -> v.paren())
                        .collect(Collectors.toList());
                return String.format("%s.%s%s", obj, method, paramList(subArgs));
            }

            case CALLPARENT: {
                VData.ID method = (VData.ID) inst.ARGS.get(0);
                List<String> subArgs = inst.ARGS
                        .subList(3, inst.ARGS.size())
                        .stream()
                        .map(v -> v.paren())
                        .collect(Collectors.toList());
                return String.format("parent.%s%s", method, paramList(subArgs));
            }

            case CALLSTATIC: {
                VData obj = inst.ARGS.get(0);
                VData.ID method = (VData.ID) inst.ARGS.get(1);
                List<String> subArgs = inst.ARGS
                        .subList(4, inst.ARGS.size())
                        .stream()
                        .map(v -> v.paren())
                        .collect(Collectors.toList());
                return String.format("%s.%s%s", obj, method, paramList(subArgs));
            }

            case NOT:
                return String.format("NOT %s", inst.ARGS.get(1).paren());

            case INEG:
            case FNEG:
                return String.format("-%s", inst.ARGS.get(1).paren());

            case ASSIGN:
                return inst.ARGS.get(1).toString();

            case CAST: {
                VData.ID dest = (VData.ID) inst.ARGS.get(0);
                VData arg = inst.ARGS.get(1);
                IString name = dest.getValue();
                IString type = types.stream().filter(t -> t.name.equals(name)).findFirst().get().TYPE;
                return String.format("(%s)%s", type, arg.paren());
            }

            case PROPGET: {
                VData obj = inst.ARGS.get(1);
                VData prop = inst.ARGS.get(0);
                return String.format("%s.%s", obj, prop);
            }

            case PROPSET: {
                return inst.ARGS.get(2).paren();
            }

            case CMP_EQ: {
                VData operand1 = inst.ARGS.get(1);
                VData operand2 = inst.ARGS.get(2);
                return String.format("%s == %s", operand1.paren(), operand2.paren());
            }

            case CMP_LT: {
                VData operand1 = inst.ARGS.get(1);
                VData operand2 = inst.ARGS.get(2);
                return String.format("%s < %s", operand1.paren(), operand2.paren());
            }

            case CMP_LE: {
                VData operand1 = inst.ARGS.get(1);
                VData operand2 = inst.ARGS.get(2);
                return String.format("%s <= %s", operand1.paren(), operand2.paren());
            }

            case CMP_GT: {
                VData operand1 = inst.ARGS.get(1);
                VData operand2 = inst.ARGS.get(2);
                return String.format("%s > %s", operand1.paren(), operand2.paren());
            }

            case CMP_GE: {
                VData operand1 = inst.ARGS.get(1);
                VData operand2 = inst.ARGS.get(2);
                return String.format("%s >= %s", operand1.paren(), operand2.paren());
            }

            case ARR_CREATE: {
                VData size = inst.ARGS.get(1);
                VData.ID dest = (VData.ID) inst.ARGS.get(0);
                IString name = dest.getValue();
                IString type = types.stream().filter(t -> t.name.equals(name)).findFirst().get().TYPE;
                String subtype = type.toString().substring(0, type.length() - 2);
                return String.format("new %s[%s]", subtype, size);
            }

            case ARR_LENGTH: {
                return String.format("%s.length", inst.ARGS.get(1));
            }

            case ARR_GET: {
                VData idx = inst.ARGS.get(2);
                VData arr = inst.ARGS.get(1);
                return String.format("%s[%s]", arr, idx);
            }

            case ARR_SET: {
                return inst.ARGS.get(2).paren();
            }

            case JMPT:
            case JMPF: {
                return inst.ARGS.get(0).paren();
            }

            case JMP:
            case ARR_FIND:
            case ARR_RFIND:
            default:
                return String.format("%s", inst.ARGS);
        }
    }

    /**
     * @param args
     * @param terms
     * @param destPos
     * @param positions
     */
    static boolean processTerm(List<VData> args, TermMap terms, int destPos, String term) {
        if (destPos >= args.size() || !(args.get(destPos) instanceof VData.ID)) {
            return false;
        }
        VData.ID dest = (VData.ID) args.get(destPos);

        if (!dest.isTemp()) {
            return false;
        }

        terms.put(dest, new VData.Term(term));
        return true;
    }

    /**
     * Replaces certain variables with terms. In particular, all temp variables
     * and autovar names should be replaced.
     *
     * @param args
     * @param terms
     * @param exclude
     */
    static void replaceVariables(List<VData> args, TermMap terms, int exclude) {
        for (int i = 0; i < args.size(); i++) {
            VData arg = args.get(i);
            if (arg instanceof VData.ID) {
                VData.ID id = (VData.ID) arg;

                if (terms.containsKey(id) && i != exclude) {
                    args.set(i, terms.get(id));

                } else if (id.isAutovar()) {
                    final Matcher MATCHER = AUTOVAR_REGEX.matcher(id.toString());
                    MATCHER.matches();
                    String prop = MATCHER.group(1);
                    terms.put(id, new VData.Term(prop));
                    args.set(i, terms.get(id));
                }

            } else if (arg instanceof VData.Str) {
                VData.Str str = (VData.Str) arg;
                args.set(i, new VData.StrLit(str.getString().toString()));
            }
        }
    }

    /**
     *
     * @param <T>
     * @param params
     * @return
     */
    static <T> String paramList(List<T> params) {
        return params.stream()
                .map(p -> p.toString())
                .collect(Collectors.joining(", ", "(", ")"));
    }

    /**
     *
     * @param n
     * @return
     */
    static public String tab(int n) {
        final StringBuilder BUF = new StringBuilder();
        for (int i = 0; i < n; i++) {
            BUF.append('\t');
        }
        return BUF.toString();
    }

    static final Pattern AUTOVAR_REGEX = Pattern.compile("^::(.+)_var$", Pattern.CASE_INSENSITIVE);
}
