/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.pex;

import java.io.DataInput;
import java.io.IOException;
import java.util.Objects;

/**
 * Describes the different opcodes that can appear in PEX functions, and stores
 * the number of arguments that they accept. If the number of arguments is
 * negative, it indicates that a variable number of arguments can follow the
 * main arguments.
 *
 * @author Mark Fairchild
 * @version 2016/07/04
 */
public enum Opcode {
    NOP(0),
    IADD(3),
    FADD(3),
    ISUB(3),
    FSUB(3),
    IMUL(3),
    FMUL(3),
    IDIV(3),
    FDIV(3),
    IMOD(3),
    NOT(2),
    INEG(2),
    FNEG(2),
    ASSIGN(2),
    CAST(2),
    CMP_EQ(3),
    CMP_LT(3),
    CMP_LE(3),
    CMP_GT(3),
    CMP_GE(3),
    JMP(1),
    JMPT(2),
    JMPF(2),
    CALLMETHOD(-3),
    CALLPARENT(-2),
    CALLSTATIC(-3),
    RETURN(1),
    STRCAT(3),
    PROPGET(3),
    PROPSET(3),
    ARR_CREATE(2),
    ARR_LENGTH(2),
    ARR_GET(3),
    ARR_SET(3),
    ARR_FIND(4),
    ARR_RFIND(4);

    public boolean isConditional() {
        return (this == JMPT) || (this == JMPF);
    }

    public boolean isBranching() {
        return (this == JMPT) || (this == JMPF) || (this == JMP);
    }

    public boolean isArithmetic() {
        return this == IADD || this == FADD || 
                this == ISUB || this == FSUB || 
                this == IMUL || this == FMUL || 
                this == IDIV || this == FDIV || 
                this == IMOD;
    }

    private Opcode(int args) {
        this.ARGS = args;
    }

    /**
     * Read a <code>DataType</code> from an input stream.
     *
     * @param input The input stream.
     * @return The <code>DataType</code>.
     */
    static Opcode read(DataInput input) throws IOException {
        Objects.requireNonNull(input);

        int index = input.readUnsignedByte();
        if (index < 0 || index >= VALUES.length) {
            throw new IOException("Invalid Opcode.");
        }

        return VALUES[index];
    }

    public final int ARGS;
    static final private Opcode[] VALUES = Opcode.values();

}
