/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.pex;

import java.io.*;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import restringer.IString;
import restringer.pex.StringTable.TString;

/**
 * Describes a Skyrim PEX script and will read and write it from streams.
 *
 * @author Mark Fairchild
 * @version 2016/07/04
 */
final public class Pex {

    /**
     * Reads a script file and creates a Pex object to represent it.
     *
     * Exceptions are not handled. At all. Not even a little bit.
     *
     * @param data An array of bytes containing the script data.
     * @return The Pex object.
     *
     * @throws IOException
     *
     */
    static public Pex readScript(byte[] data) throws IOException {
        assert null != data;

        // Prepare input stream. The DataInput interface just happen to be 
        // perfect for this kind of thing. 
        try (DataInputStream dis = new DataInputStream(new ByteArrayInputStream(data))) {
            return new Pex(dis);
        }
    }

    /**
     * Reads a script file and creates a Pex object to represent it.
     *
     * Exceptions are not handled. At all. Not even a little bit.
     *
     * @param scriptFile The script file to read, which must exist and be
     * readable.
     * @return The Pex object.
     *
     * @throws FileNotFoundException
     * @throws IOException
     *
     */
    static public Pex readScript(File scriptFile) throws FileNotFoundException, IOException {
        //assert scriptFile.isFile();
        //assert scriptFile.canRead();
        //assert scriptFile.exists();

        // Prepare input stream. The DataInput interface just happen to be 
        // perfect for this kind of thing. 
        try (DataInputStream dis = new DataInputStream(new BufferedInputStream(new FileInputStream(scriptFile)))) {
            return new Pex(dis);
        }
    }

    /**
     * Writes a Pex object to a script file.
     *
     * Exceptions are not handled. At all. Not even a little bit.
     *
     * @param script The Pex object to write.
     * @param scriptFile The script file to write. If it exists, it must be a
     * file and it must be writable.
     *
     * @throws FileNotFoundException
     * @throws IOException
     *
     */
    static public void writeScript(Pex script, File scriptFile) throws FileNotFoundException, IOException {
        assert !scriptFile.exists() || scriptFile.isFile();
        assert !scriptFile.exists() || scriptFile.canWrite();

        // Prepare output streams. The DataOutput interface just happen to be 
        // perfect for this kind of thing.
        try (DataOutputStream dos = new DataOutputStream(new BufferedOutputStream(new FileOutputStream(scriptFile)))) {
            script.write(dos);
        }
    }

    /**
     * Creates a Pex by reading from a DataInput.
     *
     * @param input A datainput for a Skyrim PEX file.
     * @throws IOException Exceptions aren't handled.
     */
    private Pex(DataInput input) throws IOException {
        this.HEADER = new Header(input);

        this.STRINGS = new StringTable(input);
        this.DEBUG = new DebugInfo(input, this.STRINGS);

        int flagCount = input.readUnsignedShort();
        this.USERFLAGDEFS = new ArrayList<>(flagCount);
        while (0 < flagCount) {
            this.USERFLAGDEFS.add(new UserFlag(input, this.STRINGS));
            flagCount--;
        }

        int objectCount = input.readUnsignedShort();
        if (objectCount != 1) {
            throw new IllegalStateException("Scripts must contain exactly one object.");
        }

        this.OBJECT = new PexObject(input, this.USERFLAGDEFS, this.STRINGS);
        /*this.OBJECTS = new ArrayList<>(objectCount);
        while (0 < objectCount) {
            this.OBJECTS.add(new PexObject(input, this.USERFLAGDEFS, this.STRINGS));
            objectCount--;
        }*/
    }

    /**
     * Write the object to a <code>DataOutput</code>.
     *
     * @param output The <code>DataOutput</code> to write.
     * @throws IOException IO errors aren't handled at all, they are simply
     * passed on.
     */
    public void write(DataOutput output) throws IOException {
        this.HEADER.write(output);

        this.STRINGS.write(output);
        this.DEBUG.write(output);

        output.writeShort((short) this.USERFLAGDEFS.size());
        for (UserFlag flag : this.USERFLAGDEFS) {
            flag.write(output);
        }

        output.writeShort(1);
        this.OBJECT.write(output);

        /*
        output.writeShort((short) this.OBJECTS.size());
        for (PexObject obj : this.OBJECTS) {
            obj.write(output);
        }*/
    }

    /**
     * Rebuilds the string table. This is necessary if ANY strings in ANY of the
     * Pex's members has changed at all. Otherwise, writing the Pex will produce
     * an invalid file.
     *
     */
    public void rebuildStringTable() {
        final Set<TString> INUSE = new java.util.LinkedHashSet<>();
        this.DEBUG.collectStrings(INUSE);
        this.USERFLAGDEFS.forEach(flag -> flag.collectStrings(INUSE));
        //this.OBJECTS.forEach(obj -> obj.collectStrings(INUSE));
        this.OBJECT.collectStrings(INUSE);
        this.STRINGS.rebuildStringTable(INUSE);
    }

    /**
     * Tries to disassemble the script.
     *
     * @param out The outputstream.
     * @param level Partial disassembly flag.
     */
    public void disassemble(java.io.Writer out, AssemblyLevel level) {
        try (java.io.PrintWriter writer = new java.io.PrintWriter(out)) {
            /*this.OBJECTS.forEach(o -> {
                o.disassemble(writer, level);
            });*/
            this.OBJECT.disassemble(writer, level);
        }
    }

    /**
     * Pretty-prints the Pex.
     *
     * @return A string representation of the Pex.
     */
    @Override
    public String toString() {
        StringBuilder buf = new StringBuilder();
        buf.append(this.HEADER.toString());
        buf.append(this.DEBUG.toString());

        buf.append("USER FLAGS\n");
        buf.append(this.USERFLAGDEFS.toString());

        buf.append("\n\nOBJECT\n");
        buf.append(this.OBJECT);
        buf.append("\n");

        /*
        this.OBJECTS.forEach(obj -> {
            buf.append(obj);
            buf.append("\n");
        });*/
        return buf.toString();

    }

    /**
     * Read an IString from a PEX file.
     *
     * @param input The <code>DataInput</code> to read.
     * @return The IString that was read.
     * @throws IOException IO errors aren't handled at all, they are simply
     * passed on.
     */
    private IString readIString(DataInput input) throws IOException {
        int index = input.readUnsignedShort();

        if (index < 0 || index >= STRINGS.size()) {
            throw new IOException();
        }

        return this.STRINGS.get(index);
    }

    /**
     * Write an IString to a PEX file.
     *
     * @param str The IString to write.
     * @param output The <code>DataOutput</code> to write.
     * @throws IOException IO errors aren't handled at all, they are simply
     * passed on.
     */
    private void writeIString(IString str, DataOutput output) throws IOException {
        short index = (short) STRINGS.indexOf(str);
        output.writeShort(index);
    }

    /**
     * @return The compilation date of the <code>Pex</code>.
     */
    public long getDate() {
        return this.HEADER.compilationTime;
    }

    /**
     * @return The filename of the <code>Pex</code>, determined from the header.
     */
    public IString getFilename() {
        final String SOURCE = this.HEADER.soureFilename;
        final String REGEX = "(psc)$";
        final String REPLACEMENT = "pex";
        final Pattern PATTERN = Pattern.compile(REGEX, Pattern.CASE_INSENSITIVE);
        final Matcher MATCHER = PATTERN.matcher(SOURCE);
        final String COMPILED = MATCHER.replaceAll(REPLACEMENT);
        return IString.get(COMPILED);
    }

    /**
     * @return A <code>ScriptStats</code> object for the <code>PexObject</code>.
     */
    public ScriptStats analyze() {
        final ScriptStats STATS = new ScriptStats();
        //this.OBJECTS.forEach(obj -> obj.analyze(STATS));
        this.OBJECT.analyze(STATS);
        return STATS;
    }

    final public Header HEADER;
    final public StringTable STRINGS;
    final public DebugInfo DEBUG;
    final public List<UserFlag> USERFLAGDEFS;
    //final public List<PexObject> OBJECTS;
    final public PexObject OBJECT;

    /**
     * Describes the header of a Pex file. Useless beyond that.
     *
     */
    final public class Header {

        /**
         * Creates a Header by reading from a DataInput.
         *
         * @param input A datainput for a Skyrim PEX file.
         * @throws IOException Exceptions aren't handled.
         */
        private Header(DataInput input) throws IOException {
            this.magic = input.readInt();
            this.version = input.readInt();
            this.compilationTime = input.readLong();
            this.soureFilename = input.readUTF();
            this.userName = input.readUTF();
            this.machineName = input.readUTF();
        }

        /**
         * Write the object to a <code>DataOutput</code>.
         *
         * @param output The <code>DataOutput</code> to write.
         * @throws IOException IO errors aren't handled at all, they are simply
         * passed on.
         */
        private void write(DataOutput output) throws IOException {
            output.writeInt(this.magic);
            output.writeInt(this.version);
            output.writeLong(this.compilationTime);
            output.writeUTF(this.soureFilename);
            output.writeUTF(this.userName);
            output.writeUTF(this.machineName);
        }

        /**
         * Pretty-prints the Header.
         *
         * @return A string representation of the Header.
         */
        @Override
        public String toString() {
            StringBuilder buf = new StringBuilder();
            buf.append(String.format("%s compiled at %d by %s on %s.\n", this.soureFilename, this.compilationTime, this.userName, this.machineName));
            buf.append(String.format("%h v%d\n\n", this.magic, this.version));
            return buf.toString();
        }

        private int magic = 0;
        private int version = 0;
        private long compilationTime = 0;
        private String soureFilename = "";
        private String userName = "";
        private String machineName = "";

    }

    /**
     * Describe the debugging info section of a PEX file.
     *
     */
    final public class DebugInfo {

        /**
         * Creates a DebugInfo by reading from a DataInput.
         *
         * @param input A datainput for a Skyrim PEX file.
         * @param strings The <code>StringTable</code> for the <code>Pex</code>.
         * @throws IOException Exceptions aren't handled.
         */
        private DebugInfo(DataInput input, StringTable strings) throws IOException {
            this.hasDebugInfo = input.readByte();

            if (this.hasDebugInfo == 0) {
                this.DEBUGFUNCTIONS = new ArrayList<>(0);

            } else {
                this.modificationTime = input.readLong();
                int functionCount = input.readUnsignedShort();
                this.DEBUGFUNCTIONS = new ArrayList<>(functionCount);
                for (int i = 0; i < functionCount; i++) {
                    this.DEBUGFUNCTIONS.add(new DebugFunction(input, strings));
                }
            }
        }

        /**
         * Write the object to a <code>DataOutput</code>.
         *
         * @param output The <code>DataOutput</code> to write.
         * @throws IOException IO errors aren't handled at all, they are simply
         * passed on.
         */
        private void write(DataOutput output) throws IOException {
            output.write(this.hasDebugInfo);

            if (this.hasDebugInfo != 0) {
                output.writeLong(this.modificationTime);
                output.writeShort(this.DEBUGFUNCTIONS.size());

                for (DebugFunction function : this.DEBUGFUNCTIONS) {
                    function.write(output);
                }
            }
        }

        /**
         * Removes all debug info.
         */
        public void clear() {
            this.hasDebugInfo = 0;
            this.DEBUGFUNCTIONS.clear();

        }

        /**
         * Collects all of the strings used by the DebugInfo and adds them to a
         * set.
         *
         * @param strings The set of strings.
         */
        public void collectStrings(Set<TString> strings) {
            this.DEBUGFUNCTIONS.forEach(func -> func.collectStrings(strings));
        }

        /**
         * Pretty-prints the DebugInfo.
         *
         * @return A string representation of the DebugInfo.
         */
        @Override
        public String toString() {
            StringBuilder buf = new StringBuilder();
            buf.append("DEBUGINFO\n");
            this.DEBUGFUNCTIONS.forEach(function -> {
                buf.append("\t");
                buf.append(function.toString());
                buf.append("\n");
            });
            buf.append("\n");
            return buf.toString();
        }

        private byte hasDebugInfo;
        private long modificationTime;
        final private List<DebugFunction> DEBUGFUNCTIONS;

    }

    /**
     * Describes the debugging information for a function.
     *
     */
    final class DebugFunction {

        /**
         * Creates a DebugFunction by reading from a DataInput.
         *
         * @param input A datainput for a Skyrim PEX file.
         * @param strings The <code>StringTable</code> for the <code>Pex</code>.
         * @throws IOException Exceptions aren't handled.
         */
        private DebugFunction(DataInput input, StringTable strings) throws IOException {
            this.OBJECTNAME = strings.read(input);
            this.STATENAME = strings.read(input);
            this.FUNCNAME = strings.read(input);
            this.FUNCTYPE = input.readByte();

            int instructionCount = input.readUnsignedShort();
            this.INSTRUCTIONS = new ArrayList<>(instructionCount);

            for (int i = 0; i < instructionCount; i++) {
                this.INSTRUCTIONS.add(input.readUnsignedShort());
            }
        }

        /**
         * Write the object to a <code>DataOutput</code>.
         *
         * @param output The <code>DataOutput</code> to write.
         * @throws IOException IO errors aren't handled at all, they are simply
         * passed on.
         */
        private void write(DataOutput output) throws IOException {
            this.OBJECTNAME.write(output);
            this.STATENAME.write(output);
            this.FUNCNAME.write(output);
            output.writeByte(this.FUNCTYPE);
            output.writeShort(this.INSTRUCTIONS.size());

            for (int instr : this.INSTRUCTIONS) {
                output.writeShort(instr);
            }
        }

        /**
         * Collects all of the strings used by the DebugFunction and adds them
         * to a set.
         *
         * @param strings The set of strings.
         */
        public void collectStrings(Set<TString> strings) {
            strings.add(this.OBJECTNAME);
            strings.add(this.STATENAME);
            strings.add(this.FUNCNAME);
        }

        /**
         * Generates a qualified name for the object of the form
         * "OBJECT.FUNCTION".
         *
         * @return A qualified name.
         *
         */
        public IString getFullName() {
            return IString.format("%s.%s", this.OBJECTNAME, this.FUNCNAME);
        }

        /**
         * Pretty-prints the DebugFunction.
         *
         * @return A string representation of the DebugFunction.
         */
        @Override
        public String toString() {
            StringBuilder buf = new StringBuilder();
            buf.append(String.format("%s %s.%s (type %d): ", this.OBJECTNAME, this.STATENAME, this.FUNCNAME, this.FUNCTYPE));
            this.INSTRUCTIONS.forEach(instr -> buf.append(String.format("%04x ", instr)));
            return buf.toString();
        }

        final private TString OBJECTNAME;
        final private TString STATENAME;
        final private TString FUNCNAME;
        final private byte FUNCTYPE;
        final private List<Integer> INSTRUCTIONS;

    }

}
