/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.pex;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import restringer.IString;
import restringer.Scheme;
import restringer.pex.StringTable.TString;

/**
 * Describes an object from a PEX file.
 *
 * @author Mark Fairchild
 * @version 2016/07/04
 */
final public class PexObject {

    /**
     * Creates a PexObject by reading from a DataInput.
     *
     * @param input A datainput for a Skyrim PEX file.
     * @param strings The <code>StringTable</code> for the <code>Pex</code>.
     * @param flag The <code>UserFlag</code> list.
     * @throws IOException Exceptions aren't handled.
     */
    PexObject(DataInput input, List<UserFlag> flags, StringTable strings) throws IOException {
        Objects.requireNonNull(input);
        this.USERFLAGDEFS = Objects.requireNonNull(flags);
        this.STRINGS = Objects.requireNonNull(strings);

        this.NAME = strings.read(input);
        this.size = input.readInt();
        this.PARENTNAME = strings.read(input);
        this.docString = strings.read(input);
        this.USERFLAGS = input.readInt();
        this.AUTOSTATENAME = strings.read(input);
        this.AUTOVARMAP = new java.util.HashMap<>();

        int numVariables = input.readUnsignedShort();
        this.VARIABLES = new ArrayList<>(numVariables);
        for (int i = 0; i < numVariables; i++) {
            this.VARIABLES.add(new Variable(input, strings));
        }

        int numProperties = input.readUnsignedShort();
        this.PROPERTIES = new ArrayList<>(numProperties);
        for (int i = 0; i < numProperties; i++) {
            this.PROPERTIES.add(new Property(input, strings));
        }

        int numStates = input.readUnsignedShort();
        this.STATES = new ArrayList<>(numStates);
        for (int i = 0; i < numStates; i++) {
            this.STATES.add(new State(input, strings));
        }

        this.PROPERTIES.forEach(prop -> {
            if (prop.hasAutoVar()) {
                for (Variable var : this.VARIABLES) {
                    if (prop.autoVarName.equals(var.name)) {
                        this.AUTOVARMAP.put(prop, var);
                        break;
                    }
                }
                assert this.AUTOVARMAP.containsKey(prop);
            }

        });
    }

    /**
     * Write the object to a <code>DataOutput</code>.
     *
     * @param output The <code>DataOutput</code> to write.
     * @throws IOException IO errors aren't handled at all, they are simply
     * passed on.
     *
     */
    void write(DataOutput output) throws IOException {
        this.NAME.write(output);

        this.size = this.calculateSize();
        output.writeInt(this.size);

        this.PARENTNAME.write(output);
        this.docString.write(output);
        output.writeInt(this.USERFLAGS);
        this.AUTOSTATENAME.write(output);

        output.writeShort(this.VARIABLES.size());
        for (Variable var : this.VARIABLES) {
            var.write(output);
        }

        output.writeShort(this.PROPERTIES.size());
        for (Property prop : this.PROPERTIES) {
            prop.write(output);
        }

        output.writeShort(this.STATES.size());
        for (State state : this.STATES) {
            state.write(output);
        }
    }

    /**
     * Calculates the size of the PexObject, in bytes.
     *
     * @return The size of the PexObject.
     *
     */
    public int calculateSize() {
        int sum = 0;
        sum += 4; // size
        sum += 2; // parentClassName
        sum += 2; // docString
        sum += 4; // userFlags
        sum += 2; // autoStateName
        sum += 6; // array sizes
        sum += this.VARIABLES.stream().mapToInt(v -> v.calculateSize()).sum();
        sum += this.PROPERTIES.stream().mapToInt(v -> v.calculateSize()).sum();
        sum += this.STATES.stream().mapToInt(v -> v.calculateSize()).sum();
        return sum;
    }

    /**
     * Simplified restringing algorithm. If the script is a parent to other
     * scripts, they will NOT be able to access any AUTO properties. Do not run
     * it on parent scripts for ANY reason.
     *
     * @param remapVars Remap object variables.
     * @param remapProps Remap properties.
     * @param remapParams Remap function parameters.
     * @param remapLocals Remap local variables.
     * @param stripDocs Strip documentation strings.
     *
     */
    public void restring2(boolean remapVars, boolean remapProps, boolean remapParams, boolean remapLocals, boolean stripDocs) {
        // Strip the docstring.
        if (stripDocs && !this.docString.isEmpty()) {
            this.docString = this.STRINGS.blank();
        }

        // Get/create the generator for new identifier names. 
        final TokenGenerator GEN = new TokenGenerator();

        // Create the remapping scheme. 
        final Scheme SCHEME = new Scheme();

        // First, rename autovars and store the mapping in the scheme.
        if (remapProps) {
            this.PROPERTIES
                    .stream()
                    .filter(prop -> prop.hasAutoVar())
                    .filter(prop -> this.AUTOVARMAP.containsKey(prop))
                    .filter(prop -> !this.AUTOVARMAP.get(prop).isConditional())
                    .forEach(prop -> {
                        TString newName = this.STRINGS.addString(GEN.next());
                        SCHEME.put(prop.autoVarName, newName);
                        prop.autoVarName = newName;
                        final Variable VAR = this.AUTOVARMAP.get(prop);
                        VAR.name = newName;
                    });
        }

        // Give each variable a new name. 
        if (remapVars) {
            this.VARIABLES
                    .stream()
                    .filter(v -> !v.isConditional())
                    .filter(v -> !this.AUTOVARMAP.containsValue(v))
                    .forEach(var -> {
                        TString newName = this.STRINGS.addString(GEN.next());
                        SCHEME.put(var.name, newName);
                        var.name = this.STRINGS.addString(newName);
                    });
        }

        // Apply the scheme to the object's properties.
        this.PROPERTIES.forEach(prop -> {
            // Remove the docstrings.
            if (stripDocs) {
                if (!prop.docString.isEmpty()) {
                    prop.docString = this.STRINGS.blank();
                }
                if (!prop.READHANDLER.docString.isEmpty()) {
                    prop.READHANDLER.docString = this.STRINGS.blank();
                }
                if (!prop.WRITEHANDLER.docString.isEmpty()) {
                    prop.WRITEHANDLER.docString = this.STRINGS.blank();
                }
            }

            // Update the instructions in the read function.
            if (prop.hasReadHandler()) {
                prop.READHANDLER.INSTRUCTIONS.forEach(instr -> {
                    instr.remapVariables(SCHEME);
                });
            }
            // Update the instructions in the write function.
            if (prop.hasWriteHandler()) {
                prop.WRITEHANDLER.INSTRUCTIONS.forEach(instr -> {
                    instr.remapVariables(SCHEME);
                });
            }
        });

        // For each function of each state, give the locals
        // and paramters new names.
        this.STATES.forEach(state -> {
            state.FUNCTIONS.forEach((PexObject.Function func) -> {
                // Strip the docstring.
                if (stripDocs && !func.docString.isEmpty()) {
                    func.docString = this.STRINGS.blank();
                }

                // Fork the token generator to generator local and 
                // parameter names.
                final TokenGenerator FN_GEN = GEN.clone();

                // The general variable name replacement scheme for 
                // function parameters and function local variables. It
                // needs to include the object variable scheme.
                //final Scheme FN_SCHEME = VAR_SCHEME.clone();
                final Scheme FN_SCHEME = SCHEME.clone();

                // Rename the parameters.
                if (remapParams) {
                    func.PARAMS.forEach(param -> {
                        TString newName = this.STRINGS.addString(FN_GEN.next());
                        FN_SCHEME.put(param.name, newName);
                        param.name = newName;
                    });
                }

                // Rename the local variables (skipping the ones that
                // are autogenerated, since they get reused anyway).
                if (remapLocals) {
                    func.LOCALS
                            .stream()
                            .filter(local -> !local.name.matches("::.+"))
                            .forEach(local -> {
                                IString newName = FN_GEN.next();
                                FN_SCHEME.put(local.name, newName);
                                local.name = this.STRINGS.addString(newName);
                            });
                }

                // Update all of the instructions to use the new 
                // identifiers.
                func.INSTRUCTIONS.forEach(instr -> {
                    instr.remapVariables(FN_SCHEME);
                });
            });
        });
    }

    /**
     * Remaps the names of the scripts variables, including object-level
     * variables, function local variables, and function parameters.
     *
     * @param autovarSchemes Uses to retrieve the autovar name replacement
     * schemes from parent classes.
     * @param autovarGens Used to retrieve the autovar TokenGenerators of
     * @param stats Used to record some statistics about what this method does.
     * parent classes.
     */
    public void remapVariables(Map<IString, Scheme> autovarSchemes, Map<IString, TokenGenerator> autovarGens, RemappingStats stats) {
        stats.incScripts();

        // Strip the docstring.
        if (!this.docString.isEmpty()) {
            this.docString = this.STRINGS.blank();
            stats.incDocStrings();
        }

        // Get/create the generator for new identifier names. If the 
        // parent class already has a generator, use that. Otherwise
        // make a new one.
        final TokenGenerator AUTOVAR_GEN;
        if (autovarGens.containsKey(this.PARENTNAME)) {
            final TokenGenerator PARENT_GEN = autovarGens.get(this.PARENTNAME);
            AUTOVAR_GEN = PARENT_GEN.clone();
        } else {
            AUTOVAR_GEN = new TokenGenerator();
        }
        autovarGens.put(this.NAME, AUTOVAR_GEN);

        // Get/create the autovar scheme. If the parent class already has
        // a scheme, use that. Otherwise make a new one.
        final Scheme AUTOVAR_SCHEME;
        if (autovarSchemes.containsKey(this.PARENTNAME)) {
            final Scheme PARENT_SCHEME = autovarSchemes.get(this.PARENTNAME);
            AUTOVAR_SCHEME = PARENT_SCHEME.clone();
        } else {
            AUTOVAR_SCHEME = new Scheme();
        }
        autovarSchemes.put(this.NAME, AUTOVAR_SCHEME);

        // First, rename autovars and store the mapping in the autovarScheme.
        this.PROPERTIES.forEach(prop -> {
            if (prop.hasAutoVar() && this.AUTOVARMAP.containsKey(prop)) {
                // Find the matching object Variable.
                Variable var = this.AUTOVARMAP.get(prop);

                // If it's conditional, don't rename the autovar unless the
                // appropriate flag is set.
                if (!var.isConditional()) {
                    IString newName = AUTOVAR_GEN.next();
                    AUTOVAR_SCHEME.put(prop.autoVarName, newName);
                    prop.autoVarName = this.STRINGS.addString(newName);
                }
            }
        });

        // The general variable name replacement scheme for object 
        // variables. It needs to include the autovar scheme.
        final Scheme VAR_SCHEME = AUTOVAR_SCHEME.clone();

        // Fork the token generator to generate variable names.
        final TokenGenerator VAR_GEN = AUTOVAR_GEN.clone();

        // Give each variable a new name. 
        this.VARIABLES
                .stream()
                .filter(v -> !v.isConditional())
                .forEach(var -> {

                    stats.incObjectVariables();

                    // If this is an autovar, give it the name we assigned earlier.
                    if (AUTOVAR_SCHEME.containsKey(var.name)) {
                        IString newName = AUTOVAR_SCHEME.get(var.name);
                        var.name = this.STRINGS.addString(newName);
                        stats.incAutoVariables();
                    } // If it's not an autovar and it's not excluded, just give it a new name.
                    else {
                        IString newName = VAR_GEN.next();
                        VAR_SCHEME.put(var.name, newName);
                        var.name = this.STRINGS.addString(newName);
                    }
                });

        // Apply the scheme to the object's properties.
        this.PROPERTIES.forEach(prop -> {
            // Remove the docstring.
            if (!prop.docString.isEmpty()) {
                prop.docString = this.STRINGS.blank();
                stats.incDocStrings();
            }

            // Strip docstring and update the instructions in the 
            // read function.
            if (prop.hasReadHandler()) {
                stats.incFunctions();
                prop.READHANDLER.INSTRUCTIONS.forEach(instr -> {
                    instr.remapVariables(VAR_SCHEME);
                });
                if (!prop.READHANDLER.docString.isEmpty()) {
                    prop.READHANDLER.docString = this.STRINGS.blank();
                    stats.incDocStrings();
                }
            }
            // Strip docstring and update the instructions in the 
            // write function.
            if (prop.hasWriteHandler()) {
                stats.incFunctions();
                prop.WRITEHANDLER.INSTRUCTIONS.forEach(instr -> {
                    instr.remapVariables(VAR_SCHEME);
                });
                if (!prop.WRITEHANDLER.docString.isEmpty()) {
                    prop.WRITEHANDLER.docString = this.STRINGS.blank();
                    stats.incDocStrings();
                }
            }
        });

        // For each function of each state, give the locals
        // and paramters new names.
        this.STATES.forEach(state -> {
            state.FUNCTIONS.forEach((PexObject.Function func) -> {
                stats.incFunctions();

                // Strip the docstring.
                if (!func.docString.isEmpty()) {
                    func.docString = this.STRINGS.blank();
                    stats.incDocStrings();
                }

                // Fork the token generator to generator local and 
                // parameter names.
                final TokenGenerator FN_GEN = VAR_GEN.clone();

                // The general variable name replacement scheme for 
                // function parameters and function local variables. It
                // needs to include the object variable scheme.
                //final Scheme FN_SCHEME = VAR_SCHEME.clone();
                final Scheme FN_SCHEME = VAR_SCHEME.clone();

                // Rename the parameters.
                func.PARAMS.forEach(param -> {
                    IString newName = FN_GEN.next();
                    FN_SCHEME.put(param.name, newName);
                    param.name = this.STRINGS.addString(newName);
                    stats.incParameters();
                });

                // Rename the local variables (skipping the ones that
                // are autogenerated, since they get reused anyway).
                func.LOCALS.forEach(local -> {
                    if (!local.name.matches("::.+")) {
                        IString newName = FN_GEN.next();
                        FN_SCHEME.put(local.name, newName);
                        local.name = this.STRINGS.addString(newName);
                        stats.incLocalVariables();
                    }
                });

                // Update all of the instructions to use the new 
                // identifiers.
                func.INSTRUCTIONS.forEach(instr -> {
                    instr.remapVariables(FN_SCHEME);
                });
            });
        });
    }

    /**
     * Collects all of the strings used by the PexObject and adds them to a set.
     *
     * @param strings The set of strings.
     */
    public void collectStrings(Set<TString> strings) {
        strings.add(this.NAME);
        strings.add(this.PARENTNAME);
        strings.add(this.docString);
        strings.add(this.AUTOSTATENAME);
        this.VARIABLES.forEach(var -> var.collectStrings(strings));
        this.PROPERTIES.forEach(prop -> prop.collectStrings(strings));
        this.STATES.forEach(state -> state.collectStrings(strings));
    }

    /**
     * Retrieves a set of the property names in this <code>PexObject</code>.
     *
     * @return A <code>Set</code> of property names.
     *
     */
    public Set<IString> getPropertyNames() {
        return this.PROPERTIES.stream().map(p -> p.name).collect(Collectors.toSet());
    }

    /**
     * Retrieves a set of the variable names in this <code>PexObject</code>.
     *
     * @return A <code>Set</code> of property names.
     *
     */
    public Set<IString> getVariableNames() {
        return this.VARIABLES.stream().map(p -> p.name).collect(Collectors.toSet());
    }

    /**
     * Retrieves a set of the function names in this <code>PexObject</code>.
     *
     * @return A <code>Set</code> of function names.
     *
     */
    public Set<IString> getFunctionNames() {
        final Set<IString> NAMES = new java.util.HashSet<>();
        this.STATES.forEach(state -> state.FUNCTIONS.forEach(func -> NAMES.add(func.getFullName())));
        return NAMES;
    }

    /**
     * Returns a set of UserFlag objects matching a userFlags field.
     *
     * @param userFlags The flags to match.
     * @return The matching UserFlag objects.
     */
    public Set<UserFlag> getFlags(int userFlags) {
        final Set<UserFlag> FLAGS = new java.util.HashSet<>();

        this.USERFLAGDEFS.forEach(flag -> {
            if (flag.matches(userFlags)) {
                FLAGS.add(flag);
            }
        });

        return FLAGS;
    }

    /**
     * @param stats A <code>ScriptStats</code> into which to place the analysis.
     */
    public void analyze(ScriptStats stats) {
        if (!this.docString.isEmpty()) {
            stats.modDocStrings(+1);
        }

        this.PROPERTIES.forEach(prop -> {
            if (!prop.docString.isEmpty()) {
                stats.modDocStrings(+1);
            }
            if (prop.hasReadHandler()) {
                stats.modLocalVariables(prop.READHANDLER.LOCALS.size());
                if (!prop.READHANDLER.docString.isEmpty()) {
                    stats.modDocStrings(+1);
                }
            }
            if (prop.hasWriteHandler()) {
                stats.modLocalVariables(prop.WRITEHANDLER.LOCALS.size());
                if (!prop.WRITEHANDLER.docString.isEmpty()) {
                    stats.modDocStrings(+1);
                }
            }
        });

        this.VARIABLES.forEach(var -> {
            stats.modObjectVariables(+1);

            if (var.isConditional()) {
                stats.modConditionals(+1);
            } else if (this.AUTOVARMAP.containsValue(var)) {
                stats.modAutoVariables(+1);
            }
        });

        // For each function of each state, give the locals
        // and paramters new names.
        this.STATES.forEach(state -> {
            state.FUNCTIONS.forEach((PexObject.Function func) -> {

                if (!func.docString.isEmpty()) {
                    stats.modDocStrings(+1);
                }

                stats.modParameters(func.PARAMS.size());
                stats.modLocalVariables(func.LOCALS.size());
            });
        });
    }

    /**
     * Tries to disassemble the script.
     *
     * @param out The outputstream.
     * @param level Partial disassembly flag.
     */
    public void disassemble(java.io.PrintWriter out, AssemblyLevel level) {
        if (this.PARENTNAME == null) {
            out.printf("ScriptName %s", this.NAME);
        } else {
            out.printf("ScriptName %s extends %s", this.NAME, this.PARENTNAME);
        }

        final Set<UserFlag> FLAGOBJS = this.getFlags(this.USERFLAGS);
        FLAGOBJS.forEach(flag -> out.print(" " + flag));
        out.println();

        if (null != this.docString && !this.docString.isEmpty()) {
            out.printf("{%s}\n", this.docString);
        }

        out.println();

        final Map<Property, Variable> AUTOVARS = new java.util.HashMap<>();
        this.PROPERTIES.stream().filter(p -> p.hasAutoVar()).forEach(p -> {
            this.VARIABLES.stream().filter(v -> v.name.equals(p.autoVarName)).forEach(v -> AUTOVARS.put(p, v));
        });

        this.PROPERTIES.forEach(v -> v.disassemble(out, level, AUTOVARS));
        this.VARIABLES.stream().filter(v -> !AUTOVARS.containsValue(v)).forEach(v -> v.disassemble(out, level));
        this.STATES.forEach(v -> v.disassemble(out, level, this.AUTOSTATENAME.equals(v.NAME), AUTOVARS));
    }

    /**
     * Pretty-prints the PexObject.
     *
     * @return A string representation of the PexObject.
     */
    @Override
    public String toString() {
        StringBuilder buf = new StringBuilder();
        buf.append(String.format("Object %s extends %s %s\n", this.NAME, this.PARENTNAME, getFlags(this.USERFLAGS)));
        buf.append(String.format("\tDoc: %s\n", this.docString));
        buf.append(String.format("\tInitial state: %s\n", this.AUTOSTATENAME));
        buf.append("\n");

        this.PROPERTIES.forEach(prop -> buf.append(prop.toString()));
        this.VARIABLES.forEach(var -> buf.append(var.toString()));
        this.STATES.forEach(state -> buf.append(state.toString()));

        return buf.toString();
    }

    final public TString NAME;
    public int size;
    final public TString PARENTNAME;
    public TString docString;
    final public int USERFLAGS;
    final public TString AUTOSTATENAME;
    final private List<Variable> VARIABLES;
    final private List<Property> PROPERTIES;
    final private List<State> STATES;
    final private Map<Property, Variable> AUTOVARMAP;

    final private List<UserFlag> USERFLAGDEFS;
    final private StringTable STRINGS;

    /**
     * Describes a Property from a PEX file.
     *
     */
    public final class Property {

        /**
         * Creates a Property by reading from a DataInput.
         *
         * @param input A datainput for a Skyrim PEX file.
         * @param strings The <code>StringTable</code> for the <code>Pex</code>.
         * @throws IOException Exceptions aren't handled.
         */
        private Property(DataInput input, StringTable strings) throws IOException {
            this.name = strings.read(input);
            this.TYPE = strings.read(input);
            this.docString = strings.read(input);
            this.USERFLAGS = input.readInt();
            this.FLAGS = input.readByte();

            if (this.hasAutoVar()) {
                this.autoVarName = strings.read(input);
            }

            if (this.hasReadHandler()) {
                this.READHANDLER = new Function(input, false, strings);
            } else {
                this.READHANDLER = null;
            }

            if (this.hasWriteHandler()) {
                this.WRITEHANDLER = new Function(input, false, strings);
            } else {
                this.WRITEHANDLER = null;
            }
        }

        /**
         * Write the this.ct to a <code>DataOutput</code>. No IO error handling
         * is performed.
         *
         * @param output The <code>DataOutput</code> to write.
         * @throws IOException IO errors aren't handled at all, they are simply
         * passed on.
         */
        private void write(DataOutput output) throws IOException {
            this.name.write(output);
            this.TYPE.write(output);
            this.docString.write(output);
            output.writeInt(this.USERFLAGS);
            output.writeByte(this.FLAGS);

            if (this.hasAutoVar()) {
                this.autoVarName.write(output);
            }

            if (this.hasReadHandler()) {
                this.READHANDLER.write(output);
            }

            if (this.hasWriteHandler()) {
                this.WRITEHANDLER.write(output);
            }
        }

        /**
         * Calculates the size of the Property, in bytes.
         *
         * @return The size of the Property.
         *
         */
        public int calculateSize() {
            int sum = 0;
            sum += 2; // name
            sum += 2; // type
            sum += 2; // docstring
            sum += 4; // userflags;
            sum += 1; // flags

            if (this.hasAutoVar()) {
                sum += 2; // autovarname
            }

            if (this.hasReadHandler()) {
                sum += this.READHANDLER.calculateSize();
            }
            if (this.hasWriteHandler()) {
                sum += this.WRITEHANDLER.calculateSize();
            }

            return sum;
        }

        /**
         * Indicates whether the <code>Property</code> is conditional.
         *
         * @return True if the <code>Property</code> is conditional, false
         * otherwise.
         */
        public boolean isConditional() {
            return (this.USERFLAGS & 2) != 0;
        }

        /**
         * Indicates whether the <code>Property</code> is conditional.
         *
         * @return True if the <code>Property</code> is conditional, false
         * otherwise.
         */
        public boolean isHidden() {
            return (this.USERFLAGS & 1) != 0;
        }

        /**
         * Indicates whether the <code>Property</code> has an autovar.
         *
         * @return True if the <code>Property</code> has an autovar, false
         * otherwise.
         */
        public boolean hasAutoVar() {
            return (this.FLAGS & 4) != 0;
        }

        /**
         * Indicates whether the <code>Property</code> has a read handler
         * function or not.
         *
         * @return True if the <code>Property</code> has a read handler, false
         * otherwise.
         */
        public boolean hasReadHandler() {
            return (this.FLAGS & 5) == 1;
        }

        /**
         * Indicates whether the <code>Property</code> has a write handler
         * function or not.
         *
         * @return True if the <code>Property</code> has a write handler, false
         * otherwise.
         */
        public boolean hasWriteHandler() {
            return (this.FLAGS & 6) == 2;
        }

        /**
         * Collects all of the strings used by the Function and adds them to a
         * set.
         *
         * @param strings The set of strings.
         */
        public void collectStrings(Set<TString> strings) {
            strings.add(this.name);
            strings.add(this.TYPE);
            strings.add(this.docString);

            if (this.hasAutoVar()) {
                strings.add(this.autoVarName);
            }

            if (this.hasReadHandler()) {
                this.READHANDLER.collectStrings(strings);
            }

            if (this.hasWriteHandler()) {
                this.WRITEHANDLER.collectStrings(strings);
            }
        }

        /**
         * Generates a qualified name for the object.
         *
         * @return A qualified name.
         */
        public IString getFullName() {
            return IString.format("%s.%s", PexObject.this.NAME, this.name);
        }

        /**
         * Tries to disassemble the script.
         *
         * @param out The outputstream.
         * @param level Partial disassembly flag.
         * @param autovars Map of properties to their autovariable.
         */
        public void disassemble(java.io.PrintWriter out, AssemblyLevel level, Map<Property, Variable> autovars) {
            Objects.requireNonNull(autovars);

            out.printf("%s Property %s", this.TYPE, this.name);

            if (autovars.containsKey(this) || this.hasAutoVar()) {
                assert autovars.containsKey(this);
                assert this.hasAutoVar();
                assert autovars.get(this).name.equals(this.autoVarName);

                final Variable AUTOVAR = autovars.get(this);
                if (AUTOVAR.DATA.getType() != DataType.NONE) {
                    out.print(" = ");
                    out.print(AUTOVAR.DATA);
                }

                out.print(" AUTO");
                final Set<UserFlag> FLAGOBJS = PexObject.this.getFlags(AUTOVAR.USERFLAGS);
                FLAGOBJS.forEach(flag -> out.print(" " + flag.toString()));
            }

            final Set<UserFlag> FLAGOBJS = PexObject.this.getFlags(this.USERFLAGS);
            FLAGOBJS.forEach(flag -> out.print(" " + flag.toString()));
            out.println();

            if (null != this.docString && !this.docString.isEmpty()) {
                out.printf("{%s}\n", this.docString);
            }

            if (this.hasReadHandler()) {
                assert null != this.READHANDLER;
                this.READHANDLER.disassemble(out, level, "GET", autovars, 1);
            }

            if (this.hasWriteHandler()) {
                assert null != this.WRITEHANDLER;
                this.WRITEHANDLER.disassemble(out, level, "SET", autovars, 1);
            }

            if (this.hasReadHandler() || this.hasWriteHandler()) {
                out.println("EndProperty");
            }

            out.println();
        }

        /**
         * Pretty-prints the Property.
         *
         * @return A string representation of the Property.
         */
        @Override
        public String toString() {
            StringBuilder buf = new StringBuilder();
            buf.append(String.format("\tProperty %s %s", this.TYPE.toString(), this.name.toString()));

            if (this.hasAutoVar()) {
                buf.append(String.format(" AUTO(%s) ", this.autoVarName));
            }

            buf.append(getFlags(this.USERFLAGS));

            buf.append(String.format("\n\t\tDoc: %s\n", this.docString));
            buf.append(String.format("\t\tFlags: %d\n", this.FLAGS));

            if (this.hasReadHandler()) {
                buf.append("ReadHandler: ");
                buf.append(this.READHANDLER.toString());
            }

            if (this.hasWriteHandler()) {
                buf.append("WriteHandler: ");
                buf.append(this.WRITEHANDLER.toString());
            }

            buf.append("\n");
            return buf.toString();
        }

        public TString name;
        final public TString TYPE;
        public TString docString;
        final public int USERFLAGS;
        final public byte FLAGS;
        public TString autoVarName;
        final private Function READHANDLER;
        final private Function WRITEHANDLER;
    }

    /**
     * Describes a State in a PEX file.
     *
     */
    public final class State {

        /**
         * Creates a State by reading from a DataInput.
         *
         * @param input A datainput for a Skyrim PEX file.
         * @param strings The <code>StringTable</code> for the <code>Pex</code>.
         * @throws IOException Exceptions aren't handled.
         */
        private State(DataInput input, StringTable strings) throws IOException {
            this.NAME = strings.read(input);

            int numFunctions = input.readUnsignedShort();
            this.FUNCTIONS = new ArrayList<>(numFunctions);
            for (int i = 0; i < numFunctions; i++) {
                this.FUNCTIONS.add(new Function(input, true, strings));
            }
        }

        /**
         * Write the object to a <code>DataOutput</code>.
         *
         * @param output The <code>DataOutput</code> to write.
         * @throws IOException IO errors aren't handled at all, they are simply
         * passed on.
         */
        private void write(DataOutput output) throws IOException {
            this.NAME.write(output);
            output.writeShort(this.FUNCTIONS.size());
            for (Function function : this.FUNCTIONS) {
                function.write(output);
            }
        }

        /**
         * Calculates the size of the State, in bytes.
         *
         * @return The size of the State.
         *
         */
        public int calculateSize() {
            int sum = 0;
            sum += 2; // name
            sum += 2; // array size
            sum += this.FUNCTIONS.stream().mapToInt(v -> v.calculateSize()).sum();
            return sum;
        }

        /**
         * Collects all of the strings used by the State and adds them to a set.
         *
         * @param strings The set of strings.
         */
        public void collectStrings(Set<TString> strings) {
            strings.add(this.NAME);
            this.FUNCTIONS.forEach(function -> function.collectStrings(strings));
        }

        /**
         * Tries to disassemble the script.
         *
         * @param out The outputstream.
         * @param level Partial disassembly flag.
         * @param autostate A flag indicating if this state is the autostate.
         * @param autovars Map of properties to their autovariable.
         */
        public void disassemble(java.io.PrintWriter out, AssemblyLevel level, boolean autostate, Map<Property, Variable> autovars) {
            final Set<IString> RESERVED = new java.util.HashSet<>();
            RESERVED.add(IString.get("GoToState"));
            RESERVED.add(IString.get("GetState"));

            if (null == this.NAME || this.NAME.isEmpty()) {
                out.print(";");
            }

            if (autostate) {
                out.print("AUTO ");
            }

            out.print("State ");
            out.println(this.NAME);
            out.println();

            final int INDENT = (autostate ? 0 : 1);

            this.FUNCTIONS.stream()
                    .filter(f -> !RESERVED.contains(f.name))
                    .forEach(f -> f.disassemble(out, level, null, autovars, INDENT));

            if (null == this.NAME || this.NAME.isEmpty()) {
                out.println(";EndState");
            } else {
                out.println("EndState");
            }

            out.println();
        }

        /**
         * Pretty-prints the State.
         *
         * @return A string representation of the State.
         */
        @Override
        public String toString() {
            StringBuilder buf = new StringBuilder();
            buf.append(String.format("\tState %s\n", this.NAME));
            this.FUNCTIONS.forEach(function -> buf.append(function.toString()));
            return buf.toString();
        }

        final public TString NAME;
        final public List<Function> FUNCTIONS;

    }

    /**
     * Describes a Function and it's code.
     *
     */
    public final class Function {

        /**
         * Creates a Function by reading from a DataInput.
         *
         * @param input A datainput for a Skyrim PEX file.
         * @param named A flag indicating whether to read a named function or a
         * nameless function.
         * @param strings The <code>StringTable</code> for the <code>Pex</code>.
         * @throws IOException Exceptions aren't handled.
         */
        private Function(DataInput input, boolean named, StringTable strings) throws IOException {
            if (named) {
                this.name = strings.read(input);
            } else {
                this.name = null;
            }

            this.RETURNTYPE = strings.read(input);
            this.docString = strings.read(input);
            this.USERFLAGS = input.readInt();
            this.FLAGS = input.readByte();

            int paramsCount = input.readUnsignedShort();
            this.PARAMS = new ArrayList<>(paramsCount);
            for (int i = 0; i < paramsCount; i++) {
                this.PARAMS.add(new VariableType(input, strings));
            }

            int localsCount = input.readUnsignedShort();
            this.LOCALS = new ArrayList<>(localsCount);
            for (int i = 0; i < localsCount; i++) {
                this.LOCALS.add(new VariableType(input, strings));
            }

            int instructionsCount = input.readUnsignedShort();
            this.INSTRUCTIONS = new ArrayList<>(instructionsCount);
            for (int i = 0; i < instructionsCount; i++) {
                this.INSTRUCTIONS.add(new Instruction(input, strings));
            }
        }

        /**
         * Write the object to a <code>DataOutput</code>. No IO error handling
         * is performed.
         *
         * @param output The <code>DataOutput</code> to write.
         * @throws IOException IO errors aren't handled at all, they are simply
         * passed on.
         */
        private void write(DataOutput output) throws IOException {
            if (null != this.name) {
                this.name.write(output);
            }

            this.RETURNTYPE.write(output);
            this.docString.write(output);
            output.writeInt(this.USERFLAGS);
            output.writeByte(this.FLAGS);

            output.writeShort(this.PARAMS.size());
            for (VariableType vt : this.PARAMS) {
                vt.write(output);
            }

            output.writeShort(this.LOCALS.size());
            for (VariableType vt : this.LOCALS) {
                vt.write(output);
            }

            output.writeShort(this.INSTRUCTIONS.size());
            for (Instruction inst : this.INSTRUCTIONS) {
                inst.write(output);
            }
        }

        /**
         * Calculates the size of the Function, in bytes.
         *
         * @return The size of the Function.
         *
         */
        public int calculateSize() {
            int sum = 0;

            if (null != this.name) {
                sum += 2; // name
            }

            sum += 2; // returntype
            sum += 2; // docstring
            sum += 4; // userflags
            sum += 1; // flags
            sum += 6; // array sizes
            sum += this.PARAMS.stream().mapToInt(v -> v.calculateSize()).sum();
            sum += this.LOCALS.stream().mapToInt(v -> v.calculateSize()).sum();
            sum += this.INSTRUCTIONS.stream().mapToInt(v -> v.calculateSize()).sum();

            return sum;
        }

        /**
         * Collects all of the strings used by the Function and adds them to a
         * set.
         *
         * @param strings The set of strings.
         */
        public void collectStrings(Set<TString> strings) {
            if (null != this.name) {
                strings.add(this.name);
            }

            strings.add(this.RETURNTYPE);
            strings.add(this.docString);

            this.PARAMS.forEach(param -> param.collectStrings(strings));
            this.LOCALS.forEach(local -> local.collectStrings(strings));
            this.INSTRUCTIONS.forEach(instr -> instr.collectStrings(strings));

        }

        /**
         * Generates a qualified name for the Function of the form
         * "OBJECT.FUNCTION".
         *
         * @return A qualified name.
         */
        public IString getFullName() {
            if (this.name != null) {
                return IString.format("%s.%s", PexObject.this.NAME, this.name);
            } else {
                return IString.format("%s.()", PexObject.this.NAME);
            }
        }

        /**
         * @return True if the function is global, false otherwise.
         */
        public boolean isGlobal() {
            return (this.FLAGS & 0x01) != 0;
        }

        /**
         * @return True if the function is native, false otherwise.
         */
        public boolean isNative() {
            return (this.FLAGS & 0x02) != 0;
        }

        /**
         * Tries to disassemble the script.
         *
         * @param out The outputstream.
         * @param level Partial disassembly flag.
         * @param nameOverride Provides the function name; useful for functions
         * that don't have a name stored internally.
         * @param autovars A map of properties to their autovars.
         * @param indent The indent level.
         */
        public void disassemble(java.io.PrintWriter out, AssemblyLevel level, String nameOverride, Map<Property, Variable> autovars, int indent) {
            out.print(Disassembler.tab(indent));
            if (null != this.RETURNTYPE && !this.RETURNTYPE.isEmpty() && !this.RETURNTYPE.equals("NONE")) {
                out.print(this.RETURNTYPE);
                out.print(" ");
            }

            if (null != nameOverride) {
                out.printf("Function %s%s", nameOverride, Disassembler.paramList(this.PARAMS));
            } else {
                out.printf("Function %s%s", this.name, Disassembler.paramList(this.PARAMS));
            }

            final Set<UserFlag> FLAGOBJS = PexObject.this.getFlags(this.USERFLAGS);
            FLAGOBJS.forEach(flag -> out.print(" " + flag.toString()));

            if (this.isGlobal()) {
                out.print(" GLOBAL");
            }
            if (this.isNative()) {
                out.print(" NATIVE");
            }

            out.println();

            if (null != this.docString && !this.docString.isEmpty()) {
                out.print(Disassembler.tab(indent + 1));
                out.printf("{%s}\n", this.docString);
            }

            List<VariableType> locals = new ArrayList<>(this.LOCALS);
            List<VariableType> types = new java.util.ArrayList<>(this.PARAMS);
            types.addAll(this.LOCALS);

            TermMap terms = new TermMap();
            autovars.forEach((p, v) -> {
                terms.put(new VData.ID(v.name), new VData.Term(p.name.toString()));
            });

            List<Instruction> block = new ArrayList<>(this.INSTRUCTIONS);

            switch (level) {
                case STRIPPED:
                    Disassembler.preMap(block, locals, types, terms);
                case BYTECODE:
                    block.forEach(v -> {
                        out.print(Disassembler.tab(indent + 1));
                        out.println(v);
                    });
                    break;
                case FULL:
                    try {
                        if (this.getFullName().toString().contains("StartManualDrain")) {
                            int k = 0;
                        }
                        Disassembler.disassemble(out, this.INSTRUCTIONS, locals, types, terms, indent + 1);
                    } catch (Exception|Error ex) {
                        final String MSG = String.format("Error disassembling %s.", this.getFullName());
                        throw new IllegalStateException(MSG);
                    }
            }

            out.print(Disassembler.tab(indent));
            out.println("EndFunction");
            out.println();
        }

        /**
         * Pretty-prints the Function.
         *
         * @return A string representation of the Function.
         */
        @Override
        public String toString() {
            StringBuilder buf = new StringBuilder();

            if (this.name != null) {
                buf.append(String.format("Function %s ", this.name));
            } else {
                buf.append("Function (UNNAMED) ");
            }

            buf.append(this.PARAMS.toString());
            buf.append(String.format(" returns %s\n", this.RETURNTYPE.toString()));
            buf.append(String.format("\tDoc: %s\n", this.docString.toString()));
            buf.append(String.format("\tFlags: %s\n", getFlags(this.USERFLAGS)));
            buf.append("\tLocals: ");
            buf.append(this.LOCALS.toString());
            buf.append("\n\tBEGIN\n");

            this.INSTRUCTIONS.forEach(instruction -> {
                buf.append("\t\t");
                buf.append(instruction.toString());
                buf.append("\n");
            });

            buf.append("\tEND\n\n");

            return buf.toString();
        }

        public TString name;
        final public TString RETURNTYPE;
        public TString docString;
        final public int USERFLAGS;
        final public byte FLAGS;
        final private List<VariableType> PARAMS;
        final private List<VariableType> LOCALS;
        final private List<Instruction> INSTRUCTIONS;

        /**
         * Describes a single executable Instruction in a Function.
         *
         */
        public final class Instruction {

            /**
             * Creates a new Instruction.
             *
             * @param code
             * @param args
             */
            public Instruction(Opcode code, List<VData> args) {
                this.OP = (byte) code.ordinal();
                this.OPCODE = code;
                this.ARGS = new ArrayList<>(args);
            }

            /**
             * Creates an Instruction by reading from a DataInput.
             *
             * @param input A datainput for a Skyrim PEX file.
             * @param strings The <code>StringTable</code> for the
             * <code>Pex</code>.
             * @throws IOException Exceptions aren't handled.
             */
            private Instruction(DataInput input, StringTable strings) throws IOException {
                this.OPCODE = Opcode.read(input);
                this.OP = (byte) this.OPCODE.ordinal();

                if (this.OPCODE.ARGS > 0) {
                    this.ARGS = new ArrayList<>(this.OPCODE.ARGS);
                    for (int i = 0; i < OPCODE.ARGS; i++) {
                        this.ARGS.add(VData.readVariableData(input, strings));
                    }
                } else if (this.OPCODE.ARGS < 0) {
                    this.ARGS = new ArrayList<>(-this.OPCODE.ARGS);
                    for (int i = 0; i < 1 - this.OPCODE.ARGS; i++) {
                        this.ARGS.add(VData.readVariableData(input, strings));
                    }

                    VData count = this.ARGS.get(-this.OPCODE.ARGS);
                    if (!(count instanceof VData.Int)) {
                        throw new IOException();
                    }

                    int numVargs = ((VData.Int) count).getValue();
                    for (int i = 0; i < numVargs; i++) {
                        this.ARGS.add(VData.readVariableData(input, strings));
                    }

                } else {
                    this.ARGS = new ArrayList<>(0);
                }
            }

            /**
             * Write the object to a <code>DataOutput</code>.
             *
             * @param output The <code>DataOutput</code> to write.
             * @throws IOException IO errors aren't handled at all, they are
             * simply passed on.
             */
            private void write(DataOutput output) throws IOException {
                output.writeByte(this.OP);

                for (VData vd : this.ARGS) {
                    vd.write(output);
                }
            }

            /**
             * Calculates the size of the Instruction, in bytes.
             *
             * @return The size of the Instruction.
             *
             */
            public int calculateSize() {
                int sum = 0;
                sum += 1; // opcode
                sum += ARGS.stream().mapToInt(v -> v.calculateSize()).sum();
                return sum;
            }

            /**
             * Collects all of the strings used by the Instruction and adds them
             * to a set.
             *
             * @param strings The set of strings.
             */
            public void collectStrings(Set<TString> strings) {
                this.ARGS.forEach(arg -> arg.collectStrings(strings));
            }

            /**
             * Pretty-prints the Instruction.
             *
             * @return A string representation of the Instruction.
             */
            @Override
            public String toString() {
                final String FORMAT = "%s %s";
                return String.format(FORMAT, this.OPCODE, this.ARGS);
            }

            /**
             * Checks for instruction arguments that are in a replacement
             * scheme, and replaces them.
             *
             * @param scheme The replacement scheme.
             *
             */
            public void remapVariables(Scheme scheme) {
                int firstArg;

                // These five instruction types include identifiers to
                // properties or functions, which are separate 
                // namespaces. We use firstArg to skip over those 
                // identifiers
                switch (this.OPCODE) {
                    case CALLSTATIC:
                        firstArg = 2;
                        break;
                    case CALLMETHOD:
                    case CALLPARENT:
                    case PROPGET:
                    case PROPSET:
                        firstArg = 1;
                        break;
                    default:
                        firstArg = 0;
                        break;
                }

                // Remap identifiers 
                for (int i = firstArg; i < this.ARGS.size(); i++) {
                    VData arg = this.ARGS.get(i);

                    if (arg.getType() == DataType.IDENTIFIER) {
                        VData.ID id = (VData.ID) arg;
                        if (id.getValue().equals("::isresetting_var")) {
                            int k = 0;
                        }
                        if (scheme.containsKey(id.getValue())) {
                            IString newValue = scheme.get(id.getValue());
                            TString newStr = PexObject.this.STRINGS.addString(newValue);
                            id.setValue(newStr);
                        }
                    }
                }
            }

            final public byte OP;
            final public Opcode OPCODE;
            final public List<VData> ARGS;
        }
    }

    /**
     * Describes a PEX file variable entry. A variable consists of a name, a
     * type, user flags, and VData.
     *
     */
    public final class Variable {

        /**
         * Creates a Variable by reading from a DataInput.
         *
         * @param input A datainput for a Skyrim PEX file.
         * @param strings The <code>StringTable</code> for the <code>Pex</code>.
         * @throws IOException Exceptions aren't handled.
         */
        private Variable(DataInput input, StringTable strings) throws IOException {
            this.name = strings.read(input);
            this.TYPE = strings.read(input);
            this.USERFLAGS = input.readInt();
            this.DATA = VData.readVariableData(input, strings);
        }

        /**
         * Write the object to a <code>DataOutput</code>.
         *
         * @param output The <code>DataOutput</code> to write.
         * @throws IOException IO errors aren't handled at all, they are simply
         * passed on.
         */
        private void write(DataOutput output) throws IOException {
            this.name.write(output);
            this.TYPE.write(output);
            output.writeInt(this.USERFLAGS);
            this.DATA.write(output);
        }

        /**
         * Calculates the size of the VData, in bytes.
         *
         * @return The size of the VData.
         *
         */
        public int calculateSize() {
            int sum = 0;
            sum += 2; // name
            sum += 2; // type
            sum += 4; // userflags
            sum += this.DATA.calculateSize();
            return sum;
        }

        /**
         * Collects all of the strings used by the Variable and adds them to a
         * set.
         *
         * @param strings The set of strings.
         */
        public void collectStrings(Set<TString> strings) {
            strings.add(this.name);
            strings.add(this.TYPE);
            this.DATA.collectStrings(strings);
        }

        /**
         * Indicates whether the <code>Property</code> is conditional.
         *
         * @return True if the <code>Property</code> is conditional, false
         * otherwise.
         */
        public boolean isConditional() {
            return (this.USERFLAGS & 2) != 0;
        }

        /**
         * Tries to disassemble the script.
         *
         * @param out The outputstream.
         * @param level Partial disassembly flag.
         */
        public void disassemble(java.io.PrintWriter out, AssemblyLevel level) {
            if (this.DATA.getType() != DataType.NONE) {
                out.printf("%s %s = %s", this.TYPE, this.name, this.DATA);
            } else {
                out.printf("%s %s", this.TYPE, this.name);
            }

            final Set<UserFlag> FLAGOBJS = PexObject.this.getFlags(this.USERFLAGS);
            FLAGOBJS.forEach(flag -> out.print(" " + flag.toString()));
            out.println();
            out.println();
        }

        /**
         * Pretty-prints the Variable.
         *
         * @return A string representation of the Variable.
         */
        @Override
        public String toString() {
            final String FORMAT = "\tVariable %s %s = %s %s\n\n";
            return String.format(FORMAT, this.TYPE, this.name, this.DATA, getFlags(this.USERFLAGS));
        }

        public TString name;
        final public TString TYPE;
        final public int USERFLAGS;
        final public VData DATA;
    }

}
