local Freezablefood = Class(function(self, inst)
    self.inst = inst
	self.isFrozen = false
	self.isInFreezer = false
end)

local FROZENCOMPS = {"freezablefood", "stackable", "inspectable", "inventoryitem", "perishable", "moisturelistener"}

function Freezablefood:GetDebugString()
	return "freezable, frozen:"..self.isFrozen
end

function Freezablefood:IsFrozen()
	return self.isFrozen
end

function Freezablefood:GetCorePrefabName()
	local nBeg, nEnd = self.inst.prefab:find(STRINGS.MODS.LHC.PREPEND)
	nEnd = nEnd or 0
	return self.inst.prefab:sub(nEnd + 1)
end

function Freezablefood:FreezeItem(bDeepFreeze)
	local inst = self.inst
	if not inst.components.freezablefood then --short-circuit if trying to freeze unfreezable item; should never happen
		return inst
	end

	for k in pairs(inst.components) do --remove all components that frozen items don't need; they'll be re-added when the thawed item is spawned
		local keepComp = false
		for i=0,#FROZENCOMPS do
			if (FROZENCOMPS[i] == k) then
				keepComp = true
			end
		end
		if (not keepComp) then
			inst:RemoveComponent(k)
		end
	end

	if not inst.components.perishable then --Perishable is how the thaw mechanic is implemented
		inst:AddComponent("perishable")
	end
	inst.components.perishable:StopPerishing()

	inst.components.freezablefood.isFrozen = true --we're freezing it

	if inst.components.inspectable then
		inst.components.inspectable:SetDescription("This needs to thaw!")
	end

	inst.components.perishable.onperishreplacement = inst.components.freezablefood:GetCorePrefabName() --When finished thawing, replace with original item
	
	if bDeepFreeze then
		inst.components.perishable.perishremainingtime = 9000000 --Prevents items from thawing
		self.isInFreezer = true
	else
		inst.components.perishable.perishtime = TUNING.LCH_THAW_TIME --sets max time without also setting remaining time
		inst.components.perishable.perishremainingtime = math.min(TUNING.LCH_THAW_TIME, inst.components.perishable.perishremainingtime) --handles edge case of using c_give to spawn frozen prefab
		inst.components.perishable:StartPerishing() --isn't in freezer, so thawing should begin
		self.isInFreezer = false
	end
	
	inst.components.inventoryitem:ChangeImageName(inst.components.freezablefood:GetCorePrefabName()) --Tells UI to display same image as original item
	
	inst.prefab = STRINGS.MODS.LHC.PREPEND..inst.components.freezablefood:GetCorePrefabName() --Change prefab name to prevent stacking with non-frozen versions
	
	return inst
end

function Freezablefood:OnSave()
	return {isFrozen = self.isFrozen, isInFreezer = self.isInFreezer} --only need to record whether the item was actually frozen, and whether it is actively thawing (i.e. not in a freezer)
end

function Freezablefood:GetAdjective()
	if self.isFrozen then
		return STRINGS.UI.HUD.LHC_FROZENFOOD --preprends "Frozen " when viewing item in inventory
	end
end

function Freezablefood:OnLoad(data, newents)
	if newents then
		for k,v in pairs(newents) do
			if (v.entity.components.freezablefood and v.data.isFrozen and not v.data.isInFreezer) then
				v.entity.components.freezablefood:FreezeItem(false) --OnLoad will have spawned in unfrozen versions; if we've recorded that they were frozen, re-freeze them
			elseif (v.entity.components.freezablefood and v.data.isFrozen and v.data.isInFreezer) then
				v.entity.components.freezablefood:FreezeItem(true) --if it's in a freezer, we need to deep-freeze it
			end
		end
	else --edge case for different load mechanic when loading cave world
		self.isFrozen = data.isFrozen
		self.isInFreezer = data.isInFreezer
		if self.isFrozen then
			self.inst.components.freezablefood:FreezeItem(self.isInFreezer)
		end
	end
end

return Freezablefood
