﻿using RimWorld;
using Verse;
using Verse.AI;
using System.Collections.Generic;

//For RimWorld Alpha 16

//Note to self: darkness no longer affects shooting accuracy!

namespace HuntingRestricted
{
	public class MarvsHuntWhenSane : WorkGiver_HunterHunt
	{

		const float hungerBlockPercent = 0.4f;
		const float sleepBlockPercent = 0.3f;
		const float accuracyPenaltyBlock = 0.7f;
		const float movementCapacityBlock = 0.5f;
		const int distFar = 25;
		const string terribleSightMsg = "CantHuntTerribleSight";
		const string shootingAccuracy = "ShootingAccuracy";


		private Trait GetNamedTrait(Pawn pawn, string TraitName)
		{

			List<Trait> pawnTraits = pawn.story.traits.allTraits;
			for (int i = 1; i < pawnTraits.Count; i++)
			{
				if (pawnTraits[i].def.defName == TraitName)
				{
					return pawnTraits[i];
				}
			}
			return null;
		}

		public override bool ShouldSkip(Pawn pawn)
		{
			if (!HasHuntingWeapon(pawn))
			{
				return true;
			}

			if (HasShieldAndRangedWeapon(pawn))
			{
				return true;
			}

			bool terribleSight = pawn.health.capacities.GetEfficiency(PawnCapacityDefOf.Sight) < 0.55;

			if (terribleSight)
			{
				pawn.workSettings.SetPriority(WorkTypeDefOf.Hunting, 0);
				Messages.Message(terribleSightMsg.Translate(), MessageSound.RejectInput);
				return true;
			}

			bool night = GenLocalDate.HourOfDay(pawn.Map) < 6 || GenLocalDate.HourOfDay(pawn.Map) > 21;
			//no need to check for eclipses
			//bool eclipse = pawn.Map.mapConditionManager.ConditionIsActive(MapConditionDefOf.Eclipse);
			bool badweather = pawn.Map.weatherManager.CurWeatherAccuracyMultiplier < accuracyPenaltyBlock;
			bool hungry = pawn.needs.food.CurLevelPercentage < hungerBlockPercent;
			bool tired = pawn.needs.rest.CurLevelPercentage < sleepBlockPercent;
			Trait shooterVariant;
			bool triggerHappyHunter = false;
			bool carefulHunter = false;
			shooterVariant = GetNamedTrait(pawn, shootingAccuracy);
			if (shooterVariant != null)
			{
				triggerHappyHunter = shooterVariant.Degree == -1;
				carefulHunter = shooterVariant.Degree == 1;
			}

			//New check. Trigger happy pawns have even less chance of hitting things in bad weather.
			if (badweather && triggerHappyHunter)
			{
				return true;
			}

			//New check. Stops pawns from wasting time hunting at night
			//when about to become hungry or tired, or when weather conditions are bad.
			//
			//Reason: Animals tend to sleep at night, which means they may be lying down.
			//
			//Careful hunters will hunt in bad weather and at night, because they're amazing
			//at shooting things.

			if (night)
			{
				if (hungry || tired)
				{
					return true;
				}

				if (badweather && !carefulHunter)
				{
					return true;
				}
			}

			return false;

		}



		public override bool HasJobOnThing(Pawn pawn, Thing t)
		{
			Pawn pawn2 = t as Pawn;
			//Vanilla checks, but inverted conditions & return values.
			if (pawn2 == null || !pawn2.RaceProps.Animal)
			{
				return false;
			}

			if (!pawn.CanReserve(t, 1))
			{
				return false;
			}

			if (pawn.Map.designationManager.DesignationOn(t, DesignationDefOf.Hunt) == null)
			{
				return false;
			}

			bool night = GenLocalDate.HourOfDay(pawn.Map) < 6 || GenLocalDate.HourOfDay(pawn.Map) > 21;
			//No need to check for eclipses.
			//bool eclipse = pawn.Map.mapConditionManager.ConditionIsActive(MapConditionDefOf.Eclipse);
			bool badSight = pawn.health.capacities.GetEfficiency(PawnCapacityDefOf.Sight) < 0.85;
			bool smallGame = pawn2.BodySize < 0.65;
			bool bigGame = pawn2.BodySize > 1.3;

			Trait shooterVariant;
			bool triggerHappyHunter = false;
			bool carefulHunter = false;
			shooterVariant = GetNamedTrait(pawn, shootingAccuracy);
			if (shooterVariant != null)
			{
				triggerHappyHunter = shooterVariant.Degree == -1;
				carefulHunter = shooterVariant.Degree == 1;
			}

			//New check. Trigger-happy hunters should never try to hunt small game, only medium or big.
			if (triggerHappyHunter && smallGame) return false;

			//New check. If hunting small game at night, need Careful Shooter trait.
			//If hunting medium or small game at night, need good eyesight.
			if (night)
			{
				if (smallGame && !carefulHunter) return false;

				if (badSight && !bigGame) return false;
			}

			IntVec3 distToTarget = pawn.Position - t.Position;
			bool far = distToTarget.LengthManhattan > distFar;
			bool hungry = pawn.needs.food.CurLevelPercentage < hungerBlockPercent;
			bool tired = pawn.needs.rest.CurLevelPercentage < sleepBlockPercent;
			bool slow = pawn.health.capacities.GetEfficiency(PawnCapacityDefOf.Moving) < movementCapacityBlock;

			//New check. Stops pawns which are about to become tired or hungry from
			//going after distant targets.
			if ((hungry || tired) && far)
			{
				return false;
			}

			//New check. Stops pawns which are slow from going after distant targets.
			if (slow && far)
			{
				return false;
			}

			bool hotBlock = pawn.Map.mapTemperature.OutdoorTemp > pawn.SafeTemperatureRange().max;
			bool coldBlock = pawn.Map.mapTemperature.OutdoorTemp < pawn.SafeTemperatureRange().min;

			//New check. Stops pawns from going after targets in uncomfortable temperatures
			//if target is far or pawn is slow.
			if ((hotBlock || coldBlock) && (far || slow))
			{
				return false;
			}

			bool fallout = pawn.Map.mapConditionManager.ConditionIsActive(MapConditionDefOf.ToxicFallout);

			//New check. Stops pawns from going after targets in fallout if target is far or pawn is slow.
			if (fallout && (far || slow))
			{
				return false;
			}

			return true;

		}
	}
}

