# -*- coding:utf-8 -*-

# Speedflow Add-on
# Copyright (C) 2018 Cedric Lepiller aka Pitiwazou & Legigan Jeremy AKA Pistiwique and Stephen Leger
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

# <pep8 compliant>

import bpy
from bpy_extras import view3d_utils
from bpy.types import Operator
from bpy.props import (StringProperty,
                       IntProperty,
                       FloatProperty,
                       BoolProperty,
                       EnumProperty,
                       FloatVectorProperty
                       )
from .utils.text_to_draw import *
from .utils.functions import *
import math
import mathutils
from mathutils import Matrix, Vector


def local_rotate(obj, axis, angle):
    obj.matrix_world @= Matrix.Rotation(math.radians(angle), 4, axis)

class SPEEDFLOW_OT_array(SpeedApi, Operator):
    """
    ARRAY

    CLICK - Add and adjust
    SHIFT - Apply all
    CTRL  - Remove all
    ALT    - Hide all

    Use default exit method
    Does change display of objects
    """
    bl_idname = "speedflow.array"
    bl_label = "Modal Array"
    bl_options = {'REGISTER', 'UNDO', 'GRAB_CURSOR', 'BLOCKING'}

    modal_action : StringProperty(default='count')
    on_curve : BoolProperty(default=False)

    # @classmethod
    # def poll(cls, context):
    #     return SpeedApi.can_add_modifier(context.object, types={'MESH', 'CURVE'})

    def __init__(self):

        SpeedApi.__init__(self)

        # NOTE: les valeurs par défaut sont définies dans les addon prefs
        # idéalement dans un propertyGroup pour chaque modifier
        # et set à la volée à la création du modifier seulement

        self.axis_index = 0
        self.empty_matrix_save = None
        # self.offset_type = self.prefs.array.offset_type

        # xxx_action dicts here make an abstraction level between key and action
        # this way we may change keys without changing code !!
        # keys may contains modifiers in definition like CTRL+ALT+SHIFT+key
        # action_name should be exact modifier parameters names
        # No space allowed between modifiers+key
        # {[CTRL+|ALT+|SHIFT+]key: action_name}

        # Mouse action list (float and int values), allow keyboard values input
        # allow change of alt/shift/ctrl, no setup_modal_keymap here
        self.mouse_actions = self.setup_modal_keymap({
            'S': 'count',
            'L': 'fit_length',
            # 'F': 'constant_offset_displace',
            # 'D': 'relative_offset_displace',
            'J': 'merge_threshold',
            'V': 'strength',

            'X': 'relative_X',
            'Y': 'relative_Y',
            'Z': 'relative_Z',

            'CTRL+X': 'constant_X',
            'CTRL+Y': 'constant_Y',
            'CTRL+Z': 'constant_Z',
        })

        # Generic modifier toggle action list (boolean vars)
        self.toggle_actions = self.setup_modal_keymap({
            'G': 'use_merge_vertices',
            'U': 'use_merge_vertices_cap',
            'F': 'use_object_offset',
            # 'O': 'use_offset_object',
            # 'C': 'use_normal_calculate',

        })

        # "special actions" not changing modifier attributes in a regular way (mouse/toggle)

        self.special_actions = self.setup_modal_keymap({
            'W': 'reset_axis',
            'SHIFT+X': 'new_mod_relative_X',
            'SHIFT+Y': 'new_mod_relative_Y',
            'SHIFT+Z': 'new_mod_relative_Z',

            'CTRL+SHIFT+X': 'new_mod_constant_X',
            'CTRL+SHIFT+Y': 'new_mod_constant_Y',
            'CTRL+SHIFT+Z': 'new_mod_constant_Z',


            'C': 'array_circular',
            'D': 'fit_type',
            'F1': 'set_origin',
            'SHIFT+F1': 'set_origin_1',
            'SHIFT+RIGHT_ARROW': 'show_modifier',
            'SHIFT+LEFT_ARROW': 'hide_modifier',

            'QUOTE': 'show_hide_all_modifiers',
            'NUMPAD_SLASH': 'localview',
            'NUMPAD_PLUS': 'boolean_operation',
            # 'W': 'use_object_offset',
            # 'B': 'count'
            # 'W': 'add_move_ref_object',
            # 'B': 'set_origin',
            # 'ALT+B': 'set_origin_1'
        })

        # Pick/remove objects
        self.pick_actions = self.setup_modal_keymap({
            # 'P': 'curve',  # Profile
            'E': 'start_cap',  # Caps
            'R': 'end_cap',
            'O': 'offset_object',
            'P': 'curve',

        })

        # default actions available for all modifiers based modals
        self.default_actions = self.setup_modal_keymap({
            # {[CTRL+|ALT+|SHIFT+]key: action_name}
            'A': 'apply_modifier',
            'SHIFT+SPACE': 'add_modifier',
            'DEL': 'remove_modifier',
            'BACK_SPACE': 'remove_modifier',

            'H': 'show_viewport',
            'CTRL+H': 'show_hide_keymap',
            'SHIFT+H': 'beginner_mode',

            'UP_ARROW': 'move_modifier_up',
            'DOWN_ARROW': 'move_modifier_down',

            'GRLESS': 'same_modifier_up',
            'RIGHT_ARROW': 'same_modifier_up',
            'SHIFT+GRLESS': 'same_modifier_down',
            'LEFT_ARROW': 'same_modifier_down',

            'CTRL+GRLESS': 'next_modifier_up',
            'CTRL+RIGHT_ARROW': 'next_modifier_up',
            'CTRL+SHIFT+GRLESS': 'next_modifier_down',
            'CTRL+LEFT_ARROW': 'next_modifier_down',

            'SPACE': 'call_pie',
            'ONE': 'keep_wire',
            'TWO': 'show_wire_in_modal',
            'THREE': 'set_xray',
            'FOUR': 'shading_mode',
            'FIVE': 'random',
            'SIX': 'overlays',
            'SEVEN': 'face_orientation',
            'EIGHT' : 'hide_grid',
            'NINE': 'bool_mode',
            'F2': 'active_snap',
            'F3': 'snap_vertex',
            'F4': 'snap_face',
            'F5': 'snap_grid',
            'F6': 'origin_to_selection',
            'F7': 'apply_location',

            'ESC': 'cancel',
            self.key_cancel: 'cancel',
            'SHIFT+%s' % self.key_cancel: 'cancel',
            'CTRL+SHIFT+%s' % self.key_cancel: 'cancel',
            'CTRL+%s' % self.key_cancel: 'cancel',

            'RET': 'confirm',
            'NUMPAD_ENTER': 'confirm',

            self.key_confirm: 'confirm',
            'SHIFT+%s' % self.key_confirm: 'confirm',
            'CTRL+SHIFT+%s' % self.key_confirm: 'confirm',
            'CTRL+%s' % self.key_confirm: 'confirm',

            # 'SPACE': 'confirm'
        })

    # def change_array_name(self, mod):
    #
    #     if self.act_mod.constant_offset_displace[0] and self.act_mod.constant_offset_displace[1] and self.act_mod.constant_offset_displace[2]:
    #         self.act_mod.name = "Array - CO - X Y Z"
    #
    #     elif self.act_mod.constant_offset_displace[0] and self.act_mod.constant_offset_displace[1]:
    #         self.act_mod.name = "Array - CO - X Y"
    #
    #     elif self.act_mod.constant_offset_displace[0] and self.constant_offset_displace.use_axis[2]:
    #         self.act_mod.name = "Array - CO - X Z"
    #
    #     elif self.act_mod.constant_offset_displace[1] and self.act_mod.constant_offset_displace[2]:
    #         self.act_mod.name = "Array - CO - Y Z"
    #
    #     elif self.act_mod.relative_offset_displace[0] and self.act_mod.relative_offset_displace[1] and self.act_mod.relative_offset_displace[2]:
    #         self.act_mod.name = "Array - RO - X Y Z"
    #
    #     elif self.act_mod.relative_offset_displace[0] and self.act_mod.relative_offset_displace[1]:
    #         self.act_mod.name = "Array - RO - X Y"
    #
    #     elif self.act_mod.relative_offset_displace[0] and self.relative_offset_displace.use_axis[2]:
    #         self.act_mod.name = "Array - RO - X Z"
    #
    #     elif self.act_mod.relative_offset_displace[1] and self.act_mod.relative_offset_displace[2]:
    #         self.act_mod.name = "Array - RO - Y Z"
    #
    #     else:
    #         self.act_mod.name = "Array"



    # ----------------------------------------------------------------------------------------------
    # Override store_init_value to handle resolution_u
    # ----------------------------------------------------------------------------------------------

    def store_init_value(self, event, action, index=None):
        # if action == 'resolution_u':
        #     # This is a "special" action not working on modifier itself
        #     # disallow resolution_u action when act_obj is not a curve
        #     if self.act_obj.type == 'CURVE':
        #         value = self.get_property_value(self.act_obj.data, 'resolution_u')
        #     else:
        #         action = 'free'
        # else:
        #     # Those actions work on modifier
        value = self.get_property_value(self.act_mod, action, index)
        # next modal run will handle this action so store action name

        self.last_value = value
        self.init_value = value
        self.mouse_x = event.mouse_x
        return action


    def create_arrow_array_circular(self):

        faces = []
        mesh_data = bpy.data.meshes.new("Arrow - Circular Array")

        verts = [(0, 0, 0), (0, 0, 0.5), (0.1, 0, 0.5), (0, 0, 1)]
        mesh_data.from_pydata(verts, [(0, 1), (1, 2), (2, 3)], faces)

        mesh_data.update()

        self.array_empty = bpy.data.objects.new("Arrow - Circular Array", mesh_data)

        scene = bpy.context.scene
        scene = bpy.context.view_layer.active_layer_collection.collection
        scene.objects.link(self.array_empty)

        self.array_empty.select_set(state=True)
        bpy.context.view_layer.objects.active = self.array_empty

    # ----------------------------------------------------------------------------------------------
    #
    # ----------------------------------------------------------------------------------------------

    def modal(self, context, event):
        axis_list = "XYZ"

        context.area.tag_redraw()

        modal_action = self.modal_action
        act_obj = self.act_obj

        # index = None

        if modal_action == 'free':
            if event.type in {'NUMPAD_1','NUMPAD_3','NUMPAD_7','NUMPAD_5'} or event.ctrl and event.type in {'NUMPAD_1','NUMPAD_3','NUMPAD_7','NUMPAD_5'}:
                return {'PASS_THROUGH'}

        if event.alt:
            return {'PASS_THROUGH'}

        if modal_action == 'free':
            if event.alt and event.type in {'LEFTMOUSE','RIGHTMOUSE','MIDDLEMOUSE'}:
                return {'PASS_THROUGH'}


        if event.type in { 'A', 'B','C','D','E','F','G','H','I','J','K','L','M','N','O','P','Q','R','S','T',
                           'U','V','W','X','Y','Z' 'SHIFT', 'LEFTMOUSE', 'RIGHTMOUSE','CTRL','ALT', 'ONE', 'TWO',
                           'SLASH'} and act_obj.mode == 'EDIT':
            return {'PASS_THROUGH'}

        # VERTEX
        if event.type == 'TAB' and event.value == 'PRESS':
            if context.object != self.act_obj :
                self.modal_action = 'free'
                bpy.ops.object.select_all(action='DESELECT')
                context.view_layer.objects.active = self.act_obj
                self.act_obj.select_set(state=True)


            bpy.ops.wm.tool_set_by_id(name="builtin.select", cycle=False, space_type='VIEW_3D')
            obj_mode = any([obj.mode == 'OBJECT' for obj in self.sel])
            if obj_mode:
                bpy.ops.object.mode_set(mode='EDIT')
                if self.act_obj.type == 'MESH':
                    context.scene.tool_settings.use_mesh_automerge = False
            else:
                bpy.ops.object.mode_set(mode='OBJECT')
                return {'RUNNING_MODAL'}
            self.mode = self.act_obj.mode

        # pass through events
        if event.type == 'MIDDLEMOUSE' or (
                # allow zoom in free mode and for pen mode
                (modal_action == 'free' or self.work_tool != 'mouse') and
                event.type in {'WHEELUPMOUSE', 'WHEELDOWNMOUSE'}):
            return {'PASS_THROUGH'}

        # event ctrl/shift/alt state last time a key was pressed
        event_alt = self.event_alt

        # ---------------------------------
        # Mouse actions
        # ---------------------------------
        if modal_action in self.mouse_actions[event_alt].values():

            if event.unicode in self.input_list:
                self.get_input_value(event.unicode)
                # No need to go further
                return {'RUNNING_MODAL'}

            if event.type in {'LEFT_SHIFT', 'RIGHT_SHIFT'}:
                index = None
                prop = modal_action

                if "relative_" in modal_action:
                    index = self.axis_index
                    prop = "relative_offset_displace"


                elif "constant_" in modal_action:
                    index = self.axis_index
                    prop = "constant_offset_displace"

                if modal_action == 'strength':
                    next = self.get_modifier_down(act_obj, self.act_mod)
                    if next is not None and next.type == 'DISPLACE':
                        value = self.get_property_value(next, modal_action)
                else:
                    value = self.get_property_value(self.act_mod, prop, index)


                # disallow slow_down for integer values
                slow_down = type(value).__name__ != 'int'

                if slow_down:
                    if event.value == 'PRESS':
                        if not self.slow_down:
                            self.mouse_x = event.mouse_x
                            self.last_value = value
                            self.slow_down = True

                    elif event.value == 'RELEASE':
                        self.mouse_x = event.mouse_x
                        self.last_value = value
                        self.slow_down = False



            if event.type in {'MOUSEMOVE', 'WHEELUPMOUSE', 'WHEELDOWNMOUSE'}:

                if event.type == 'MOUSEMOVE':

                    # prevent mouse move in 'mouse' mode (we could allow mouse in both modes)
                    if self.work_tool == 'mouse':
                        return {'PASS_THROUGH'}

                    # relative diff between last action and current
                    delta = event.mouse_x - self.mouse_x

                else:

                    # absolute delta for this step
                    wheel_delta = 1
                    if event.type == 'WHEELDOWNMOUSE':
                        wheel_delta = -wheel_delta

                    # relative diff between last action and current
                    self.wheel_delta += wheel_delta

                    delta = self.wheel_delta * 40 * self.modal_speed

                if self.slow_down:
                    delta /= 8

                pixels = self.get_pixels_from_rna(self.act_mod, modal_action)

                prop = modal_action
                if "relative_" in modal_action:
                    self.axis_index = {"X": 0, "Y": 1, "Z": 2}[modal_action[-1]]
                    prop = "relative_offset_displace"

                if "constant_" in modal_action:
                    self.axis_index = {"X": 0, "Y": 1, "Z": 2}[modal_action[-1]]
                    prop = "constant_offset_displace"



                value = self.limit_value(self.last_value + delta / pixels / 5 , self.act_mod, prop)
                if event.ctrl:
                    value = round(value, 1)

                # # if not has_array_circular:
                # if modal_action == 'relative_offset_displace':
                #     value = self.limit_value(self.last_value + delta / pixels / 20, self.act_mod, modal_action)
                #     if event.ctrl:
                #         value = round(value, 1)
                # #
                # elif modal_action == 'constant_offset_displace':
                #     value = self.limit_value(self.last_value + delta / pixels *4, self.act_mod, modal_action)
                #     if event.ctrl:
                #         value = round(value, 1)
                # else:
                #     value = self.limit_value(self.last_value + delta / pixels / 2, self.act_mod, modal_action)
                #
                #     if event.ctrl:
                #         value = round(value, 1)

                if self.act_mod.offset_object is not None:
                    self.array_empty = self.act_mod.offset_object

                    # if self.array_empty:
                    # if self.array_empty.name.startswith("Empty - "):
                    self.array_empty.matrix_world = self.empty_matrix_save
                    angle = 360 / int(self.act_mod.count)
                    local_rotate(self.array_empty, 'Z', angle)

                index = None
                prop = modal_action

                if "relative_" in modal_action:
                    index = self.axis_index
                    prop = "relative_offset_displace"

                elif "constant_" in modal_action:
                    index = self.axis_index
                    prop = "constant_offset_displace"


                if modal_action == 'strength':
                    next = self.get_modifier_down(act_obj, self.act_mod)
                    if next is not None and next.type == 'DISPLACE':
                        next.strength = value

                else:
                    # Set la valeur
                    self.set_same_modifiers_value(prop, value, index)

        # ---------------------------------
        # Objects picking
        # ---------------------------------

        if modal_action in self.pick_actions[event_alt].values():

            bpy.ops.wm.tool_set_by_id(name="builtin.select", cycle=False, space_type='VIEW_3D')

            if event.type == self.key_select:

                if event.value == 'PRESS':
                    # self.curve = context.active_object
                    # allow to pick objects by selecting through regular blender select key
                    return {'PASS_THROUGH'}

                # Use RELEASE event to get selected object
                # Unlike ray cast method we are able to select curves too !!
                if event.value == 'RELEASE':
                    obj = context.active_object

                    self.curve = obj

                    if obj is not None:
                        obj.select_set(state=False)
                        context.view_layer.objects.active = self.act_obj

                    if obj != self.act_obj:
                        # filter obj by type and action to avoid errors on set ?
                        self.set_same_modifiers_value(modal_action, obj)
                        self.modal_action = 'free'

                act_obj.select_set(state=True)

                # Add Curve Modifier
                if self.act_mod.fit_type == 'FIT_CURVE':
                    for o in self.sel:
                        context.view_layer.objects.active = o
                        o.select_set(state=True)
                        self.curve_mod = self.get_or_create_modifier(o, 'CURVE')
                        self.curve_mod.deform_axis = 'POS_Z'
                        self.curve_mod.object = self.curve
                        bpy.ops.object.modifier_move_up(modifier=self.curve_mod.name)

                act_obj.select_set(state=True)

            if event.type == 'MOUSEMOVE':
                return {'PASS_THROUGH'}




        # ---------------------------------
        # Select another object while modal running
        # ---------------------------------
        # event_alt = (event.ctrl, event.shift, event.alt)
        #
        # if modal_action == "free" and event.type == self.key_select and any(event_alt):
        #     if event.shift:
        #         return self.copy_modifier_if_missing(context, event, types={'MESH'})
        #     elif event.ctrl:
        #         return self.remove_same_modifier(context, event)
        #     else:
        #         action = 'free'
        #         return self.run_modal_with_another_object(context, event, action, types={'MESH'})

        # ---------------------------------
        # Keyboard events, check PRESS state once
        # ---------------------------------

        if event.value == 'PRESS':

            # Store event alternatives
            self.event_alt = event_alt = (event.ctrl, event.shift, event.alt)

            # set default value for action
            action = 'free'

            # ---------------------------------
            # Step 1 : Process events : turn events into actions
            # ---------------------------------

            # ---------------------------------
            # Mouse actions (startup - store current values)
            # ---------------------------------

            if event.type in self.mouse_actions[event_alt].keys() and not self.input:
                # Mouse actions
                action = self.mouse_actions[event_alt][event.type]

                index = None
                next = self.get_modifier_down(act_obj, self.act_mod)

                if action == 'strength' and next is not None and next.type == 'DISPLACE':

                    value = self.get_property_value(next, modal_action)

                    self.last_value = value
                    self.init_value = value
                    self.mouse_x = event.mouse_x

                    self.modal_action = action

                    return {'RUNNING_MODAL'}


                prop = action


                if "relative_" in action:
                    self.axis_index = {"X": 0, "Y": 1, "Z": 2}[action[-1]]
                    index = self.axis_index
                    prop = "relative_offset_displace"

                if "constant_" in action:
                    self.axis_index = {"X": 0, "Y": 1, "Z": 2}[action[-1]]
                    index = self.axis_index
                    prop = "constant_offset_displace"

                self.store_init_value(event, prop, index)

                self.modal_action = action

                return {'RUNNING_MODAL'}

            # ---------------------------------
            # Special actions
            # ---------------------------------

            elif event.type in self.special_actions[event_alt].keys():
                action = self.special_actions[event_alt][event.type]
                # let process actions below once

            elif event.type == 'BACK_SPACE' and self.input:
                self.input = self.input[:-1]
                # No need to process action
                return {'RUNNING_MODAL'}

            # ---------------------------------
            # Default actions
            # ---------------------------------

            elif event.type in self.default_actions[event_alt].keys():
                action = self.default_actions[event_alt][event.type]
                # let process actions below once

            # ---------------------------------
            # Pick / Remove Objects
            # ---------------------------------

            elif event.type in self.pick_actions[event_alt].keys():

                action = self.pick_actions[event_alt][event.type]
                if self.get_property_value(self.act_mod, action) is not None:

                    if self.act_mod.curve == True:
                        curve_mod = self.get_last_modifier_by_type(act_obj, 'CURVE')
                        if curve_mod:
                            self.remove_modifiers_by_type(act_obj, 'CURVE')

                    self.set_same_modifiers_value(action, None)

                else:

                        # self.set_same_modifiers_value(action, None)

                    # next modal run will handle this action so store action
                    self.modal_action = action

                    return {'RUNNING_MODAL'}




                # No need to process action


            # ---------------------------------
            # Toggle actions
            # ---------------------------------

            elif event.type in self.toggle_actions[event_alt].keys():
                action = self.toggle_actions[event_alt][event.type]
                self.toggle_same_modifiers_value(action)
                # No need to process action
                return {'RUNNING_MODAL'}

            # ---------------------------------
            # Step 2 : Process actions
            # ---------------------------------

            # ---------------------------------
            # Confirm
            # ---------------------------------
            if action == 'confirm':

                if self.input:

                    index = None
                    prop = modal_action

                    if "relative_" in modal_action:
                        index = self.axis_index
                        prop = "relative_offset_displace"

                    if "constant_" in modal_action:
                        index = self.axis_index
                        prop = "constant_offset_displace"

                    value = self.input_as_value(self.act_mod, prop)
                    self.set_same_modifiers_value(prop, value, index)

                    self.input = ""

                if modal_action == 'free':
                    self.exit(context)

                    # restore Overlays
                    context.space_data.overlay.show_overlays = self.overlay_state
                    context.space_data.overlay.show_wireframes = self.overlay_wire_state
                    return {'FINISHED'}

                self.modal_action = 'free'

            # ---------------------------------
            # Cancel
            # ---------------------------------
            elif action == 'cancel':

                if self.input:
                    self.input = ""

                if modal_action in self.mouse_actions[event_alt].values():
                    index = None
                    prop = modal_action

                    if "relative_" in modal_action:
                        index = self.axis_index
                        prop = "relative_offset_displace"

                    if "constant_" in modal_action:
                        index = self.axis_index
                        prop = "constant_offset_displace"
                    self.set_same_modifiers_value(prop, self.init_value, index)

                elif modal_action == 'free':
                    self.exit(context)

                    # restore Overlays
                    context.space_data.overlay.show_overlays = self.overlay_state
                    context.space_data.overlay.show_wireframes = self.overlay_wire_state
                    return {'CANCELLED'}

                self.modal_action = 'free'

            # ---------------------------------
            # Add Modifier on Axis
            # ---------------------------------
            elif "new_mod_" in action:
                # print("ta mère")
                axis = action[-1]

                for obj in self.sel:
                    mod = self.add_modifier(obj, self.mod_type)
                    self.same_modifs[obj] = mod
                    if obj.type == 'MESH':
                        self.move_weighted_normals_down(context, obj)

                    # self.change_array_name(obj, mod, action)

                    mod.count = 2
                    mod.constant_offset_displace[:] = (0, 0, 0)
                    mod.relative_offset_displace[:] = (0, 0, 0)

                # Update same modifiers cache
                self.act_mod = self.same_modifs[act_obj]

                index = {"X": 0, "Y": 1, "Z": 2}[axis]
                prop = "relative_offset_displace"
                if "relative_" in action:
                    modal_action = "relative_%s" % (axis)

                elif "constant_" in action:
                    modal_action = "constant_%s" % (axis)
                    prop = "constant_offset_displace"

                self.modal_action = modal_action

                self.store_init_value(event, prop, index)


            # ---------------------------------
            # Add Modifier  (Generic)
            # ---------------------------------
            elif action == 'add_modifier':

                # On peut imaginer une action similaire, pour
                # rajouter les modifiers manquants dans les objets selectionnés

                for obj in self.sel:
                    mod = self.add_modifier(obj, self.mod_type)
                    self.same_modifs[obj] = mod
                    if obj.type == 'MESH':
                        self.move_weighted_normals_down(context, obj)

                    # self.change_array_name(obj, mod, action)

                    mod.count = 2
                    mod.constant_offset_displace[:] = (0,0,0)
                    mod.relative_offset_displace[:] = (0,0,0)



                    # print(mod.constant_offset_displace[:])
                # Update same modifiers cache
                self.act_mod = self.same_modifs[act_obj]



                self.modal_action = 'free'
                # self.store_init_value(event, self.modal_action)


            # ---------------------------------
            # Apply Modifier (Generic)
            # ---------------------------------
            elif action == 'apply_modifier':

                next = self.next_supported_modifier_down(act_obj, self.act_mod)

                for obj, mod in self.same_modifs.items():
                    # this will make object unique !!!
                    self.apply_modifier(context, obj, mod)

                self.run_modal(act_obj, next, call_from_pie=False)
                self.exit(context)
                return {'FINISHED'}

            # ---------------------------------
            # Remove Modifier (Generic)
            # ---------------------------------
            elif action == 'remove_modifier':

                next = self.next_supported_modifier_down(act_obj, self.act_mod)

                for obj, mod in self.same_modifs.items():
                    self.remove_modifier(obj, mod)

                self.run_modal(act_obj, next)
                self.exit(context)

                return {'FINISHED'}

            # ---------------------------------
            # Change Modifier Position (Generic)
            # ---------------------------------
            elif action == 'move_modifier_up':
                for obj, mod in self.same_modifs.items():
                    self.move_modifier_up(context, obj, mod)
                self.set_index_for_companion(self.act_mod)

            elif action == 'move_modifier_down':
                for obj, mod in self.same_modifs.items():
                    self.move_modifier_down(context, obj, mod)
                self.set_index_for_companion(self.act_mod)

            # ---------------------------------
            # Switch Between modifiers (Generic)
            # ---------------------------------
            elif action == 'next_modifier_up':
                res = self.run_modal_up(act_obj, self.act_mod)
                if res:
                    self.exit(context)
                    return {'FINISHED'}

            elif action == 'next_modifier_down':
                res = self.run_modal_down(act_obj, self.act_mod)
                if res:
                    self.exit(context)
                    return {'FINISHED'}

            elif action == 'same_modifier_up':
                self.act_mod = self.get_modifier_up_same_type(act_obj, self.act_mod)

            elif action == 'same_modifier_down':
                self.act_mod = self.get_modifier_down_same_type(act_obj, self.act_mod)

            # ---------------------------------
            # Call pie menu (Generic)
            # ---------------------------------
            elif action == 'call_pie':
                self.exit(context)
                if self.prefs.choose_menu_type == 'pie':
                    bpy.ops.wm.call_menu_pie('INVOKE_DEFAULT', name="SPEEDFLOW_MT_pie_menu")
                else:
                    bpy.ops.wm.call_menu('INVOKE_DEFAULT', name="SPEEDFLOW_MT_simple_menu")
                return {'FINISHED'}

            # ---------------------------------
            # Change Fit Type
            # ---------------------------------
            elif action == 'fit_type':
                value = self.get_next_enum(self.act_mod, action)
                self.set_same_modifiers_value(action, value)

            # ---------------------------------
            # Displace array circular
            # ---------------------------------
            elif action == 'strength':
                next = self.get_modifier_down(act_obj, self.act_mod)
                if next is not None and next.type == 'DISPLACE':
                    # print("next")
                    self.modal_action = 'strength'

            # ---------------------------------
            # Toggle Keymaps (Generic)
            # ---------------------------------
            elif action == 'show_hide_keymap':
                self.prefs.display_keymaps = not self.prefs.display_keymaps

            # ---------------------------------
            # SHIFT + H Toggle Beginner Mode
            # ---------------------------------
            elif action == 'beginner_mode':
                self.prefs.beginner_mode = not self.prefs.beginner_mode

            # ---------------------------------
            # Toggle Show / Hide Modifier (Generic)
            # ---------------------------------
            elif action == 'show_viewport':
                # same modifiers
                self.toggle_same_modifiers_value(action)


# ---------------------------------
# Special actions
# ---------------------------------

            # ---------------------------------
            # Change Boolean operation
            # ---------------------------------
            elif action == 'boolean_operation':
                self.edit_boolean_operation(context, self)

            # ---------------------------------
            # Toggle local view in Modal
            # ---------------------------------
            elif action == 'localview':
                bpy.ops.view3d.localview()

            # ---------------------------------
            # Toggle Hide all modifiers
            # ---------------------------------
            elif action == 'show_hide_all_modifiers':
                self.show_hide_all_modifiers(act_obj, self.act_mod)

            # ---------------------------------
            # Reset Axis
            # ---------------------------------
            if action == 'reset_axis':
                # mod = self.get_or_create_modifier(act_obj, 'ARRAY')

                # for obj in self.sel:
                    # self.same_modifs[obj] = mod
                for obj, mod in self.same_modifs.items():
                    mod.constant_offset_displace[:] = (0, 0, 0)
                    mod.relative_offset_displace[:] = (0, 0, 0)

                    # self.change_array_name(obj, mod, action)

                # Update same modifiers cache
                self.act_mod = self.same_modifs[act_obj]

                self.modal_action = 'free'

            # ---------------------------------
            # Array Circular
            # ---------------------------------
            if action == 'array_circular':

                # has_array_circular = self.get_modifiers_by_type_and_attr(self.act_obj, self.mod_type,
                #                                                          'offset_object',
                #                                                          self._act_mod.relative_offset_displace[0])
                #
                #
                # if not has_array_circular:
                    # for obj, mod in self.same_modifs.items():
                    #     mod.constant_offset_displace[:] = mod.relative_offset_displace[:] = (0,0,0)
                    #     mod.count = 3
                    #     mod.use_object_offset = True
                    #     mod.name = "Circular_Array"
                    #
                    #     displace_mod = self.create_modifier(obj, 'DISPLACE', "Displace")
                    #     displace_mod.mid_level = 0
                    #     displace_mod.direction = 'X'
                    #     if self.prefs.auto_rename_modifiers:
                    #         displace_mod.name = "Displace - %s" % (displace_mod.direction)
                    #
                    #     disp_strength = float(self.act_obj.dimensions[0] * 0.9)
                    #
                    #     displace_mod.strength = disp_strength
                    #     displace_mod.show_expanded = self.show_expanded
                    #     displace_mod.show_in_editmode = True
                    #
                    #     self.move_modifier_before(context, obj, displace_mod, self.act_mod)
                    #
                    #     #creation de l'empty
                    #     bpy.ops.view3d.snap_cursor_to_selected()
                    #     bpy.context.scene.cursor.rotation_euler[0:3] = self.act_obj.rotation_euler[0:3]
                    #     bpy.ops.object.empty_add(type='SINGLE_ARROW', radius=1, align='CURSOR')
                    #     # self.create_arrow_array_circular()
                    #     bpy.ops.view3d.snap_selected_to_cursor(use_offset=False)
                    #
                    #     array_empty = context.active_object
                    #     array_empty.name = "Empty - " + self.act_mod.name
                    #
                    #     self.angle = 360 / self.act_obj.modifiers['Circular_Array'].count
                    #
                    #     array_empty.matrix_world = self.act_obj.matrix_world
                    #     self.empty_matrix_save = self.act_obj.matrix_world
                    #     local_rotate(array_empty, 'Z', self.angle)
                    #
                    #
                    #     mod.offset_object = array_empty
                    #
                    #     context.view_layer.objects.active = self.act_obj
                    #     self.act_obj.select_set(state=True)
                    #     bpy.ops.object.parent_set(type='OBJECT', keep_transform=True)
                    #
                    #     self.act_obj.select_set(state=False)
                    #     context.view_layer.objects.active = array_empty
                    #     array_empty.select_set(state=True)

                has_array_circular = self.get_modifiers_by_type_and_attr(self.act_obj, self.mod_type,
                                                                         'offset_object',
                                                                         self._act_mod.relative_offset_displace[0])

                if not has_array_circular:
                    self.act_mod.constant_offset_displace[:] = self.act_mod.relative_offset_displace[:] = (0, 0, 0)
                    self.act_mod.count = 3
                    self.act_mod.use_object_offset = True
                    self.act_mod.name = "Circular_Array"

                    displace_mod = self.create_modifier(self.act_obj, 'DISPLACE', "Displace")
                    displace_mod.mid_level = 0
                    displace_mod.direction = 'X'
                    if self.prefs.auto_rename_modifiers:
                        displace_mod.name = "Displace - %s" % (displace_mod.direction)

                    disp_strength = float(self.act_obj.dimensions[0] * 0.9)

                    displace_mod.strength = disp_strength
                    displace_mod.show_expanded = self.show_expanded
                    displace_mod.show_in_editmode = True

                    self.move_modifier_before(context, self.act_obj, displace_mod, self.act_mod)

                    # creation de l'empty
                    bpy.ops.view3d.snap_cursor_to_selected()
                    bpy.context.scene.cursor.rotation_euler[0:3] = act_obj.rotation_euler[0:3]
                    bpy.ops.object.empty_add(type='SINGLE_ARROW', radius=1, align='CURSOR')
                    # self.create_arrow_array_circular()
                    bpy.ops.view3d.snap_selected_to_cursor(use_offset=False)

                    array_empty = context.active_object
                    array_empty.name = "Empty - " + self.act_mod.name

                    self.angle = 360 / self.act_obj.modifiers['Circular_Array'].count

                    array_empty.matrix_world = self.act_obj.matrix_world
                    self.empty_matrix_save = self.act_obj.matrix_world
                    local_rotate(array_empty, 'Z', self.angle)

                    self.act_mod.offset_object = array_empty

                    context.view_layer.objects.active = self.act_obj
                    self.act_obj.select_set(state=True)
                    bpy.ops.object.parent_set(type='OBJECT', keep_transform=True)

                    self.act_obj.select_set(state=False)
                    context.view_layer.objects.active = array_empty
                    array_empty.select_set(state=True)
            # ---------------------------------
            # Show modifiers
            # ---------------------------------
            elif action == 'show_modifier':
                for obj in context.selected_objects:
                    to_show = self.get_first_modifier_by_dict(obj, {'show_viewport' : False})
                    if to_show is not None:
                        to_show.show_viewport = True

            # ---------------------------------
            # Hide modifiers
            # ---------------------------------
            elif action == 'hide_modifier':
                for obj in context.selected_objects:
                    to_show = self.get_last_modifier_by_dict(obj, {'show_viewport' : True})
                    if to_show is not None:
                        to_show.show_viewport = False

    # ---------------------------------
    # Shading / Snap
    # ---------------------------------

            # ---------------------------------
            # 1 - Toggle Keep Wire in Modal (Generic)
            # ---------------------------------
            elif action == 'keep_wire':
                self.prefs.keep_wire = not self.prefs.keep_wire

            # ---------------------------------
            # 2 - Toggle wire in the modal (Generic)
            # ---------------------------------
            elif action == 'show_wire_in_modal':

                self.shading_wire_in_modal(context, self)

            # ---------------------------------
            # 3 - Toggle XRAY
            # ---------------------------------
            elif action == 'set_xray':
                self.shading_xray(context, self)

            # ---------------------------------
            # 4 - Toggle shading mode type
            # ---------------------------------
            elif action == 'shading_mode':
                self.shading_mode_type(context, self)

            # ---------------------------------
            # 5 - Toggle Random Mode
            # ---------------------------------
            elif action == 'random':
                self.shading_random_mode(context, self)

            # ---------------------------------
            # 6 - Toggle OVERLAYS
            # ---------------------------------
            elif action == 'overlays':
                self.shading_overlays(context, self)

            # ---------------------------------
            # 7 - Toggle FACE ORIENTATION
            # ---------------------------------
            elif action == 'face_orientation':
                self.shading_face_orientation(context, self)

            # ---------------------------------
            # 8 - Toggle Hide Grid
            # ---------------------------------
            elif action == 'hide_grid':
                self.shading_hide_grid(context, self)

            # ---------------------------------
            # 9 - Toggle Bool Shading Mode
            # ---------------------------------
            elif action == 'bool_mode':
                self.shading_bool_mode(context, self)

            # ---------------------------------
            # F2 - Toggle Activate Snap
            # ---------------------------------
            elif action == 'active_snap':
                self.snap_activate_snap(context, self)

            # ---------------------------------
            # F3 - Toggle Snap Vertex
            # ---------------------------------
            elif action == 'snap_vertex':
                self.snap_vertex(context, self)

            # ---------------------------------
            # F4 - Toggle Snap Face
            # ---------------------------------
            elif action == 'snap_face':
                self.snap_face(context, self)

            # ---------------------------------
            # F5 - Toggle Snap Grid
            # ---------------------------------
            elif action == 'snap_grid':
                self.snap_grid(context, self)

            # ---------------------------------
            # F6 - Origin to selection
            # ---------------------------------
            elif action == 'origin_to_selection':
                self.snap_origin_to_selection(context)

            # ---------------------------------
            # F7 - Toggle Apply Origin
            # ---------------------------------
            elif action == 'apply_location':
                self.snap_origin_to_grid(context)

            # ---------------------------------
            # START Origin
            # ---------------------------------
            if action == 'set_origin':
                self.act_mod.show_viewport = False

                bpy.ops.wm.tool_set_by_id(name="builtin.select", cycle=False, space_type='VIEW_3D')

                self.mode_snap = self.act_obj.mode
                context.scene.tool_settings.use_snap = True
                context.scene.tool_settings.snap_elements = {'VERTEX'}

                bpy.ops.view3d.snap_cursor_to_selected()

                if act_obj.mode != 'OBJECT':
                    bpy.ops.object.mode_set(mode='OBJECT')

                bpy.ops.transform.translate('INVOKE_DEFAULT', cursor_transform=True)

                self.modal_action = 'origin'

            if action == 'set_origin_1':
                self.act_mod.show_viewport = False

                self.mode_snap = self.act_obj.mode
                context.scene.tool_settings.use_snap = True
                context.scene.tool_settings.snap_elements = {'INCREMENT'}
                context.scene.tool_settings.use_snap_grid_absolute = True

                bpy.ops.view3d.snap_cursor_to_selected()

                if act_obj.mode != 'OBJECT':
                    bpy.ops.object.mode_set(mode='OBJECT')

                bpy.ops.transform.translate('INVOKE_DEFAULT', cursor_transform=True)

                self.modal_action = 'origin'

            # ---------------------------------
            # END ORIGIN
            # ---------------------------------
            if modal_action == 'origin':
                bpy.ops.object.origin_set(type='ORIGIN_CURSOR')
                self.act_mod.show_viewport = True

                bpy.ops.object.mode_set(mode=self.mode_snap)
                self.modal_action = 'free'
        return {'RUNNING_MODAL'}



    # ----------------------------------------------------------------------------------------------
    #
    # ----------------------------------------------------------------------------------------------

    def invoke(self, context, event):

        if context.area and context.area.type == 'VIEW_3D':

            # Get modifier type from modal class name
            mod_type = self.mod_type

            # retrieve addon prefs
            prefs = self.prefs

            act_obj = context.object
            # store context mode, selected and active
            curve = self.filter_objects_by_type(context.selected_objects, {'CURVE'})
            mesh = self.filter_objects_by_type(context.selected_objects, {'MESH'})
            self.mode = act_obj.mode
            self.act_obj = act_obj

            self.curve = curve
            self.mesh = mesh
            self.shading_viewport_type = context.space_data.shading.type
            self.overlay_state = context.space_data.overlay.show_overlays
            self.overlay_wire_state = context.space_data.overlay.show_wireframes
            self.shading_random = context.space_data.shading.color_type

            # filter objects from selection, act_obj is part of sel
            self.sel_obj = None


            # On check si il y a des enfants et on les sélectionne tous
            self.hide_unhide(self.act_obj, states=False, select=True)
            for obj in context.selected_objects:
                self.sel.append(obj)


            sel = self.filter_objects_by_type(context.selected_objects, {'MESH', 'CURVE', 'FONT'})
            self.sel = sel
            # self.sel_obj = context.selected_objects[0] if context.selected_objects[0] != self.act_obj and context.selected_objects.type == 'CURVE' else \
            #     context.selected_objects[1]

            # self.curve_selected = ([obj for obj in context.selected_objects if obj.type == 'CURVE'])

            # if self.act_obj == self.curve_selected:
            #     print(self.curve_selected)
            #
            # return {'FINISHED'}

            # if self.act_obj.type == 'CURVE':
            #     self.act_curve = self.act_obj

            # if mesh and curve and self.act_curve:
            #     self.act_curve.select_set(state=False)
            #     sel = self.filter_objects_by_type(context.selected_objects, {'MESH','CURVE','FONT'})
            #     print(sel)

            # for obj in self.sel:
            #     if obj.type == 'MESH':
            #         if self.prefs.shading_smooth:
            #             context.object.data.use_auto_smooth = True
            # if act_obj.name.startswith("Empty - "):
            #     act_obj.select_set(state=False)

            # Screw is not visible in edit mode
            if act_obj.mode != 'OBJECT':
                bpy.ops.object.mode_set(mode="OBJECT")

            # Set Curve as Ref for array curve
            self.use_curve = False
            self.curve = None
            if self.act_obj.type == 'CURVE' and (self.act_obj.data.extrude and self.act_obj.data.bevel_depth) == 0:
                self.use_curve = True
                self.curve = self.act_obj

                self.act_obj.select_set(state=False)
                self.sel.remove(self.act_obj)

                for obj in self.sel:
                    obj.select_set(state=True)
                    context.view_layer.objects.active = obj

                    act_obj = context.object
                    self.act_obj = act_obj

            # flag indicate call method, through pie menu or object selection change or <-> with specified modifier
            call_through_pie = self.call_from_pie
            # flag when call through selection change using alt key, prevent special actions
            allow_special_actions = self.modifier_index == -2

            # Find if we do have a modifier of this modal type
            has_mod = self.has_modifier_of_type(act_obj, mod_type)
            if has_mod and call_through_pie and self.work_tool == 'pen' and prefs.default_modal_action:
                self.modal_action = 'free'

            if call_through_pie:
                act_mod = self.get_or_create_modifier(act_obj, mod_type)
                # Array Curve
                if self.use_curve:
                    act_mod.fit_type = 'FIT_CURVE'
                    act_mod.curve = self.curve
                    # mod.use_merge_vertices = True
                    mod_curve = self.add_modifier(act_obj, 'CURVE')
                    mod_curve.object = self.curve
                # act_mod.relative_offset_displace[0] = 1
                # act_mod.relative_offset_displace[1] = 0
                # act_mod.relative_offset_displace[2] = 0
                # act_mod.constant_offset_displace[0] = 0
                # act_mod.constant_offset_displace[1] = 0
                # act_mod.constant_offset_displace[2] = 0
                # act_mod.name = "Array - %s" % act_mod.fit_type


                # if mesh and curve:
                #     act_mod.fit_type = 'FIT_CURVE'
                #     act_mod.curve = self.curve_selected
                #
                #     curve_mod = self.get_or_create_modifier(act_obj, 'CURVE')
                #     curve_mod.object = self.curve_selected
                #     curve_mod.deform_axis = 'POS_X'
                #     curve_mod.name = "Curve - %s" % curve_mod.deform_axis


                # self.change_array_name()
            else:
                # use specified act_mod
                act_mod = self.get_modifier_by_index(act_obj, self.modifier_index)

            # if children:
            #     self.mod.relative_offset_displace[:] = (0, 0, 0)
            #     self.mod.constant_offset_displace[:] = (5, 0, 0)
            #     self.mod.count = 2

            # specials actions
            special = (event.shift, event.ctrl, event.alt)

            if allow_special_actions and any(special):

                if check_valid_input(special):

                    for obj in sel:
                        fun, args = {
                            (True, False, False): [self.apply_modifiers_by_type, (context, obj, mod_type)],
                            (False, True, False): [self.remove_modifiers_by_type, (obj, mod_type)],
                            (False, False, True): [self.toggle_modifiers_visibility_by_type, (obj, mod_type)],
                        }[special]

                        # hi hi hi, have some fun
                        fun(*args)

                    bpy.ops.object.mode_set(mode=self.mode)

                    return {'FINISHED'}

            # if len(context.selected_objects) == 2:
            #     curve_obj = False
            #     if any([context.object.type == 'CURVE']):
            #         curve_obj = True
            #     print(curve_obj)
            #
            #     return {'CANCELLED'}



            # Add modifiers to selected objects
            for obj in sel:

                #Apply Scale to avoid feformations
                # if obj.data.users == 1:
                #     if prefs.auto_apply_scale:
                #         self.apply_transform_scale(context, act_obj)
                # find existing modifier with same data than act_mod
                self.mod = self.find_same_in_list(act_mod, obj.modifiers)

                # when not found, add a new one
                if self.mod is None:
                    # condition to add a modifier on selected objects:
                    # if we are browsing act_obj stack using <->
                    # and no similar modifier is found then do nothing (ignore object)

                    # if we call through pie, and no modifier of this kind
                    # is found then add a new one
                    if call_through_pie and not self.has_modifier_of_type(obj, mod_type):
                        mod = self.add_modifier(obj, mod_type)

                        # Array Curve
                        if self.use_curve:
                            mod.fit_type = 'FIT_CURVE'
                            mod.curve = self.curve
                            # mod.use_merge_vertices = True

                            mod_curve = self.add_modifier(obj, 'CURVE')
                            mod_curve.object = self.curve
                            # if has_children:
                            #     mod.relative_offset_displace[:2] = 0
                            #     mod.constant_offset_displace[:2] = 0

                        # self.change_array_name()
                        # selection may change between 2 pie calls
                        # then maybe act_obj already hold a non default modifier
                        # so we copy params from it
                        if has_mod:
                            self.copy_modifier_params(act_mod, mod)


                        #     self.modal_action = 'constant_offset_displace'
                        #     mod.constant_offset_displace[0] = 1
                        #     mod.relative_offset_displace[0] = 0

                            # for child in self.act_obj.children:
                            #     child.constant_offset_displace[0] = 1
                            #     child.relative_offset_displace[0] = 0

                    # subsequent modifiers must be added through modal commands

                # if self.mod is not None:

                if obj.type == 'MESH':
                    self.move_weighted_normals_down(context, obj)

            # Store same modifiers in cache
            self.act_mod = act_mod

            # for obj, mod in self.same_modifs.items():
            #     mod.constant_offset_displace[:] = (0, 0, 0)
            #     mod.relative_offset_displace[:] = (0, 0, 0)

            # self.change_array_name(self.act_obj, self.act_mod, self.modal_action)

            # self.change_array_name()

            # Rotation locale de l'empty
            # if self.act_mod.name == 'Circular_Array':
            if self.act_mod.name.startswith("Circular_"):
            # has_array_circular = self.get_modifiers_by_type_and_attr(self.act_obj, self.mod_type,
            #                                                              'offset_object',
            #                                                              self._act_mod.relative_offset_displace[0])
            # if has_array_circular:
                self.empty_obj = self.act_mod.offset_object
                self.empty_matrix_save = self.act_obj.matrix_world.copy()
                # angle = 360 / self.act_obj.modifiers['Circular_Array'].count
                # self.local_rotate(self.empty_obj, 'Z', angle * -1)
                # self.empty_matrix_save = self.empty_obj.matrix_world.copy()
                # self.local_rotate(self.empty_obj, 'Z', angle)

            # store initial value for mouse action to restore on exit and last to change mouse steps on the fly


            if self.modal_action in self.mouse_actions[special].values():
                self.store_init_value(event, self.modal_action)




            # # Check if face, if not edit the modifier
            # if not self.act_obj.data.polygons:
            #     self.mod.screw_offset = 1
            #     self.mod.angle = 0
            #     self.mod.steps = 1
            #     self.mod.render_steps = 1
            #     self.modal_action = 'free'


            # Setup display of objects
            display_type = None
            if prefs.show_wire_in_modal and act_obj.display_type == 'BOUNDS':
                display_type = 'WIRE'

            if prefs.show_wire_in_modal:
                context.space_data.overlay.show_overlays = True
                context.space_data.overlay.show_wireframes = False

            self.setup_display(display_type)

            # setup draw handler
            self.setup_draw_handler(context)

            context.window_manager.modal_handler_add(self)

            return {'RUNNING_MODAL'}

        else:
            self.report({'WARNING'}, "View3D not found, cannot run operator")
            return {'CANCELLED'}

def register():
    try:
        bpy.utils.register_class(SPEEDFLOW_OT_array)
    except:
        print("Array already registred")

def unregister():
    bpy.utils.unregister_class(SPEEDFLOW_OT_array)