# -*- coding:utf-8 -*-

# Speedflow Add-on
# Copyright (C) 2018 Cedric Lepiller aka Pitiwazou & Legigan Jeremy AKA Pistiwique and Stephen Leger
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

# <pep8 compliant>

import bmesh
import bpy
from math import pi
from bpy.types import Operator
from bpy.props import StringProperty, FloatVectorProperty
from .utils.text_to_draw import *
from .utils.functions import *


class SPEEDFLOW_OT_bevel(SpeedApi, Operator):
    """
    BEVEL

    CLICK - Add and adjust
    SHIFT - act_obj_mode = Apply / EDT_mode = Add bevel weight
    CTRL  - act_obj_mode = Remove / EDT_mode = Remove bevel_weight
    ALT    - Hide all

    Use custom exit method
    Does change display of objects
    """
    bl_idname = "speedflow.bevel"
    bl_label = "Modal Bevel"
    bl_options = {'REGISTER', 'UNDO', 'GRAB_CURSOR', 'BLOCKING'}


    # Default action for this modal
    # prefs = self.get_addon_preferences()
    # if prefs.default_modal_action:
    #     modal_action : StringProperty(default='free')
    # else:
    modal_action: StringProperty(default='width')


    @classmethod
    def poll(cls, context):
        return context.object and context.object.type in {'MESH', 'CURVE', 'FONT'}

    def __init__(self):

        SpeedApi.__init__(self)

        # Mouse action list (float and int values), allow keyboard values input
        # allow change of alt/shift/ctrl, no setup_modal_keymap here
        self.mouse_actions = {
            "S": "width",
            "D": "segments",
            "F": "profile",
            "X": "angle_limit",
        }

        self.toggle_actions = self.setup_modal_keymap({
            "J": "loop_slide",
            "L": "use_clamp_overlap",

        })

        self.special_actions = self.setup_modal_keymap({
            "E": "destructive_mode",
            "G": "limit_method",
            "K": "use_only_vertices",
            "T": "offset_type",
            "Q": "auto_update",
            "C": "vertex_group",
            'CTRL+C': 'clean_vertex_group',
            "B": "miter_outer",
            "M": "offset_type",
            "N": "miter_inner",
            'F1': 'set_origin',
            'SHIFT+F1': 'set_origin_1',
            'SHIFT+RIGHT_ARROW': 'show_modifier',
            'SHIFT+LEFT_ARROW': 'hide_modifier',
            'QUOTE': 'show_hide_all_modifiers',
            'NUMPAD_SLASH': 'localview',
            'NUMPAD_PLUS': 'boolean_operation',

        })

        # default actions available for all modifiers based modals
        self.default_actions = self.setup_modal_keymap({
            # {[CTRL+|ALT+|SHIFT+]key: action_name}
            'A': 'apply_modifier',
            'SHIFT+SPACE': 'add_modifier',
            'DEL': 'remove_modifier',
            'BACK_SPACE': 'remove_modifier',

            'H': 'show_viewport',
            'CTRL+H': 'show_hide_keymap',
            'SHIFT+H': 'beginner_mode',

            'UP_ARROW': 'move_modifier_up',
            'ALT+SHIFT+GRLESS': 'move_modifier_up',
            'DOWN_ARROW': 'move_modifier_down',
            'ALT+GRLESS': 'move_modifier_down',

            'GRLESS': 'same_modifier_up',
            'RIGHT_ARROW': 'same_modifier_up',
            'SHIFT+GRLESS': 'same_modifier_down',
            'LEFT_ARROW': 'same_modifier_down',

            'CTRL+GRLESS': 'next_modifier_up',
            'CTRL+RIGHT_ARROW': 'next_modifier_up',
            'CTRL+SHIFT+GRLESS': 'next_modifier_down',
            'CTRL+LEFT_ARROW': 'next_modifier_down',

            'SPACE': 'call_pie',
            'ONE': 'keep_wire',
            'TWO': 'show_wire_in_modal',
            'THREE': 'set_xray',
            'FOUR': 'shading_mode',
            'FIVE': 'random',
            'SIX': 'overlays',
            'SEVEN': 'face_orientation',
            'EIGHT' : 'hide_grid',
            'NINE': 'bool_mode',
            'F2': 'active_snap',
            'F3': 'snap_vertex',
            'F4': 'snap_face',
            'F5': 'snap_grid',
            'F6': 'origin_to_selection',
            'F7': 'apply_location',

            'ESC': 'cancel',
            self.key_cancel: 'cancel',
            'SHIFT+%s' % self.key_cancel: 'cancel',
            'CTRL+SHIFT+%s' % self.key_cancel: 'cancel',
            'CTRL+%s' % self.key_cancel: 'cancel',

            'RET': 'confirm',
            'NUMPAD_ENTER': 'confirm',

            self.key_confirm: 'confirm',
            'SHIFT+%s' % self.key_confirm: 'confirm',
            'CTRL+SHIFT+%s' % self.key_confirm: 'confirm',
            'CTRL+%s' % self.key_confirm: 'confirm',

            # 'SPACE': 'confirm'
        })

        self.auto_update = self.prefs.bevel.auto_update
        self.destructive_mode = self.prefs.bevel.destructive_mode

        self.first_bevel = False

        # Enable bevel weight mode
        self.use_bevel_weight = False
        self.clamp_bevel = False

        # self.same_modifs = {object:modifier}
        # {width:modif.width ,selected:set(), unselected:set()}
        self.edges_datas = {'width': 0, 'selected': {}, 'unselected': {}}
        # {object:mesh}
        self.mesh_backup = {}

    # ----------------------------------------------------------------------------------------------
    #
    # ----------------------------------------------------------------------------------------------

    def select_edges_boundary_loop(self, bm):
        print("select_edges_boundary_loop")
        for ed in bm.edges:
            ed.select = ed.select and len([f for f in ed.link_faces if f.select]) == 1

    # ----------------------------------------------------------------------------------------------
    #
    # ----------------------------------------------------------------------------------------------

    def setup_edges_bevel_weight(self, context, obj, subdiv, value, all_edges):
        """ Setup the weight, crease and sharp depending on the bevel mode """
        print("setup_edges_bevel_weight")
        me = obj.data
        bm = bmesh.new()
        bm.from_mesh(me)



        is_initialized = bm.edges.layers.bevel_weight.get('BevelWeight')

        # to be sure that the Bevel modifier was initialized
        if not is_initialized:
            bm.free()
            del bm

            self.clear_edges_weights(context, obj)

            me = obj.data
            bm = bmesh.new()
            bm.from_mesh(me)

        bm.edges.index_update()
        bm.edges.ensure_lookup_table()

        if all_edges:
            edges = (ed.index for ed in bm.edges)
        else:
            edges = self.get_sharp_edges_index(bm, 30)

        bm.free()
        # del bm

        ode = obj.data.edges

        for idx in edges:
            ode[idx].bevel_weight = value



        # print("bevel_weight value:",value)
        #
        # if not subdiv:
        #
        #     sharp_value = value
        #     if value == -1:
        #         sharp_value = 0
        #
        #
        #     for idx in edges:
        #         ode[idx].crease = value
        #         ode[idx].use_edge_sharp = sharp_value

        # print("sharp_value:", value)


    # ----------------------------------------------------------------------------------------------
    # Manage SpeedFlow Bevel tag
    # ----------------------------------------------------------------------------------------------

    # TODO: add this flag on obj.data instead, with direct access !! so instances are up to date
    @staticmethod
    def tag_object_with_destructive_mode(obj, destructive_mode=None):

        if obj.get('SpeedFlow') is None:
            obj['SpeedFlow'] = {}

        if destructive_mode is not None:
            obj['SpeedFlow']['Bevel'] = destructive_mode
        # if destructive_mode == 'DESTRUCTIVE':
        #     obj['SpeedFlow']['Bevel'] = destructive_mode

        if 'Bevel' not in obj['SpeedFlow']:
            subdiv = any([ed for ed in obj.data.edges if ed.bevel_weight and ed.crease])
            obj['SpeedFlow']['Bevel'] = not subdiv

    @staticmethod
    def has_object_destructive_tag(obj):
        return obj.get('SpeedFlow') and 'Bevel' in obj['SpeedFlow']

    def is_object_in_destructive_mode(self, obj):
        return self.has_object_destructive_tag(obj) and obj['SpeedFlow']['Bevel']

    # ----------------------------------------------------------------------------------------------
    #
    # ----------------------------------------------------------------------------------------------

    def edit_bevel_weight(self, bm, me, value):
        """ Add or Remove bevel weight on selected edges """
        # print("edit_bevel_weight")
        self.edit_edges_layer_weight(bm, 'bevel_weight', value)

        if not self.is_object_in_destructive_mode(self.act_obj):
            self.edit_edges_layer_weight(bm, 'crease', value)
            self.edit_edges_smooth(bm, value < 0)

        bmesh.update_edit_mesh(me)

    # ----------------------------------------------------------------------------------------------
    #
    # ----------------------------------------------------------------------------------------------

    def bevel_weight_from_active(self, bm, me):
        """ Set weight value on selected edges from active edge weight value """
        # print("bevel_weight_from_active")

        self.edge_weight_from_active(bm, 'bevel_weight')
        bmesh.update_edit_mesh(me)

    # ----------------------------------------------------------------------------------------------
    #
    # ----------------------------------------------------------------------------------------------

    # def clean_useless_bevel(self, obj):
    #     """ Remove the weight, crease and sharp on the edges less than 30 degrees """
    #     print("clean_useless_bevel")
    #
    #     me = obj.data
    #     bm = bmesh.new()
    #     bm.from_mesh(me)
    #     edges = self.get_smooth_edges_index(bm, 30)
    #     bm.free()
    #     del bm
    #     ode = obj.data.edges
    #     for idx in edges:
    #         e = ode[idx]
    #         e.use_edge_sharp = False
    #         e.bevel_weight = -1
    #         e.crease = -1

    # ----------------------------------------------------------------------------------------------
    #
    # ----------------------------------------------------------------------------------------------

    def restore_values(self):
        """ Restore the previous values """
        action = self.modal_action
        if action != 'free':
            if self.use_bevel_weight:
                if action == 'width':
                    self.apply_width(self.init_value)
                else:
                    self.set_same_modifiers_value(action, self.init_value)
                # Certain de ça ??? -> no, apply width already do it
                # self.reset_edges_weight()
            else:
                self.set_same_modifiers_value(action, self.init_value)

    # ----------------------------------------------------------------------------------------------
    #
    # ----------------------------------------------------------------------------------------------

    def apply_width(self, value):
        """Set edges weight and modifier.width to match value
        :param value:
        :return:
        """
        sel_edges = self.edges_datas['selected']

        if value == 0:
            for ed in sel_edges:
                ed.bevel_weight = -1
                ed.crease = -1
                ed.use_edge_sharp = False
        else:
            # reduction ratio for weights so bevel dosent change
            # allow a ratio up to 1/10k in bevels
            uns_edges = self.edges_datas['unselected']

            # modifier.width is max(uns_width, desired)
            width = value

            if value < self.edges_datas['width']:
                width = self.edges_datas['width']

            ratio = self.edges_datas['width'] / width
            weight = value / width

            setattr(self.act_mod, 'width', width)

            to_clamp = any([val * ratio <= 0.0001 for val in uns_edges.values()])
            if not to_clamp:
                for ed, val in uns_edges.items():
                    ed.bevel_weight = val * ratio

            for ed in sel_edges.keys():
                ed.bevel_weight = weight

    def apply_input(self):
        """ Apply the input value """

        action = self.modal_action
        value = self.input_as_value(self.act_mod, action)

        if action == 'width' and self.act_mod.limit_method == 'WEIGHT':
            self.apply_width(value)

        else:
            self.set_same_modifiers_value(action, value)

    # ----------------------------------------------------------------------------------------------
    # Apply all modifs, kept for future use ??
    # ----------------------------------------------------------------------------------------------

    # def apply_all_bevels(self, context):
    #     """ Apply the Bevels modifiers on selected objects """
    #     for obj, modifs in self.same_modifs.items():
    #         for mod in modifs:
    #             if mod.limit_method == 'WEIGHT':
    #                 # self.setup_edges_bevel_weight(context, obj, 0, -1, True)
    #                 if self.has_object_destructive_tag(obj):
    #                     del obj['SpeedFlow']['Bevel']
    #                 break
    #
    #         self.apply_modifiers_by_type(context, obj, 'BEVEL')
    #
    #     bpy.context.view_layer.objects.active = self.act_obj

    # ----------------------------------------------------------------------------------------------
    # Remove all modifs, kept for future use ??
    # ----------------------------------------------------------------------------------------------
    #
    # def remove_all_bevels(self, context):
    #     """ Remove the Bevels modifiers on selected objects """
    #     for obj, modifs in self.same_modifs.items():
    #         for mod in modifs:
    #             if mod.limit_method == 'WEIGHT':
    #                 self.setup_edges_bevel_weight(context, obj, 0, -1, True)
    #                 if self.has_object_destructive_tag(obj):
    #                     del obj['SpeedFlow']['Bevel']
    #                 break
    #
    #         self.remove_modifiers_by_type(obj, 'BEVEL')
    #
    #     self.shading_mode(context, self.act_obj, 'FLAT')

    # ----------------------------------------------------------------------------------------------
    #
    # ----------------------------------------------------------------------------------------------

    def exit(self, context):
        """ Finalization before modal output """

        # clean up backup mesh
        if self.act_obj.type == 'MESH':
            for obj, me in self.mesh_backup.items():
                bpy.data.meshes.remove(me)

            self.mesh_backup.clear()
            self.edges_datas.clear()

        # call parent's exit method
        SpeedApi.exit(self, context)


    def store_edges_weight(self, obj, mod):
        """Store edges weight for object as {edge: backup_weight}
        in a dict {selected, unselected}
        :param obj: blender object
        :return:
        """
        sel_edges = {}
        uns_edges = {}
        for ed in obj.data.edges:
            if ed.select:
                sel_edges[ed] = ed.bevel_weight
            else:
                if ed.bevel_weight > 0:
                    uns_edges[ed] = ed.bevel_weight

        self.edges_datas = {
            'width': mod.width,
            'selected': sel_edges,
            'unselected': uns_edges
        }

    def reset_edges_weight(self):
        """ Restore edge weights in backup state
        :return:
        """
        # print("reset_edges_weight")
        for key in ('selected', 'unselected'):
            for ed, weight in self.edges_datas[key]:
                ed.bevel_weight = weight

    # ----------------------------------------------------------------------------------------------
    #
    # ----------------------------------------------------------------------------------------------

    def modal(self, context, event):

        context.area.tag_redraw()

        modal_action = self.modal_action
        act_obj = self.act_obj
        act_mod = self.act_mod
        # self.mode = self.act_obj.mode

        if modal_action == 'free':
            if event.type in {'NUMPAD_1','NUMPAD_3','NUMPAD_7','NUMPAD_5'} or event.ctrl and event.type in {'NUMPAD_1','NUMPAD_3','NUMPAD_7','NUMPAD_5'}:
                return {'PASS_THROUGH'}

        if event.alt:
            return {'PASS_THROUGH'}

        if modal_action == 'free':
            if event.alt and event.type in {'LEFTMOUSE','RIGHTMOUSE','MIDDLEMOUSE'}:
                return {'PASS_THROUGH'}


        if event.type in { 'A', 'B','C','D','E','F','G','H','I','J','K','L','M','N','O','P','Q','R','S','T',
                           'U','V','W','X','Y','Z' 'SHIFT', 'LEFTMOUSE', 'RIGHTMOUSE','CTRL','ALT', 'ONE', 'TWO',
                           'SLASH'} and act_obj.mode == 'EDIT':
            return {'PASS_THROUGH'}

        # VERTEX
        if event.type == 'TAB' and event.value == 'PRESS':
            bpy.ops.wm.tool_set_by_id(name="builtin.select", cycle=False, space_type='VIEW_3D')
            obj_mode = any([obj.mode == 'OBJECT' for obj in self.sel])
            if obj_mode:
                bpy.ops.object.mode_set(mode='EDIT')
                if self.act_obj.type == 'MESH':
                    context.scene.tool_settings.use_mesh_automerge = False
            else:
                bpy.ops.object.mode_set(mode='OBJECT')
                return {'RUNNING_MODAL'}
            self.mode = self.act_obj.mode


        # pass through events
        if event.type == 'MIDDLEMOUSE' or (
                # allow zoom in free mode and for pen mode
                (modal_action == 'free' or self.work_tool != 'mouse') and
                event.type in {'WHEELUPMOUSE', 'WHEELDOWNMOUSE'}):
            return {'PASS_THROUGH'}

        # event ctrl/shift/alt state last time a key was pressed
        event_alt = self.event_alt

        # ---------------------------------
        # Mouse actions
        # ---------------------------------
        if modal_action in self.mouse_actions.values():

            # mouse action enabled
            if event.unicode in self.input_list:
                self.get_input_value(event.unicode)
                # No need to go further
                return {'RUNNING_MODAL'}

            if event.type in {'LEFT_SHIFT', 'RIGHT_SHIFT'}:

                # Sauve la dernière valeur pour modifier les steps
                value = self.get_property_value(self.act_mod, modal_action)

                # disallow slow_down for integer values
                slow_down = type(value).__name__ != 'int'

                if slow_down:
                    if event.value == 'PRESS':
                        if not self.slow_down:
                            self.mouse_x = event.mouse_x
                            # self.wheel_delta = 0
                            self.last_value = value
                            self.slow_down = True

                    elif event.value == 'RELEASE':
                        self.mouse_x = event.mouse_x
                        # self.wheel_delta = 0
                        self.last_value = value
                        self.slow_down = False

            if event.type in {'MOUSEMOVE', 'WHEELUPMOUSE', 'WHEELDOWNMOUSE'}:

                if event.type == 'MOUSEMOVE':

                    # prevent mouse move in 'mouse' mode (we could allow mouse in both modes)
                    if self.work_tool == 'mouse':
                        return {'PASS_THROUGH'}

                    # relative diff between last action and current
                    delta = event.mouse_x - self.mouse_x

                else:

                    # absolute delta for this step
                    wheel_delta = 1
                    if event.type == 'WHEELDOWNMOUSE':
                        wheel_delta = -wheel_delta

                    # relative diff between last action and current
                    self.wheel_delta += wheel_delta

                    delta = self.wheel_delta * 40 * self.modal_speed

                if self.slow_down:
                    delta /= 8

                pixels = self.get_pixels_from_rna(self.act_mod, modal_action)

                # limit to min / max for property
                if modal_action == 'width':
                    if self.act_mod.offset_type == 'PERCENT':
                        value = min(100, max(0.001, self.limit_value(self.last_value + delta / pixels * 150, self.act_mod, modal_action)))

                    else:
                        value = min(10000, max(0.001, self.limit_value(self.last_value + delta / pixels, self.act_mod,
                                                                       modal_action)))

                else:
                    value = self.limit_value(self.last_value + delta / pixels, self.act_mod, modal_action)

                if modal_action == 'width' and self.act_mod.limit_method == 'WEIGHT':
                    self.apply_width(value)

                else:
                    self.set_same_modifiers_value(modal_action, value)

        # ---------------------------------
        # Select another object while modal running
        # ---------------------------------
        event_alt = (event.ctrl, event.shift, event.alt)

        if modal_action == "free" and event.type == self.key_select and any(event_alt):
            if event.shift:
                return self.copy_modifier_if_missing(context, event, types={'MESH'})
            elif event.ctrl:
                return self.remove_same_modifier(context, event)
            else:
                action = 'free'
                return self.run_modal_with_another_object(context, event, action, types={'MESH'})

        if event.value == 'PRESS':

            # Store event alternatives
            self.event_alt = event_alt = (event.ctrl, event.shift, event.alt)

            # set default value for action
            action = 'free'

            # ---------------------------------
            # Step 1 : Process events : turn events into actions
            # ---------------------------------

            # ---------------------------------
            # Mouse actions (startup - store current values)
            # ---------------------------------

            if event.type in self.mouse_actions.keys() and not self.input:
                # Start action, store init_value to cancel
                action = self.mouse_actions[event.type]
                # store initial value for mouse action to restore on exit and last to change mouse steps on the fly
                self.store_init_value(event, action)
                self.modal_action = action
                return {'RUNNING_MODAL'}

            # ---------------------------------
            # Special actions
            # ---------------------------------
            elif event.type in self.special_actions[event_alt].keys():
                action = self.special_actions[event_alt][event.type]
                # let process actions below once

            elif event.type == 'BACK_SPACE' and self.input:
                self.input = self.input[:-1]
                # No need to process action
                return {'RUNNING_MODAL'}

            # ---------------------------------
            # Default actions
            # ---------------------------------

            elif event.type in self.default_actions[event_alt].keys():
                action = self.default_actions[event_alt][event.type]
                # let process actions below once

            # ---------------------------------
            # Toggle actions
            # ---------------------------------

            elif event.type in self.toggle_actions[event_alt].keys():
                action = self.toggle_actions[event_alt][event.type]
                self.toggle_same_modifiers_value(action)
                # No need to process action
                return {'RUNNING_MODAL'}

            # ---------------------------------
            # Autres actions
            # ---------------------------------

            # ---------------------------------
            # Step 2 : Process actions
            # ---------------------------------

            # ---------------------------------
            # Confirm
            # ---------------------------------
            if action == 'confirm':

                if self.input:

                    value = self.input_as_value(self.act_mod, modal_action)

                    if modal_action == 'width' and self.act_mod.limit_method == 'WEIGHT':
                        self.apply_width(value)
                    else:
                        self.set_same_modifiers_value(modal_action, value)


                    self.input = ""

                if modal_action == 'free':
                    self.exit(context)

                    # restore Overlays
                    context.space_data.overlay.show_overlays = self.overlay_state
                    context.space_data.overlay.show_wireframes = self.overlay_wire_state
                    return {'FINISHED'}

                # self.correct_normals(self.mode)

                self.modal_action = 'free'

            # ---------------------------------
            # Cancel (Generic)
            # ---------------------------------
            elif action == 'cancel':
                if self.input:
                    self.input = ""

                if modal_action != 'free':
                    # cancel l'action en cours
                    self.restore_values()

                else:
                    # exit on est en free, il n'y a rien à restaurer
                    self.exit(context)

                    # restore Overlays
                    context.space_data.overlay.show_overlays = self.overlay_state
                    context.space_data.overlay.show_wireframes = self.overlay_wire_state
                    return {'CANCELLED'}

                self.modal_action = 'free'

            # ---------------------------------
            # Add Bevel Modifier
            # ---------------------------------
            elif action == 'add_modifier':

                act_mod = self.act_mod
                for obj in self.sel:
                    mod = self.add_modifier(obj, self.mod_type)
                    limit_method = 'ANGLE'
                    if self.prefs.shading_smooth:
                        self.shading_mode(self, context, self.act_obj, 'SMOOTH')
                    else:
                        self.shading_mode(self, context, self.act_obj, 'FLAT')

                    if self.prefs.auto_rename_modifiers:
                        mod.name = "Bevel - {}".format(limit_method)

                    skin = self.get_modifiers_by_type(obj, 'SKIN')

                    mod.angle_limit = radians(28)


                    mod.loop_slide = not self.destructive_mode

                    if skin:
                        mod.loop_slide = True

                    mod.limit_method = limit_method
                    mod.show_viewport = True

                    if obj == act_obj:
                        act_mod = mod


                    mods = self.get_modifiers_by_type_and_attr(obj, self.mod_type, 'limit_method', 'ANGLE')
                    if len(mods) > 1:
                        mod.segments = 4
                        mod.angle_limit = radians(28)
                        mod.loop_slide = True

                # Store new modifiers in cache
                self.act_mod = act_mod

                action = 'width'
                # store initial value for mouse action to restore on exit and last to change mouse steps on the fly
                self.store_init_value(event, action)
                self.modal_action = action

                if self.act_obj.type == 'MESH':
                    for obj in self.sel:
                        if self.prefs.add_weighted_normal:
                            self.move_weighted_normals_down(context, obj)

            # ---------------------------------
            # Apply Modifier
            # ---------------------------------
            elif action == 'apply_modifier':
                # Only work for active object, What about multi object support here ??
                # if self.act_obj.mode == 'OBJECT':

                # Flag to exit if object has no more modifiers

                # self.modal_action = 'free'

                act_mod = None

                limit_method = self.act_mod.limit_method

                next = self.next_supported_modifier_down(act_obj, self.act_mod)

                for obj, mod in self.same_modifs.items():

                    # this will make object unique !!!
                    self.apply_modifier(context, obj, mod)

                    # if obj.type == 'MESH':
                    #     if limit_method == 'WEIGHT':
                    #         # Remove beveil Weight
                    #         bpy.ops.object.mode_set(mode='EDIT')
                    #         bpy.ops.mesh.select_all(action='SELECT')
                    #         bpy.ops.transform.edge_bevelweight(value=-1)
                    #
                    #         # self.setup_edges_bevel_weight(context, obj, 0, -1, True)
                    #         del obj['SpeedFlow']['Bevel']

                # restore Overlays before running next one
                context.space_data.overlay.show_overlays = self.overlay_state
                context.space_data.overlay.show_wireframes = self.overlay_wire_state

                self.run_modal(act_obj, next, call_from_pie=False)
                self.exit(context)

                return {'FINISHED'}

            # ---------------------------------
            # Remove Modifier
            # ---------------------------------
            elif action == 'remove_modifier':

                # why not remove in use_bevel_weight mode ??
                # if not self.use_bevel_weight:

                # use as flag to exit if act_obj has no more modifiers
                next = self.next_supported_modifier_down(act_obj, self.act_mod)

                for obj, mod in self.same_modifs.items():

                    if obj.type == 'MESH':
                        if mod.limit_method == 'WEIGHT':
                            # self.setup_edges_bevel_weight(context, obj, 0, -1, True)
                            if self.has_object_destructive_tag(obj):
                                del obj['SpeedFlow']['Bevel']
                            else:
                                del obj['SpeedFlow']

                    self.remove_modifier(obj, mod)


                for obj in self.same_modifs.keys():
                    self.shading_mode(self, context, obj, mode='FLAT')


                self.run_modal(act_obj, next)
                self.exit(context)

                # restore Overlays before running next one
                context.space_data.overlay.show_overlays = self.overlay_state
                context.space_data.overlay.show_wireframes = self.overlay_wire_state

                return {'FINISHED'}

            # ---------------------------------
            # Change Modifier Position (Generic)
            # ---------------------------------
            elif action == 'move_modifier_up':
                for obj, mod in self.same_modifs.items():
                    self.move_modifier_up(context, obj, mod)
                self.set_index_for_companion(self.act_mod)

            elif action == 'move_modifier_down':
                for obj, mod in self.same_modifs.items():
                    self.move_modifier_down(context, obj, mod)
                self.set_index_for_companion(self.act_mod)

            # ---------------------------------
            # Switch Between modifiers (Generic)
            # ---------------------------------
            elif action == 'next_modifier_up':
                # restore Overlays before running next one
                context.space_data.overlay.show_overlays = self.overlay_state
                context.space_data.overlay.show_wireframes = self.overlay_wire_state

                res = self.run_modal_up(act_obj, self.act_mod)
                if res:
                    self.exit(context)
                    return {'FINISHED'}

            elif action == 'next_modifier_down':
                # restore Overlays before running next one
                context.space_data.overlay.show_overlays = self.overlay_state
                context.space_data.overlay.show_wireframes = self.overlay_wire_state

                res = self.run_modal_down(act_obj, self.act_mod)
                if res:
                    self.exit(context)
                    return {'FINISHED'}

            elif action == 'same_modifier_up':
                self.act_mod = self.get_modifier_up_same_type(act_obj, self.act_mod)

            elif action == 'same_modifier_down':
                self.act_mod = self.get_modifier_down_same_type(act_obj, self.act_mod)


            # ---------------------------------
            # Call pie menu (Generic)
            # ---------------------------------
            elif action == 'call_pie':
                # restore Overlays before running next one
                context.space_data.overlay.show_overlays = self.overlay_state
                context.space_data.overlay.show_wireframes = self.overlay_wire_state

                self.exit(context)
                if self.prefs.choose_menu_type == 'pie':
                    bpy.ops.wm.call_menu_pie('INVOKE_DEFAULT', name="SPEEDFLOW_MT_pie_menu")
                else:
                    bpy.ops.wm.call_menu('INVOKE_DEFAULT', name="SPEEDFLOW_MT_simple_menu")
                return {'FINISHED'}

            # ---------------------------------
            # CHTR + H Toggle Keymaps (Generic)
            # ---------------------------------
            elif action == 'show_hide_keymap':
                self.prefs.display_keymaps = not self.prefs.display_keymaps

            # ---------------------------------
            # SHIFT + H Toggle Beginner Mode
            # ---------------------------------
            elif action == 'beginner_mode':
                self.prefs.beginner_mode = not self.prefs.beginner_mode

            # ---------------------------------
            # H - Toggle Show / Hide Modifier (Generic)
            # ---------------------------------
            elif action == 'show_viewport':
                # same modifiers
                self.toggle_same_modifiers_value('show_viewport')

# ---------------------------------
# Special actions
# ---------------------------------

            # ---------------------------------
            # Change Boolean operation
            # ---------------------------------
            elif action == 'boolean_operation':
                self.edit_boolean_operation(context, self)

            # ---------------------------------
            # Toggle local view in Modal
            # ---------------------------------
            elif action == 'localview':
                bpy.ops.view3d.localview()

            # ---------------------------------
            # Toggle Hide all modifiers
            # ---------------------------------
            elif action == 'show_hide_all_modifiers':
                self.show_hide_all_modifiers(act_obj, self.act_mod)

            # ---------------------------------
            # Toggle Auto Update Bevel
            # ---------------------------------
            elif action == 'auto_update':
                if self.act_obj.type == 'MESH':
                    # on all objects including when no shared mod are found ???
                    if not self.first_bevel and not self.is_object_in_destructive_mode(act_obj):

                        for obj in self.sel:
                            if obj.type == "MESH" and self.has_object_destructive_tag(obj):
                                d = obj.data
                                obj.data = self.mesh_backup[obj].copy()
                                obj.data.update()
                                bpy.data.meshes.remove(d)
                                if not self.auto_update:
                                    self.setup_edges_bevel_weight(context, obj, obj['SpeedFlow']['Bevel'], -1, True)
                                    self.setup_edges_bevel_weight(context, obj, obj['SpeedFlow']['Bevel'], 1, False)

                        self.prefs.auto_update = self.auto_update = not self.auto_update

            # ---------------------------------
            # Mode Only Vertex
            # ---------------------------------
            elif action == 'use_only_vertices':
                self.toggle_same_modifiers_value(action)
                if self.prefs.auto_rename_modifiers:
                    for obj, mod in self.same_modifs.items():
                        if self.act_mod.use_only_vertices:
                            mod.name = self.act_mod.name + " - Only V"
                        else:
                            mod.name = self.act_mod.name.replace(" - Only V", "")

            # ---------------------------------
            # Change Limit Method
            # ---------------------------------
            elif action == 'limit_method':
                value = self.get_next_enum(self.act_mod, action)
                # vgroup_name = self.get_vertex_group_by_name(self.act_obj, name)

                # TODO: optimize
                for obj, mod in self.same_modifs.items():

                    mod.limit_method = value
                    if self.prefs.auto_rename_modifiers:
                        mod.name = 'Bevel - ' + value  # ------------------------------- Modif for name

                    if value != 'WEIGHT':
                        if value == 'ANGLE':
                            mod.angle_limit = radians(28)
                    #
                        # if value == 'VGROUP':
                        #     mod.vertex_group = ''


                            # for vgroups in self.act_obj.vertex_groups:
                            #     if vgroups is not None:
                            #         vgroups_name = vgroups.active.namec
                            #
                            #         if mod.vertex_groups:
                            #             mod.vertex_group = vgroups_name
                        # if self.act_obj.type == 'MESH':
                        #     if self.has_object_destructive_tag(obj):
                        #         # self.setup_edges_bevel_weight(context, obj, 0, -1, True)
                        #
                        #         if len(obj['SpeedFlow']) > 1:
                        #             del obj['SpeedFlow']['Bevel']
                        #         else:
                        #             del obj['SpeedFlow']

                    # else:
                    #     if self.act_obj.type == 'MESH':
                            # self.tag_object_with_destructive_mode(obj, destructive_mode=0)
                            # self.setup_edges_bevel_weight(context, obj, self.destructive_mode, 1, False)

            # ---------------------------------
            # Change Offset Type
            # ---------------------------------
            elif action == 'offset_type':
                # width = self.act_mod.width
                #
                # if self.act_mod.offset_type == 'PERCENT':
                #     self.act_mod.width = 100
                # else:
                #     self.act_mod.width = width

                value = self.get_next_enum(self.act_mod, action)
                self.set_same_modifiers_value(action, value)

            # ---------------------------------
            # Change Miter Outer
            # ---------------------------------
            elif action == 'miter_outer':
                value = self.get_next_enum(self.act_mod, action)
                self.set_same_modifiers_value(action, value)

            # ---------------------------------
            # Change Miter Inner
            # ---------------------------------
            elif action == 'miter_inner':
                value = self.get_next_enum(self.act_mod, action)
                self.set_same_modifiers_value(action, value)

            # ---------------------------------
            # Mode VGROUP
            # ---------------------------------

            elif action == 'vertex_group':
                vgroups = act_obj.vertex_groups
                if len(vgroups) > 0:
                    vgroups.active_index = (vgroups.active_index + 1) % len(vgroups)
                    self.act_mod.vertex_group = vgroups.active.name

                if self.prefs.auto_rename_modifiers:

                    if self.act_mod.vertex_group:
                        if not " - VGROUP" in self.act_mod.name :
                            self.act_mod.name = self.act_mod.name + " - VGROUP"

                self.modal_action = 'vertex_group'

            # ---------------------------------
            # Remove VGROUP
            # ---------------------------------

            elif action == 'clean_vertex_group':
                for mod in self.same_modifs.values():
                    self.act_mod.vertex_group = ""

                    if self.prefs.auto_rename_modifiers:
                        mod.name = mod.name.replace(" - VGROUP", "")

            # ---------------------------------
            # Toggle Subdiv/NoSubdiv
            # ---------------------------------
            elif action == 'destructive_mode':
                if self.act_obj.type == 'MESH':
                    # si l'objet a 1 seul modifier et qu'il est en angle > passe en weight

                    if self.has_object_destructive_tag(act_obj):

                        # obj et modif sont les objets qui ont le même modifier et leur modifier
                        for obj, modif in self.same_modifs.items():

                            if self.has_object_destructive_tag(obj):

                                # on recup la liste des bevels de l'objet courant
                                modifs = self.get_modifiers_by_type(obj, self.mod_type)

                                # Gniii ?? le premier modifier ???
                                bevel_modifier = [mod for mod in modifs if
                                                  mod.limit_method == 'WEIGHT' or
                                                  # Gniii ?? ces conditions ..
                                                  # mod.limit_method == 'ANGLE' and mod.use_only_vertices or
                                                  mod.limit_method == 'ANGLE'][0]

                                if self.is_object_in_destructive_mode(obj):

                                    if not self.use_bevel_weight and len(modifs) == 1:
                                        for ed in obj.data.edges:
                                            ed.bevel_weight = -1

                                        self.setup_edges_bevel_weight(context, obj, 0, 1, False)

                                    else:
                                        for ed in obj.data.edges:
                                            if ed.bevel_weight:
                                                ed.crease = 1
                                                ed.use_edge_sharp = True

                                    bevel_modifier.segments = self.prefs.bevel.nosubdiv_segments
                                    bevel_modifier.profile = self.prefs.bevel.nosubdiv_profile
                                    bevel_modifier.loop_slide = True

                                    if self.prefs.shading_smooth:
                                        self.shading_mode(self, context, obj, 'SMOOTH')
                                    else:
                                        self.shading_mode(self, context, obj, 'FLAT')

                                else:
                                    for ed in obj.data.edges:
                                        ed.crease = -1
                                        ed.use_edge_sharp = False

                                    bevel_modifier.segments = self.prefs.bevel.segments
                                    bevel_modifier.profile = self.prefs.bevel.profile
                                    bevel_modifier.loop_slide = False

                                    if self.prefs.shading_smooth:
                                        self.shading_mode(self, context, obj, 'SMOOTH')
                                    else:
                                        self.shading_mode(self, context, obj, 'FLAT')
                                    # self.shading_mode(context, obj, 'FLAT')

                                obj['SpeedFlow']['Bevel'] = not obj['SpeedFlow']['Bevel']

                        self.prefs.destructive_mode = act_obj['SpeedFlow']['Bevel']

                if self.prefs.auto_rename_modifiers:
                    for mod in self.same_modifs.values():
                        mod.name = "Bevel - %s" % (mod.limit_method)

            # ---------------------------------
            # Show modifiers
            # ---------------------------------
            elif action == 'show_modifier':
                for obj in context.selected_objects:
                    to_show = self.get_first_modifier_by_dict(obj, {'show_viewport' : False})
                    if to_show is not None:
                        to_show.show_viewport = True

            # ---------------------------------
            # Hide modifiers
            # ---------------------------------
            elif action == 'hide_modifier':
                for obj in context.selected_objects:
                    to_show = self.get_last_modifier_by_dict(obj, {'show_viewport' : True})
                    if to_show is not None:
                        to_show.show_viewport = False
    # ---------------------------------
    # Shading / Snap
    # ---------------------------------

            # ---------------------------------------
            # 1 - Toggle Keep Wire in Modal (Generic)
            # ---------------------------------------
            elif action == 'keep_wire':
                self.prefs.keep_wire = not self.prefs.keep_wire

            # --------------------------------------
            # 2 - Toggle wire in the modal (Generic)
            # --------------------------------------
            elif action == 'show_wire_in_modal':
                self.shading_wire_in_modal(context, self)

            # ---------------------------------
            # 3 - Toggle XRAY
            # ---------------------------------
            elif action == 'set_xray':
                self.shading_xray(context, self)

            # ---------------------------------
            # 4 - Toggle shading mode type
            # ---------------------------------
            elif action == 'shading_mode':
                self.shading_mode_type(context, self)

            # ---------------------------------
            # 5 - Toggle Random Mode
            # ---------------------------------
            elif action == 'random':
                self.shading_random_mode(context, self)

            # ---------------------------------
            # 6 - Toggle OVERLAYS
            # ---------------------------------
            elif action == 'overlays':
                self.shading_overlays(context, self)

            # ---------------------------------
            # 7 - Toggle FACE ORIENTATION
            # ---------------------------------
            elif action == 'face_orientation':
                self.shading_face_orientation(context, self)

            # ---------------------------------
            # 8 - Toggle Hide Grid
            # ---------------------------------
            elif action == 'hide_grid':
                self.shading_hide_grid(context, self)

            # ---------------------------------
            # 9 - Toggle Bool Shading Mode
            # ---------------------------------
            elif action == 'bool_mode':
                self.shading_bool_mode(context, self)

            # ---------------------------------
            # F2 - Toggle Activate Snap
            # ---------------------------------
            elif action == 'active_snap':
                self.snap_activate_snap(context, self)

            # ---------------------------------
            # F3 - Toggle Snap Vertex
            # ---------------------------------
            elif action == 'snap_vertex':
                self.snap_vertex(context, self)

            # ---------------------------------
            # F4 - Toggle Snap Face
            # ---------------------------------
            elif action == 'snap_face':
                self.snap_face(context, self)

            # ---------------------------------
            # F5 - Toggle Snap Grid
            # ---------------------------------
            elif action == 'snap_grid':
                self.snap_grid(context, self)

            # ---------------------------------
            # F6 - Origin to selection
            # ---------------------------------
            elif action == 'origin_to_selection':
                self.snap_origin_to_selection(context)

            # ---------------------------------
            # F7 - Toggle Apply Origin
            # ---------------------------------
            elif action == 'apply_location':
                self.snap_origin_to_grid(context)

            # ---------------------------------
            # START Origin
            # ---------------------------------
            if action == 'set_origin':
                # self.act_mod.show_viewport = False

                bpy.ops.wm.tool_set_by_id(name="builtin.select", cycle=False, space_type='VIEW_3D')

                self.mode_snap = self.act_obj.mode
                context.scene.tool_settings.use_snap = True
                # context.scene.tool_settings.snap_elements = {'VERTEX'}
                context.scene.tool_settings.use_snap_align_rotation = False

                bpy.ops.view3d.snap_cursor_to_selected()

                if act_obj.mode != 'OBJECT':
                    bpy.ops.object.mode_set(mode='OBJECT')

                bpy.ops.transform.translate('INVOKE_DEFAULT', cursor_transform=True)

                self.modal_action = 'origin'

            if action == 'set_origin_1':
                # self.act_mod.show_viewport = False

                self.mode_snap = self.act_obj.mode
                context.scene.tool_settings.use_snap = True
                context.scene.tool_settings.snap_elements = {'INCREMENT'}
                context.scene.tool_settings.use_snap_grid_absolute = True

                bpy.ops.view3d.snap_cursor_to_selected()

                if act_obj.mode != 'OBJECT':
                    bpy.ops.object.mode_set(mode='OBJECT')

                bpy.ops.transform.translate('INVOKE_DEFAULT', cursor_transform=True)

                self.modal_action = 'origin'

            # ---------------------------------
            # END ORIGIN
            # ---------------------------------
            if modal_action == 'origin':

                bpy.ops.object.origin_set(type='ORIGIN_CURSOR')
                # self.act_mod.show_viewport = True

                bpy.ops.object.mode_set(mode=self.mode_snap)
                self.modal_action = 'free'
        return {'RUNNING_MODAL'}

    # ----------------------------------------------------------------------------------------------
    #
    # ----------------------------------------------------------------------------------------------

    def invoke(self, context, event):

        if context.area and context.area.type == 'VIEW_3D':
            self.saved_location = context.scene.cursor.location.copy()
            # Get modifier type from modal class name
            mod_type = self.mod_type

            prefs = self.prefs

            act_obj = context.object

            sel = self.filter_objects_by_type(context.selected_objects, {'MESH', 'CURVE'})

            self.act_obj = act_obj
            self.sel = sel
            self.mode = act_obj.mode
            self.shading_viewport_type = context.space_data.shading.type
            self.overlay_state = context.space_data.overlay.show_overlays
            self.overlay_wire_state = context.space_data.overlay.show_wireframes
            self.shading_random = context.space_data.shading.color_type

            if act_obj.mode == 'EDIT':
                bpy.ops.object.mode_set(mode='OBJECT')
                if self.prefs.shading_smooth:
                    bpy.ops.object.shade_smooth()
                bpy.ops.object.mode_set(mode='EDIT')

            # flag indicate call method, through pie menu or object selection change or <-> with specified modifier
            call_through_pie = self.call_from_pie
            # flag when call through selection change using alt key, prevent special actions
            allow_special_actions = self.modifier_index == -2

            act_mod = None

            # Free Navigation


            has_mod = self.has_modifier_of_type(act_obj, mod_type)

            if self.prefs.default_modal_action and has_mod:
                self.modal_action = 'free'
            else:
                self.modal_action = 'width'



            # Comportement attendu ici ?? par défaut si on le call avec le pie, l'action est width
            if has_mod and call_through_pie and self.work_tool == 'pen' and prefs.default_modal_action:
                if not act_obj.mode == 'EDIT':
                    self.modal_action = 'free'

            # handle <-> object mode
            if not call_through_pie:
                act_mod = self.get_modifier_by_index(act_obj, self.modifier_index)
                # self.modifier_index = -3

                if act_obj.type=='MESH': # check for curves
                    if act_mod.limit_method == 'WEIGHT':
                        bpy.ops.object.mode_set(mode="EDIT")

                    elif act_obj.mode != 'OBJECT':
                        bpy.ops.object.mode_set(mode="OBJECT")

                elif act_obj.type == 'CURVE':
                    act_mod.loop_slide = True

            # try to get a weight modifier on active object as active modifier
            if act_mod is None:
                act_mod = self.get_first_modifier_by_type_and_attr(act_obj, mod_type, 'limit_method', 'WEIGHT')
                # act_mod = self.get_first_modifier_by_type_and_attr(act_obj, mod_type, 'limit_method', 'ANGLE')

            # IF EDIT MODE

            if act_obj.mode == 'EDIT':
                if self.act_obj.type == 'MESH':
                    # Bevel in vertex group mode
                    if tuple(context.tool_settings.mesh_select_mode) == (True, False, False):

                        vgroups = act_obj.vertex_groups

                        if call_through_pie and vgroups and (event.shift or event.ctrl):

                            if event.shift:
                                context.scene.tool_settings.vertex_group_weight = 1
                                bpy.ops.object.vertex_group_assign()

                            elif event.ctrl:
                                bpy.ops.object.vertex_group_remove_from()

                            return {'FINISHED'}

                        else:
                            # we only reach this part if object is in edit vertex mode
                            # at this point same_modifs should only store this object
                            bpy.ops.object.vertex_group_add()
                            context.scene.tool_settings.vertex_group_weight = 1
                            bpy.ops.object.vertex_group_assign()

                            act_mod = self.add_modifier(act_obj, mod_type)



                            #-------------------------------------------
                            me = act_obj.data
                            bm = bmesh.from_edit_mesh(me)
                            selected_verts = [v.index for v in bm.verts if v.select]
                            bm.verts.ensure_lookup_table()

                            if [v for v in bm.verts if v.select]:
                                if len(bm.verts[selected_verts[0]].link_edges) == 2:

                                    if len([mod for mod in act_obj.modifiers if mod.type == 'SCREW']):
                                        act_mod.use_only_vertices = False
                                    else:
                                        act_mod.use_only_vertices = True
                                else:
                                    act_mod.use_only_vertices = False



                            else:
                                bpy.ops.object.modifier_remove(modifier=act_mod.name)
                                self.report({'WARNING'}, "You need to select one or more vertices")
                                return {'FINISHED'}


                            act_mod.loop_slide = not self.destructive_mode
                            act_mod.limit_method = 'VGROUP'
                            act_mod.vertex_group = vgroups.active.name

                            if self.prefs.auto_rename_modifiers:
                                if act_mod.use_only_vertices and act_mod.vertex_group:
                                    act_mod.name = "Bevel - %s" % act_mod.limit_method + " - Only V"

                                elif act_mod.vertex_group:
                                    act_mod.name = "Bevel - %s" % act_mod.limit_method

                                elif act_mod.use_only_vertices:
                                    act_mod.name = "Bevel - %s" % act_mod.limit_method + " - Only V"

                            self.tag_object_with_destructive_mode(act_obj, destructive_mode=self.destructive_mode)

                            solidify = self.get_last_modifier_by_type(act_obj, 'SOLIDIFY')
                            screw = self.get_last_modifier_by_type(act_obj, 'SCREW')
                            if solidify is not None and screw is None:
                                self.move_modifier_top(context, act_obj, act_mod)

                            bpy.ops.object.mode_set(mode='OBJECT')

                    else:

                        # if the active object haven't go Bevel modifier with a WEIGHT
                        # limit_method, we create it.
                        if act_mod is None:

                            # act_mod = self.add_modifier(act_obj, mod_type, 'Bevel - WEIGHT')
                            act_mod = self.add_modifier(act_obj, mod_type, 'Bevel')


                            # ne dépend pas directement du destructive_mode, mais on le set comme ça par défaut
                            act_mod.loop_slide = not self.destructive_mode
                            act_mod.limit_method = 'WEIGHT'

                            if self.prefs.auto_rename_modifiers:
                                act_mod.name = "Bevel - %s" % act_mod.limit_method

                            # init using nosubdiv defaults
                            if not self.destructive_mode:
                                for attr in ('width', 'segments', 'profile'):
                                    setattr(act_mod, attr, getattr(self.prefs.bevel, "nosubdiv_{}".format(attr)))

                            self.tag_object_with_destructive_mode(act_obj, destructive_mode=self.destructive_mode)

                            # Cleanup selected edges weight
                            self.clear_edges_weights(context, act_obj)

                        # Update mesh from edit state
                        act_obj.update_from_editmode()

                        # compute Bevel speedflow user attr if not found
                        self.tag_object_with_destructive_mode(act_obj)

                        # to avoid bevel deformations, we must apply the object scale
                        # we can apply transforms on mesh with only 1 user
                        if act_obj.data.users == 1:
                            if prefs.auto_apply_scale:
                                self.apply_transform_scale(context, act_obj)

                        me = act_obj.data
                        bm = bmesh.from_edit_mesh(me)

                        # Use selected faces to select edges
                        if tuple(context.tool_settings.mesh_select_mode) == (False, False, True):
                            self.select_edges_boundary_loop(bm)

                        if any([ed for ed in bm.edges if ed.select]):

                            if call_through_pie and event.shift or event.ctrl:
                                edit_direct_action = {
                                    (True, False): [self.edit_bevel_weight, (bm, me, 1)],
                                    (False, True): [self.edit_bevel_weight, (bm, me, -1)],
                                    (True, True): [self.bevel_weight_from_active, (bm, me)],
                                }
                                fun, args = edit_direct_action[(event.shift, event.ctrl)]
                                fun(*args)

                                return {'FINISHED'}

                            bpy.ops.object.mode_set(mode='OBJECT')

                            # Move on top of stack
                            self.move_modifier_top(context, act_obj, act_mod)

                            # Stockage des differentes info des edges possedant un bevel_weight
                            self.store_edges_weight(act_obj, act_mod)

                            if not self.is_object_in_destructive_mode(act_obj):
                                for ed in self.edges_datas.keys():
                                    ed.crease = 1
                                    ed.use_edge_sharp = True

                            if self.prefs.shading_smooth:
                                self.shading_mode(self, context, act_obj, 'SMOOTH')
                            else:
                                self.shading_mode(self, context, act_obj, 'FLAT')

                            # Flag inutile !!! on l'a sur le modifier 'WEIGHT'
                            self.use_bevel_weight = True

                        else:
                            self.report({'WARNING'}, "No selected edges")
                            return {'CANCELLED'}


                    # Store modifiers in cache
                    # self.store_all_modifs_of_type('BEVEL')
                    self.mesh_backup[act_obj, 'MESH'] = act_obj.data.copy()

            else:
                # in object mode, many modifiers for many objects

                # Not user friendly !
                # Should only work on selected bevel modifier so result is predictible
                special = (event.shift, event.ctrl, event.alt)

                if allow_special_actions and any(special):

                    if check_valid_input(special):

                        for obj in sel:
                            fun, args = {
                                (True, False, False): [self.apply_modifiers_by_type, (context, obj, mod_type)],
                                (False, True, False): [self.remove_modifiers_by_type, (obj, mod_type)],
                                (False, False, True): [self.toggle_modifiers_visibility_by_type, (obj, mod_type)],
                            }[special]

                            # hi hi hi, have some fun
                            fun(*args)

                        bpy.ops.object.mode_set(mode=self.mode)

                        return {'FINISHED'}

                # find act_mod if no WEIGHT bevel found
                if act_mod is None:
                    act_mod = self.get_last_modifier_by_type(act_obj, mod_type)

                for obj in sel:

                    # to avoid bevel deformations, we must apply the object scale
                    # we can apply transforms on mesh with only 1 user
                    if obj.data.users == 1:
                        if prefs.auto_apply_scale:
                            self.apply_transform_scale(context, obj)

                    # after transform apply
                    self.mesh_backup[obj, 'MESH'] = obj.data.copy()

                    # find existing modifier with same data than act_mod
                    mod = self.find_same_in_list(act_mod, obj.modifiers)
                    # exclude vertex_group as name likely not be the same across objects ???
                    # mod = self.find_same_in_list(act_mod, obj.modifiers, blacklist={'vertex_group'})

                    if mod is None:
                        # condition to add a modifier on selected objects:
                        # if we are browsing act_obj stack using <->
                        # and no similar modifier is found then do nothing (ignore object)

                        # elif act_obj.type == 'CURVE':
                        # act_mod.loop_slide = True

                        # if we call through pie, and no modifier of this kind
                        # is found then add a new one
                        if call_through_pie and not self.has_modifier_of_type(obj, mod_type):

                            if obj.type == 'MESH':
                                shading = 'SMOOTH'
                                if self.destructive_mode:
                                    self.limit_method = 'ANGLE'

                                else:
                                    self.limit_method = 'WEIGHT'

                                self.shading_mode(self, context, obj, mode=shading)

                                # mod = self.add_modifier(obj, mod_type, "Bevel_{}".format(self.limit_method))
                                mod = self.add_modifier(obj, mod_type, "Bevel")

                                if self.prefs.auto_rename_modifiers:
                                    mod.name = "Bevel - %s" % mod.limit_method

                            elif act_obj.type == 'CURVE':
                                self.limit_method = 'ANGLE'

                                # mod = self.add_modifier(obj, mod_type, "Bevel_{}".format(self.limit_method))
                                mod = self.add_modifier(obj, mod_type, "Bevel")

                                if self.prefs.auto_rename_modifiers:
                                    mod.name = "Bevel - %s" % mod.limit_method

                            # selection may change between 2 pie calls
                            # then maybe act_obj already hold a non default modifier
                            # so we copy params from it
                            if has_mod:
                                self.copy_modifier_params(act_mod, mod)
                            else:
                                mod.angle_limit = radians(28)
                                # print(act_mod)
                                #
                                # return {'FINISHED'}

                                if act_obj.type == 'MESH':
                                    mod.loop_slide = not self.destructive_mode
                                    mod.limit_method = self.limit_method

                                elif act_obj.type == 'CURVE':
                                    mod.loop_slide = True
                                    mod.limit_method = 'ANGLE'


                                mod.show_viewport = True

                            if obj.type == 'MESH':
                                self.tag_object_with_destructive_mode(obj, destructive_mode=self.destructive_mode)

                                self.select_mesh_components(obj, True)
                                self.clear_edges_weights(context, obj)

                                # what does this flag ??
                                self.first_bevel = True

                                if self.limit_method == 'WEIGHT':
                                    self.setup_edges_bevel_weight(context, obj, self.destructive_mode, 1, False)
                                    self.move_modifier_top(context, obj, mod)

                        # subsequent modifiers must be added through modal commands

                    if mod is not None:
                        # temporary store moodifiers so we are able to retrieve act_mod later
                        self.same_modifs[obj] = mod

                        if self.prefs.shading_smooth:
                            self.shading_mode(self, context, act_obj, 'SMOOTH')
                        else:
                            self.shading_mode(self, context, act_obj, 'FLAT')

                    # Rename
                    # if act_mod.vertex_group:
                    #     act_mod.name = 'Bevel - %s - VGROUP' % act_mod.limit_method
                    # elif act_mod.use_only_vertices:
                    #     act_mod.name = 'Bevel - %s - Only V' % act_mod.limit_method
                    # elif act_mod.use_only_vertices and act_mod.vertex_group:
                    #     act_mod.name = 'Bevel - %s - Only V - VGROUP' % act_mod.limit_method


                # no bevel at startup, use new one as act
                if act_mod is None:
                    act_mod = self.same_modifs[act_obj]

                # this allow to set segments for all bevels at once
                # self.store_all_modifs_of_type('BEVEL')

                # disable weighted if only one modifier
                # mods = self.get_modifiers_by_type_and_attr(obj, self.mod_type, 'limit_method', 'ANGLE')
                # # if len(mods) > 1:

            # Store modifiers in cache
            self.act_mod = act_mod
            if self.act_obj.type == 'MESH':
                if self.prefs.add_weighted_normal:
                    for obj in sel:
                        has_weighted_normal = self.get_modifiers_by_type(obj, 'WEIGHTED_NORMAL')
                        if not has_weighted_normal:
                            self.add_modifier(obj, 'WEIGHTED_NORMAL')
                        self.move_weighted_normals_down(context, act_obj)

                # self.if_weighted_on_bool_and_not_on_ref_remove_weighted_from_bool(act_obj, context)

            # store initial value for action to restore on exit and last to change mouse steps on the fly
            if self.modal_action in self.mouse_actions.values():
                self.store_init_value(event, self.modal_action)

            # setup display
            display_type = None
            if prefs.show_wire_in_modal and act_obj.display_type == 'BOUNDS':
                display_type = 'WIRE'

            self.setup_display(display_type)



            # setup draw handler
            self.setup_draw_handler(context)


            bpy.context.window_manager.modal_handler_add(self)
            return {'RUNNING_MODAL'}

        else:
            self.report({'WARNING'}, "View3D not found")
            return {'CANCELLED'}

def register():
    try:
        bpy.utils.register_class(SPEEDFLOW_OT_bevel)
    except:
        print("Bevel already registred")

def unregister():
    bpy.utils.unregister_class(SPEEDFLOW_OT_bevel)