# -*- coding:utf-8 -*-

# Speedflow Add-on
# Copyright (C) 2018 Cedric Lepiller aka Pitiwazou & Legigan Jeremy AKA Pistiwique and Stephen Leger
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

# <pep8 compliant>
import bpy
import bmesh
from bpy.types import Operator
from .utils.functions import *
import math
import mathutils
from mathutils import Matrix, Vector


def local_rotate(obj, axis, angle):
    obj.matrix_world @= Matrix.Rotation(math.radians(angle), 4, axis)

# self.ref_object = []
bool_mesh = []
class SPEEDFLOW_OT_boolean(SpeedApi, Operator):

    """
    BOOLEAN

    CLICK - Add and adjust
    SHIFT - Apply all
    CTRL  - Remove all
    ALT    - Hide all
    CTRL + SHIFT - Direct rebool with apply
    CTRL + ALT   - Direct rebool without apply
    CTRL + ALT + SHIFT - Inset
    """
    bl_idname = "speedflow.boolean"
    bl_label = "Modal Boolean"
    bl_options = {'REGISTER', 'UNDO'}


    @classmethod
    def poll(cls, context):
        return context.object and context.object.type == 'MESH'

    def __init__(self):
        SpeedApi.__init__(self)
        CommonProperties.__init__(self)

        self.mouse_actions = {'S': 'UNION',
                            'D': 'DIFFERENCE',
                            'F': 'INTERSECT',
                            }
        self.boolean_layer = None
        self.is_invisible = False
        self.use_bound = self.prefs.boolean.use_bound
        self.bools_color = self.prefs.boolean.booleans_color
        self.delete_bool_obj = self.prefs.boolean.delete_bool_obj




# ---------------------------------------------------------------------------------------------
#
# ---------------------------------------------------------------------------------------------

    def add_boolean_modifier(self, bool_obj):
        """ Create a new boolean modifier """

        new_boolean = self.act_obj.modifiers.new(bool_obj.name, 'BOOLEAN')
        new_boolean.object = bool_obj
        if self.prefs.auto_rename_modifiers:
            new_boolean.name = "Boolean - %s" % new_boolean.operation
        new_boolean.show_viewport = True
        new_boolean.show_expanded = self.show_expanded


        if not bool_obj.name.startswith('BB_'):
            if not self.use_bound:
                bool_obj.display_type = 'WIRE'
            else:
                bool_obj.display_type = 'BOUNDS'
                bool_obj.show_wire = False
                bool_obj.show_all_edges = False

        bool_obj.hide_render = True

        #modif ced
        bool_obj.cycles_visibility.camera = \
        bool_obj.cycles_visibility.shadow = \
        bool_obj.cycles_visibility.transmission = \
        bool_obj.cycles_visibility.scatter = \
        bool_obj.cycles_visibility.diffuse = \
        bool_obj.cycles_visibility.glossy = False
        bool_obj.hide_render = True

        has_bevel_modifiers = self.get_modifiers_by_type_and_attr(self.act_obj, 'BEVEL', 'limit_method', 'ANGLE')
        has_triangulate_modifiers = self.get_modifiers_by_type(self.act_obj, 'TRIANGULATE')
        has_weigthed_normal = self.get_modifiers_by_type(self.act_obj, 'WEIGHTED_NORMAL')

        if has_bevel_modifiers:
            bpy.ops.object.modifier_move_up(modifier=new_boolean.name)

        if has_weigthed_normal:
            bpy.ops.object.modifier_move_up(modifier=new_boolean.name)

        if has_triangulate_modifiers:
            bpy.ops.object.modifier_move_up(modifier=new_boolean.name)

        # has_bevel_modifiers = self.get_modifiers_by_type(self.act_obj, 'BEVEL')
        # if has_bevel_modifiers:
        #     bpy.ops.object.modifier_move_up(modifier=new_boolean.name)
        #     bpy.ops.object.modifier_move_up(modifier=new_boolean.name)
        else:
            bpy.ops.object.shade_flat()

        # Shading
        if self.prefs.shading_smooth:
            bpy.ops.object.shade_smooth()
            self.act_obj.data.auto_smooth_angle = radians(self.prefs.auto_smooth_value)
            self.act_obj.data.use_auto_smooth = True
            bool_obj.data.use_auto_smooth = True
            bool_obj.data.auto_smooth_angle = self.act_obj.data.auto_smooth_angle
        return new_boolean


    def add_boolean_to_ref_object(self, obj):
        bpy.ops.object.select_all(action='DESELECT')


        for obj in self.ref_object:
            bpy.context.view_layer.objects.active = obj
            obj.select_set(state=True)

            mod = obj.modifiers.new("Boolean", 'BOOLEAN')
            if self.prefs.auto_rename_modifiers:
                mod.name = "Boolean - %s" % mod.operation
            mod.object = self.new_bool_obj

            has_bevel_modifiers = self.get_modifiers_by_type_and_attr(obj, 'BEVEL', 'limit_method', 'ANGLE')
            has_triangulate_modifiers = self.get_modifiers_by_type(obj, 'TRIANGULATE')
            has_weigthed_normal = self.get_modifiers_by_type(obj, 'WEIGHTED_NORMAL')

            if has_bevel_modifiers:
                bpy.ops.object.modifier_move_up(modifier=mod.name)

            if has_weigthed_normal:
                bpy.ops.object.modifier_move_up(modifier=mod.name)

            if has_triangulate_modifiers:
                bpy.ops.object.modifier_move_up(modifier=mod.name)


            # Shading
            if self.prefs.shading_smooth and has_bevel_modifiers:
                bpy.ops.object.shade_smooth()
                self.act_obj.data.auto_smooth_angle = radians(self.prefs.auto_smooth_value)
                self.act_obj.data.use_auto_smooth = True
                self.new_bool_obj.data.use_auto_smooth = True
                self.new_bool_obj.data.auto_smooth_angle = self.act_obj.data.auto_smooth_angle
            else:
                bpy.ops.object.shade_flat()
                self.act_obj.data.use_auto_smooth = False
                self.new_bool_obj.data.use_auto_smooth = False

            bpy.ops.object.select_all(action='DESELECT')

# ---------------------------------------------------------------------------------------------
#
# ---------------------------------------------------------------------------------------------

    def get_Hops_insert(self, obj):
        """ Check if the boolean object is an Hops boolean or not """

        if obj.name.startswith('BB_'):
            if obj.parent and obj.parent.name.startswith('AP_'):
                return obj.parent

        return


# ----------------------------------------------------------------------------------------------
#
# ----------------------------------------------------------------------------------------------

    def update_autosmooth(self, act_obj, bool_list):

        if self.prefs.shading_smooth:
            bpy.ops.object.shade_smooth()
            act_obj.data.auto_smooth_angle = radians(self.prefs.auto_smooth_value)
        else:
            bpy.ops.object.shade_flat()

        for obj in bool_list:
            if self.prefs.shading_smooth:
                bpy.ops.object.shade_smooth()
                obj.data.use_auto_smooth = True
                obj.data.auto_smooth_angle = act_obj.data.auto_smooth_angle
            else:
                bpy.ops.object.shade_flat()



# ---------------------------------------------------------------------------------------------
#
# ---------------------------------------------------------------------------------------------

    def get_boolean_objects(self, act_obj):
        """ Return the boolean objects """

        def get_BB_object(obj):
            for ch in obj.children:
                if ch.name.startswith('BB_'):
                    return ch

        booleanList = list()

        for obj in bpy.context.selected_objects:
            if obj != act_obj and obj.type == 'MESH':
                if obj.name.startswith('AP_'):
                    BB_object = get_BB_object(obj)
                    booleanList.append(BB_object)

                else:
                    booleanList.append(obj)

        if act_obj.data.use_auto_smooth:
            self.update_autosmooth(act_obj, booleanList)

        return booleanList

# ---------------------------------------------------------------------------------------------
#
# ---------------------------------------------------------------------------------------------

    def get_bevel_limit_method_(self, obj):
        """ Return the limit method of all bevel modifier object """

        bevel = [mod.limit_method for mod in obj.modifiers if mod.type == 'BEVEL']

        return bevel

# ---------------------------------------------------------------------------------------------
#
# ---------------------------------------------------------------------------------------------

    def get_destructive_mode(self, act_obj):
        """ Get Bevel 'WEIGHT' destructive_mode """

        if act_obj.get('SpeedFlow') and 'Bevel' in act_obj['SpeedFlow']:
            return act_obj['SpeedFlow']['Bevel']

        nosubdiv = [e for e in act_obj.data.edges if e.bevel_weight and e.crease]
        return len(nosubdiv) == 0

# ---------------------------------------------------------------------------------------------
#
# ---------------------------------------------------------------------------------------------

    def apply_modifiers(self, act_obj, bool_obj):
        """ Apply all the bevels and solidify modifiers of the boolean
        object """

        bpy.context.view_layer.objects.active = bool_obj

        if self.get_Hops_insert(bool_obj):
            bool_obj.hide_select = False
            bool_obj.select_set(state=True)


        if not self.delete_bool_obj:
            for obj in bpy.context.selected_objects:
                if obj != bool_obj:
                    obj.select_set(state=False)

            bpy.ops.object.duplicate()

            if self.get_Hops_insert(bool_obj):
                bool_obj.hide_select = True

            bool_obj = bpy.context.active_object
            bool_obj.name = "TMP_%s" %bool_obj.name
            self.act_mod.object = bool_obj

# add ced------------------------------------------------------------------
        # Check et applique les shape keys
        if bool_obj.active_shape_key:
            bpy.ops.object.shape_key_add(from_mix=True)
            bool_obj.active_shape_key.value = 1.0
            bool_obj.active_shape_key.name = "Freeze"

            i = bool_obj.active_shape_key_index
            for n in range(1, i):
                bool_obj.active_shape_key_index = 1
                bpy.ops.object.shape_key_remove()

            bpy.ops.object.shape_key_remove()
# add ced------------------------------------------------------------------

        for mod in bool_obj.modifiers:
            if mod.type in ['SOLIDIFY', 'BEVEL']:
                if bool_obj.data.users > 1:
                    bpy.ops.object.make_single_user(type = 'SELECTED_OBJECTS', object = True, obdata = True, material = False,\
                                                                            texture = False, animation = False)

                bpy.ops.object.modifier_apply(apply_as = 'DATA', modifier = mod.name)

        act_obj.select_set(state=True)

        for e in bool_obj.data.edges:
            e.bevel_weight = -1
            e.crease = -1
            e.use_edge_sharp = False

        bpy.context.view_layer.objects.active = act_obj

        return bool_obj

# ---------------------------------------------------------------------------------------------
#
# ---------------------------------------------------------------------------------------------

    def get_valid_boolean_obj(self, act_obj, bool_obj):
        # Les modifiers Bevel doivent etre applique pour pouvoir recuperer les nouvelles faces
        # apres apply du boolen.
        # On en profite pour supprimer les weight, crease et sharp pour eviter qu'ils ne soient tranfere
        # lors de l'apply du booleen

        if get_modifier_list(bool_obj, 'BEVEL'):
            return self.apply_modifiers(act_obj, bool_obj), 'BEVEL'

        return bool_obj, None

# ---------------------------------------------------------------------------------------------
#
# ---------------------------------------------------------------------------------------------

    def check_bmesh_validity(self, act_obj):
        """ Check if the conditions for applying Boolean  with BMESH solver are met """

        # si l'objet actif possede un bevel dont le limit_method est en WEIGHT,
        # on s'assure d'etre en CARVE car BMESH fait sauter les bevels weight
        bevels_limit_method = self.get_bevel_limit_method_(act_obj)
        if bevels_limit_method and 'WEIGHT' in bevels_limit_method:
            return False

        def check_is_manifold_object(obj):
            me = obj.data
            bm = bmesh.new()
            bm.from_mesh(me)

            for e in bm.edges:
                if not e.is_manifold:
                    bm.free()
                    del bm
                    return False

            bm.free()
            del bm
            return True

        if not check_is_manifold_object(act_obj):
            return False

        if not check_is_manifold_object(self.act_mod.object):
            return False

        return True

# ---------------------------------------------------------------------------------------------
#
# ---------------------------------------------------------------------------------------------

    def apply_boolean(self, act_obj, bool_obj):
        # select all in boolean mesh
        custom_selection(bool_obj, True)

        # deselect all in act_obj mesh
        custom_selection(act_obj, False)

        if act_obj.data.users > 1:
            bpy.ops.object.make_single_user(type = 'SELECTED_OBJECTS', object = True, obdata = True, material = False,\
                                                                    texture = False, animation = False)

        # if hasattr(act_obj.modifiers[self.act_mod.name], 'solver') and not self.check_bmesh_validity(act_obj):
        #     act_obj.modifiers[self.act_mod.name].solver = 'CARVE'

        bpy.ops.object.modifier_apply(apply_as = 'DATA', modifier = self.act_mod.name)

        if len(act_obj.material_slots) and len(bool_obj.material_slots):
            act_obj_materials = [mat.material for mat in act_obj.material_slots]
            if not bool_obj.material_slots[0].material in act_obj_materials:
                bpy.ops.object.material_slot_add()
                bpy.context.object.active_material = bool_obj.material_slots[0].material
                for f in act_obj.data.polygons:
                    if f.select:
                        f.material_index = act_obj.active_material_index

# ---------------------------------------------------------------------------------------------
#
# ---------------------------------------------------------------------------------------------

    def update_bevel(self, act_obj, is_bool_bevel, subdiv_bevel_update):

        me = act_obj.data
        bm = bmesh.new()
        bm.from_mesh(me)

        beveWeightLayer = bm.edges.layers.bevel_weight['BevelWeight']

        destructive_mode = self.get_destructive_mode(act_obj)
        # Si le Bevel est en mode subdiv
        if destructive_mode:
            bevel_name = [mod.name for mod in act_obj.modifiers if mod.type == 'BEVEL'][0]
            # on passe le weight des edges peripherique a 1
            edges_boundary = get_edges_boundary_loop(bm, 1)

            for e in edges_boundary:
                e[beveWeightLayer] = 1
                e.select = False

            if not is_bool_bevel and subdiv_bevel_update:
                bevel_value = getattr(act_obj.modifiers[bevel_name], 'width')

                for e in bm.edges:
                    if e.select and len(e.link_faces) == 2:
                        face0, face1 = e.link_faces
                        if face0.normal.angle(face1.normal) < 0.523599:
                            e.select = False

                bm.select_flush(False) # permet de deselectionner les vertex qui vont avec
                bm.to_mesh(me)
                bm.free()
                del bm
                me.update()

                # A modifier pour eviter le toggle de changement de mode
                # CF: https://blender.stackexchange.com/questions/50437/beveling-edges-with-varying-number-of-segments-preferably-programatically
                bpy.ops.object.mode_set(mode = 'EDIT')
                bpy.ops.mesh.bevel(offset=bevel_value, segments = 2, profile = 1, vertex_only = False)
                bpy.ops.object.mode_set(mode = 'OBJECT')

            else:
                bm.select_flush(False) # permet de deselectionner les vertex qui vont avec
                bm.to_mesh(me)
                bm.free()
                del bm
                me.update()

        else:
            creaseLayer = bm.edges.layers.crease['SubSurfCrease']

            # if is_bool_bevel:
            edges_boundary = get_edges_boundary_loop(bm, 1)

            for e in edges_boundary:
                e[beveWeightLayer] = 1
                e[creaseLayer] = 1
                e.smooth = False

            # on selectionne les sharp uniquement sur les edges selectionnes
            for e in bm.edges:
                if e.select and len(e.link_faces) == 2:
                    face0, face1 = e.link_faces
                    if face0.normal.angle(face1.normal) >= 0.523599:
                        e[beveWeightLayer] = 1
                        e[creaseLayer] = 1
                        e.smooth = False

            # get edges to clean
            edges = get_edges(bm, True)

            bm.to_mesh(me)
            bm.free()
            del bm
            me.update()

            # clean useless bevels
            for idx in edges:
                e = act_obj.data.edges[idx]
                e.use_edge_sharp = False
                e.bevel_weight = -1
                e.crease = -1

            if self.prefs.shading_smooth:
                self.shading_mode(self, act_obj, 'SMOOTH')
            else:
                self.shading_mode(self, act_obj,'FLAT')
            # shading_mode('SMOOTH')

        custom_selection(act_obj, False)

# ---------------------------------------------------------------------------------------------
#
# ---------------------------------------------------------------------------------------------

    def finalize(self, act_obj, bool_obj):

        if bool_obj.children:
            act_obj.select_set(state=False)
            bool_obj.select_set(state=False)

            # act_obj.select = False
            # bool_obj.select = False

            is_constraint = False
            for ch in bool_obj.children:
                ch.hide_select = False
                ch.select_set(state=True)
                # ch.select = True
                if ch.constraints:
                    is_constraint = True

            # if self.addon_prefs.delete_bool_obj:
            if self.delete_bool_obj:
                if is_constraint:
                    bool_obj.hide = True
                    bpy.ops.object.select_all(action = 'DESELECT')
#                    for ch in bool_obj.children:
#                        ch.select = False

                else:
                    bpy.ops.object.parent_clear(type = 'CLEAR_KEEP_TRANSFORM')
                    bpy.ops.object.select_all(action = 'DESELECT')

                    bpy.data.objects.remove(bool_obj, do_unlink = True)

            else:
                if is_constraint:
                    bool_obj.select_set(state=True)
                    # bool_obj.select = True
                    bpy.ops.object.duplicate()
                    for ob in bpy.context.selected_objects:
                        if not ob.parent:
                            ob.hide = True

                else:
                    bpy.ops.object.duplicate()
                    bpy.ops.object.parent_clear(type = 'CLEAR_KEEP_TRANSFORM')

                for ob in bpy.context.selected_objects:
                    # ob.select = False
                    ob.select_set(state=False)

            act_obj.select_set(state=True)
            # act_obj.select = True

        else:
            # if self.addon_prefs.delete_bool_obj:
            if self.delete_bool_obj:
                users = self.get_users(bool_obj)

                if not users:
                    bpy.data.objects.remove(bool_obj, do_unlink = True)
                    # act_obj.select = True
                    act_obj.select_set(state=True)

# ---------------------------------------------------------------------------------------------
#
# ---------------------------------------------------------------------------------------------

    def hardops_finalize(self, act_obj, AP_insert, bool_obj):

        act_obj.select_set(state=False)
        # act_obj.select = False

        def get_OB_insert(AP_insert):
            inserts = []

            def find_OB(obj):
                for ob in obj.children:
                    if ob.name.startswith('OB_'):
                        inserts.append(ob)

                    if ob.children:
                        find_OB(ob)

            if AP_insert.children:
                find_OB(AP_insert)

            return inserts

        OB_inserts = get_OB_insert(AP_insert)

        is_constraint = False
        for ob in OB_inserts:
            ob.hide_select = False
            ob.select_set(state=True)
            # ob.select = True
            if ob.constraints:
                is_constraint = True

        # if self.addon_prefs.delete_bool_obj:
        if self.delete_bool_obj:
            # dans le cas ou l'insert serait anime, il faut concerver le parent
            if is_constraint:
                AP_insert.hide = True
                # AP_insert.select = False
                AP_insert.select_set(state=False)
                bpy.data.objects.remove(bool_obj, do_unlink = True)

            else:
                bpy.ops.object.parent_clear(type = 'CLEAR_KEEP_TRANSFORM')
                bpy.data.objects.remove(AP_insert, do_unlink = True)
                bpy.data.objects.remove(bool_obj, do_unlink = True)

            for ob in OB_inserts:
                ob.select_set(state=False)
                # ob.select = False

        else:
            if is_constraint:
                AP_insert.select_set(state=True)
                # AP_insert.select = True
                bpy.ops.object.duplicate()
                for ob in bpy.context.selected_objects:
                    if ob.name.startswith('AP_'):
                        ob.hide = True
                    elif ob.name.startswith('BB_'):
                        bpy.data.objects.remove(ob, do_unlink = True)
            else:
                AP_insert.select_set(state=False)
                # AP_insert.select = False
                bpy.ops.object.duplicate()
                bpy.ops.object.parent_clear(type = 'CLEAR_KEEP_TRANSFORM')

            for ob in bpy.context.selected_objects:
                ob.select_set(state=False)
                # ob.select = False

            for ob in OB_inserts:
                ob.hide_select = True

        act_obj.select_set(state=True)
        # act_obj.select = True

# ---------------------------------------------------------------------------------------------
#
# ---------------------------------------------------------------------------------------------

    def get_users(self, bool_obj):

        users = list()

        for obj in bpy.context.scene.objects:
            for mod in obj.modifiers:
                if mod.type == 'BOOLEAN' and mod.object == bool_obj:
                    users.append(obj)

        return users



# ---------------------------------------------------------------------------------------------
#
# ---------------------------------------------------------------------------------------------

    def create_rebool_obj(self, act_obj):
        bpy.ops.object.select_all(action = 'DESELECT')
        act_obj.select_set(state=True)
        bpy.ops.object.duplicate()
        rev_bool = bpy.context.active_object



        rev_bool.modifiers[self.act_mod.name].operation = 'INTERSECT'
        rev_bool.modifiers[self.act_mod.name].show_expanded = self.show_expanded

        rev_bool.name = "%s_rebool" %act_obj.name
        rev_bool.select_set(state=False)

        bpy.context.view_layer.objects.active = act_obj
        act_obj.select_set(state=True)

        return rev_bool

# ---------------------------------------------------------------------------------------------
#
# ---------------------------------------------------------------------------------------------

    def apply_boolean_modifier(self, act_obj, event_shift):
        bool_obj = self.act_mod.object
        Hops_insert = self.get_Hops_insert(bool_obj)

        valid_bool_obj, is_bool_bevel = self.get_valid_boolean_obj(act_obj, bool_obj)

        if valid_bool_obj.name.startswith("TMP_"):
            self.act_mod.object = valid_bool_obj

        self.apply_boolean(act_obj, valid_bool_obj)

        # Pour pouvoir faire l'update du Bevel, il faut:
        #   - que l'objet n'ait qu'un seul Bevel
        #   - que le limit_method du Bevel soit en WEIGHT
        # cela pour eviter que l'update ne s'applique aux edges concernes par les autres Bevel
        bevels_limit_method = self.get_bevel_limit_method_(act_obj)
        if len(bevels_limit_method) == 1 and bevels_limit_method[0] == 'WEIGHT':
            self.update_bevel(act_obj, is_bool_bevel, event_shift)

        if valid_bool_obj.name.startswith("TMP_"):
            bpy.data.objects.remove(valid_bool_obj, do_unlink = True)

        if Hops_insert:
            self.hardops_finalize(self.act_obj, Hops_insert, bool_obj)

        else:
            self.finalize(self.act_obj, bool_obj)

# ---------------------------------------------------------------------------------------------
#
# ---------------------------------------------------------------------------------------------

    def reverse_boolean(self, act_obj, rev_bool, booleanCut, subdive_bevel_update):

        bool_obj = self.act_mod.object
        valid_bool_obj, is_bool_bevel = self.get_valid_boolean_obj(act_obj, bool_obj)

        if valid_bool_obj.name.startswith("TMP_"):
            self.act_mod.object = valid_bool_obj
            rev_bool.modifiers[bool_obj.name].object = valid_bool_obj

        # application du booleen sur le rebool

        bpy.context.view_layer.objects.active = rev_bool
        act_obj.select_set(state=False)
        act_obj.select_set(state=True)



        self.apply_boolean(rev_bool, valid_bool_obj)

        if not booleanCut:
            bevels_limit_method = self.get_bevel_limit_method_(rev_bool)
            if len(bevels_limit_method) == 1 and bevels_limit_method[0] == 'WEIGHT':
                self.update_bevel(rev_bool, is_bool_bevel, subdive_bevel_update)

        # application du booleen sur l'object actif
        bpy.context.view_layer.objects.active = act_obj

        self.apply_boolean(act_obj, valid_bool_obj)

        valid_bool_obj.select_set(state=False)
        rev_bool.select_set(state=True)

        if booleanCut:
            if rev_bool.data.materials:
                rev_bool.data.materials.clear()

            if bool_obj.data.materials:
                for mat in bool_obj.data.materials:
                    rev_bool.data.materials.append(mat)

            bpy.ops.object.join()
            new_obj = bpy.context.active_object
            me = new_obj.data
            bm = bmesh.new()
            bm.from_mesh(me)

            # suppression des edges internes
            boundary_edges = get_edges_boundary_loop(bm, 1)
            for e in bm.edges:
                if e.select and not e in boundary_edges:
                    bm.edges.remove(e)

            for v in bm.verts:
                if v.select and not v.link_edges:
                    bm.verts.remove(v)

            # remove double
            bmesh.ops.remove_doubles(bm, verts = [v for v in bm.verts if v.select], dist = 0.001)

            bm.to_mesh(me)
            bm.free()
            del bm
            me.update()

        else:
            bevels_limit_method = self.get_bevel_limit_method_(act_obj)
            if len(bevels_limit_method) == 1 and bevels_limit_method[0] == 'WEIGHT':
                self.update_bevel(act_obj, is_bool_bevel, subdive_bevel_update)

        if valid_bool_obj.name.startswith("TMP_"):
            bpy.data.objects.remove(valid_bool_obj, do_unlink = True)

        # elif self.addon_prefs.delete_bool_obj:
        elif self.delete_bool_obj:
            users = self.get_users(valid_bool_obj)
            if not users:
                bpy.data.objects.remove(valid_bool_obj, do_unlink = True)

# ---------------------------------------------------------------------------------------------
#
# ---------------------------------------------------------------------------------------------

    def direct_rebool(self, apply):

        bool_objects = [obj for obj in bpy.context.selected_objects if obj != self.act_obj]

        bpy.ops.object.select_all(action = 'DESELECT')

        for bool_obj in bool_objects:

            bool_obj.select_set(state=True)
            # bool_obj.select = True

            # Si notre objet actif ne possede pas de modifier booleen, on en cree un
            if not get_modifier_list(self.act_obj, 'BOOLEAN'):
                self.act_mod = self.add_boolean_modifier(bool_obj)

            else:
                # on verifie que le bool_obj n'est pas deja utilise
                # Si c'est le cas, on le passe en booleen  actif
                # sinon on cree un nouveau booleen
                if not bool_obj in [mod.object for mod in self.act_obj.modifiers if mod.type == 'BOOLEAN']:
                    self.act_mod = self.add_boolean_modifier(bool_obj)
                else:
                    self.act_mod = self.act_obj.modifiers[bool_obj.name]


            # creation de l'objet rebool
            rev_bool = self.create_rebool_obj(self.act_obj)

            if int(apply):
                bool_obj.select_set(state=True)
                self.reverse_boolean(self.act_obj, rev_bool, False, False)

            if not self.delete_bool_obj:
                bool_obj.select_set(state=False)

        if not apply or (apply and not self.delete_bool_obj):
            bpy.ops.object.select_all(action = 'DESELECT')
            for ob in bool_objects:
                ob.select_set(state=True)
                bpy.context.view_layer.objects.active = ob

        else:
            bpy.ops.object.select_all(action = 'DESELECT')

            self.act_obj.select_set(state=True)
            bpy.context.view_layer.objects.active = self.act_obj


# ---------------------------------------------------------------------------------------------
#
# ---------------------------------------------------------------------------------------------

    def apply_all_boolean(self):
        """ Apply all Boolean modifier of the active object """


        OBJS = [obj for obj in bpy.context.selected_objects if [mod for mod in obj.modifiers if mod.type == 'BOOLEAN']]
        bpy.ops.object.select_all(action = 'DESELECT')

        for obj in OBJS:
            bool_modifiers = get_modifier_list(obj, 'BOOLEAN')
            obj.select_set(state=True)
            bpy.context.view_layer.objects.active = obj

            for bool in bool_modifiers:
                self.act_mod = obj.modifiers[bool]
                bool_obj = self.act_mod.object
                Hops_insert = self.get_Hops_insert(bool_obj)
                if Hops_insert:
                    bool_obj = Hops_insert
                bool_obj.select_set(state=True)
                bool_obj_is_hidden = bool_obj.hide_set(True)
                if bool_obj_is_hidden:
                    bool_obj.hide_set(False)


                self.apply_boolean_modifier(obj, False)

                for ob in bpy.context.selected_objects:
                    if ob != obj:
                        ob.select_set(state=False)
                if bool_obj_is_hidden:
                    bool_obj.hide_set(True)

            obj.select_set(state=False)

        bpy.context.view_layer.objects.active = self.act_obj
        self.act_obj.select_set(state=True)



# ---------------------------------------------------------------------------------------------
#
# ---------------------------------------------------------------------------------------------

    def boolean_inset(self):

        self.act_obj.select_set(state=False)
        for obj in bpy.context.selected_objects:
            obj.select_set(state=True)
            bpy.context.view_layer.objects.active = obj

        bool_obj = bpy.context.active_object

        bpy.ops.object.select_all(action='DESELECT')

        self.act_obj.select_set(state=True)
        bpy.context.view_layer.objects.active = self.act_obj
        new_boolean = self.create_modifier(self.act_obj, 'BOOLEAN', "Boolean - Difference")
        new_boolean.object = bool_obj

        has_bevel_modifiers = self.get_modifiers_by_type_and_attr(self.act_obj, 'BEVEL', 'limit_method', 'ANGLE')
        has_triangulate_modifiers = self.get_modifiers_by_type(self.act_obj, 'TRIANGULATE')
        has_weigthed_normal = self.get_modifiers_by_type(self.act_obj, 'WEIGHTED_NORMAL')

        if has_bevel_modifiers:
            bpy.ops.object.modifier_move_up(modifier=new_boolean.name)

        if has_weigthed_normal:
            bpy.ops.object.modifier_move_up(modifier=new_boolean.name)

        if has_triangulate_modifiers:
            bpy.ops.object.modifier_move_up(modifier=new_boolean.name)

        # bevel_angle = False
        # weigthed_normals = False
        # triangulate = False
        # for mod in self.act_obj.modifiers:
        #     if mod.type == 'BEVEL' and mod.limit_method == 'ANGLE':
        #         bevel_angle = True
        #     if mod.type == 'WEIGHTED_NORMAL':
        #         weigthed_normals = True
        #     if mod.type == 'TRIANGULATE':
        #         triangulate = True
        #
        # if bevel_angle:
        #     bpy.ops.object.modifier_move_up(modifier=new_boolean.name)
        # if weigthed_normals:
        #     bpy.ops.object.modifier_move_up(modifier=new_boolean.name)
        # if triangulate:
        #     bpy.ops.object.modifier_move_up(modifier=new_boolean.name)

        bpy.ops.object.select_all(action='DESELECT')
        bool_obj.select_set(state=True)
        bpy.context.view_layer.objects.active = bool_obj
        if bool_obj.modifiers:
            for mod in bool_obj.modifiers:
                bpy.ops.object.modifier_apply(modifier=mod.name)
        bpy.ops.object.mode_set(mode='EDIT')
        bpy.ops.mesh.select_mode(type='FACE')
        bpy.ops.mesh.select_all(action='SELECT')
        bpy.ops.object.mode_set(mode='OBJECT')

        bpy.ops.object.select_all(action='DESELECT')

        self.act_obj.select_set(state=True)
        bpy.context.view_layer.objects.active = self.act_obj
        bpy.ops.object.mode_set(mode='EDIT')
        bpy.ops.mesh.select_mode(type='FACE')
        bpy.ops.mesh.select_all(action='DESELECT')
        # bpy.ops.mesh.select_mode(type='VERT')
        bpy.ops.object.mode_set(mode='OBJECT')

        bpy.ops.object.duplicate_move(OBJECT_OT_duplicate={"linked": False, "mode": 'TRANSLATION'},
                                      TRANSFORM_OT_translate={"value": (0, 0, 0), "orient_type": 'GLOBAL',
                                                              "orient_matrix": ((0, 0, 0), (0, 0, 0), (0, 0, 0)),
                                                              "orient_matrix_type": 'GLOBAL'})

        new_act_obj = bpy.context.active_object

        new_mod = self.get_last_modifier_by_type(new_act_obj, 'BOOLEAN')
        new_mod.operation = 'INTERSECT'
        bpy.ops.object.modifier_apply(apply_as='DATA', modifier=new_mod.name)

        if new_act_obj.modifiers:
            for mod in new_act_obj.modifiers:
                bpy.ops.object.modifier_remove(modifier=mod.name)

        bpy.ops.object.mode_set(mode='EDIT')
        bpy.ops.mesh.delete(type='FACE')
        bpy.ops.object.mode_set(mode='OBJECT')

        # self.exit(context)
        bool_obj.select_set(state=False)
        bpy.data.objects.remove(bool_obj, do_unlink=True)
        new_act_obj.select_set(state=False)

        self.act_obj.select_set(state=True)
        bpy.context.view_layer.objects.active = self.act_obj

        displace_mod = self.get_last_modifier_by_type(self.act_obj, 'BOOLEAN')
        displace_mod.object = new_act_obj

        bpy.ops.object.select_all(action='DESELECT')
        new_act_obj.select_set(state=True)
        new_act_obj.display_type = 'BOUNDS'
        bpy.context.view_layer.objects.active = new_act_obj

        bpy.ops.speedflow.solidify('INVOKE_DEFAULT', True)

        return {'FINISHED'}



# ---------------------------------------------------------------------------------------------
#
# ---------------------------------------------------------------------------------------------

    def modal(self, context, event):
        act_obj = self.act_obj
        # modal_action = self.modal_action

        # if modal_action == 'free':
        if event.type in {'NUMPAD_1','NUMPAD_3','NUMPAD_7','NUMPAD_5'} or event.ctrl and event.type in {'NUMPAD_1','NUMPAD_3','NUMPAD_7','NUMPAD_5'}:
            return {'PASS_THROUGH'}

        if event.alt:
            return {'PASS_THROUGH'}


        if event.alt and event.type in {'LEFTMOUSE','RIGHTMOUSE','MIDDLEMOUSE'}:
            return {'PASS_THROUGH'}


        if event.type in { 'A', 'B','C','D','E','F','G','H','I','J','K','L','M','N','O','P','Q','R','S','T',
                           'U','V','W','X','Y','Z' 'SHIFT', 'LEFTMOUSE', 'RIGHTMOUSE','CTRL','ALT', 'ONE', 'TWO',
                           'SLASH'} and act_obj.mode == 'EDIT':
            return {'PASS_THROUGH'}

        # # VERTEX
        # if event.type == 'TAB' and event.value == 'PRESS':
        #     bpy.ops.wm.tool_set_by_id(name="builtin.select", cycle=False, space_type='VIEW_3D')
        #     obj_mode = any([obj.mode == 'OBJECT' for obj in self.sel])
        #     if obj_mode:
        #         bpy.ops.object.mode_set(mode='EDIT')
        #         if self.act_obj.type == 'MESH':
        #             context.scene.tool_settings.use_mesh_automerge = False
        #     else:
        #         bpy.ops.object.mode_set(mode='OBJECT')
        #         # return {'RUNNING_MODAL'}
        #     self.mode = self.act_obj.mode

        if event.type in {'MIDDLEMOUSE', 'WHEELUPMOUSE', 'WHEELDOWNMOUSE'}:
            return {'PASS_THROUGH'}

        # --------------------------------
        # Selection Operation
        # --------------------------------
        if event.type in self.mouse_actions and event.value =='PRESS':
            bool_obj = self.act_mod.object
            self.act_mod.operation = self.mouse_actions[event.type]

            if self.prefs.auto_rename_modifiers:
                for mod in self.same_modifs.values():
                    mod.name = "Boolean - %s" % mod.operation

        # --------------------------------
        # Apply Modifier
        # --------------------------------

        if event.type == 'A' and event.value == 'PRESS':
            bool_obj = self.act_mod.object
            Hops_insert = self.get_Hops_insert(bool_obj)

            self.apply_boolean_modifier(self.act_obj, event.shift)

            if not get_modifier_list(self.act_obj, 'BOOLEAN'):
                self.exit(context)

                # restore Overlays
                context.space_data.overlay.show_overlays = self.overlay_state
                context.space_data.overlay.show_wireframes = self.overlay_wire_state

                if not self.delete_bool_obj:
                    if Hops_insert:
                        Hops_insert.select_set(state=True)
                        bpy.context.view_layer.objects.active = Hops_insert

                    else:
                        for obj in context.selected_objects:
                            obj.select_set(state=False)
                        bool_obj.select_set(state=True)
                        bpy.context.view_layer.objects.active =bool_obj

                    self.act_obj.select_set(state=False)

                return {'FINISHED'}

            boolean_name = get_modifier_list(self.act_obj, 'BOOLEAN')[-1]
            self.act_mod = self.act_obj.modifiers[boolean_name]
            for obj in context.selected_objects:
                if obj != self.act_obj:
                    obj.select_set(state=False)

            new_bool_obj = self.act_mod.object
            if self.get_Hops_insert(new_bool_obj):
                new_bool_obj = self.get_Hops_insert(new_bool_obj)

            new_bool_obj.select_set(state=True)


        # --------------------------------
        # Switch Between Booleans
        # --------------------------------
        if event.type in {'GRLESS', 'LEFT_ARROW', 'RIGHT_ARROW'} and event.value =='PRESS':
        # if event.type == 'GRLESS' and event.value =='PRESS':
            bool_modifiers = [mod for mod in self.act_obj.modifiers if mod.type == 'BOOLEAN']
            prev_bool_obj = self.act_mod.object
            index = bool_modifiers.index(self.act_mod)
            if event.type == 'GRLESS' and event.shift or event.type == 'RIGHT_ARROW':
            # if event.type == 'GRLESS' and event.shift:
                self.act_mod = bool_modifiers[index + 1 if index != len(bool_modifiers) - 1 else 0]
            else:
                self.act_mod = bool_modifiers[index - 1 if index != 0 else -1]

            new_bool_obj = self.act_mod.object

            if self.get_Hops_insert(prev_bool_obj):
                prev_bool_obj = self.get_Hops_insert(prev_bool_obj)


            # prev_bool_obj.hide_set(True)

            prev_bool_obj.select_set(state=False)



            # new_bool_obj.hide_viewport = False
            new_bool_obj.hide_set(False)

            # if self.boolean_layer != None:
            #     prev_bool_obj.layers[bpy.context.scene.active_layer] = False
            #     self.boolean_layer = None
            #
            # if self.is_invisible:
            #     prev_bool_obj.hide = True
            #     self.is_invisible = False

            if self.get_Hops_insert(new_bool_obj):
                new_bool_obj = self.get_Hops_insert(new_bool_obj)

            new_bool_obj.select_set(state=True)
            context.view_layer.objects.active = new_bool_obj
            # new_bool_obj.select=True

            # self.display_boolean_object(new_bool_obj)

# ---------------------------------
# Special actions
# ---------------------------------
        # --------------------------------
        # Rotate
        # --------------------------------
        # rotation
        if event.value == 'PRESS' and event.type == 'Z':
            bool_obj = self.act_mod.object
            if event.ctrl and event.shift:
                local_rotate(bool_obj, 'Z', 5)
            elif event.shift:
                local_rotate(bool_obj, 'Z', 45)
            else:
                local_rotate(bool_obj, 'Z', 90)
        if event.value == 'PRESS' and event.type == 'Y':
            bool_obj = self.act_mod.object
            if event.ctrl and event.shift:
                local_rotate(bool_obj, 'Y', 5)
            elif event.shift:
                local_rotate(bool_obj, 'Y', 45)
            else:
                local_rotate(bool_obj, 'Y', 90)
        if event.value == 'PRESS' and event.type == 'X':
            bool_obj = self.act_mod.object
            if event.ctrl and event.shift:
                local_rotate(bool_obj, 'X', 5)
            elif event.shift:
                local_rotate(bool_obj, 'X', 45)
            else:
                local_rotate(bool_obj, 'X', 90)


        # --------------------------------
        # Rebool
        # --------------------------------
        if event.type in {'R', 'W'} and event.value == 'PRESS':
            bool_obj = self.act_mod.object

            rev_bool = self.create_rebool_obj(self.act_obj)
            bool_obj.select_set(state=True)
            # bool_obj.select = True

            self.reverse_boolean(self.act_obj, rev_bool, event.type == 'W', event.shift)

            if not get_modifier_list(context.active_object, 'BOOLEAN'):
                self.exit(context)

                # restore Overlays
                context.space_data.overlay.show_overlays = self.overlay_state
                context.space_data.overlay.show_wireframes = self.overlay_wire_state

                # if not self.addon_prefs.delete_bool_obj:
                if not self.delete_bool_obj:
                    self.act_obj.select_set(state=False)
                    rev_bool.select_set(state=False)
                    bool_obj.select_set(state=True)
                    bpy.context.view_layer.objects.active = bool_obj


                return {'FINISHED'}

            rev_bool.select_set(state=False)
            # rev_bool.select = False
            boolean_name = get_modifier_list(bpy.context.active_object, 'BOOLEAN')[-1]
            self.act_mod = self.act_obj.modifiers[boolean_name]
            new_bool_obj = self.act_mod.object

            if not self.delete_bool_obj:
                bool_obj.select_set(state=False)

            new_bool_obj.select_set(state=True)

        # --------------------------------
        # Inset
        # --------------------------------
        if event.type == 'I' and event.value == 'PRESS':

            bool_operaton_mode = self.act_mod.operation
            bool_obj = self.act_mod.object

            bpy.ops.object.select_all(action='DESELECT')

            bool_obj.select_set(state=True)
            bpy.context.view_layer.objects.active = bool_obj

            if bool_obj.modifiers:
                for mod in bool_obj.modifiers:
                    bpy.ops.object.modifier_apply(modifier=mod.name)

            bpy.ops.object.mode_set(mode='EDIT')
            bpy.ops.mesh.select_mode(type='FACE')
            bpy.ops.mesh.select_all(action='SELECT')
            bpy.ops.object.mode_set(mode='OBJECT')


            bpy.ops.object.select_all(action='DESELECT')

            # On deselectionne les faces
            self.act_obj.select_set(state=True)
            bpy.context.view_layer.objects.active = self.act_obj
            bpy.ops.object.mode_set(mode='EDIT')
            bpy.ops.mesh.select_mode(type='FACE')
            bpy.ops.mesh.select_all(action='DESELECT')
            bpy.ops.object.mode_set(mode='OBJECT')


            bpy.ops.object.duplicate_move(OBJECT_OT_duplicate={"linked": False, "mode": 'TRANSLATION'},
                                          TRANSFORM_OT_translate={"value": (0, 0, 0), "orient_type": 'GLOBAL',
                                                                  "orient_matrix": ((0, 0, 0), (0, 0, 0), (0, 0, 0)),
                                                                  "orient_matrix_type": 'GLOBAL'})

            new_act_obj = context.active_object

            new_mod = self.get_last_modifier_by_type(new_act_obj, 'BOOLEAN')
            new_mod.operation = 'INTERSECT'
            bpy.ops.object.modifier_apply(apply_as='DATA', modifier=new_mod.name)

            if new_act_obj.modifiers:
                for mod in new_act_obj.modifiers:
                    bpy.ops.object.modifier_remove(modifier=mod.name)


            bpy.ops.object.mode_set(mode='EDIT')
            bpy.ops.mesh.delete(type='FACE')
            bpy.ops.object.mode_set(mode='OBJECT')

            self.exit(context)
            bool_obj.select_set(state=False)
            bpy.data.objects.remove(bool_obj, do_unlink=True)
            new_act_obj.select_set(state=False)

            self.act_obj.select_set(state=True)
            bpy.context.view_layer.objects.active = self.act_obj

            displace_mod = self.get_last_modifier_by_type(act_obj, 'BOOLEAN')
            displace_mod.object = new_act_obj
            displace_mod.operation = bool_operaton_mode

            bpy.ops.object.select_all(action='DESELECT')
            new_act_obj.select_set(state=True)
            new_act_obj.display_type = 'BOUNDS'
            bpy.context.view_layer.objects.active = new_act_obj

            bpy.ops.speedflow.solidify('INVOKE_DEFAULT', True)

            return {'FINISHED'}

        # --------------------------------
        # Delete Bool Object
        # --------------------------------
        if event.type == 'E' and event.value == 'PRESS':
            self.delete_bool_obj = not self.delete_bool_obj

        # --------------------------------
        # Change Modifier Position
        # --------------------------------
        if event.type == 'UP_ARROW' and event.value == 'PRESS':
            bpy.ops.object.modifier_move_up(modifier=self.act_mod.name)

        if event.type == 'DOWN_ARROW' and event.value == 'PRESS':
            bpy.ops.object.modifier_move_down(modifier=self.act_mod.name)


        # --------------------------------
        # Hide Modifier/Keymaps
        # --------------------------------
        if event.type == 'H' and event.value == 'PRESS':
            if event.shift:
                self.prefs.beginner_mode = not self.prefs.beginner_mode
                context.area.tag_redraw()

            elif event.ctrl:
                self.addon_prefs.display_keymaps = not self.addon_prefs.display_keymaps

            else:
                self.act_mod.show_viewport = not self.act_mod.show_viewport

        # ---------------------------------
        # Use Bound
        # ---------------------------------
        if event.type == 'J' and event.value == 'PRESS':
            self.use_bound = self.addon_prefs.use_bound = not self.addon_prefs.use_bound

            bool_modifiers = get_modifier_list(self.act_obj, 'BOOLEAN')
            if bool_modifiers:
                for bool in bool_modifiers:
                    obj_bool = self.act_obj.modifiers[bool].object

                    if not self.use_bound :

                        obj_bool.display_type = 'WIRE'
                    elif self.use_bound:
                        self.use_bound = True
                        obj_bool.display_type = 'BOUNDS'

        # ---------------------------------
        # Show modifiers
        # ---------------------------------

        if event.shift and event.type == 'RIGHT_ARROW' and event.value == 'PRESS':
            to_show = self.get_first_modifier_by_dict(act_obj, {'show_viewport' : False})
            if to_show is not None:
                to_show.show_viewport = True

        # ---------------------------------
        # Hide modifiers
        # ---------------------------------

        if event.shift and event.type == 'LEFT_ARROW' and event.value == 'PRESS':
            to_show = self.get_last_modifier_by_dict(act_obj, {'show_viewport' : True})
            if to_show is not None:
                to_show.show_viewport = False

        # ---------------------------------
        # Toggle Hide all modifiers
        # ---------------------------------
        if event.type == 'QUOTE' and event.value == 'PRESS':
            self.show_hide_all_modifiers(act_obj, self.act_mod)

        # ---------------------------------
        # Toggle Local View
        # ---------------------------------
        if event.type == 'NUMPAD_SLASH' and event.value == 'PRESS':
            bpy.ops.view3d.localview()


    # ---------------------------------
    # Shading / Snap
    # ---------------------------------

        # ---------------------------------------
        # 1 - Toggle Keep Wire in Modal
        # ---------------------------------------
        if event.type == 'ONE' and event.value == 'PRESS':
            self.prefs.keep_wire = not self.prefs.keep_wire

        # --------------------------------------
        # 2 - Toggle wire in the modal
        # --------------------------------------
        if event.type == 'TWO' and event.value == 'PRESS':
            self.shading_wire_in_modal(context, self)

        # ---------------------------------
        # Toggle XRAY
        # ---------------------------------
        if event.type == 'THREE' and event.value == 'PRESS':
                self.shading_xray(context, self)

        # ---------------------------------
        # Toggle shading mode type
        # ---------------------------------
        if event.type == 'FOUR' and event.value == 'PRESS':
            self.shading_mode_type(context, self)

        # ---------------------------------
        # Toggle Bool Shading Mode
        # ---------------------------------
        if event.type == 'FIVE' and event.value == 'PRESS':
            self.shading_random_mode(context, self)

        # ---------------------------------
        # Toggle OVERLAYS
        # ---------------------------------
        if event.type == 'SIX' and event.value == 'PRESS':
            self.shading_overlays(context, self)

        # ---------------------------------
        # Toggle FACE ORIENTATION
        # ---------------------------------
        if event.type == 'SEVEN' and event.value == 'PRESS':
            self.shading_face_orientation(context, self)

        # ---------------------------------
        # Toggle Hide Grid
        # ---------------------------------
        if event.type == 'EIGHT' and event.value == 'PRESS':
            self.shading_hide_grid(context, self)

        # ---------------------------------
        # 9 - Toggle Bool Shading Mode
        # ---------------------------------
        if event.type == 'NINE' and event.value == 'PRESS':
            self.shading_bool_mode(context, self)

        # ---------------------------------
        # Toggle Activate Snap
        # ---------------------------------
        if event.type == 'F2' and event.value == 'PRESS':
            self.snap_activate_snap(context, self)

        # ---------------------------------
        # Toggle Snap Vertex
        # ---------------------------------
        if event.type == 'F3' and event.value == 'PRESS':
            self.snap_vertex(context, self)

        # ---------------------------------
        # Toggle Snap Face
        # ---------------------------------
        if event.type == 'F4' and event.value == 'PRESS':
            self.snap_grid(context, self)

        # ---------------------------------
        # Toggle Snap Grid
        # ---------------------------------
        if event.type == 'F5' and event.value == 'PRESS':
            self.snap_grid(context, self)

        # ---------------------------------
        # Origin to selection
        # ---------------------------------
        if event.type == 'F6' and event.value == 'PRESS':
            self.snap_origin_to_selection(context)

        # ---------------------------------
        # Toggle Apply Origin
        # ---------------------------------
        if event.type == 'F7' and event.value == 'PRESS':
            self.snap_origin_to_grid(context)

        # ---------------------------------
        # Call pie menu (Generic)
        # ---------------------------------
        if event.type == 'SPACE' and event.value == 'PRESS':
            self.exit(context)
            if self.prefs.choose_menu_type == 'pie':
                bpy.ops.wm.call_menu_pie('INVOKE_DEFAULT', name="SPEEDFLOW_MT_pie_menu")
            else:
                bpy.ops.wm.call_menu('INVOKE_DEFAULT', name="SPEEDFLOW_MT_simple_menu")
            return {'FINISHED'}

        # --------------------------------
        # Delete Action
        # --------------------------------
        if event.type in {'DEL', 'BACK_SPACE'} and event.value == 'PRESS':
            bool_modifiers = get_modifier_list(self.act_obj, 'BOOLEAN')
            if len(bool_modifiers) > 1:
                prev_bool_obj = self.act_mod.object
                self.act_obj.modifiers.remove(self.act_mod)
                boolean_name = get_modifier_list(self.act_obj, 'BOOLEAN')[-1]
                self.act_mod = self.act_obj.modifiers[boolean_name]
                new_bool_obj = self.act_mod.object
                prev_bool_obj.select_set(state=False)
                # prev_bool_obj.select = False
                if self.get_Hops_insert(new_bool_obj):
                    self.get_Hops_insert(new_bool_obj).select_set(state=True)
                    # self.get_Hops_insert(new_bool_obj).select = True
                else:
                    new_bool_obj.select_set(state=True)
                    # new_bool_obj.select = True

            else:
                bool_obj = self.act_mod.object
                self.act_obj.modifiers.remove(self.act_mod)
                self.exit(context)

                bool_obj.select_set(state=True)
                context.view_layer.objects.active = bool_obj

                # restore Overlays
                context.space_data.overlay.show_overlays = self.overlay_state
                context.space_data.overlay.show_wireframes = self.overlay_wire_state

                # bool_obj.select = True
                # context.scene.objects.active = bool_obj
                return {'FINISHED'}

        # --------------------------------
        # Exit Action
        # --------------------------------
        # if event.type in {'LEFTMOUSE', 'RIGHTMOUSE', 'ESC', 'SPACE'} and event.value == 'PRESS':
        if event.type in {self.key_confirm, self.key_cancel, 'ESC', 'SPACE'} and event.value == 'PRESS':
            self.exit(context)

            boolean_name = get_modifier_list(self.act_obj, 'BOOLEAN')[-1]
            self.act_mod = self.act_obj.modifiers[boolean_name]

            self.act_obj.select_set(state=False)
            bool_obj = self.act_mod.object
            bool_obj.select_set(state=True)
            context.view_layer.objects.active = bool_obj

            # restore Overlays
            context.space_data.overlay.show_overlays = self.overlay_state
            context.space_data.overlay.show_wireframes = self.overlay_wire_state

            #move bool objects to bool_objects collection
            # bool_objects = self.get_boolean_objects(self.act_obj)
            #
            # for bool_ob in bool_objects:
            #     bool_ob.color = self.bools_color
            #
            #     for coll in bool_ob.users_collection:
            #         coll.objects.unlink(bool_ob)
            #
            #     bpy.data.collections['Bool_Objects'].objects.link(bool_ob)

            return {'CANCELLED'}

        return {'RUNNING_MODAL'}

    def invoke(self, context, event):

        SF = context.window_manager.SF

        act_obj = context.object
        sel = self.filter_objects_by_type(context.selected_objects, {'MESH'})
        sel_hidden = self.sel_hidden

        prefs = self.prefs
        self.act_obj = act_obj
        self.sel = sel
        self.mode = self.act_obj.mode
        self.shading_viewport_type = context.space_data.shading.type
        self.overlay_state = context.space_data.overlay.show_overlays
        self.overlay_wire_state = context.space_data.overlay.show_wireframes
        self.parent_bool_objects = self.prefs.boolean.parent_bool_objects
        self.shading_random = context.space_data.shading.color_type
        self.ref_object = [obj for obj in context.scene.objects if obj.get("is_ref")]

        self.bool_mode_obj = False
        if len(self.sel) == 1:
            # has_boolean_modifiers = self.get_modifiers_by_type(self.act_obj, 'BOOLEAN')
            # if has_boolean_modifiers:
            #     self.ref_object = self.act_obj
                # print("REF_OBJECT: ", self.ref_object)

            # On check si les objets de la scene utilisent l objet actif comme boolean
            for obj in context.scene.objects:
                for mod in obj.modifiers:
                    if mod.type == 'BOOLEAN' and mod.object == self.act_obj:
                        self.bool_mode_obj = True

            # on copie l object actif sur l objet ref dans ref_object
            if self.bool_mode_obj :
                if event.ctrl:
                    bpy.ops.view3d.cursor3d('INVOKE_DEFAULT')

                # On affiche les enfant caches de l objet actif et on les selectionne
                has_children = False
                if self.act_obj.children:
                    has_children = True
                    self.hide_unhide(self.act_obj, states=False, select=True) # state = afficher / select=selectionner

                # on duplique l objet actif
                bpy.ops.object.duplicate_move()

                self.new_bool_obj = context.active_object

                if has_children:
                    bpy.ops.object.select_all(action='DESELECT')
                    context.view_layer.objects.active = self.new_bool_obj
                    self.new_bool_obj.select_set(state=True)

                bpy.ops.object.select_all(action='DESELECT')

                self.add_boolean_to_ref_object(self.new_bool_obj)

                context.view_layer.objects.active = self.new_bool_obj
                self.new_bool_obj.select_set(state=True)
                self.new_bool_obj.display_type = 'BOUNDS'

                if event.shift:
                    cursor_rot = bpy.context.scene.cursor.rotation_euler.copy()
                    bpy.context.object.rotation_euler = cursor_rot
                    bpy.ops.view3d.snap_selected_to_cursor(use_offset=False)

                if event.alt and event.shift:
                    bpy.ops.transform.resize('INVOKE_DEFAULT')

                if event.ctrl:
                    bpy.ops.view3d.cursor3d('INVOKE_DEFAULT')
                    bpy.ops.view3d.snap_selected_to_cursor(use_offset=False)
                    bpy.ops.transform.translate('INVOKE_DEFAULT')

                self.bool_mode_obj = False
                return {'FINISHED'}

        # if len(sel) >= 2:
        #     if self.parent_bool_objects:
        #         bpy.ops.object.parent_set(type='OBJECT', keep_transform=True)

        if context.area.type == 'VIEW_3D':

            # Si un modifier 'BOOLEAN' ne possede pas d'objet, on le supprime
            for mod in act_obj.modifiers:
                if mod.type == 'BOOLEAN' and not mod.object:
                    self.remove_modifier(act_obj, mod)

            if event.shift or event.ctrl or event.alt:
                input = (event.shift, event.ctrl, event.alt)

                extend_valid_input = [(True, True, False),
                                      (False, True, True),
                                      (True, True, True),
                                      ]

                if check_valid_input(input) or input in extend_valid_input:

                    direct_action = {(True, True, True): ['boolean_inset', ()],
                                    (True, True, False): ['direct_rebool', ("1")],
                                    (False, True, True): ['direct_rebool', ("0")],
                                    (True, False, False): ['apply_all_boolean', ()],
                                    (False, True, False): [remove_all_modifier_by_type,
                                                           (context, 'BOOLEAN')],
                                    (False, False, True): [hide_all_modifier_by_type,
                                                           (context, 'BOOLEAN')],


                                    }

                    function, args = direct_action[input]

                    if input in [(False, True, False), (False, False, True)]:
                        function(*args)

                    else:
                        getattr(self, function)(*args)

                return {'FINISHED'}

            # bool_mode_obj = False
            if len(sel) == 1:

                # if self.ref_object:
                #     bpy.ops.object.select_all(action='DESELECT')
                #     bpy.context.view_layer.objects.active = self.ref_object[0]
                #     self.ref_object[0].select_set(state=True)


                bool_modifiers = get_modifier_list(act_obj, 'BOOLEAN')
                if bool_modifiers:
                    boolean_name = bool_modifiers[-1]
                    self.act_mod = act_obj.modifiers[boolean_name]
                    bool_object = self.act_mod.object

                    Hops_insert = self.get_Hops_insert(bool_object)
                    if Hops_insert:
                        Hops_insert.select_set(state=True)
                    else:
                        bool_object.select_set(state=True)
                        bool_object.hide_set(False)

                else:
                    # self.ref_object = self.act_obj
                    # del (self.ref_object[:])
                    # self.ref_object.append(self.act_obj)
                    self.report({'WARNING'}, "Need second object")
                    return {'FINISHED'}

            else:
                del (self.ref_object[:])
                self.ref_object.append(self.act_obj)
                print("REF_OBJECT NAME: ", self.ref_object)
                bool_objects = self.get_boolean_objects(act_obj)

                # Check si collection existe, si non, on la cree
                if self.prefs.add_bool_objects_to_collection:
                    if not bpy.data.collections.get('Bool_Objects'):
                        coll = bpy.data.collections.new("Bool_Objects")
                        bpy.context.collection.children.link(coll)


                for bool_ob in bool_objects:
                    if bool_ob.data.users == 1:
                        if not bool_ob.children:
                            if prefs.auto_apply_scale:
                                self.apply_transform_scale(context, bool_ob)

                        if self.prefs.add_bool_objects_to_collection:
                            in_collection = bpy.data.collections['Bool_Objects'].objects.get(bool_ob.name)
                            if not in_collection:
                                self.unlink_object_from_scene(bool_ob)
                                bpy.data.collections['Bool_Objects'].objects.link(bool_ob)
                                # bpy.context.scene.collection.objects.unlink(bool_ob)


                for obj in bool_objects:
                    obj.color = self.bools_color

                    # for coll in obj.users_collection:
                    #     coll.objects.unlink(obj)
                    # obj.users_collection

                    # bpy.data.collections['Bool_Objects'].objects.unlink(obj)
                    # bpy.data.collections['Bool_Objects'].objects.link(obj)

                    # si l'objet booleen selectionne n'est pas utilise par un modifier Boolean,
                    # on cree un nouveau Boolean pour cet objet

                    if not self.get_first_modifier_by_type_and_attr(act_obj, 'BOOLEAN', 'object', obj):
                        self.add_boolean_modifier(obj)

                    Hops = self.get_Hops_insert(obj)
                    if Hops:
                        Hops.select_set(state=False)
                        # Hops.select = False
                    else:
                        obj.select_set(state=False)
                        obj.hide_set(False)

                Hops_insert = self.get_Hops_insert(bool_objects[-1])
                if Hops_insert:
                    Hops_insert.select_set(state=True)
                else:
                    bool_objects[-1].select_set(state=True)
                    # bool_objects[-1].select = True

                for mod in act_obj.modifiers:
                    if mod.type == 'BOOLEAN' and mod.object in bool_objects:
                        self.act_mod = mod
                        break


                # self.display_boolean_object(self.act_mod.object)

            for mod in act_obj.modifiers:
                if mod is not None:
                    if self.prefs.add_weighted_normal:
                        has_weighted = self.get_modifiers_by_type(self.act_obj, 'WEIGHTED_NORMAL')
                        if has_weighted:
                            self.move_weighted_normals_down(context, act_obj)



            # setup display
            display_type = None
            if self.prefs.show_wire_in_modal and act_obj.display_type == 'BOUNDS':
                display_type = 'WIRE'

            if prefs.show_wire_in_modal:
                context.space_data.overlay.show_overlays = True
                context.space_data.overlay.show_wireframes = False

            self.setup_display(display_type)

            self.setup_draw_handler(context)
            context.window_manager.modal_handler_add(self)
            return {'RUNNING_MODAL'}
 
        else:
            self.report({'WARNING'}, "View3D not found, cannot run operator")
            return {'FINISHED'}

def register():
    try:
        bpy.utils.register_class(SPEEDFLOW_OT_boolean)
    except:
        print("Boolean already registred")

def unregister():
    bpy.utils.unregister_class(SPEEDFLOW_OT_boolean)