# -*- coding:utf-8 -*-

# Speedflow Add-on
# Copyright (C) 2018 Cedric Lepiller aka Pitiwazou & Legigan Jeremy AKA Pistiwique and Stephen Leger
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

# <pep8 compliant>

import bpy
from math import radians
from mathutils import Matrix
from bpy_extras import view3d_utils
from bpy.types import Operator
from bpy.props import (FloatProperty,
                       IntProperty,
                       EnumProperty,
                       BoolProperty,
                       StringProperty
                       )
from .utils.text_to_draw import *
from .utils.functions import *


class SPEEDFLOW_OT_rotate(SpeedApi, Operator):
    """
    ROTATE
    
    CLICK - Adjust
    SHIFT - Apply Rotation
    CTRL  - Clear Rotation
    """
    bl_idname = "speedflow.rotate"
    bl_label = "Modal Rotate"
    bl_options = {'REGISTER', 'UNDO'}


    axis : IntProperty(default=0)
    update_up : IntProperty(default=0)
    update_down : IntProperty(default=0)
    event_enabled : BoolProperty(default=False)
    count = 0

    def __init__(self):
        SpeedApi.__init__(self)
        self.modal_action = 'rotate'

# ---------------------------------------------------------------------------------------------
#
# ---------------------------------------------------------------------------------------------

    def setup_axis(self):
        if bpy.context.space_data.transform_orientation == 'LOCAL':
            activeObj = bpy.context.active_object
            activeObjMatrix = activeObj.matrix_world
            theAxis = (activeObjMatrix[0][self.axis], activeObjMatrix[1][self.axis], activeObjMatrix[2][self.axis])
        
        else:
            theAxis = [0, 0, 0]
            theAxis[self.axis] = 1
            
        return theAxis

# ---------------------------------------------------------------------------------------------
#
# ---------------------------------------------------------------------------------------------

    def update_properties(self):
        self.update_up = 0
        self.update_down = 0

# ---------------------------------------------------------------------------------------------
#
# ---------------------------------------------------------------------------------------------

    def get_angle_rotation(self, obj, value):
        axis = ['X', 'Y', 'Z']
        objectMatrix = obj.matrix_world
        rot_mat = Matrix.Rotation(value, 4, axis[self.axis])
        
        return rot_mat

# ---------------------------------------------------------------------------------------------
#
# ---------------------------------------------------------------------------------------------

    def set_cursor_location(self, event):
        scene = bpy.context.scene
        region = bpy.context.region
        rv3d = bpy.context.region_data
        mouse_co = event.mouse_region_x, event.mouse_region_y
        depth = view3d_utils.region_2d_to_vector_3d(region, rv3d, mouse_co)
        emp_co = view3d_utils.region_2d_to_location_3d(region, rv3d, mouse_co, depth)
        bpy.ops.object.select_all(action='DESELECT')
        bpy.ops.object.empty_add(type='PLAIN_AXES', radius=0.2)
        bpy.context.object.location = emp_co
 
        for obj in bpy.context.selected_objects:
            obj.name = "Empty_cursor"
 
        # Lock rotations
        bpy.context.object.lock_rotation = (True, True, True)
 
        # Activate Snap
        bpy.context.scene.tool_settings.use_snap = True
        bpy.context.scene.tool_settings.snap_element = 'VERTEX'
        bpy.context.scene.tool_settings.snap_target = 'MEDIAN'
    
# ---------------------------------------------------------------------------------------------
#
# ---------------------------------------------------------------------------------------------
    def modal(self, context, event):
        context.area.tag_redraw()
        # MPM = context.window_manager.MPM
        event_list = ["x", "y", "z"]
        pivot_p = ['BOUNDING_BOX_CENTER', 'CURSOR', 'INDIVIDUAL_ORIGINS', 'MEDIAN_POINT', 'ACTIVE_ELEMENT']
        self.count += 1
        
        
        if self.count == 1 and self.modal_action == 'cursor':
            bpy.ops.transform.translate('INVOKE_DEFAULT')

        # if event.type == 'LEFTMOUSE' and self.modal_action == 'cursor':
        if event.type == self.key_confirm and self.modal_action == 'cursor':
            bpy.ops.view3d.snap_cursor_to_selected()
            bpy.ops.object.delete()
            for obj in self.obj_list:
                obj.select_set(state=True)
            context.view_layer.objects.active = self.act_obj
            self.modal_action = 'rotate'
            self.update_properties()
            self.mouse_x = event.mouse_x
                

        if event.type == 'MIDDLEMOUSE':
            return {'PASS_THROUGH'}

        # if event.type in {'RET', 'NUMPAD_ENTER', 'LEFTMOUSE'} and event.value == 'PRESS':
        if event.type in {'RET', 'NUMPAD_ENTER', self.key_confirm} and event.value == 'PRESS':
            # if event.alt and event.type == 'LEFTMOUSE':
            if event.alt and event.type == self.key_confirm:
                return {'PASS_THROUGH'}
            
            elif self.modal_action == 'rotate':
                self.update_up = 0
                self.update_down = 0
                self.modal_action = 'free'
            
            else:
                bpy.ops.object.mode_set(mode=self.mode)
                bpy.context.space_data.pivot_point = self.pivot
                bpy.context.space_data.transform_orientation = self.orientation
                bpy.types.SpaceView3D.draw_handler_remove(self._rotate_handle, 'WINDOW')
                return {'FINISHED'}
                
        # --------------------------------
        # Mode Rotation
        # --------------------------------
        if event.type == 'S' and event.value == 'PRESS':
            self.modal_action = 'rotate'
            self.mouse_x = event.mouse_x
        
        # --------------------------------
        # Change Axis
        # --------------------------------
        if event.unicode and event.unicode in event_list:
            self.axis = event_list.index(event.unicode)
            self.modal_action = 'rotate'
            self.mouse_x = event.mouse_x
        
        # --------------------------------
        # Change Transform Orientation
        # --------------------------------
        if event.type == 'D' and event.value == 'PRESS':
            if bpy.context.space_data.transform_orientation == 'GLOBAL':
                bpy.context.space_data.transform_orientation = 'LOCAL'
            else:
                bpy.context.space_data.transform_orientation = 'GLOBAL'
        
        # --------------------------------
        # Change Transform Orientation
        # --------------------------------
        if event.type == 'F' and event.value == 'PRESS':
            self.set_cursor_location(event)
            self.modal_action = 'cursor'
            bpy.context.space_data.pivot_point = 'CURSOR'
            self.count = 0
               
        # --------------------------------
        # Setup Values
        # --------------------------------
        if self.work_tool == 'pen':
            if event.type in {'WHEELUPMOUSE', 'WHEELDOWNMOUSE'}:
                return {'PASS_THROUGH'}
            
            if event.type in {'LEFT_SHIFT', 'RIGHT_SHIFT', 'LEFT_CTRL', 'RIGHT_CTRL'}:
                if event.value == 'PRESS':
                    if not self.event_enabled:
                        self.mouse_x = event.mouse_x
                        self.update_properties()
                        self.event_enabled = True
                
                elif event.value == 'RELEASE':
                    self.mouse_x = event.mouse_x
                    self.update_properties()
                    self.event_enabled = False
                    
            if event.type == 'MOUSEMOVE' and self.modal_action == 'rotate':
                delta = (event.mouse_x - self.mouse_x)/3 * self.modal_speed
                
                if event.ctrl:
                    angle = 5
                    value = radians(5)
                 
                elif event.shift:
                    angle = 10
                    value = radians(10)
                 
                else:
                    angle = 45
                    value = radians(45)
                   
                axis = self.setup_axis()
                    
                if self.update_up == self.update_down == 0:
                    if delta >= angle:
                        if bpy.context.space_data.pivot_point == 'INDIVIDUAL_ORIGINS' and bpy.context.space_data.transform_orientation == 'LOCAL':
                            for obj in context.selected_objects:
                                obj.matrix_world *= self.get_angle_rotation(obj, value)
                        else:
                            bpy.ops.transform.rotate(value=value, axis=(axis[0], axis[1], axis[2]))
  
                        self.update_up += (angle*2)
                        self.update_down += angle
                        
                    elif delta <= -angle:
                        if bpy.context.space_data.pivot_point == 'INDIVIDUAL_ORIGINS' and bpy.context.space_data.transform_orientation == 'LOCAL':
                            for obj in context.selected_objects:
                                obj.matrix_world *= self.get_angle_rotation(obj, -value)
                        else:
                            bpy.ops.transform.rotate(value=-value, axis=(axis[0], axis[1], axis[2]))
                        
                        self.update_up -= angle
                        self.update_down -= (angle*2)
     
                elif delta >= self.update_up:
                    if bpy.context.space_data.pivot_point == 'INDIVIDUAL_ORIGINS' and bpy.context.space_data.transform_orientation == 'LOCAL':
                        for obj in context.selected_objects:
                            obj.matrix_world *= self.get_angle_rotation(obj, value)
                    else:
                        bpy.ops.transform.rotate(value=value, axis=(axis[0], axis[1], axis[2]))
                    
                    self.update_up += angle
                    self.update_down += angle
     
                elif delta <= self.update_down:
                    if bpy.context.space_data.pivot_point == 'INDIVIDUAL_ORIGINS' and bpy.context.space_data.transform_orientation == 'LOCAL':
                        for obj in context.selected_objects:
                            obj.matrix_world *= self.get_angle_rotation(obj, -value)
                    else:
                        bpy.ops.transform.rotate(value=-value, axis=(axis[0], axis[1], axis[2]))
                    
                    self.update_up -= angle
                    self.update_down -= angle
                
        
        elif self.work_tool == 'mouse':
            if self.modal_action == 'free' and event.type in {'WHEELUPMOUSE', 'WHEELDOWNMOUSE'}:
                return {'PASS_THROUGH'}
            
            else:
                if event.ctrl:
                    value = radians(5)
                    
                elif event.shift:
                    value = radians(10)
                    
                else:
                    value = radians(45)
                
                axis = self.setup_axis()
                
                if event.type == 'WHEELUPMOUSE':
                    if bpy.context.space_data.pivot_point == 'INDIVIDUAL_ORIGINS' and bpy.context.space_data.transform_orientation == 'LOCAL':
                        for obj in context.selected_objects:
                            obj.matrix_world *= self.get_angle_rotation(obj, value)
                    else:
                        bpy.ops.transform.rotate(value=value, axis=(axis[0], axis[1], axis[2]))
                        
                if event.type == 'WHEELDOWNMOUSE':
                    if bpy.context.space_data.pivot_point == 'INDIVIDUAL_ORIGINS' and bpy.context.space_data.transform_orientation == 'LOCAL':
                        for obj in context.selected_objects:
                            obj.matrix_world *= self.get_angle_rotation(obj, -value)
                    else:
                        bpy.ops.transform.rotate(value=-value, axis=(axis[0], axis[1], axis[2]))
                    
        # --------------------------------
        # Switch Between Arrays
        # --------------------------------
        if (event.type == 'GRLESS' or event.type == 'LEFT_ARROW') and event.value=='PRESS':
            index = pivot_p.index(bpy.context.space_data.pivot_point)
            bpy.context.space_data.pivot_point = pivot_p[index-1 if index != 0 else -1]

            
        if (event.type == 'GRLESS' and event.shift) or (event.type == 'RIGHT_ARROW' and event.value=='PRESS'):
            index = pivot_p.index(bpy.context.space_data.pivot_point)
            bpy.context.space_data.pivot_point = pivot_p[index+1 if index != len(pivot_p)-1 else 0]

        # ---------------------------------
        # Restart Speedflow
        # ---------------------------------
        if event.type == 'TAB' and event.value == 'PRESS':
            bpy.ops.object.mode_set(mode='OBJECT')
            bpy.types.SpaceView3D.draw_handler_remove(self._rotate_handle, 'WINDOW')
            # self.exit_subsurf()
            bpy.ops.wm.call_menu_pie('INVOKE_DEFAULT', name="view3d.speedflow_ui_menu")
            return {'FINISHED'}

        # --------------------------------
        # Exit Action
        # --------------------------------
        # if event.type in {'RIGHTMOUSE', 'ESC', 'SPACE'} and event.value == 'PRESS':
        if event.type in {self.quit, 'ESC', 'SPACE'} and event.value == 'PRESS':
            # if event.alt and (event.type == 'LEFTMOUSE' or event.type == 'RIGHTMOUSE'):
            if event.alt and (event.type == self.key_confirm or event.type == self.quit):
                return {'PASS_THROUGH'}

            # elif self.modal_action != 'free' and (event.type == 'LEFTMOUSE' or event.type == 'ESC'):
            elif self.modal_action != 'free' and (event.type == self.key_confirm or event.type == 'ESC'):
                self.modal_action = 'free'
                
            else:
                bpy.ops.object.mode_set(mode=self.mode)
                bpy.context.space_data.pivot_point = self.pivot
                bpy.context.space_data.transform_orientation = self.orientation
                bpy.types.SpaceView3D.draw_handler_remove(self._rotate_handle, 'WINDOW')
                return {'CANCELLED'}
 
 
        return {'RUNNING_MODAL'}

# ---------------------------------------------------------------------------------------------
#
# ---------------------------------------------------------------------------------------------
 
    def invoke(self, context, event):
        # Validating
        self.key_confirm = get_addon_preferences().valid

        if context.window_manager.keyconfigs.active.preferences.select_mouse  == "LEFT":
        # if context.preferences.inputs.select_mouse == "LEFT":
            self.key_confirm = "LEFTMOUSE"

        self.quit = "RIGHTMOUSE" if self.key_confirm == "LEFTMOUSE" else "LEFTMOUSE"

        if context.area.type == 'VIEW_3D':
            # MPM = context.window_manager.MPM
            self.modal_speed = get_addon_preferences().modal_speed
            self.work_tool = get_addon_preferences().work_tool
            self.default_axis = get_addon_preferences().axis_rotation




            # self.default_axis = ['X']


            self.mode = context.object.mode
            bpy.ops.object.mode_set(mode='OBJECT')
            if context.object:
                edit_mode=False
                if event.shift or event.ctrl or event.alt:
                    for obj in context.selected_objects:
                        bpy.context.view_layer.objects.active = obj
                            
                        #If Edit
                        if bpy.context.object.mode == "EDIT":
                            edit_mode=True
                            bpy.ops.object.mode_set(mode = 'OBJECT')
                            
                        # SHIFT
                        if event.shift:
                            bpy.ops.object.transform_apply(location=False, rotation=True, scale=False)

                        # CTRL
                        if event.ctrl:
                            bpy.ops.object.rotation_clear(clear_delta=False)

                    #Comeback in Edit Mode            
                    if edit_mode == True:
                        bpy.ops.object.mode_set(mode = 'EDIT')
                
                else:
                    self.modal_action = 'rotate'
                    bpy.ops.object.transform_apply(location=False, rotation=False, scale=True)
                    self.orientation = bpy.context.space_data.transform_orientation
                    self.pivot = bpy.context.space_data.pivot_point
                    self.axis = int(self.default_axis)
                    self.act_obj = context.active_object
                    self.obj_list = [obj for obj in context.selected_objects]
                    
                    if bpy.context.space_data.transform_orientation not in {'GLOBAL', 'LOCAL'}:
                        bpy.context.space_data.transform_orientation = 'GLOBAL'
                        
                    context.window_manager.modal_handler_add(self)
                    
                    args = (self, context, 'ROTATE')
                    self.mouse_x = event.mouse_x
                    self._rotate_handle = bpy.types.SpaceView3D.draw_handler_add(draw_text_callback_mpm, args, 'WINDOW', 'POST_PIXEL')
     
                return {'RUNNING_MODAL'}
        
        else:
            self.report({'WARNING'}, "View3D not found, cannot run operator")
            return {'CANCELLED'}