# -*- coding:utf-8 -*-

# Speedflow Add-on
# Copyright (C) 2018 Cedric Lepiller aka Pitiwazou & Legigan Jeremy AKA Pistiwique and Stephen Leger
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

# <pep8 compliant>

import bpy
import bmesh
from math import pi, radians, degrees
from bpy.props import IntProperty, BoolProperty
from .text_to_draw import draw_text_callback_mpm

# blacklisted params
BLACKLIST = {"bl_rna", "name", "rna_type", "id_data", "show_on_cage", "show_in_editmode"}


class deselect_and_restore():
    """Context Manager
    For use bpy.ops like transform.translate or any op working on selection

    Usage:

    with deselect_and_restore(context, obj, object_mode='EDIT'):
        .. obj is selected and active
        bpy.ops.transform.translate ...

    with deselect_and_restore(context, obj, object_mode='OBJECT') as (new_act):
        .. new_act is selected and active and in object_mode if set
        .. old_sel objects are no more selected (if different of obj)
        .. old_act is no more active (if different of obj)
        bpy.ops.transform.translate ...

    .. obj is not active nor selected unless it was selected or active before
    .. old_sel are selected
    .. old_act is active
    :param context: blender context
    :param obj: blender object
    :param object_mode: object mode to switch in ['EDIT', 'OBJECT', ..]
    """
    def __init__(self, context, obj, object_mode=None):
        self._ctx = context
        self._act = context.object
        self._act_mode = None
        self._sel = context.selected_objects[:]
        self._obj = obj

        obj_mode = obj.mode

        # ensure act.mode allow deselect all
        # other objects are all in 'OBJECT' mode
        if self._act is not None:
            self._act_mode = self._act.mode
            bpy.ops.object.mode_set(mode='OBJECT')

        # deselect require object mode
        bpy.ops.object.select_all(action="DESELECT")

        obj.select_set(state=True)
        bpy.context.view_layer.objects.active = obj

        # obj.select = True
        # context.scene.objects.active = obj

        if object_mode is not None:
            # obj == act -> obj was in _act_mode, exit set obj mode back to _act_mode
            # obj != act -> obj was in 'OBJECT' mode, exit set obj mode back to 'OBJECT'
            if obj_mode != object_mode:
                bpy.ops.object.mode_set(mode=object_mode)

    def __enter__(self):
        return self._obj

    def __exit__(self, exc_type, exc_val, exc_tb):

        bpy.ops.object.mode_set(mode='OBJECT')

        # deselect require object mode
        bpy.ops.object.select_all(action="DESELECT")

        for obj in self._sel:
            obj.select_set(state=True)
            # obj.select = True

        self._ctx.view_layer.objects.active = self._act
        # self._ctx.scene.objects.active = self._act

        # restore act mode
        if self._act_mode is not None and self._act.mode != self._act_mode:
            bpy.ops.object.mode_set(mode=self._act_mode)

        # false let raise exception if any
        return False


class set_and_restore_active():
    """Context Manager
    For use with simple bpy.ops working on active object

    with set_and_restore_active(context, obj):
        .. obj is active
        bpy.ops. ...

    .. active is restored to last state
    """
    def __init__(self, context, obj, object_mode=None):
        self._ctx = context
        self._act = context.object
        self._obj = obj
        self._obj_select = obj.select_set(state=True)


        obj.select_set(state=True)
        context.view_layer.objects.active = obj

        # switch object mode
        self._obj_mode = None
        if object_mode is not None and obj.mode != object_mode:
            self._obj_mode = obj.mode
            bpy.ops.object.mode_set(mode=object_mode)

    def __enter__(self):
        return self._obj

    def __exit__(self, exc_type, exc_val, exc_tb):
        # restore context
        if self._obj_mode is not None:
            bpy.ops.object.mode_set(mode=self._obj_mode)


        # self._obj.select_set(state=self._obj_select)
        self._obj_select = self._obj.select_set(state=True)
        self._ctx.view_layer.objects.active = self._act
        return False



class SpeedApi:

    def __init__(self):


        # Store same modifiers as act_mod found in selected objects
        # Same modifs is automatically set / update when act_mod is set
        # so you typically dont have to bother with it
        # {obj: mod}
        self.same_modifs = {}

        # Private, modifier of initial active object, use act_mod getter and setter
        self._act_mod = None

        # context at startup
        self.act_obj = None
        self.sel = []
        self.sel_hidden = []

        # self.last_selection = []
        # blender's active object mode at startup
        self.mode = None

        # addon preferences
        prefs = self.get_addon_preferences()
        user_prefs = self.get_user_preferences()


        self.prefs = prefs

        # self.autosmoothvalue = self.prefs.auto_smooth_value
        # self.data.auto_smooth_angle = self.prefs.auto_smooth_value

        # Init modals options
        self.modal_speed = prefs.modal_speed
        self.work_tool = prefs.work_tool

        # default modifier ui
        self.show_expanded = prefs.show_expanded

        # store object's display state to handle show_wire_in_modal and use_bound
        self._display_state = {}

        # Keyboard / mouse selection preferences
        # active_keymaps = bpy.context.window_manager.keyconfigs.active

        select_mode = bpy.context.window_manager.keyconfigs[0].preferences.select_mouse
        # select_mode = bpy.context.window_manager.keyconfigs[0].preferences.select_mouse
        # print(select_mode)

        # select_mode = user_prefs.inputs.select_mouse
        # select_inverse = {'LEFT': 'RIGHT', 'RIGHT': 'LEFT'}[select_mode]
        key_confirm = {'LEFTSELECT': 'LEFT', 'RIGHTSELECT': 'RIGHT', 'AUTO': 'LEFT'}[prefs.valid]
        self.key_confirm = "{0}MOUSE".format(key_confirm)
        self.key_cancel =  "{0}MOUSE".format({'LEFT': 'RIGHT', 'RIGHT': 'LEFT'}[key_confirm])
        self.key_select = "{0}MOUSE".format(select_mode)

        # # Set the key to call inside the pie, identical to the key to call the addon
        #         # wm = bpy.context.window_manager
        #         # kc = wm.keyconfigs.user
        #         #
        #         # self.addon_key = None
        #         # for kmi in kc.keymaps["3D View Generic"].keymap_items:
        #         #     if kmi.idname == "wm.call_menu_pie" and kmi.properties.name == "SPEEDFLOW_MT_pie_menu":
        #         #         self.addon_key = kmi.type
        #         # print(self.addon_key)

        # keyboard input
        self.input_list = {'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '-', '+', '.'}
        self.input = ""

        # mouse input / mouse wheel
        self.mouse_x = 0
        self.wheel_delta = 0

        # [mixed] store initial value to reset on exit and last to change step size for current action
        self.last_value = None
        self.init_value = None
        self.slow_down = False

        # store for draw handler
        self._draw_handler = None

        # store (event.ctrl, event.shift, event.alt) when pressed
        self.event_alt = (False, False, False)


    # define as prop so we are able to call modal on specified modifier
    # default is -2 as last index on stack may be -1
    # when called from pie this value is -2
    # when called from <-> this value >= -1
    modifier_index : IntProperty(default=-2)
    call_from_pie: BoolProperty(default=True)
    # Generic modal management helpers

    def unlink_object_from_scene(self, obj):
        for coll in obj.users_collection:
            coll.objects.unlink(obj)

    def get_addon_preferences(self):
        addon_key = __package__.split(".")[0]
        return bpy.context.preferences.addons[addon_key].preferences

    def get_user_preferences(self):
        return bpy.context.preferences

    def setup_display(self, display_type):
        """Setup display state for selected objects
        Use in modal.invoke() when required
        :param display_type: in ['WIRE', 'BOUNDS', None] override object draw type when set
        :return:
        """
        show_wire = self.prefs.show_wire_in_modal
        for obj in self.sel:
            # store initial draw types / show_wire for each object
            self._display_state[obj] = (obj.display_type, obj.show_wire)

            obj.show_wire = obj.show_all_edges = show_wire
            if display_type is not None:
                obj.display_type = display_type

    def restore_display(self):
        """Restore display state for selected objects,
        .exit() call this one automatically
        :return:
        """
        keep_wire = self.prefs.keep_wire
        for obj, display in self._display_state.items():
            obj.display_type, show_wire = display
            # force wire if keep wire is true
            obj.show_wire = obj.show_all_edges =  keep_wire
            # obj.show_wire = obj.show_all_edges = show_wire or keep_wire
        # clean up
        self._display_state.clear()

    def setup_modal_keymap(self, keymap):
        """ Setup modal keymap, parse human readable keymap definition to handle ctrl+alt+shift
        This func provide an abstraction layer between key and actions so
        we may change keys without changing code
        keymap definition:
        modifiers separated by + without spaces: action name
        {[CTRL+|ALT+|SHIFT+]key: action_name}
        :param self:
        :param keymap: {[CTRL+|ALT+|SHIFT+]key: action_name}
        :return: keymap able to handle event.shift/event.alt/event.ctrl
        """
        km = {
            # ctrl, shift, alt
            (False, False, False): {},  # regular
            (True, False, False): {},  # ctrl
            (False, True, False): {},  # shift
            (False, False, True): {},  # alt
            (True, True, False): {},  # ctrl + shift
            (False, True, True): {},  # shift + alt
            (True, False, True): {},  # ctrl + alt
            (True, True, True): {}  # ctrl + shift + alt
        }
        for keys, action in keymap.items():
            k = keys.split("+")
            key = k[-1]
            km[("CTRL" in k, "SHIFT" in k, "ALT" in k)][key] = action
        return km

    def store_init_value(self, event, action, index=None):
        """ Store initial value when starting mouse_action
        Override this method when attributes are not all from modifier
        :param event: modal event
        :param action: action name, modifier parameter name
        :param index: index of iterable properties
        :return:
        """
        value = self.get_property_value(self.act_mod, action, index)
        self.last_value = value
        self.init_value = value
        self.mouse_x = event.mouse_x
        self.wheel_delta = 0



    def exit(self, context):
        """ Default modal exit implementation,
        restore active, selected, mode, object's draw methods and remove draw handler
        modal might override this method and call super().exit(context)
        """
        self.restore_display()

        bpy.types.SpaceView3D.draw_handler_remove(self._draw_handler, 'WINDOW')

        # ensure OBJECT mode before deselect
        # if context.object and context.object.mode != 'OBJECT':
        #     bpy.ops.object.mode_set(mode='OBJECT')
        #new
        edit_mode = False
        if context.object and context.object.mode == 'EDIT':
            edit_mode = True
            bpy.ops.object.mode_set(mode='OBJECT')

        bpy.ops.object.select_all(action="DESELECT")

        for obj in self.sel:
            obj.select_set(state=True)


        context.view_layer.objects.active = self.act_obj

        # Go in object mode if bevel not in weight and vgroup
        # if self.act_mod.type == 'BEVEL' and self.act_mod.limit_method not in {'WEIGHT','VGROUP'}:
        #     bpy.ops.object.mode_set(mode='OBJECT')
        # else:
        #     bpy.ops.object.mode_set(mode=self.mode)

        # new
        if edit_mode:
            bpy.ops.object.mode_set(mode='EDIT')

        self.same_modifs.clear()
        self.sel.clear()

        self.act_mod = None

        self.prefs.display_texts = True

    def setup_draw_handler(self, context):

        args = (self, context, self.mod_type)
        self._draw_handler = bpy.types.SpaceView3D.draw_handler_add(
            draw_text_callback_mpm,
            args,
            'WINDOW',
            'POST_PIXEL')

    def copy_modifier_if_missing(self, context, event, types={'MESH'}):
        """Copy modifier to active object on shift+select
        :param context:
        :param event:
        :param types: filter object by type
        :return: modal return value in {'FINISHED', 'RUNNING_MODAL', 'PASS_THROUGH'}
        """
        # NOTE: shift adds active to selection
        if event.value == 'RELEASE':

            if context.object:
                obj = context.object

                # nothing to do when obj already has same modifier as act_mod
                if obj.type in types and obj != self.act_obj and obj not in self.same_modifs.keys():

                    # add modifier by type and copy params if missing
                    act_obj = self.act_obj
                    act_mod = self.act_mod
                    mod_type = act_mod.type

                    if obj not in self.sel:
                        self.sel.append(obj)

                    if obj not in self._display_state.keys():
                        # setup display state
                        self._display_state[obj] = (obj.display_type, obj.show_wire)
                        # self._display_state[obj] = (obj.draw_type, obj.show_wire)
                        obj.show_all_edges = act_obj.show_all_edges
                        obj.show_wire = act_obj.show_wire
                        obj.display_type = act_obj.display_type
                        # obj.draw_type = act_obj.draw_type

                    # try to find same modifier
                    mod = self.find_same_in_list(act_mod, obj.modifiers)

                    if mod is None:
                        mod = self.add_modifier(obj, mod_type)
                        mod.name = act_mod.name
                        self.copy_modifier_params(act_mod, mod)

                    self.same_modifs[obj] = mod

                # restore active
                self.act_obj.select_set(state=True)
                context.view_layer.objects.active = self.act_obj

                # self.act_obj.select = True
                # context.scene.objects.active = self.act_obj

                return {'RUNNING_MODAL'}

        # Allow active object selection change with shift pressed
        return {'PASS_THROUGH'}

    def remove_same_modifier(self, context, event):
        """Remove same modifier from active object on ctrl+select
        :param context:
        :param event:
        :param types: filter object by type
        :return: modal return value in {'FINISHED', 'RUNNING_MODAL', 'PASS_THROUGH'}
        """
        # NOTE: ctrl+select deselect all
        if event.value == 'RELEASE':

            if context.object:
                obj = context.object

                # find same modifier
                mod = self.find_same_in_list(self.act_mod, obj.modifiers)

                if mod is not None:

                    # restore display
                    if obj in self._display_state.keys():
                        keep_wire = self.prefs.keep_wire
                        obj.display_type, show_wire = self._display_state[obj]
                        # obj.draw_type, show_wire = self._display_state[obj]
                        # force wire if keep wire is true
                        obj.show_wire = obj.show_all_edges = show_wire or keep_wire
                        del self._display_state[obj]

                    self.remove_modifier(obj, mod)

                    # clear from same_modifs cache dict
                    if obj in self.same_modifs.keys():
                        del self.same_modifs[obj]

                # special case: remove from act_obj
                if obj == self.act_obj:

                    if len(self.same_modifs) < 1:
                        return {'FINISHED'}

                    # find next obj using same modifier and make it active
                    for obj, mod in self.same_modifs.items():
                        context.view_layer.objects.active = obj
                        self.act_obj = obj
                        # rebuild same_modifs cache
                        self.act_mod = mod
                        break

                else:
                    # other cases: make act_obj active again
                    context.view_layer.objects.active = self.act_obj

                # restore selection as ctrl+select deselect all
                for obj in self.sel:
                    obj.select_set(state=True)

                return {'RUNNING_MODAL'}

        # Allow active object selection change with ctrl pressed
        return {'PASS_THROUGH'}

    def run_modal_with_another_object(self, context, event, action, types={'MESH'}):
        """Change active object on alt+select
        :param context:
        :param event:
        :param action: default action in use when restart modal
        :param types: filter object by type
        :return: modal return value in {'FINISHED', 'RUNNING_MODAL', 'PASS_THROUGH'}
        """
        # NOTE: alt+select deselect all
        if event.value == 'RELEASE':

            if context.object:
                obj = context.object
                if obj.type in types and obj != self.act_obj:

                    # restart modal using another object as act_obj
                    self.exit(context)

                    # exit restore context so we must reselect and make obj active
                    obj.select_set(state=True)
                    context.view_layer.objects.active = obj
                    # obj.select = True
                    # context.scene.objects.active = obj

                    self.run_modal_by_type(self.mod_type, action)
                    return {'FINISHED'}


                return {'RUNNING_MODAL'}

        # Allow active object selection change with alt pressed
        return {'PASS_THROUGH'}

    # Automatically handle cache update for same modifiers in selection
    # when act_mod does change
    # NOTE: only set self.act_mode when realy needed,
    # rely on local act_mod in the between
    @property
    def act_mod(self):
        return self._act_mod

    @act_mod.setter
    def act_mod(self, act_mod):
        self._act_mod = act_mod
        # Update cache for same modifiers in selection
        self._store_same_modifiers()
        # set index of active modifier for companion
        self.set_index_for_companion(act_mod)

    @property
    def mod_type(self):
        """Modifier type name CAPITALISED from modal class name
        :return: modifier TYPE eg: speedflow.boolean -> BOOLEAN
        """
        return self.__class__.bl_idname.split(".")[-1].upper()

    def set_index_for_companion(self, act_mod):
        """Store index of current modifier in wm
        So companion is able to know wich modifier on stack is active one
        XXX rely on bpy as we cant retrieve context here
        :param act_mod:
        :return:
        """
        # TODO: setup wm.sf_companion.active_modifier propertygroup in companion


        wm = bpy.context.window_manager
        if hasattr(wm, 'sf_companion') and hasattr(wm.sf_companion, 'active_modifier'):
            if act_mod is None:
                index = -1
            else:
                obj = act_mod.id_data
                index = self.get_modifier_index(obj, act_mod)

            wm.sf_companion.active_modifier = index

    # Fonts and colors for draw methods

    def get_font_size(self):
        span = self.prefs.text_size
        h1 = int(span * 3)
        h2 = int(span * 2.25)
        h3 = int(span * 1.75)
        h4 = int(span * 1.25)
        return span, h1, h2, h3, h4

    def get_font_colors(self):
        text_color = self.prefs.text_color
        color_1 = self.prefs.text_color_1
        color_2 = self.prefs.text_color_2
        return text_color, color_1, color_2

    def get_modifier_owner_obj(self, mod):
        return mod.id_data

    # Modals
    @staticmethod
    def has_modal(_type):
        """Return True if speedflow is able to handle the modifier
        :param _type: modifier type
        :return:
        """
        _t = _type.lower()
        res = False
        try:
            getattr(bpy.ops.speedflow, _t).poll()
            # getattr(bpy.ops.speedflow, _t)
            res = True
        except:
            res = False
            # print(_t, "not supported")
            # import traceback
            # traceback.print_exc()
            pass
        return res
        # return hasattr(bpy.ops.speedflow, _t)

    def run_modal_by_type(self, _type, modal_action='free', modifier_index=-3, call_from_pie=True):
        """Run modal for specified modifier type
        :param _type: modifier type
        :param modal_action: action
        :return: True if new modal is running
        """
        if self.has_modal(_type):
            _t = _type.lower()
            if modifier_index > -1:
                mod = self.get_modifier_by_index(self.act_obj, modifier_index)
                self.set_index_for_companion(mod)
            getattr(bpy.ops.speedflow, _t)('INVOKE_DEFAULT',
                                           modal_action=modal_action,
                                           modifier_index=modifier_index,
                                           call_from_pie=call_from_pie)
            return True
        return False

    def run_modal(self, obj, mod, modal_action='free', call_from_pie=True):
        """Run modal for specified modifier
        :param obj:
        :param mod:
        :return: True if new modal is running
        """
        # When not specified use default modal action for each modal instead of 'free'
        # if modal_action is None:
        #    modal_action = self.modal_action
        # if mod is not None:
        if mod is not None and self.has_modal(mod.type):
            index = self.get_modifier_index(obj, mod)
            # print(index)
            return self.run_modal_by_type(mod.type,
                                          modal_action=modal_action,
                                          modifier_index=index,
                                          call_from_pie=call_from_pie)
        return False

    def next_supported_modifier_down(self, obj, mod):
        """Return next supported modifier down in the stack
        :param obj:
        :param mod:
        :return: another modifier or none
        """
        index = self.get_modifier_index(obj, mod)
        next = mod
        while next is mod or not self.has_modal(next.type):
            next = self.get_modifier_down(obj, next)
            if index == self.get_modifier_index(obj, next):
                return None
        return next

    def next_supported_modifier(self, obj, mod):
        """Return next supported modifier down in the stack
        :param obj:
        :param mod:
        :return: another modifier or none
        """
        index = self.get_modifier_index(obj, mod)
        next = mod
        # print(next)
        # while next is mod or not self.has_modal(next.type):
        #     next = obj.modifiers[0]
        #     if index == self.get_modifier_index(obj, next):
        #         return None
        return next


    def next_supported_modifier_up(self, obj, mod):
        """Return next supported modifier up in the stack
        :param obj:
        :param mod:
        :return: another modifier or none
        """
        index = self.get_modifier_index(obj, mod)
        next = mod
        while next is mod or not self.has_modal(next.type):
            next = self.get_modifier_up(obj, next)
            if index == self.get_modifier_index(obj, next):
                return None
        return next

    def change_mirror_name(self, obj, mod):
        """Return mirror name
        :param obj:
        :param mod:
        :return: another modifier or none
        """
        for obj, mod in self.same_modifs.items():
            if mod.use_axis[0] and mod.use_axis[1] and mod.use_axis[2]:
                mod.name = "Mirror - X Y Z"

            elif mod.use_axis[0] and mod.use_axis[1]:
                mod.name = "Mirror - X Y"

            elif mod.use_axis[0] and mod.use_axis[2]:
                mod.name = "Mirror - X Z"

            elif mod.use_axis[1] and mod.use_axis[2]:
                mod.name = "Mirror - Y Z"

            elif mod.use_axis[0]:
                mod.name = "Mirror - X"
            elif mod.use_axis[1]:
                mod.name = "Mirror - Y"
            elif mod.use_axis[2]:
                mod.name = "Mirror - Z"
            else:
                mod.name = "Mirror"

    def change_array_name(self, obj, mod, action):

        # axis = action
        axis = ""

        for obj, mod in self.same_modifs.items():
            # if mod.relative_offset_displace[0] > 0:
            #     mod.name = "Array - RO - X"
            #
            # elif mod.relative_offset_displace[1] > 0:
            #     mod.name = "Array - RO - Y"
            #
            # elif mod.relative_offset_displace[2] > 0:
            #     mod.name = "Array - RO - Z"


            if any([mod.relative_offset_displace[0], mod.relative_offset_displace[1], mod.relative_offset_displace[2]]) >0:
                mod.name = "Array - RO - %s" % (axis)

            elif any([mod.constant_offset_displace[0], mod.constant_offset_displace[1], mod.constant_offset_displace[2]]) >0:
                mod.name = "Array - CO - %s" % (axis)


            # elif (mod.constant_offset_displace[0] and mod.constant_offset_displace[1] and mod.constant_offset_displace[2]) > 0:
            #    mod.name = "Array - CO - X Y Z"
            #
            # elif (mod.constant_offset_displace[0] and mod.constant_offset_displace[1]) > 0:
            #     mod.name = "Array - CO - X Y"
            #
            # elif (mod.constant_offset_displace[0] and mod.constant_offset_displace[2]) > 0:
            #     mod.name = "Array - CO - X Z"
            #
            # elif (mod.constant_offset_displace[1] and mod.constant_offset_displace[2]) > 0:
            #     mod.name = "Array - CO - Y Z"
            #
            # elif (mod.relative_offset_displace[0] and mod.relative_offset_displace[1] and mod.relative_offset_displace[2]) > 0:
            #     mod.name = "Array - RO - X Y Z"
            #
            # elif (mod.relative_offset_displace[0] and mod.relative_offset_displace[1]) > 0:
            #     mod.name = "Array - RO - X Y"
            #
            # elif (mod.relative_offset_displace[0] and mod.relative_offset_displace[2]) > 0:
            #     mod.name = "Array - RO - X Z"
            #
            # elif (mod.relative_offset_displace[1] and mod.relative_offset_displace[2]) > 0:
            #     mod.name = "Array - RO - Y Z"

            else:
                mod.name = "Array"


    def run_modal_up(self, obj, mod, modal_action='free'):
        """Run modal for supported modifier up in the stack
        :param obj:
        :param mod:
        :return: True if new modal is running
        """
        next = self.next_supported_modifier_up(obj, mod)
        return self.run_modal(obj, next, modal_action, call_from_pie=False)

    def run_modal_down(self, obj, mod, modal_action='free'):
        """Run modal for supported modifier down in the stack
        :param obj:
        :param mod:
        :return: True if new modal is running
        """
        next = self.next_supported_modifier_down(obj, mod)
        return self.run_modal(obj, next, modal_action, call_from_pie=False)

    @staticmethod
    def copy_modifier_params(src, target):
        """Copy parameters of src on target 
        :param src: 
        :param target: 
        :return: 
        """
        for attr in dir(src):
            if not (attr.startswith("__") or attr in BLACKLIST or attr == 'type'):
                setattr(target, attr, getattr(src, attr))
        
    @staticmethod
    def can_add_modifier(obj, types=None):
        """Check if object is modifiable
        :param obj:
        :param types: filter types
        :return: True if we can add a modifier
        """
        if types is None:
            types = {'MESH', 'CURVE', 'FONT'}
        return obj is not None and obj.type in types

    def can_apply_modifier(self, obj):
        """Check whenever it is possible to apply modifier
        :param obj:
        :return: True if we can apply
        """
        return self.can_add_modifier(obj) and obj.data.users == 1

    # Filtering
    @staticmethod
    def filter_objects_by_type(sel, types={'MESH', 'CURVE', 'FONT'}):
        """Filter objects in sel by type
        :param sel: input collection of objects
        :param types: dict with types to filter
        :return: collection of matching objects
        """
        return [obj for obj in sel if obj.type in types]

    # @staticmethod
    # def compare_strict(ref, obj, blacklist={}):
    #     """Compare two objects for same attr: values excepted name
    #     :param ref:
    #     :param obj:
    #     :param blacklist: set of attr names to skip
    #     :return: True if attr: values are equals
    #     """
    #     return not any((not (hasattr(obj, attr) and getattr(ref, attr) == getattr(obj, attr))
    #                     for attr in dir(ref)
    #                     if not (attr.startswith("__") or attr in BLACKLIST or attr in blacklist)))

    @staticmethod
    def compare_strict(ref, obj, blacklist={}):
        """Compare two objects for same attr: values excepted name
        :param ref:
        :param obj:
        :param blacklist: set of attr names to skip
        :return: True if attr: values are equals
        """
        for attr in dir(ref):
            if not (attr.startswith("__") or attr in BLACKLIST or attr in blacklist):
                if hasattr(obj, attr):
                    u, v = getattr(ref, attr), getattr(obj, attr)
                    if type(u).__name__ == 'bpy_prop_array':
                        for x, y in zip(u, v):
                            if x != y:
                                return False
                    elif u != v:
                        # print("la valeur est diff", attr, u, v)
                        return False

        # return not any((not (hasattr(obj, attr) and getattr(ref, attr) == getattr(obj, attr))
        #                 for attr in dir(ref)
        #                 if not (attr.startswith("__") or attr in BLACKLIST or attr in blacklist)))
        return True

    @staticmethod
    def correct_normals(mode):
        """Flip normals to the correct direction
        :param obj:
        :param: mode:
        """
        if mode == 'OBJECT':
            bpy.ops.object.mode_set(mode='EDIT')
        bpy.ops.mesh.select_all(action='SELECT')
        bpy.ops.mesh.normals_make_consistent(inside=False)
        bpy.ops.mesh.select_all(action='DESELECT')
        bpy.ops.object.mode_set(mode=mode)

    @staticmethod
    def compare_by_dict(filter, obj):
        """Compare dict key: values to objects attr: values
        :param filter: dict, expected {key: values}
        :param obj:
        :return: True if key: values are equals
        """
        return not any((not (hasattr(obj, attr) and getattr(obj, attr) == val) for attr, val in filter.items()))

    def filter_first(self, filter, seek_list):
        """Return first item in list with attr: values matching filter dict
        :param filter: dict {attr: value} to compare
        :param seek_list: enumerable list to filter
        :return: first item or None
        """
        for target in seek_list:
            if self.compare_by_dict(filter, target):
                return target
        return None

    def filter_list(self, filter, seek_list):
        """Return items in list with attr: values matching filter dict
        :param filter: dict {attr: value} to compare
        :param seek_list: enumerable list to filter
        :return: list
        """
        return [target for target in seek_list if self.compare_by_dict(filter, target)]

    def find_same_in_list(self, ref, seek_list, blacklist={}):
        """Find first item in list strictly matching attrs: values of ref
        :param ref: object to match
        :param seek_list: list to filter
        :param blacklist: set of attr names to skip
        :return: strictly matching object or None
        """
        if ref is None:
            return None
        for target in seek_list:
            if self.compare_strict(ref, target, blacklist):
                return target
        return None

    # Modifiers get

    @staticmethod
    def has_modifier_of_type(obj, mod_type):
        return any([modif.type == mod_type for modif in obj.modifiers])

    @staticmethod
    def get_modifier_by_index(obj, index):
        if len(obj.modifiers) > index:
            return obj.modifiers[index]

    def is_array_on_curve(self, obj, mod):
        #         """Return True if next modifier is a CURVE
        #         Array on curve typically is made of 2 modifier in ARRAY + CURVE order
        #         :param self:
        #         :param obj:
        #         :param mod: an ARRAY modifier to check
        #         :return: True if current modifier is array_on_curve
        #         """
        next = self.get_modifier_up(obj, mod)
        return next is not None and next.type == 'CURVE'

    def get_next_modifier_by_dict(self, obj, filter):
        """Return first modifier of object with attr: values matching filter dict
        :param obj: blender object
        :param filter: dict {attr: value} to compare
        :return: first modifier or None
        """
        return self.filter_first(filter, obj.modifiers)


    def get_first_modifier_by_dict(self, obj, filter):
        """Return first modifier of object with attr: values matching filter dict
        :param obj: blender object
        :param filter: dict {attr: value} to compare
        :return: first modifier or None
        """
        return self.filter_first(filter, obj.modifiers)

    def get_last_modifier_by_dict(self, obj, filter):
        """Return last modifier of object with attr: values matching filter dict
        :param obj: blender object
        :param filter: dict {attr: value} to compare
        :return: last modifier or None
        """
        return self.filter_first(filter, reversed(obj.modifiers))

    def get_modifiers_by_dict(self, obj, filter):
        """Return modifier of object with attr: values matching filter dict
        :param obj: blender object
        :param filter: dict {attr: value} to compare
        :return: list of modifiers
        """
        return self.filter_list(filter, obj.modifiers)

    def get_first_modifier_by_type_and_attr(self, obj, mod_type, attr, value):
        """Get first modifier of a kind with given attr set to value
        :param obj: blender object
        :param mod_type: string modifier type
        :param attr: attribute name
        :param value: expected attribute value
        :return: modifier or None if not found
        """
        filter = {'type': mod_type}
        filter[attr] = value
        return self.filter_first(filter, obj.modifiers)

    def get_last_modifier_by_type_and_attr(self, obj, mod_type, attr, value):
        """Get first modifier of a kind with given attr set to value
        :param obj: blender object
        :param mod_type: string modifier type
        :param attr: attribute name
        :param value: expected attribute value
        :return: modifier or None if not found
        """
        filter = {'type': mod_type}
        filter[attr] = value
        return self.filter_first(filter, reversed(obj.modifiers))

    def get_modifiers_by_type_and_attr(self, obj, mod_type, attr, value):
        """Get first modifier of a kind with given attr set to value
        :param obj: blender object
        :param mod_type: string modifier type
        :param attr: attribute name
        :param value: expected attribute value
        :return: modifier or None if not found
        """
        filter = {'type': mod_type}
        filter[attr] = value
        return self.filter_list(filter, obj.modifiers)

    def get_first_modifier_by_type(self, obj, mod_type):
        """Get first modifier of a kind
        :param obj: blender object
        :param mod_type: string modifier type
        :return: modifier or None if not found
        """
        return self.filter_first({'type': mod_type}, obj.modifiers)

    def get_last_modifier_by_type(self, obj, mod_type):
        """Get first modifier of a kind
        :param obj: blender object
        :param mod_type: string modifier type
        :return: modifier or None if not found
        """
        return self.filter_first({'type': mod_type}, reversed(obj.modifiers))

    def get_modifiers_by_type(self, obj, mod_type):
        return self.filter_list({'type': mod_type}, obj.modifiers)

    def get_modifier_up(self, obj, mod):
        """Get next modifier up in stack
        :param obj: object
        :param mod: current modifier
        :return: modifier
        """
        index = self.get_modifier_index(obj, mod)
        idx = (index + 1) % len(obj.modifiers)
        return obj.modifiers[idx]

    def get_modifier_down(self, obj, mod):
        """Get next modifier down in stack
        :param obj: object
        :param mod: current modifier
        :return: modifier
        """
        index = self.get_modifier_index(obj, mod)
        return obj.modifiers[index - 1]

    def get_modifier_up_same_type(self, obj, mod):
        """Get next modifier up in stack with same type as current
        :param obj: object
        :param mod: current modifier
        :return: next modifier
        """
        modifs = self.get_modifiers_by_type(obj, mod.type)
        index = modifs.index(mod)
        return modifs[(index + 1) % len(modifs)]

    def get_modifier_down_same_type(self, obj, mod):
        """Get next modifier down in stack with same type as current
        :param obj: object
        :param mod: current modifier
        :return: next modifier
        """
        modifs = self.get_modifiers_by_type(obj, mod.type)
        index = modifs.index(mod)
        return modifs[index - 1]


    def get_or_create_modifier(self, obj, mod_type, mod_name=None):
        """Get top modifier of a kind or create a new one
        :param obj: blender object
        :param mod_type: string modifier type
        :param mod_name: default modifier name
        :return: modifier
        """
        mod = self.get_last_modifier_by_type(obj, mod_type)
        if mod is not None:
            return mod

        return self.add_modifier(obj, mod_type, mod_name)

    @staticmethod
    def get_modifier_index(obj, mod):
        """Get modifier index
        :param obj: blender object
        :param mod: modifier
        :return: int index
        """
        return obj.modifiers.find(mod.name)

    @staticmethod
    def create_modifier(obj, mod_type, mod_name=None):
        """Add a modifier of any kind, doesnt init any parameter
        :param obj: blender object
        :param mod_type: string modifier type
        :param mod_name: optional modifier name
        :return: modifier
        """
        if mod_name is None:
            mod_name = mod_type.capitalize()

        mod = obj.modifiers.new(mod_name, mod_type.upper())
        return mod

    def add_modifier(self, obj, mod_type, mod_name=None):
        """Add a modifier
        and setup property from modal properties
        :param obj: blender object
        :param mod_type: string modifier type
        :param mod_name: optional modifier name
        :return: modifier
        """
        mod = self.create_modifier(obj, mod_type, mod_name)

        BLACKLIST = {"bl_rna", "name", "rna_type"}

        # TODO: remove this when prefs are all done
        if not hasattr(self.prefs, mod_type.lower()):
            # print("Prefs datablock not found for modifier : %s" % mod_type)
            return mod

        # init modifier property using addon prefs
        prefs = getattr(self.prefs, mod_type.lower())

        _dir = dir(prefs)

        for attr in dir(mod):
            if attr not in _dir \
                    or attr.startswith("__") \
                    or attr in BLACKLIST:
                continue

            value = getattr(prefs, attr)
            setattr(mod, attr, value)

        mod.show_expanded = self.prefs.show_expanded

        return mod

    def setup_object_data_from_prefs(self, obj, mod_type):
        """Add a modifier
        and setup property from modal properties
        :param obj: blender object
        :param mod_type: string modifier type
        :param mod_name: optional modifier name
        :return: modifier
        """
        mod = obj.data

        BLACKLIST = {"bl_rna", "name", "rna_type"}

        # TODO: remove this when prefs are all done
        if not hasattr(self.prefs, mod_type.lower()):
            # print("Prefs datablock not found for modifier : %s" % mod_type)
            return mod

        # init modifier property using addon prefs
        prefs = getattr(self.prefs, mod_type.lower())

        _dir = dir(prefs)

        for attr in dir(mod):
            if attr not in _dir \
                    or attr.startswith("__") \
                    or attr in BLACKLIST:
                continue

            value = getattr(prefs, attr)
            setattr(mod, attr, value)

        return mod

    # Utils to store modifiers in selection
    def _store_same_modifiers(self):
        """PRIVATE Store same modifiers as act_mod in selected objects
        :param act_mod:
        :return:
        """
        ref = self.act_mod
        self.same_modifs.clear()
        for obj in self.sel:
            mod = self.find_same_in_list(ref, obj.modifiers)
            if mod is not None:
                self.same_modifs[obj] = mod

    def has_same_modifier(self, obj):
        return obj in self.same_modifs

    def get_same_modifier(self, obj):
        """Return modifier of obj with same values as act_mod
        :param obj: object where to seek for same modifier
        :return: equivalent modifier or None
        """
        return self.same_modifs.get(obj)

    def _clear_same_modifs_reference(self, obj, mod):
        """PRIVATE Clear deleted modifier reference from same_modifs
        keep obj as key, but with a None value instead of deleted modifier
        to prevent any ACCESS_VIOLATION exception
        :param obj:
        :param mod:
        :return:
        """
        if obj in self.same_modifs and self.same_modifs[obj] == mod:
            self.same_modifs[obj] = None

    def remove_modifier(self, obj, mod):
        if mod is not None:
            # clear ref to deleted mod in cache
            self._clear_same_modifs_reference(obj, mod)
            obj.modifiers.remove(mod)

    def remove_modifiers_by_type(self, obj, mod_type):
        modifs = self.get_modifiers_by_type(obj, mod_type)
        for mod in modifs:
            self.remove_modifier(obj, mod)

    def toggle_modifiers_visibility_by_type(self, obj, mod_type):
        modifs = self.get_modifiers_by_type(obj, mod_type)
        for mod in modifs:
            mod.show_viewport = not mod.show_viewport

    def set_modifiers_visibility_by_type(self, obj, mod_type, visibility):
        modifs = self.get_modifiers_by_type(obj, mod_type)
        for mod in modifs:
            mod.show_viewport = visibility

    def apply_modifier(self, context, obj, mod):
        """Apply specified object's modifier, preserve context
        WARNING: this will make object single_user !!!
        :param context: blender context pointer
        :param obj: blender object, selected and active
        :param mod: blender modifier
        :return:
        """
        if mod is not None:
            self.make_single_user(context, obj)
            self._clear_same_modifs_reference(obj, mod)
            with set_and_restore_active(context, obj, 'OBJECT'):
                bpy.ops.object.modifier_apply(
                    apply_as='DATA',
                    modifier=mod.name)

    def apply_modifiers_by_type(self, context, obj, mod_type):
        """Apply all modifier of a kind on object
        :param context: blender context pointer
        :param obj: blender object
        :param mod_type: string modifier type
        :return:
        """
        modifs = self.get_modifiers_by_type(obj, mod_type)
        for mod in modifs:
            self.apply_modifier(context, obj, mod)

    # Change modifier order in stack

    @staticmethod
    def move_modifier_up(context, obj, mod):
        with set_and_restore_active(context, obj):
            bpy.ops.object.modifier_move_up(modifier=mod.name)

    @staticmethod
    def move_modifier_down(context, obj, mod):
        with set_and_restore_active(context, obj):
            bpy.ops.object.modifier_move_down(modifier=mod.name)

    def move_modifier_bottom(self, context, obj, mod):
        delta = len(obj.modifiers) - self.get_modifier_index(obj, mod)
        with set_and_restore_active(context, obj):
            for i in range(delta):
                bpy.ops.object.modifier_move_down(modifier=mod.name)

    def move_modifier_top(self, context, obj, mod):
        delta = self.get_modifier_index(obj, mod)
        with set_and_restore_active(context, obj):
            for i in range(delta):
                bpy.ops.object.modifier_move_up(modifier=mod.name)

    def move_modifier_before(self, context, obj, mod, target):
        delta = self.get_modifier_index(obj, target) - self.get_modifier_index(obj, mod)
        print(delta)
        with set_and_restore_active(context, obj):
            for i in range(0, delta, -1):
                bpy.ops.object.modifier_move_up(modifier=mod.name)

    def move_modifier_after(self, context, obj, mod, target):
        delta = 1 + self.get_modifier_index(obj, target) - self.get_modifier_index(obj, mod)
        with set_and_restore_active(context, obj):
            if delta > 0:
                for i in range(delta):
                    bpy.ops.object.modifier_move_up(modifier=mod.name)
            else:
                for i in range(0, delta, -1):
                    bpy.ops.object.modifier_move_down(modifier=mod.name)

    # def move_modifier_before_by_type_and_attr(self, context, obj, mod, target, attr, value):
    #     """Move modifier before another with type, attr set and value
    #     :param obj: blender object
    #     :param mod_type: string modifier type
    #     :param attr: attribute name
    #     :param value: expected attribute value
    #     :return: modifier or None if not found
    #     """
    #     filter = {'type': target}
    #     filter[attr] = value
    #
    #     delta = self.get_modifier_index(obj, target) - self.get_modifier_index(obj, mod)
    #     print(delta)
    #     with set_and_restore_active(context, obj):
    #         for i in range(0, delta, -1):
    #             bpy.ops.object.modifier_move_up(modifier=mod.name)
    #
    #     return self.filter_list(filter, obj.modifiers)


    # def get_modifier_up_limited(self, obj, mod):
    #     """Get next modifier up in stack
    #     :param obj: object
    #     :param mod: current modifier
    #     :return: modifier
    #     """
    #     index = self.get_modifier_index(obj, mod)
    #
    #     if index + 1 < len(obj.modifiers):
    #         return obj.modifiers[index + 1]
    #     return None
    #
    # def get_modifier_down_limited(self, obj, mod):
    #     """Get next modifier down in stack
    #     :param obj: object
    #     :param mod: current modifier
    #     :return: modifier
    #     """
    #     index = self.get_modifier_index(obj, mod)
    #
    #     if index - 1 > 0:
    #         return obj.modifiers[index - 1]
    #     return None
    #
    # def next_modifier_index_down(self, obj, mod):
    #     """Return next modifier down in the stack
    #        Recup l'index du prochain modifier dans la liste
    #     :param obj:
    #     :param mod:
    #     :return: index  or -1 when not found
    #     """
    #     index = -1
    #     next = self.get_modifier_down_limited(obj, mod)
    #     if next is not None:
    #         index = self.get_modifier_index(obj, next)
    #     return index
    #
    #
    # def next_modifier_index_up(self, obj, mod):
    #     """Return next modifier down in the stack
    #        Recup l'index du précédent modifier de la liste
    #     :param obj:
    #     :param mod:
    #     :return: index  or +1 when not found
    #     """
    #     index = -1
    #     next = self.get_modifier_up_limited(obj, mod)
    #     if next is not None:
    #         index = self.get_modifier_index(obj, next)
    #     return index
    #
    #
    # def set_modifier_visibility(self, obj, mod, visible):
    #     """Hide next modifiers
    #     :param obj:
    #     :param mod:
    #     :return: another modifier or none
    #     """
    #     next = mod
    #     while next is not None:
    #         next = self.get_modifier_down_limited(obj, next)
    #         if next is not None:
    #             next.show_viewport = visible
    #     return next


    def get_enums_from_rna(self, obj, prop):
        """
        :param obj: any object with rna attributes
        :param prop: property name
        :return: Enums keys, flag for is_enum_flag (enum use int as keys)
        """
        res = [], False
        if hasattr(obj, prop):
            pdef = obj.rna_type.properties[prop]
            if pdef.type == 'ENUM':
                res = pdef.enum_items.keys(), pdef.is_enum_flag
        return res

    def get_next_enum(self, obj, prop):
        """
        :param obj: any object with rna attributes
        :param prop: property name
        :return: Next value for enum
        """
        enum, is_enum_flag = self.get_enums_from_rna(obj, prop)
        curr = getattr(obj, prop)
        index = enum.index(curr)
        return enum[(index + 1) % len(enum)]

    def get_start_cap(self):
        """
        :param obj: any object with start as name
        :return: last array object
        """
        start = False
        for obj in bpy.context.selected_objects:
            if obj.name.startswith("start"):
                start = True

        return start

    def get_end_cap(self):
        """
        :param obj: any object with start as name
        :return: last array object
        """
        end = False
        for obj in bpy.context.selected_objects:
            if obj.name.startswith("end"):
                end = True

        return end

    def get_label_from_rna(self, obj, prop):
        """
        :param obj: any object with rna attributes
        :param prop: property name
        :return: Label
        """
        res = ""
        if hasattr(obj, prop):
            res = obj.rna_type.properties[prop].name
        return res

    def get_type_from_rna(self, obj, prop):
        """
        :param obj: any object with rna attributes
        :param prop: property name
        :return: type, subtype
        """
        res = 'NONE', 'NONE'
        if hasattr(obj, prop):
            pdef = obj.rna_type.properties[prop]
            res = pdef.type, pdef.subtype
        return res

    def get_limits_from_rna(self, obj, prop):
        """
        :param obj: any object with rna attributes
        :param prop: property name
        :return: hard min / max for a value
        """
        typ = 'NONE'
        res = -1e64, 1e64, typ
        if hasattr(obj, prop):
            pdef = obj.rna_type.properties[prop]
            typ = pdef.type
            if typ in {'INT', 'FLOAT'}:
                res = pdef.hard_min, pdef.hard_max, typ
        return res


    def get_pixels_from_rna(self, obj, prop):
        """
        :param obj: any object with rna attributes
        :param prop: property name
        :return: Pixels to move from one step
        """
        pixels = 70
        if hasattr(obj, prop):
            pdef = obj.rna_type.properties[prop]
            step = pdef.step
            if pdef.type == 'INT':
                step = 2
            if pdef.subtype == 'ANGLE':
                step = (0.5 * pi) / 15
            pixels /= step
        return pixels * self.modal_speed

    # act on same modifiers than act_mod
    def set_same_modifiers_value(self, prop, value, index=None):
        """Set property value of same_modifiers
        :param prop: property name
        :param value: value
        :param index: optional index for iterable property
        :return:
        """
        if index is None:
            for mod in self.same_modifs.values():
                # print(mod.name)
                setattr(mod, prop, value)
        else:
            for mod in self.same_modifs.values():
                self.get_property_value(mod, prop)[index] = value

    def set_same_modifiers_delta(self, prop, delta, index=None):
        """Set property delta of same_modifiers
        :param prop: property name
        :param value: value
        :param index: optional index for iterable property
        :return:
        """
        if index is None:
            for mod in self.same_modifs.values():
                setattr(mod, prop, getattr(mod, prop) + delta)
        else:
            for mod in self.same_modifs.values():
                self.get_property_value(mod, prop)[index] += delta

    def toggle_same_modifiers_value(self, prop, index=None):
        """Toggle property of same_modifs
        :param prop: property name
        :param index: optional index for iterable property
        :return:
        """
        state = not self.get_property_value(self.act_mod, prop, index)
        self.set_same_modifiers_value(prop, state, index)

    def set_same_objects_data_value(self, prop, value, index=None):
        """Set property of object data for same_modifs objects
        :param prop: property name
        :param value: value
        :param index: optional index for iterable property
        :return:
        """
        if index is None:
            for obj in self.same_modifs.keys():
                d = obj.data
                if hasattr(d, prop):
                    setattr(d, prop, value)
        else:
            for obj in self.same_modifs.keys():
                d = obj.data
                if hasattr(d, prop):
                    self.get_property_value(d, prop)[index] = value


    def set_same_objects_data_delta(self, prop, delta, index=None):
        """Set property of object data for same_modifs objects
        :param prop: property name
        :param value: value
        :param index: optional index for iterable property
        :return:
        """
        if index is None:
            for obj in self.same_modifs.keys():
                d = obj.data
                if hasattr(d, prop):
                    setattr(d, prop, getattr(d, prop) + delta)
        else:
            for obj in self.same_modifs.keys():
                d = obj.data
                if hasattr(d, prop):
                    self.get_property_value(d, prop)[index] += delta

    def toggle_same_objects_data_value(self, prop, index=None):
        """Toggle property of object data for same_modifs objects
        :param prop: property name
        :param index: optional index for iterable property
        :return:
        """
        state = not self.get_property_value(self.act_obj.data, prop, index)
        self.set_same_objects_data_value(prop, state, index)

    def show_hide_all_modifiers(self, obj, mod):
        """Show Hide All Modifiers
        :param obj: blender object
        :param mod: string modifier
        :return: modifier
        """
        for obj in bpy.context.selected_objects:
            if obj.modifiers:
                for mod in obj.modifiers:
                    mod.show_viewport = not mod.show_viewport

    @staticmethod
    def get_property_value(obj, prop, index=None):
        """ Get value of a property of any object/data/modifier
        :param obj: modifier or any object/data
        :param prop: property name
        :param index: optional use for indexed properties in array
        :return: value of property
        """
        if index is None:
            value = getattr(obj, prop)
        else:
            value = getattr(obj, prop)[index]
        return value

    # keyboard inputs
    def get_input_value(self, value):
        """ Return valid input """
        if value == "-":
            if not self.input.startswith("-"):
                self.input = "-" + self.input

        elif value == "+":
            if self.input.startswith("-"):
                self.input = self.input.split("-")[-1]

        elif value == "." and "." in self.input:
            pass

        else:
            self.input += value

    def limit_value(self, value, obj, attr):
        """Limit value using rna_type definition
        :param value: value to limit
        :param obj: any object with rna_type
        :param attr: attribute name
        :return: value
        """
        if value != 0:
            mini, maxi, typ = self.get_limits_from_rna(obj, attr)
            res = max(mini, min(maxi, value))
            if typ == 'INT':
                res = int(res)
            return res
        return value

    def input_as_value(self, obj, attr):
        """Convert string input into float or int using rna_type definition
        :param obj: any object with rna_type
        :param attr: attribute name
        :return: value
        """
        value = 0
        ptype, subtype = self.get_type_from_rna(obj, attr)

        if ptype == 'FLOAT':
            value = float(self.input)
            if subtype == 'ANGLE':
                value = radians(value)

        elif ptype == 'INT':
            value = int(self.input)

        return self.limit_value(value, obj, attr)

    # Special modifier weighted normals
    @staticmethod
    def support_weighted_normals_modifier():
        return any([mod for mod
                    in bpy.types.Modifier.bl_rna.properties['type'].enum_items
                    if mod.identifier == 'WEIGHTED_NORMAL'])

    def move_weighted_normals_down(self, context, obj):
        """Add and keep a weighted normal modifier at bottom of the stack
        :param context:
        :param obj:
        :return:
        """
        weighted_normals = self.support_weighted_normals_modifier()

        if weighted_normals:
            modifs = self.get_modifiers_by_type(obj, 'WEIGHTED_NORMAL')
            if len(modifs) == 0:
                pass
            else:
                self.move_modifier_bottom(context, obj, modifs[-1])

    # Special modifier Triangulate
    @staticmethod
    def support_triangulate_modifier():
        return any([mod for mod
                    in bpy.types.Modifier.bl_rna.properties['type'].enum_items
                    if mod.identifier == 'TRIANGULATE'])

    def move_triangulate_down(self, context, obj):
        """Add and keep a Triangulate modifier at bottom of the stack
        :param context:
        :param obj:
        :return:
        """
        triangulate = self.support_triangulate_modifier()

        if triangulate:
            modifs = self.get_modifiers_by_type(obj, 'TRIANGULATE')
            if len(modifs) == 0:
                mod = self.create_modifier(obj, 'TRIANGULATE', "Triangulate")
                mod.quad_method = 'SHORTEST_DIAGONAL'
                mod.keep_custom_normals = True
                mod.ngon_method = 'BEAUTY'
                mod.min_vertices = 5
                mod.show_expanded = self.prefs.show_expanded

            else:
                self.move_modifier_bottom(context, obj, modifs[-1])

    # Object utils
    @staticmethod
    def create_empty(context, name):
        empty = bpy.data.objects.new(name, None)
        coll = context.view_layer.active_layer_collection.collection
        coll.objects.link(empty)
        return empty

    @staticmethod
    def clear_edges_weights(context, obj):
        with set_and_restore_active(context, obj, 'EDIT'):
            bpy.ops.transform.edge_bevelweight(value=0)
            bpy.ops.transform.edge_crease(value=0)

    @staticmethod
    def shading_mode(self, context, obj, mode='SMOOTH'):
        prefs = get_addon_preferences()
        is_subsurf = self.get_last_modifier_by_type(obj, 'SUBSURF')
        if obj.type == 'MESH':
            if prefs.shading_smooth:
                obj.data.use_auto_smooth = True
                if is_subsurf:
                    obj.data.auto_smooth_angle = radians(60)
                else:
                    obj.data.auto_smooth_angle = radians(prefs.auto_smooth_value)

            with set_and_restore_active(context, obj):
                if mode == 'SMOOTH':
                    bpy.ops.object.shade_smooth()
                else:
                    bpy.ops.object.shade_flat()

    # @staticmethod
    # def shading_mode(context, obj, mode='SMOOTH', angle=radians(31)):
    #
    #     obj.data.use_auto_smooth = True
    #     obj.data.auto_smooth_angle = angle
    #
    #     with set_and_restore_active(context, obj):
    #         if mode == 'SMOOTH':
    #             bpy.ops.object.shade_smooth()
    #         else:
    #             bpy.ops.object.shade_flat()
    @staticmethod
    def shading_wire_in_modal(context, self):
        show_wire = not self.prefs.show_wire_in_modal

        # on désactive les wire des overlays
        if not show_wire:
            context.space_data.overlay.show_wireframes = False

        self.prefs.show_wire_in_modal = show_wire

        display_type = 'BOUNDS'
        if show_wire:
            display_type = 'WIRE'

        for obj in self.sel:
            obj.show_all_edges = obj.show_wire = show_wire
            if obj.display_type in {'BOUNDS', 'WIRE'}:
                obj.display_type = display_type

    @staticmethod
    def shading_mode_type(context, self):
        shading = context.space_data.shading
        if shading.type != 'WIREFRAME':
            shading.type = 'WIREFRAME'
            shading.wireframe_color_type = 'RANDOM'
            shading.show_xray_wireframe = True
            shading.xray_alpha_wireframe = 0

        elif self.shading_viewport_type == 'WIREFRAME':
            shading.type = 'SOLID'

        else:
            shading.type = self.shading_viewport_type

    @staticmethod
    def shading_random_mode(context, self): #bpy.data.screens["Layout"].shading.color_type = 'MATERIAL'

        shading = context.space_data.shading


        if shading.color_type != 'RANDOM':
            shading.color_type = 'RANDOM'

        elif shading.color_type == 'RANDOM':
            shading.color_type = 'MATERIAL'

        else:
            # if self.shading_random == 'RANDOM' and shading.color_type != 'RANDOM':
            #     self.shading_random = shading.color_type

            shading.color_type = self.shading_random

    @staticmethod
    def shading_bool_mode(context, self):

        shading = context.space_data.shading
        shading_type = shading.color_type
        bools = []

        for obj in self.sel:
            bool_modifiers = self.get_modifiers_by_type(obj, 'BOOLEAN')
            if not bool_modifiers:
                continue
            for bool in bool_modifiers:
                bools.append(bool.object)

        # Si au moins un des objets est en solid, il retourne True
        solid_statue = any([obj.display_type == 'TEXTURED' for obj in bools])
        for obj in bools:
            if solid_statue:
                obj.display_type = 'BOUNDS'
                shading.color_type = shading_type
            else:
                obj.display_type = 'TEXTURED'
                shading.color_type = 'OBJECT'

    @staticmethod
    def edit_boolean_operation(context, self):
        for obj in context.scene.objects:
            for mod in obj.modifiers:
                if mod.type == 'BOOLEAN' and mod.object == self.act_obj:
                    if mod.operation == 'DIFFERENCE':
                        mod.operation = 'UNION'
                    else:
                        mod.operation = 'DIFFERENCE'

    # si obj bool a un weighted et obj qui recoit n en a pas, on vire le weighted
    @staticmethod
    def if_weighted_on_bool_and_not_on_ref_remove_weighted_from_bool(context, self, obj):
        for bool in context.scene.objects:
            for mod in bool.modifiers:
                if mod.type == 'BOOLEAN' and mod.object == obj:
                    bool_has_weighted = self.get_modifiers_by_type(obj, 'WEIGHTED_NORMAL')
                    ref_has_weighted = self.get_modifiers_by_type(bool, 'WEIGHTED_NORMAL')

                    if bool_has_weighted and not ref_has_weighted:
                        self.remove_modifier(bool, 'WEIGHTED_NORMAL')


    @staticmethod
    def shading_overlays(context, self):
        shading = context.space_data.overlay
        shading.show_overlays = not shading.show_overlays
        self.prefs.display_texts = not self.prefs.display_texts

        # if hasattr(bpy.types, "SFC_MT_addon_prefs"):
        #     if not bpy.object.show_text_options:
        #         pass
        #     else:
        #         bpy.object.show_text_options = not bpy.object.show_text_options

    @staticmethod
    def shading_xray(context, self):
        shading = context.space_data.shading
        shading.show_xray = not shading.show_xray

    @staticmethod
    def shading_face_orientation(context, self):
        shading = context.space_data.overlay

        if shading.show_face_orientation == True:
            context.space_data.overlay.show_overlays = True

        shading.show_face_orientation = not shading.show_face_orientation

    @staticmethod
    def shading_hide_grid(context, self):
        shading = context.space_data.overlay

        if shading.show_floor:
            context.space_data.overlay.show_overlays = True

        shading.show_floor = not shading.show_floor
        shading.show_axis_x = not shading.show_axis_x
        shading.show_axis_y = not shading.show_axis_y

    @staticmethod
    def snap_activate_snap(context, self):
        context.scene.tool_settings.use_snap_align_rotation = False
        context.scene.tool_settings.use_snap = not context.scene.tool_settings.use_snap

    @staticmethod
    def snap_vertex(context, self):
        context.scene.tool_settings.use_snap = True
        context.scene.tool_settings.snap_elements = {'VERTEX'}
        context.scene.tool_settings.snap_target = 'MEDIAN'
        context.scene.tool_settings.use_snap_align_rotation = False

    @staticmethod
    def snap_face(context, self):
        context.scene.tool_settings.use_snap = True
        context.scene.tool_settings.snap_elements = {'FACE'}
        context.scene.tool_settings.snap_target = 'MEDIAN'
        context.scene.tool_settings.use_snap_align_rotation = True

    @staticmethod
    def snap_grid(context, self):
        context.scene.tool_settings.use_snap = True
        context.scene.tool_settings.snap_elements = {'INCREMENT'}
        context.scene.tool_settings.use_snap_grid_absolute = True

    @staticmethod
    def snap_origin_to_selection(context):
        for obj in context.selected_objects:
            if context.object.mode == 'EDIT':
                bpy.ops.view3d.snap_cursor_to_selected()
                bpy.ops.object.mode_set(mode='OBJECT')
                bpy.ops.object.origin_set(type='ORIGIN_CURSOR', center='MEDIAN')
                bpy.ops.object.mode_set(mode='EDIT')
            elif context.object.mode == 'OBJECT':
                bpy.ops.object.origin_set(type='ORIGIN_CENTER_OF_MASS', center='MEDIAN')

    @staticmethod
    def snap_origin_to_grid(context):
        for obj in context.selected_objects:
            if context.object.mode == 'EDIT':
                bpy.ops.object.mode_set(mode='OBJECT')
                bpy.ops.object.transform_apply(location=True, rotation=False, scale=False)
                bpy.ops.object.mode_set(mode='EDIT')
            elif context.object.mode == 'OBJECT':
                bpy.ops.object.transform_apply(location=True, rotation=False, scale=False)


    @staticmethod
    def apply_transform_scale(context, obj):
        """Apply transform scale, rescale modifiers according
        supported modifiers : Solidify, Bevel, Displace, Screw, Mirror, and Array
        :param context:
        :param obj:
        :return:
        """
        scale = min(obj.scale, key=lambda x: abs(1 - x))
        with deselect_and_restore(context, obj, object_mode='OBJECT'):
            bpy.ops.object.transform_apply(location=False, rotation=False, scale=True)

        # apply scale to modifiers
        if scale != 1.0:
            for mod in obj.modifiers:
                _type = mod.type
                # modifier type: property to rescale
                props = {
                    'SOLIDIFY':'thickness',
                    'BEVEL':'width',
                    'DISPLACE':'strength',
                    'SCREW':'screw_offset'
                }
                if _type in props.keys():
                    prop = props[_type]
                    setattr(mod, prop, getattr(mod, prop) * scale)

                elif _type == 'ARRAY':
                    offset = mod.constant_offset_displace
                    for i, val in enumerate(offset):
                        offset[i] = val * scale
                    mod.fit_length *= scale

                if hasattr(mod, 'merge_threshold'):
                    mod.merge_threshold *= scale

        # to enable support for other modifiers
        return scale

    @staticmethod
    def make_single_user(context, obj):
        """ Make given object single user
        :param context:
        :param obj:
        :return:
        """
        if obj.data.users > 1:
            with deselect_and_restore(context, obj, object_mode='OBJECT'):
                bpy.ops.object.make_single_user(
                    type='SELECTED_OBJECTS',
                    object=True,
                    obdata=True,
                    material=False,
                    texture=False,
                    animation=False
                    )

    # Mesh subs select
    @staticmethod
    def select_mesh_components(obj, select):
        mesh = obj.data
        for f in mesh.polygons:
            f.select = select
        for ed in mesh.edges:
            ed.select = select
            for v in ed.key:
                mesh.vertices[v].select = select

    # Vertex groups
    @staticmethod
    def get_vertex_group_by_name(obj, name):
        return obj.vertex_groups.get(name)

    @staticmethod
    def get_or_add_vertex_group(obj, name, weight=1.0):
        """Add a vertex group in object mode and set weight for selected vertices
        :param obj: object
        :param name: vertex group name
        :param weight: weight for selected verts
        :return: vertex group
        """
        vgroup = obj.vertex_groups.get(name)
        if vgroup is None:
            vgroup = obj.vertex_groups.new()
            vgroup.name = name
            vgroup.add([v.index for v in obj.data.vertices if v.select], weight, 'REPLACE')
        return vgroup

    @staticmethod
    def remove_vertex_group(obj, name):
        vgroup = obj.vertex_groups.get(name)
        if vgroup is not None:
            obj.vertex_groups.remove(vgroup)

    """
    NOTE:
    method using bmesh dosen't update mesh 
    to be able to work in edit and object mode
    """

    # Bmesh Vertex
    @staticmethod
    def get_selected_verts(bm):
        """Get selected vertex
        :param bm: bmesh
        :return: list of BmVerts
        """
        return [v for v in bm.verts if v.select]

    @staticmethod
    def get_selected_verts_index(bm):
        """Get selected vertex indexes
        :param bm: bmesh
        :return: set of index
        """
        return set(v.index for v in bm.verts if v.select)

    # Bmesh Edges
    @staticmethod
    def get_selected_edges(bm):
        """Get selected edges
        :param bm: bmesh
        :return: list of BmEdges
        """
        return [ed for ed in bm.edges if ed.select]

    @staticmethod
    def get_selected_edges_index(bm):
        """Get selected edges indexes
        :param bm: bmesh
        :return: set of index
        """
        return set(ed.index for ed in bm.edges if ed.select)

    @staticmethod
    def get_active_edge(bm):
        """Get active edge
        :param bm: bmesh
        :return: BmEdge Last selected edge
        """
        edge = None
        if bm.select_history:
            edge = bm.select_history[-1]
        return edge


    @staticmethod
    def edit_edges_smooth(bm, smooth):
        """Edit selected edges smooth
        :param bm: bmesh
        :param smooth: boolean
        :return:
        """
        for ed in bm.edges:
            if ed.select:
                ed.smooth = smooth

    # Bmesh Layers
    @staticmethod
    def get_bmesh_layer(bm, elem_type, layer_type):
        """
        :param bm: bmesh
        :param elem_type: bmesh component in ['verts', 'edges', 'faces']
        :param layer_type: any valid layer type eg: crease, bevel_weight, freestyle, deform ..
        :return: first layer found or create one
        """
        iter_elem = getattr(bm, elem_type)
        layers = getattr(iter_elem.layers, layer_type)
        if len(layers) > 0:
            layer = layers[0]
        else:
            layer = layers.new()
        return layer

    def edit_edges_layer_weight(self, bm, layer_type, weight):
        """Edit selected edges weight for any layer
        :param bm: bmesh
        :param layer_type: any valid edge layer type in ['crease', 'bevel_weight']
        :param value: weight
        :return:
        """
        layer = self.get_bmesh_layer(bm, 'edges', layer_type)
        for ed in bm.edges:
            if ed.select:
                ed[layer] = weight

    def edge_weight_from_active(self, bm, layer_type):
        """Set weight value on selected edges from active edge weight value on specified layer
        :param bm: bmesh
        :param layer_type: any valid edge layer type in ['crease', 'bevel_weight']
        :return:
        """
        layer = self.get_bmesh_layer(bm, 'edges', layer_type)
        edge = self.get_active_edge(bm)
        selected = self.get_selected_edges(bm)
        weight = 0
        if edge is None:
            # prevent division by 0
            if len(selected) > 0:
                weight = sum([ed[layer] for ed in selected]) / len(selected)
        else:
            weight = edge[layer]
        for ed in selected:
            ed[layer] = weight

    @staticmethod
    def get_sharp_edges(bm, angle):
        """Get sharp edges where faces diff >= than angle
        :param bm: bmesh
        :param angle: degrees minimum angle between faces
        :return: list of BmEdges
        """
        _angle = radians(angle)
        return [ed for ed in bm.edges
                if len(ed.link_faces) == 2 and
                    ed.link_faces[0].normal.angle(ed.link_faces[1].normal) >= _angle]

    @staticmethod
    def get_sharp_edges_index(bm, angle):
        """Get sharp edges indices where faces diff >= than angle
        :param bm: bmesh
        :param angle: degrees minimum angle between faces
        :return: set of indices
        """
        _angle = radians(angle)
        return set(ed.index for ed in bm.edges
                if len(ed.link_faces) == 2 and
                ed.link_faces[0].normal.angle(ed.link_faces[1].normal) >= _angle)

    @staticmethod
    def get_smooth_edges(bm, angle):
        """Get smooth edges where faces diff < than angle
        :param bm: bmesh
        :param angle: degrees minimum angle between faces
        :return: list of BmEdges
        """
        _angle = radians(angle)
        return [ed for ed in bm.edges
                if len(ed.link_faces) == 2 and
                    ed.link_faces[0].normal.angle(ed.link_faces[1].normal) < _angle]

    @staticmethod
    def get_smooth_edges_index(bm, angle):
        """Get smooth edges indices where faces diff < than angle
        :param bm: bmesh
        :param angle: degrees minimum angle between faces
        :return: set of indices
        """
        _angle = radians(angle)
        return set(ed.index for ed in bm.edges
                if len(ed.link_faces) == 2 and
                ed.link_faces[0].normal.angle(ed.link_faces[1].normal) < _angle)

    def get_edges_in_layer(self, bm, layer_type):
        """Get edges with values > 0 for layer of type with given name
        :param bm: bmesh
        :param layer_type: any valid edge layer type in ['crease', 'bevel_weight']
        :return: list of BmEdges
        """
        layer = self.get_bmesh_layer(bm, 'edges', layer_type)
        return [ed for ed in bm.edges if ed[layer]]

    def get_edges_in_layer_index(self, bm, layer_type):
        """Get edges indices with values > 0 for layer of type with given name
        :param bm: bmesh
        :param layer_type: any valid edge layer type in ['crease', 'bevel_weight']
        :return: set of indices
        """
        layer = self.get_bmesh_layer(bm, 'edges', layer_type)
        return set(ed.index for ed in bm.edges if ed[layer])

    def get_edges_not_in_layer(self, bm, layer_type):
        """Get edges with values == 0 for layer of type with given name
        :param bm: bmesh
        :param layer_type: any valid edge layer type in ['crease', 'bevel_weight']
        :return: list of BmEdges
        """
        layer = self.get_bmesh_layer(bm, 'edges', layer_type)
        return [ed for ed in bm.edges if not ed[layer]]

    def get_edges_not_in_layer_index(self, bm, layer_type):
        """Get edges indices with values == 0 for layer of type with given name
        :param bm: bmesh
        :param layer_type: any valid edge layer type in ['crease', 'bevel_weight']
        :return: set of indices
        """
        layer = self.get_bmesh_layer(bm, 'edges', layer_type)
        return set(ed.index for ed in bm.edges if not ed[layer])

    def hide_unhide(self, obj, states=False, select=False):

        is_hidden = False
        for child in obj.children:

            if child.hide_set(True):
                is_hidden = True

            child.hide_set(states)

            child.select_set(state=select)

            if is_hidden:
                self.sel_hidden.append(child)

            if child.children:
                self.hide_unhide(child, states, select)


    #########################
    # select_hierarchy
    #########################
    def find_child(self, obj, sel):
        for x in obj.children:
            sel.append(x)
            self.find_child(x, sel)

    def find_parent(self, obj, sel):
        sel.append(obj)
        if obj.parent:
            self.find_parent(obj.parent, sel)

    def select_hierarchy(self, obj, sel):
        self.find_parent(obj, sel)  # find parent

        if len(sel) > 0:
            last_parent = sel[0]
            for obj in sel:
                # parent of everything has no parent itself
                if not obj.parent: last_parent = obj

            # find children for each parent
            for obj in sel: self.find_child(obj, sel)

            # select all
            for obj in sel:
                obj.select_set(state=True)

            # make parent active
            bpy.context.view_layer.objects.active = last_parent

    """
    edges = self.get_edges_not_in_layer_index(bm, 'bevel_weight', 'BevelWeight')
    sharp = self.get_sharp_edges_index(bm, 30)
    
    sharp_not_in_layer = edges.intersection(sharp)
    
    """


class CommonProperties:

    def __init__(self):
        self.addon_prefs = get_addon_preferences()
        self.input = ""
        self.mouse_x = 0
        self.modal_speed = self.addon_prefs.modal_speed
        self.OBJ = bpy.context.active_object

class PrepareApplyScale():

    def __init__(self, selected_objects):

        self.key_confirm = selected_objects

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        for obj in self.key_confirm:
            obj.select_set(state=True)
            # obj.select = True


def get_addon_preferences():
    addon_key = __package__.split(".")[0]
    addon_prefs = bpy.context.preferences.addons[addon_key].preferences
    
    return addon_prefs


def shading_mode(obj, mode='SMOOTH'):
    prefs = get_addon_preferences()

    if obj.type == 'MESH':
        if mode == 'SMOOTH':
            bpy.ops.object.shade_smooth()
            bpy.context.object.data.use_auto_smooth = True
            bpy.context.object.data.auto_smooth_angle = radians(prefs.auto_smooth_value)
        else:
            bpy.context.object.data.use_auto_smooth = True
            bpy.context.object.data.auto_smooth_angle = radians(prefs.auto_smooth_value)
            bpy.ops.object.shade_flat()
    else:
        if mode == 'SMOOTH':
            bpy.ops.object.shade_smooth()
        else:
            bpy.ops.object.shade_flat()
    
def get_modifier_list(obj, mod_type):
    return [mod.name for mod in obj.modifiers if mod.type == mod_type]


def custom_selection(ob, select):
    mesh = ob.data
 
    for f in mesh.polygons:
        f.select = select
    for e in mesh.edges:
        e.select = select
        for v in e.key:
            mesh.vertices[v].select = select


def get_edges_boundary_loop(bm, value):
    """ Select boundary or inner loop
        value: 1 to get boundary loop
               2 to get inner loop """
           
    return [e for e in bm.edges if e.select and len([f for f in e.link_faces if f.select]) == value]


def apply_transform_scale(obj):
    coef_scale = min(obj.scale, key=lambda x: abs(1 -  x))
    bpy.ops.object.transform_apply(location=False, rotation=False, scale=True)

    return coef_scale


def get_input_value(self, value):
    """ Return valid input """

    if value == "-":
        if not self.input.startswith("-"):
            self.input = "-" + self.input

    elif value == "+":
        if self.input.startswith("-"):
            self.input = self.input.split("-")[-1]

    elif value == "." and "." in self.input:
        pass

    else:
        self.input += value


def check_valid_input(input):
    valid_input = [(True, False, False),
                   (False, True, False),
                   (False, False, True),
                   ]

    return input in valid_input


def remove_all_modifier_by_type(context, mod_type):

    for obj in context.selected_objects:
        if obj.type not in {'MESH', 'CURVE', 'FONT'}:
            # obj.select = False
            obj.select_set(state=False)

        else:
            if obj.modifiers:
                for mod in obj.modifiers:
                    if mod.type == mod_type:
                        obj.modifiers.remove(mod)


def hide_all_modifier_by_type(context, mod_type):

    for obj in context.selected_objects:
        if obj.type not in {'MESH', 'CURVE', 'FONT'}:
            obj.select_set(state=False)

        else:
            if obj.modifiers:
                for mod in obj.modifiers:
                    if mod.type == mod_type:
                        mod.show_viewport = not mod.show_viewport

def get_edges(bm, clean):
    """ If "clean": return the index of the edges whose angle is less than 30 degrees
        else: return the index of the edges whose the angle is
        greater than or equal to 30 degree """

    beveWeightLayer = bm.edges.layers.bevel_weight['BevelWeight']
    edges = set()
    for e in bm.edges:
        if clean:
            if e[beveWeightLayer]:
                if len(e.link_faces) == 2:
                    face0, face1 = e.link_faces
                    if face0.normal.angle(face1.normal) < 0.523599:
                        edges.add(e.index)

        else:
            if not e[beveWeightLayer]:
                if len(e.link_faces) == 2:
                    face0, face1 = e.link_faces
                    if face0.normal.angle(face1.normal) >= 0.523599:
                        edges.add(e.index)

    return edges


def has_weighted_normals_modifier():
    return any([mod for mod in bpy.types.Modifier.bl_rna.properties['type'].enum_items if
         mod.identifier == 'WEIGHTED_NORMAL'])


def move_weighted_normals_down(obj):
    weighted_normals = has_weighted_normals_modifier()

    if weighted_normals:
        modifs = SpeedApi.get_modifiers_by_type(obj, 'WEIGHTED_NORMAL')
        if len(modifs) == 0:
            mod = SpeedApi.create_modifier(obj, 'WEIGHTED_NORMAL', "Weighted Normal")
            mod.keep_sharp = True

        else:
            mod = modifs[-1]
            num_modifs = len(obj.modifiers)
            index = SpeedApi.get_modifier_index(obj, mod)
            if index > -1:
                num_modifs -= index
            for i in range(num_modifs):
                SpeedApi.move_modifier_down(bpy.context, obj, mod)


def check_weighted_normals(obj):
    move_weighted_normals_down(obj)




