# -*- coding:utf-8 -*-

# Speedflow Add-on
# Copyright (C) 2018 Cedric Lepiller aka Pitiwazou & Legigan Jeremy AKA Pistiwique and Stephen Leger
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

# <pep8 compliant>

import bpy
import blf
from math import degrees


# CONSTANTS
RED = (1, 0, 0, 0.9)
GREEN = (0, 1, 0, 0.9)
CR = "Carriage return"


# Default keymaps available for modifiers based modals
KEYMAP_START = [

]

KEYMAP_END = [
    ("SPACE", "Change Modal"),
    ("SHIFT + SPACE", "Add New Modifier"),
    ("H", "Hide Modifier"),
    ("A", "Apply Modifier"),
    ("DEL/SUPPR", "Remove Modifier"),
    ("UP", "Move Up"),
    ("DOWN", "Move Down"),
    ("RIGHT ARROW/>", "Next Modifier (same type)"),
    ("LEFT ARROW/<", "Previous Modifier (same type)"),
    # ("CTRL + RIGHT ARROW/>", "Next Modifier"),
    # ("CTRL + LEFT ARROW/<", "Previous Modifier"),
    ("ESC", "Exit"),
    ("CTRL + H", "Show/Hide Keymaps"),
]

KEYMAP_END_BEGINNER = [
    ("SPACE", "Change Modal"),
    ("SHIFT + SPACE", "Add New Modifier"),
    ("H", "Hide Modifier"),
    ("A", "Apply Modifier"),
    ("DEL/SUPPR", "Remove Modifier"),
    ("UP", "Move Up"),
    ("DOWN", "Move Down"),
    ("RIGHT ARROW/>", "Next Modifier (same type)"),
    ("LEFT ARROW/<", "Previous Modifier (same type)"),
    ("ESC", "Exit"),
    ("CTRL + H", "Show/Hide Keymaps"),
]

def draw_text_callback_mpm(self, context, modal):

    # TODO: filter by implemented modals until end of migration
    if modal in {'ARRAY', 'BEVEL', 'BOOLEAN', 'MIRROR', 'SCREW',
                 'SOLIDIFY', 'SUBSURF', 'DISPLACE', 'WEIGHTED_NORMAL',
                 'DECIMATE', 'TRIANGULATE', 'TUBIFY', 'EDGE_SPLIT',
                 'SHRINKWRAP', 'SIMPLE_DEFORM', 'WIREFRAME', 'CURVE',
                 'CORRECTIVE_SMOOTH', 'MOD_VISIBILITY', 'CUTTER', 'SKIN', 'CUTTER'}:


        get_actions, get_keymaps = {
            'BEVEL': (bevel_actions, bevel_keymap),
            'TUBIFY': (tubify_actions, tubify_keymap),
            'SUBSURF': (subsurf_actions, subsurf_keymap),
            'MIRROR': (mirror_actions, mirror_keymap),
            'SOLIDIFY': (solidify_actions, solidify_keymap),
            'ARRAY': (array_actions, array_keymap),
            'BOOLEAN': (boolean_actions, boolean_keymap),
            'SCREW': (screw_actions, screw_keymap),
            'DISPLACE': (displace_actions, displace_keymap),
            'DECIMATE': (decimate_actions, decimate_keymap),
            'TRIANGULATE': (triangulate_actions, triangulate_keymap),
            'WEIGHTED_NORMAL': (weighted_normal_actions, weighted_normal_keymap),
            'EDGE_SPLIT': (edge_split_actions, edge_split_keymap),
            'SHRINKWRAP': (shrinkwrap_actions, shrinkwrap_keymap),
            'SIMPLE_DEFORM': (simple_deform_actions, simple_deform_keymap),
            'WIREFRAME': (wireframe_actions, wireframe_keymap),
            'CURVE': (curve_actions, curve_keymap),
            'CORRECTIVE_SMOOTH': (corrective_smooth_actions, corrective_smooth_keymap),
            'MOD_VISIBILITY': (mod_visibility_actions, visibility_keymap),
            'SKIN': (skin_actions, skin_keymap),
            'CUTTER': (cutter_actions, cutter_keymap),


            # 'SYMMETRIZE': (symetrize_actions, symetrize_keymap),
            # 'ROTATE': (rotate_actions, rotate_keymap),
            }[modal]



        if not self.prefs.display_texts:
            # self.prefs.display_keymaps = True
            keymap = []
            get_keymaps(self, keymap)
            draw_keymap(self, context, keymap)


        if self.prefs.display_texts:
            subtitle, actions = get_actions(self, context)
            draw_actions(self, context, subtitle, actions_as_tuple(self, actions))
            # self.prefs.display_keymaps = state

        if self.prefs.display_keymaps:
            keymap = []
            get_keymaps(self, keymap)
            draw_keymap(self, context, keymap)



        return



    # TODO: remove this once xxx_actions are implemented
    text_to_draw = {'SYMMETRIZE': symmetrize_text,
                    'ROTATE': rotate_text,
                    }
    draw_text_array_mpm(self, context, text_to_draw[modal](self, context), keys_text(self, context, modal))


def actions_as_tuple(self, actions):
    """Convert actions definition into fixed size tuple
    :param self:
    :param actions: enumerable action list definition
    :return: tuple of fixed size tuples
    """
    # ensure things are defined
    has_mod = hasattr(self, 'act_mod') and self.act_mod is not None
    has_input = hasattr(self, 'input') and self.input
    has_modal_action = hasattr(self, 'modal_action')

    res = []
    for action in actions:
        attr, label, shortcut = action[0:3]
        # enable display
        enable = len(action) < 4 or action[3]
        # active action state
        active = has_modal_action and attr == self.modal_action

        # attr of modifier
        is_modifier_attr = has_mod and hasattr(self.act_mod, attr)

        value = None

        # Auto label from rna_type
        if label is None:
            label = self.get_label_from_rna(self.act_mod, attr)

        # value override by definition
        if len(action) > 4:
            value = action[4]

        # override by input
        if active and has_input:
            value = self.input

        elif is_modifier_attr and value is None:
            # value from modifier
            value = getattr(self.act_mod, attr)

        res.append((enable, active, label, shortcut, value))

    return tuple(res)


# Actions

# ARRAY
def array_actions(self, context):
    act_mod = self.act_mod
    if act_mod is not None:

        fit_type = act_mod.fit_type.split("_")[-1].capitalize()
        not_picking = self.modal_action not in self.pick_actions.values()

        show_in_red = not (act_mod.show_viewport )
        subtitle = ("", show_in_red, True)
        # relative_offset = act_mod.relative_offset_displace[self.axis_index]
        # constant_offset = act_mod.constant_offset_displace[self.axis_index]

        axis = "XYZ"[self.axis_index]
        x, y, z = act_mod.relative_offset_displace[:]
        if "relative_" in self.modal_action:
            relative_offset = "%s: %.3f" % (axis, (x, y, z)[self.axis_index])
        else:
            relative_offset = "X: %.2f Y: %.2f Z: %.2f" % (x, y, z)

        x, y, z = act_mod.constant_offset_displace[:]
        if "constant_" in self.modal_action:
            constant_offset = "%s: %.3f" % (axis, (x, y, z)[self.axis_index])
        else:
            constant_offset = "X: %.2f Y: %.2f Z: %.2f" % (x, y, z)

        strength = 0
        if self.modal_action == 'strength':
            next = self.get_modifier_down(self.act_obj, self.act_mod)
            if next is not None and next.type == 'DISPLACE':
                strength = next.strength

        if self.prefs.beginner_mode:
            actions = (
                ('free',                    "Free navigation",  None, False),
                ('name',                        "Name",                 None,           not_picking),
                ('count',                       None,                           "S",                not_picking and fit_type == 'Count'),
                ('fit_type',                    None,                           "D",                not_picking, fit_type),
                ('curve',                       "Curve",                        "P",                fit_type == 'Curve'),
                ('fit_length',                  "Length",                       "L",                not_picking and fit_type == 'Length'),

                ('relative_offset_displace',    "Relative Offset",              "X,Y or Z", not_picking, relative_offset),
                ('relative_X',                  "Relative Offset",  "X", False, act_mod.relative_offset_displace[0]),
                ('relative_Y',                  "Relative Offset",  "Y", False, act_mod.relative_offset_displace[1]),
                ('relative_Z',                  "Relative Offset",  "Z", False, act_mod.relative_offset_displace[2]),

                ('constant_offset_displace',    "Constant Offset",              "CTRL + X,Y or Z", not_picking, constant_offset),
                ('constant_X',                  "Constant Offset", "CTRL + X", False, constant_offset),
                ('constant_Y',                  "Constant Offset", "CTRL + Y", False, constant_offset),
                ('constant_Z',                  "Constant Offset", "CTRL + Z", False, constant_offset),
            )
        else:

            actions = (
                # Action names as label for title, never display in shortcuts list
                ('free', "Free navigation", None, False),

                # attr                          (action name)           Label                   key,  display, override attr value
                ('name',                        "Name",                         None,               not_picking),
                ('count',                       None,                           "S",                not_picking and fit_type == 'Count'),
                ('fit_type',                    None,                           "D",                not_picking, fit_type),
                ('curve',                       "Curve",                        "P",                fit_type == 'Curve'),
                ('fit_length',                  "Length",                       "L",                not_picking and fit_type == 'Length'),
                ('use_object_offset',           "Offset Object",                "F",                not_picking),

                ('relative_offset_displace',    "Relative Offset",              "X,Y or Z",         not_picking, relative_offset),
                ('relative_X',                  "Relative Offset",              "X", False,         act_mod.relative_offset_displace[0]),
                ('relative_Y',                  "Relative Offset",              "Y", False,         act_mod.relative_offset_displace[1]),
                ('relative_Z',                  "Relative Offset",              "Z", False,         act_mod.relative_offset_displace[2]),

                ('constant_offset_displace',    "Constant Offset",              "CTRL + X,Y or Z",  not_picking, constant_offset),
                ('constant_X',                  "Constant Offset X", "CTRL + X", False,             constant_offset),
                ('constant_Y',                  "Constant Offset Y", "CTRL + Y", False,             constant_offset),
                ('constant_Z',                  "Constant Offset Z", "CTRL + Z", False,             constant_offset),

                ('use_merge_vertices',          None,                           "G"),
                ('merge_threshold',             None,                           "J",                not_picking and act_mod.use_merge_vertices),
                ('start_cap',                   "Start Cap",                    "E",                True),
                ('end_cap',                     "End Cap",                      "R",                True),
                ('offset_object',               "Pick Object",                  "O",                True),
                ('use_merge_vertices_cap',      'First Last',                   "U",                not_picking),
                ('strength',                    'Strength',                     "V",                strength != 0, strength),
            )

    else:
        subtitle, actions = ("", False, True), (('none', "No Array", None))
    # fallback
    return subtitle, actions

# BEVEL
def bevel_actions(self, context):

    # Display options for actions:
    # 1 use "red" state for current action in action list under line
    # 2 draw action beside title and draw mod name under line

    act_mod = self.act_mod

    # values not directly related to modifier attrs
    width = None
    if self.use_bevel_weight:
        edges = self.edges_datas['selected']
        width = act_mod.width * sum([e.bevel_weight for e in edges]) / len(edges)

    # show title in red
    show_in_red = not (act_mod.show_viewport and act_mod.show_render)

    #        Subtitle,      title in red, show action label and value instead of subtitle
    # 1 Use only red state for active action
    subtitle = (act_mod.name,  show_in_red)
    # 2 Use action name + value as subtitle
    subtitle = ("",            show_in_red, True)

    # for obj in context.selected_objects:
    subdiv_mode = self.is_object_in_destructive_mode(self.act_obj)
    if subdiv_mode:
        self.mode = "NON-DESTRUCTIVE"
    else:
        self.mode = "DESTRUCTIVE"

    if self.prefs.beginner_mode:
        actions = (
            # attrib name,          Title              Key    Display condition,
            ('free',                "Free Navigation", None,       False),
            ('name',                "Name",            None),
            ('width',               None,              "S",        True, width),
            ('segments',            None,              "D"),
            ('profile',             None,              "F"),
            ('limit_method',        None,              "G"),
            ('loop_slide',          None,              "J"),
            ('use_only_vertices',   None,              "K"),
            ('angle_limit',         None,              "X",  act_mod.limit_method == 'ANGLE', degrees(act_mod.angle_limit)),
        )
    else:
        actions = (
            # attrib name,          Title               Key     Display condition,
            ('free',                "Free Navigation",  None,   False),
            ('name',                "Name",             None),
            ('width',               None,               "S",    True, width),
            ('segments',            None,               "D"),
            ('profile',             None,               "F"),
            ('limit_method',        None,               "G"),
            ('offset_type',         "Width Method",     "M"),
            ('loop_slide',          None,               "J"),
            ('use_only_vertices',   None,               "K"),
            ('use_clamp_overlap',   None,               "L"),
            ('subdiv_mode',         "Subdiv Mode",      "E", True, self.mode),
            ('angle_limit',         None,               "X",    act_mod.limit_method == 'ANGLE', degrees(act_mod.angle_limit)),
            ('miter_outer',         None,               "B"),
            ('miter_inner',         None,               "N"),
            ('vertex_group',        None,               "C"),
        )
    return subtitle, actions

# BOOLEAN
def boolean_actions(self, context):
    act_mod = self.act_mod
    # show title in red
    show_in_red = not (act_mod.show_viewport and act_mod.show_render)
    subtitle = (act_mod.operation, show_in_red, False)
    actions = (
        ('free',            "Free Navigation",      None, False),
        ('name',            "Name",                 None),
        ('union',           "Union",                "S", True, act_mod.operation == 'UNION'),
        ('difference',      "Difference",           "D", True, act_mod.operation == 'DIFFERENCE'),
        ('intersect',       "Intersect",            "F", True, act_mod.operation == 'INTERSECT'),
        # ('keep_operand',    "Keep boolean object",  "E", True, self.prefs.delete_bool_obj),
        ('object',          None,                   "V"),
    )
    return subtitle, actions

# CORRECTIVE SMOOTH
def corrective_smooth_actions(self, context):
    act_mod = self.act_mod
    # show title in red
    show_in_red = not (act_mod.show_viewport and act_mod.show_render)
    subtitle = ("", show_in_red, True)

    actions = (
        ('free',                "Free Navigation",          None, False),
        ('name',                "Name",                     None),
        ('iterations',          "Repeat",              "S"),
        ('factor',              "Factor",              "D"),
        ('smooth_type',         "Smooth Type",         "F"),
        ('use_only_smooth',     "Only Smooth",         "G"),
        ('use_pin_boundary',    "Only Boundaries",     "J"),
        ('rest_source',         "Rest Source",         "K"),
        ('vertex_group',        "Vertex Group",        "C"),
    )
    return subtitle, actions

# CURVE
def curve_actions(self, context):
    act_mod = self.act_mod
    # show title in red
    show_in_red = not (act_mod.show_viewport and act_mod.show_render and act_mod.object)
    subtitle = ("", show_in_red, True)

    if self.prefs.beginner_mode:
        actions = (
            ('free',            "Free Navigation",          None, False),
            ('name',            "Name",                     None),
            ('deform_axis',     "Deform Axis",              "S"),
            ('object',          "Object",                   "O",    True),
        )
    else:
        actions = (
            ('free',            "Free Navigation",          None, False),
            ('name',            "Name",                     None),
            ('deform_axis',     "Deform Axis",              "S"),
            ('object',          "Object",                   "O",    True),
            ('vertex_group',    "Vertex Group",             "C"),
        )
    return subtitle, actions

# CUTTER
def cutter_actions(self, context):
    show_in_red = False
    subtitle = ("", show_in_red, True)

    actions = (
        )
    return subtitle, actions

# DECIMATE
def decimate_actions(self, context):

    act_mod = self.act_mod
    show_in_red = not (act_mod.show_viewport and act_mod.show_render)
    subtitle = ("", show_in_red, True)

    if act_mod.decimate_type == 'COLLAPSE':
        if self.prefs.beginner_mode:
            actions = (
                ('free',                        "Free Navigation",  None, False),
                ('name',                        "Name",             None),
                ('decimate_type',               None,               "D"),

                ('ratio',                       "Collapse",         "E"),
            )
        else:
            actions = (
                ('free',                        "Free Navigation",  None, False),
                ('name',                        "Name",             None),

                ('decimate_type', None, "D"),
                ('ratio',                       "Collapse",         "E"),
                ('use_collapse_triangulate',    "Triangulate",      "F"),
                ('use_symmetry',                "Symmetry",         "G"),
                ('symmetry_axis',               "Symmetry Axis",    "J", act_mod.use_symmetry),
                ('vertex_group',                None,               "C"),
                ('vertex_group_factor',         "Vgroup Factor",    "X"),
            )
        return subtitle, actions

    elif act_mod.decimate_type == 'UNSUBDIV':
        actions = (
            ('free', "Free Navigation", None, False),
            ('name', "Name", None),

            ('decimate_type', None, "D"),
            ('iterations', "Collapse", "R"),
        )
        return subtitle, actions

    elif act_mod.decimate_type == 'DISSOLVE':
        actions = (
            ('free',                    "Free Navigation",  None, False),
            ('name',                    "Name",             None),
            ('angle_limit',             "Collapse",         "S"),
            ('decimate_type',           None,               "D"),
        )
        return subtitle, actions

# DISPLACE
def displace_actions(self, context):

    act_mod = self.act_mod
    show_in_red = not (act_mod.show_viewport and act_mod.show_render)
    subtitle = ("", show_in_red, True)

    if self.prefs.beginner_mode:
        actions = (
            ('free',                    "Free Navigation",      None, False),
            ('name',                    "Name",                 None),
            ('strength',                None,                   "S"),
            ('direction',               "Direction: (X,Y,Z)",   "X"),
        )
    else:
        actions = (
            ('free',                    "Free Navigation",      None, False),
            ('name',                    "Name",                 None),
            ('strength',                None,                   "S"),
            ('mid_level',               None,                   "D"),
            ('direction',               "Direction: List",      "F"),
            ('space',                   None,                   "G"),
            ('direction',               "Direction: (X,Y,Z)",   "X"),
            ('vertex_group',            "Vertex Group",         "C"),
        )
    return subtitle, actions

# EDGE SPLIT
def edge_split_actions(self, context):
    act_mod = self.act_mod
    show_in_red = not (act_mod.show_viewport and act_mod.show_render)
    subtitle = ("", show_in_red, True)
    actions = (
        ('free',                    "Free Navigation",  None, False),
        ('name',                    "Name",             None),
        ('split_angle',             None,               "S", True,  degrees(act_mod.split_angle)),
        ('use_edge_angle',          None,               "D"),
        ('use_edge_sharp',          None,               "F"),

    )
    return subtitle, actions

# MIRROR
def mirror_actions(self, context):

    act_mod = self.act_mod
    if act_mod is None:
        subtitle, actions = ("", False, True), (('free', "No Mirror", None))

    else:
        show_in_red = not (act_mod.show_viewport and act_mod.show_render)
        subtitle = ("", show_in_red, True)

        text_axis = {
            (True, True, True): "X, Y, Z",
            (True, True, False): "X, Y",
            (True, False, True): "X, Z",
            (False, True, True): "Y, Z",
            (True, False, False): "X",
            (False, True, False): "Y",
            (False, False, True): "Z",
            (False, False, False): "No axis"
        }[(act_mod.use_axis[0], act_mod.use_axis[1], act_mod.use_axis[2])]

        text_bisect_axis = {
            (True, True, True): "X, Y, Z",
            (True, True, False): "X, Y",
            (True, False, True): "X, Z",
            (False, True, True): "Y, Z",
            (True, False, False): "X",
            (False, True, False): "Y",
            (False, False, True): "Z",
            (False, False, False): "No axis"
        }[(act_mod.use_bisect_axis[0], act_mod.use_bisect_axis[1], act_mod.use_bisect_axis[2])]

        text_bisect_flip_axis = {
            (True, True, True): "X, Y, Z",
            (True, True, False): "X, Y",
            (True, False, True): "X, Z",
            (False, True, True): "Y, Z",
            (True, False, False): "X",
            (False, True, False): "Y",
            (False, False, True): "Z",
            (False, False, False): "No axis"
        }[(act_mod.use_bisect_flip_axis[0], act_mod.use_bisect_flip_axis[1], act_mod.use_bisect_flip_axis[2])]

        not_picking = not self.pick_actions
        # attrib name,                 Title               Key                Display condition,               override value
        if self.prefs.beginner_mode:
            actions = (
                ('free',                    "Free Navigation",  None,               False),
                ('name',                    "Name",             None),
                ('mirror_axis',             "Mirror Axis",      "X,Y,Z",            True,                   text_axis),
            )
        else:
            actions = (
                ('free',                    "Free Navigation",  None,               False),
                ('name',                    "Name",             None),
                ('mirror_axis',             "Mirror Axis",      "X,Y,Z",            True,                   text_axis),
                ('use_clip',                None,               "S"),
                ('use_mirror_merge',        None,               "D"),
                ('merge_threshold',         None,               "F",                True  and act_mod.use_mirror_merge),
                ('bisect_axis',             "Bisect Axis",      "SHIFT + X,Y,Z",    True,                   text_bisect_axis),
                ('bisect_flip_axis',        "Bisect Flip Axis", "CTRL + X,Y,Z",     True,                   text_bisect_flip_axis),
                ('use_mirror_vertex_groups',None,               "C"),
                ('mirror_object',           None,               "O",                True),
                ('use_mirror_u',            "Flip U",           "U"),
                ('mirror_offset_u',         None,               "SHIFT + U",         True and act_mod.use_mirror_u),
                ('use_mirror_v',            "Flip V",           "V"),
                ('mirror_offset_v',         None,               "SHIFT + V",         True and act_mod.use_mirror_v),

            )
    return subtitle, actions

# SCREW
def screw_actions(self, context):

    # Display options for actions:
    # 1 use "red" state for current action in action list under line
    # 2 draw action beside title and draw mod name under line

    act_mod = self.act_mod
    act_obj = self.act_obj
    # show title in red
    show_in_red = not (act_mod.show_viewport and act_mod.show_render)
    # subsurf = self.has_modifier_of_type(self.act_obj, 'SUBSURF')

    resolution_u = 0
    is_curve = act_obj.type == 'CURVE'
    if is_curve:
        resolution_u = act_obj.data.resolution_u
    #        Subtitle,      title in red, show action label and value instead of subtitle
    # 1 Use only red state for active action
    subtitle = (act_mod.name,  show_in_red)
    # 2 Use action name + value as subtitle
    subtitle = ("",            show_in_red, True)

    if self.prefs.beginner_mode:
        actions = (
            # attrib name,   Title              Key    Display condition,               override value bpy.context.object.modifiers["Screw"].use_merge_vertices = True
            ('free',                    "Free Navigation", None,   False),
            ('name',                    "Name",             None),
            ('axis',                    None,               "X, Y or Z"),
            ('steps',                   None,               "S"),
            ('screw_offset',            None,               "D"),
            ('iterations',              None,               "F"),
            ('angle',                   None,               "G",   True, degrees(act_mod.angle)),
            ('resolution_u',            "Resolution U",     "U", is_curve, resolution_u),
        )
    else:
        actions = (
            # attrib name,   Title              Key    Display condition,               override value bpy.context.object.modifiers["Screw"].use_merge_vertices = True
            ('free',                    "Free Navigation", None,   False),
            ('name',                    "Name",             None),
            ('axis',                    None,               "X, Y or Z"),
            ('steps',                   None,               "S"),
            ('screw_offset',            None,               "D"),
            ('iterations',              None,               "F"),
            ('angle',                   None,               "G",   True, degrees(act_mod.angle)),
            ('use_normal_calculate',    'Calc Order',       'C'),
            ('use_merge_vertices',      None,               'M'),
            ('merge_threshold',         None,               "J"),
            ('resolution_u',            "Resolution U",     "U", is_curve, resolution_u),
            ('object',                  "Ref Object",       "O",    True),

        )

    return subtitle, actions

# SHRINKWRAP
def shrinkwrap_actions(self, context):
    act_mod = self.act_mod
    show_in_red = not (act_mod.show_viewport and act_mod.show_render and act_mod.target)
    subtitle = ("", show_in_red, True)

    if act_mod.wrap_method == 'PROJECT':
        if self.prefs.beginner_mode:
            actions = (
                ('free',                    "Free Navigation",      None, False),
                ('name',                    "Name",                 None),
                ('target',                  "Target",               "O", True),
                ('offset',                  "Offset",               "S"),
                ('wrap_method',             "Wrap Method",          "D"),
                ('wrap_mode',               "Wrap Mode",            "F"),
                ('use_positive_direction',  "Positive",             "L"),
                ('use_negative_direction',  "Negative",             "M"),
            )
        else:
            actions = (
                ('free',                    "Free Navigation",      None, False),
                ('name',                    "Name",                 None),
                ('target',                  "Target",               "O", True),
                ('auxiliary_target',        "Auxiliary Target",     "P", True),
                ('offset',                  "Offset",               "S"),
                ('wrap_method',             "Wrap Method",          "D"),
                ('wrap_mode',               "Wrap Mode",            "F"),
                ('subsurf_levels',          "Subsurf Levels",       "G"),
                ('project_limit',           "Project Limit",        "J"),
                ('use_project_x',           "X",                    "X"),
                ('use_project_y',           "Y",                    "Y"),
                ('use_project_z',           "Z",                    "Z"),
                ('use_positive_direction',  "Positive",             "L"),
                ('use_negative_direction',  "Negative",             "M"),
                ('cull_face',               "Cull Faces",           "K"),
                ('use_invert_cull',         "Invert Cull",          "I"),
                ('vertex_group',            "Vertex Group",         "C"),
            )
    elif act_mod.wrap_method == 'NEAREST_VERTEX':
        if self.prefs.beginner_mode:
            actions = (
                ('free',                    "Free Navigation",      None, False),
                ('name',                    "Name",                 None),
                ('target',                  "Target",               "O", True),
                ('offset',                  "Offset",               "S"),
                ('wrap_method',             "Wrap Method",          "D"),
            )
        else:
            actions = (
                ('free',                    "Free Navigation",      None, False),
                ('name',                    "Name",                 None),
                ('target',                  "Target",               "O", True),
                ('offset',                  "Offset",               "S"),
                ('wrap_method',             "Wrap Method",          "D"),
                ('vertex_group',            "Vertex Group",         "C"),
            )
    else:
        if self.prefs.beginner_mode:
            actions = (
                ('free',                    "Free Navigation",      None, False),
                ('name',                    "Name",                 None),
                ('target',                  "Target",               "O", True),
                ('offset',                  "Offset",               "S"),
                ('wrap_method',             "Wrap Method",          "D"),
                ('wrap_mode',               "Wrap Mode",            "F"),

            )
        else:
            actions = (
                ('free',                    "Free Navigation",      None, False),
                ('name',                    "Name",                 None),
                ('target',                  "Target",               "O", True),
                ('offset',                  "Offset",               "S"),
                ('wrap_method',             "Wrap Method",          "D"),
                ('wrap_mode',               "Wrap Mode",            "F"),
                ('vertex_group',            "Vertex Group",         "C"),

            )

    return subtitle, actions

# SIMPLE DEFORM
def simple_deform_actions(self, context):

    act_mod = self.act_mod
    show_in_red = not (act_mod.show_viewport and act_mod.show_render)
    subtitle = ("", show_in_red, True)

    text_lock_axis = {
        (True, True, True): "X, Y, Z",
        (True, True, False): "X, Y",
        (True, False, True): "X, Z",
        (False, True, True): "Y, Z",
        (True, False, False): "X",
        (False, True, False): "Y",
        (False, False, True): "Z",
        (False, False, False): "No axis"
    }[(act_mod.lock_x, act_mod.lock_y, act_mod.lock_z)]

    if act_mod.deform_method in {'TAPER', 'STRETCH'}:
        if self.prefs.beginner_mode:
            actions = (
                ('free',                    "Free Navigation",      None,               False),
                ('name',                    "Name",                 None),
                ('factor',                  "Factor",               "D",             True,   degrees(act_mod.factor)),
                ('deform_method',           "Deform Method",        "F"),
                ('origin',                  "Origin",               "O", True),
                ('deform_axis',             "Deform Axis: (X,Y,Z)", "X"),
            )
        else:
            actions = (
                ('free',                    "Free Navigation",      None,               False),
                ('name',                    "Name",                 None),
                ('factor',                  "Factor",               "D",             True,   degrees(act_mod.factor)),
                ('deform_method',           "Deform Method",        "F"),
                ('origin',                  "Origin",               "O", True),
                ('lock_axis',               "Lock Axis",            "CTRL + X,Y,Z",  True, text_lock_axis),
                ('deform_axis',             "Deform Axis: (X,Y,Z)", "X"),
                ('vertex_group',            "Vertex Group",         "C"),
            )

    else:
        if self.prefs.beginner_mode:
            actions = (
                ('free',                    "Free Navigation",      None,                   False),
                ('name',                    "Name",                 None),
                ('angle',                   "Angle",                "S",            True,   degrees(act_mod.angle)),
                ('deform_method',           "Deform Method",        "F"),
                ('origin',                  "Origin",               "O", True),
                ('deform_axis',             "Deform Axis: (X,Y,Z)", "X"),
            )
        else:
            actions = (
                ('free',                    "Free Navigation",      None,                   False),
                ('name',                    "Name",                 None),
                ('angle',                   "Angle",                "S",            True,   degrees(act_mod.angle)),
                ('deform_method',           "Deform Method",        "F"),
                ('origin',                  "Origin",               "O", True),
                ('deform_axis',             "Deform Axis: (X,Y,Z)", "X"),
                ('lock_axis',               "Lock Axis",            "CTRL + X,Y,Z", True,   text_lock_axis),
                ('vertex_group',            "Vertex Group",          "C"),
            )

    return subtitle, actions

# SKIN
def skin_actions(self, context):
    act_mod = self.act_mod
    show_in_red = not (act_mod.show_viewport and act_mod.show_render)
    subtitle = ("", show_in_red, True)

    text_axis = {
        (True, True, True): "X, Y, Z",
        (True, True, False): "X, Y",
        (True, False, True): "X, Z",
        (False, True, True): "Y, Z",
        (True, False, False): "X",
        (False, True, False): "Y",
        (False, False, True): "Z",
        (False, False, False): "No axis"
    }[(act_mod.use_x_symmetry, act_mod.use_y_symmetry, act_mod.use_z_symmetry)]

    if self.prefs.beginner_mode:
        actions = (
            ('free',                    "Free Navigation",  None, False),
            ('name',                    "Name",             None),
            ('branch_smoothing',        None,               "S"),
            ('use_smooth_shade',        None,               "D"),
            ('symmetry_axis',           "Symmetry Axis",    "X,Y,Z",     True,    text_axis),
        )
    else:
        actions = (
            ('free',                    "Free Navigation",  None, False),
            ('name',                    "Name",             None),
            ('branch_smoothing',        None,               "S"),
            ('use_smooth_shade',        None,               "D"),
            ('symmetry_axis',           "Symmetry Axis",    "X,Y,Z",     True,    text_axis),
        )
    return subtitle, actions

# SOLIDIFY
def solidify_actions(self, context):

    act_mod = self.act_mod
    show_in_red = not (act_mod.show_viewport and act_mod.show_render)
    subtitle = ("", show_in_red, True)

    if self.prefs.beginner_mode:
        actions = (
            ('free', "Free Navigation", None, False),
            ('name', "Name", None),
            ('thickness', None, "S"),
            ('offset', None, "D"),
            ('use_rim_only', None, "F",),
        )
    else:
        actions = (
            ('free',                    "Free Navigation",  None, False),
            ('name',                    "Name",             None),
            ('thickness',               None,               "S"),
            ('offset',                  None,               "D"),
            ('use_rim_only',            None,               "F",),
            ('use_rim',                 None,               "G",),
            ('use_even_offset',         None,               "J",),
            ('use_flip_normals',        None,               "R",),
            ('vertex_group',            None,               "C",),
            ('thickness_vertex_group',  None,               "X", act_mod.vertex_group != "")
        )
    return subtitle, actions

# SUBSURF
def subsurf_actions(self, context):
    act_mod = self.act_mod
    show_in_red = not (act_mod.show_viewport and act_mod.show_render)
    subtitle = ("", show_in_red, True)

    if self.prefs.beginner_mode:
        actions = (
            ('free',                    "Free Navigation",  None, False),
            ('name',                    "Name",             None),
            ('levels',                  None,               "S"),
            ('subdivision_type',        None,               "F"),
        )
    else:
        actions = (
            ('free',                    "Free Navigation",  None, False),
            ('name',                    "Name",             None),
            ('levels',                  None,               "S"),
            ('render_levels',           None,               "D"),
            ('subdivision_type',        None,               "F"),
            ('show_only_control_edges', None,               "G"),
            ('quality',                 None,               "J"),
            ('uv_smooth',               None,               "K"),
        )
    return subtitle, actions

# TRIANGULATE
def triangulate_actions(self, context):

    act_mod = self.act_mod
    show_in_red = not (act_mod.show_viewport and act_mod.show_render)
    subtitle = ("", show_in_red, True)

    actions = (
        ('free',                    "Free Navigation",  None, False),
        ('name',                    "Name",             None),
        ('min_vertices',             None,               "S"),
        ('keep_custom_normals',      None,               "D"),
        ('quad_method',              None,               "F"),
        ('ngon_method',              None,               "G"),

    )
    return subtitle, actions

# TUBIFY
def tubify_actions(self, context):

    show_in_red = not (self.act_obj.data.bevel_depth and self.act_obj.data.extrude)
    subtitle = ("", show_in_red, True)
    d = self.act_obj.data

    # attrib name,                  Title                   Key             Display condition,      override value
    if self.prefs.beginner_mode:
        actions = (
            ('free',                    "Free Navigation",      None, False),
            ('name',                    "Name",                 None),
            ('bevel_depth',             "Bevel Depth",          "S",            True,                   d.bevel_depth),
            ('resolution_u',            "Resolution U",         "D",            True,                   d.resolution_u ),
            ('bevel_resolution',        "Resolution V",         "F",            True,                   d.bevel_resolution),
            ('extrude',                 "Extrude",              "G",            True,                   d.extrude),
        )
    else:
        actions = (
            ('free',                    "Free Navigation",      None, False),
            ('name',                    "Name",                 None),
            ('bevel_depth',             "Bevel Depth",          "S",            True,                   d.bevel_depth),
            ('resolution_u',            "Resolution U",         "D",            True,                   d.resolution_u ),
            ('bevel_resolution',        "Resolution V",         "F",            True,                   d.bevel_resolution),
            ('extrude',                 "Extrude",              "G",            True,                   d.extrude),
            ('offset',                  "Offset",               "J",            True,                   d.offset),
            ('fill_mode',               "Fill Mode",            "K",            True,                   d.fill_mode),
            ('bevel_object',            "Pick Bevel Object",         "O",            True),
            ('taper_object',            "Pick Taper Object",         "P",            True),
            ('bevel_factor_start',      "Start",                "C",            True,                   d.bevel_factor_start),
            ('bevel_factor_end',        "End",                  "V",            True,                   d.bevel_factor_end),
        )
    return subtitle, actions

# VISIBILITY
def mod_visibility_actions(self, context):
    act_mod = self.act_mod
    # show title in red
    show_in_red = not (act_mod.show_viewport and act_mod.show_render)
    subtitle = ("", show_in_red, False)

    actions = (
        # ('free',                "Free Navigation",          None, False),
        # ('name',                "Name",                     None),

        ('hide_previous_modifier',  "Hide Previous Modifiers",      "S"),
        ('show_next_modifier',      "Show Next Modifiers",          "D"),
        ('hide_all_modifiers',      "Hide all Modifiers",           "F"),
        ('show_all_modifiers',      "Show all Modifiers",           "G"),
        ('show_viewport',           "Show/Hide all Modifiers",      "H"),
        ('apply_all',               "Apply all Modifiers",          "A"),
        ('remove_all',              "Remove all Modifiers",         "DEL"),
    )
    return subtitle, actions

# WEIGHTED_NORMAL
def weighted_normal_actions(self, context):
    act_mod = self.act_mod
    # show title in red
    show_in_red = not (act_mod.show_viewport and act_mod.show_render)
    subtitle = ("", show_in_red, True)
    actions = (
        ('free',            "Free Navigation",      None, False),
        ('name',            "Name",                 None),
        ('weight',          None,                   "S"),
        ('mode',            None,                   "D"),
        ('thresh',          None,                    "F"),
        ('keep_sharp',      None,                   "G"),
        ('face_influence',  None,                   "J"),
    )
    return subtitle, actions

# WIREFRAME
def wireframe_actions(self, context):
    act_mod = self.act_mod
    show_in_red = not (act_mod.show_viewport and act_mod.show_render)
    subtitle = ("", show_in_red, True)

    if self.prefs.beginner_mode:
        actions = (
            ('free',                    "Free Navigation",  None, False),
            ('name',                    "Name",             None),
            ('thickness',               None,               "S"),
            ('offset',                  None,               "D"),
        )
    else:
        actions = (
            ('free',                    "Free Navigation",  None, False),
            ('name',                    "Name",             None),
            ('thickness',               None,               "S"),
            ('offset',                  None,               "D"),
            ('use_even_offset',         None,               "F"),
            ('use_relative_offset',     None,               "G"),
            ('use_boundary',            None,               "J"),
            ('use_crease',              None,               "K"),
            ('use_replace',             None,               "L"),
            ('vertex_group',            "Vertex Group",     "C", ),
            ('thickness_vertex_group',  None,               "X", act_mod.vertex_group != "")
        )
    return subtitle, actions


# ------------------------------------------------------------------------------------
# Keymaps
# ------------------------------------------------------------------------------------

def array_keymap(self, km):
    if not self.prefs.display_texts:
        km.extend([("6", "Show texts")])
    else:
        if self.prefs.beginner_mode:
            km.extend([
                    ("SHIFT + H", "Normal Mode", self.prefs.beginner_mode),
                    ("W", "Reset Axis"),
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal)
                    ])
            km.extend(KEYMAP_END_BEGINNER)
        else:
            km.extend([
                    ("C", "Circular Array"),
                    ("W", "Reset Axis"),
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal)
                    ])
            km.extend(KEYMAP_END)


def bevel_keymap(self, km):
    if not self.prefs.display_texts:
        km.extend([("6", "Show texts")])
    else:
        if self.prefs.beginner_mode:
            km.extend([
                    ("SHIFT + H", "Normal Mode", self.prefs.beginner_mode),
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                    ])
            km.extend(KEYMAP_END_BEGINNER)
        else:
            km.extend([
                    ("Q", "Auto update Bevel", self.auto_update),
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                    ])
            km.extend(KEYMAP_END)


def boolean_keymap(self, km):
    if not self.prefs.display_texts:
        km.extend([("6", "Show texts")])
    else:
        if self.prefs.beginner_mode:
            km.extend([
                    ("SHIFT + H", "Normal Mode", self.prefs.beginner_mode),
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                    ])
            km.extend(KEYMAP_END_BEGINNER)

        else:
            km.extend([
                    #("CTRL + SHIFT + A", "Apply Modifier with inner Bevel", None, self.prefs.destructive_mode),
                    ("I", "Inset"),
                    # ("R", "Reverse Modifier"),
                    # ("W", "Projection Cut"),
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                    ])
            km.extend(KEYMAP_END)

def corrective_smooth_keymap(self, km):
    if not self.prefs.display_texts:
        km.extend([("6", "Show texts")])
    else:
        if self.prefs.beginner_mode:
            km.extend([
                    ("SHIFT + H", "Normal Mode", self.prefs.beginner_mode),
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                    ])
            km.extend(KEYMAP_END_BEGINNER)
        else:
            km.extend([
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                    ])
            km.extend(KEYMAP_END)

def curve_keymap(self, km):
    if not self.prefs.display_texts:
        km.extend([("6", "Show texts")])
    else:
        if self.prefs.beginner_mode:
            km.extend([
                    ("SHIFT + H", "Normal Mode", self.prefs.beginner_mode),
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                    ])
            km.extend(KEYMAP_END_BEGINNER)
        else:
            km.extend([
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                    ])
            km.extend(KEYMAP_END)

def cutter_keymap(self, km):

    km.extend([
            ("CLICK", "Rectangle"),
            ("SHIFT", "Cylinder"),
            ("CTRL", "Line"),
            ("ALT", "On Cursor"),
            ("SPACE", "APPLY"),
            ("G", "MOVE VERTEX"),
            ("SHIFT + SPACE", "MOVE OBJECT"),
            ("ESC", "Restart Primitive"),
            ("RMB", "Exit"),
            ])

def decimate_keymap(self, km):
    if not self.prefs.display_texts:
        km.extend([("6", "Show texts")])
    else:
        if self.prefs.beginner_mode:
            km.extend([
                    ("SHIFT + H", "Normal Mode", self.prefs.beginner_mode),
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                    ])
            km.extend(KEYMAP_END_BEGINNER)
        else:
            km.extend([
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                    ])
            km.extend(KEYMAP_END)

def displace_keymap(self, km):
    if not self.prefs.display_texts:
        km.extend([("6", "Show texts")])
    else:
        if self.prefs.beginner_mode:
            km.extend([
                    ("SHIFT + H", "Normal Mode", self.prefs.beginner_mode),
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                    ])
            km.extend(KEYMAP_END_BEGINNER)
        else:
            km.extend([
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                    ])
            km.extend(KEYMAP_END)

def edge_split_keymap(self, km):
    if not self.prefs.display_texts:
        km.extend([("6", "Show texts")])
    else:
        if self.prefs.beginner_mode:
            km.extend([
                    ("SHIFT + H", "Normal Mode", self.prefs.beginner_mode),
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                    ])
            km.extend(KEYMAP_END_BEGINNER)
        else:
            km.extend([
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                    ])
            km.extend(KEYMAP_END)

def mirror_keymap(self, km):
    act_mod = self.act_mod
    if not self.prefs.display_texts:
        km.extend([("6", "Show texts")])
    else:
        if self.prefs.beginner_mode:
            km.extend([
                    ("SHIFT + H", "Normal Mode", self.prefs.beginner_mode),
                    ("O", "Pick Ref Object", act_mod.mirror_object),
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                    ])
            km.extend(KEYMAP_END_BEGINNER)
        else:
            km.extend([
                    ("O", "Pick Ref Object", act_mod.mirror_object),
                    ("W", "Create/Move Ref Object", act_mod.mirror_object),
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                    ])
            km.extend(KEYMAP_END)


def rotate_keymap(self, km):
    km.extend((# key, label,                         value or None,                  enable/disable display
        ("F", "Place Cursor 3D"),
        ("ESC", "Exit")
    ))
    if self.prefs.beginner_mode:
        km.extend(KEYMAP_END_BEGINNER)
    else:
        km.extend(KEYMAP_END)


def screw_keymap(self, km):
    act_mod = self.act_mod
    if not self.prefs.display_texts:
        km.extend([("6", "Show texts")])
    else:
        if self.prefs.beginner_mode:
            km.extend([
                    # key, label,                         value or None,                  enable/disable display
                    ("SHIFT + H", "Normal Mode", self.prefs.beginner_mode),
                    ("E", "Smooth Shading", act_mod.use_smooth_shade),
                    ("R", "Flip Normals", act_mod.use_normal_flip),
                    ("M", "Merge Vertices", act_mod.use_merge_vertices),
                    ("W", "Create/Move Ref Object", act_mod.object),
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                    ])
            km.extend(KEYMAP_END_BEGINNER)
        else:
            km.extend([
                    # key, label,                         value or None,                  enable/disable display
                    ("E", "Smooth Shading", act_mod.use_smooth_shade),
                    ("R", "Flip Normals", act_mod.use_normal_flip),
                    ("M", "Merge Vertices", act_mod.use_merge_vertices),
                    ("W", "Create/Move Ref Object", act_mod.object),
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                    ])
            km.extend(KEYMAP_END)

def shrinkwrap_keymap(self, km):
    if not self.prefs.display_texts:
        km.extend([("6", "Show texts")])
    else:
        if self.prefs.beginner_mode:
            km.extend([
                    ("SHIFT + H", "Normal Mode", self.prefs.beginner_mode),
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                    ])
            km.extend(KEYMAP_END_BEGINNER)
        else:
            km.extend([
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                    ])
            km.extend(KEYMAP_END)

def simple_deform_keymap(self, km):
    act_mod = self.act_mod
    if not self.prefs.display_texts:
        km.extend([("6", "Show texts")])
    else:
        if self.prefs.beginner_mode:
            km.extend([
                    ("SHIFT + H", "Normal Mode", self.prefs.beginner_mode),
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                    ])
            km.extend(KEYMAP_END_BEGINNER)
        else:
            km.extend([
                    ("W", "Create/Move Ref Object", act_mod.origin),
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                    ])
            km.extend(KEYMAP_END)

def skin_keymap(self, km):
    if not self.prefs.display_texts:
        km.extend([("6", "Show texts")])
    else:
        if self.prefs.beginner_mode:
            km.extend([
                    ("SHIFT + H", "Normal Mode", self.prefs.beginner_mode),
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                    ])
            km.extend(KEYMAP_END_BEGINNER)
        else:
            km.extend([
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                    ])
            km.extend(KEYMAP_END)

def solidify_keymap(self, km):
    if not self.prefs.display_texts:
        km.extend([("6", "Show texts")])
    else:
        if self.prefs.beginner_mode:
            km.extend([
                    ("SHIFT + H", "Normal Mode", self.prefs.beginner_mode),
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                    ])
            km.extend(KEYMAP_END_BEGINNER)
        else:
            km.extend([
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                    ])
            km.extend(KEYMAP_END)


def subsurf_keymap(self, km):
    if not self.prefs.display_texts:
        km.extend([("6", "Show texts")])
    else:
        if self.prefs.beginner_mode:
            km.extend([
                    ("SHIFT + H", "Normal Mode", self.prefs.beginner_mode),
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                    ])
            km.extend(KEYMAP_END_BEGINNER)
        else:
            km.extend([
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                    ])
            km.extend(KEYMAP_END)



def triangulate_keymap(self, km):
    if not self.prefs.display_texts:
        km.extend([("6", "Show texts")])
    else:
        if self.prefs.beginner_mode:
            km.extend([
                    ("SHIFT + H", "Normal Mode", self.prefs.beginner_mode),
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                    ])
            km.extend(KEYMAP_END_BEGINNER)
        else:
            km.extend([
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                    ])
            km.extend(KEYMAP_END)

def tubify_keymap(self, km):
    if not self.prefs.display_texts:
        km.extend([("6", "Show texts")])
    else:
        if self.prefs.beginner_mode:
            km.extend([
                ("SHIFT + H", "Normal Mode", self.prefs.beginner_mode),
                ("1", "Keep Wire Visible", self.prefs.keep_wire),
                ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                ])
            km.extend(KEYMAP_END_BEGINNER)
        else:
            km.extend([
                # key, label,                         value or None,                  enable/disable display
                ("A", "Convert to Poly"),
                ("Q", "Cyclic", self.act_obj.data.splines[0].use_cyclic_u),
                ("Z", "Smooth", self.act_obj.data.splines[0].use_smooth),
                ("R", "Switch Direction"),
                ("L", "Add Square Profile"),
                ("W", "Fill Cap", self.act_obj.data.use_fill_caps ),
                ("T","Toggle Spline Poly/Bezier", self.act_obj.data.splines[0].type),
                ("1", "Keep Wire Visible", self.prefs.keep_wire),
                ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                ])
            km.extend(KEYMAP_END)

def visibility_keymap(self, km):
    if not self.prefs.display_texts:
        km.extend([("6", "Show texts")])
    else:
        if self.prefs.beginner_mode:
            km.extend([
                    ("SHIFT + H", "Normal Mode", self.prefs.beginner_mode),
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                    ])
            km.extend(KEYMAP_END_BEGINNER)
        else:
            km.extend([
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                    ])
            km.extend(KEYMAP_END)

def weighted_normal_keymap(self, km):
    if not self.prefs.display_texts:
        km.extend([("6", "Show texts")])
    else:
        if self.prefs.beginner_mode:
            km.extend([
                    ("SHIFT + H", "Normal Mode", self.prefs.beginner_mode),
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                    ])
            km.extend(KEYMAP_END_BEGINNER)
        else:
            km.extend([
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                    ])
            km.extend(KEYMAP_END)

def wireframe_keymap(self, km):
    if not self.prefs.display_texts:
        km.extend([("6", "Show texts")])
    else:
        if self.prefs.beginner_mode:
            km.extend([
                    ("SHIFT + H", "Normal Mode", self.prefs.beginner_mode),
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                    ])
            km.extend(KEYMAP_END_BEGINNER)
        else:
            km.extend([
                    ("1", "Keep Wire Visible", self.prefs.keep_wire),
                    ("2", "Show Wire In the Modal", self.prefs.show_wire_in_modal),
                    ])
            km.extend(KEYMAP_END)



def enable_shadow(self):
    prefs = self.prefs
    text_shadow = prefs.text_shadow
    shadow_color = prefs.shadow_color
    shadow_x = prefs.offset_shadow_x
    shadow_y = prefs.offset_shadow_y
    if text_shadow:
        blf.enable(0, blf.SHADOW)
        blf.shadow_offset(0, shadow_x, shadow_y)
        blf.shadow(0, 3, shadow_color[0], shadow_color[1], shadow_color[2], shadow_color[3])
    return text_shadow


def draw_text(text, size, color, x, y, font_id):
    blf.size(font_id, size, 72)
    blf.color(0, *color)
    blf.position(font_id, x, y, 0)
    blf.draw(font_id, text)


def value_as_string(value, color):
    """Handle value conversion to string
    Support:
     - float (rounding to 2) - no angle conversion
     - integer
     - boolean as ON/OFF
     - Blender Objects -> object.name
     - string
    :param value: any
    :param color: desired color
    :return: string value, changed color for bool
    """
    _value = value 
    _color = color
    _fmt = "%s"
    _type = type(value).__name__

    if _type == 'float':
        # 2 after .
        _fmt = "%.3f"

    elif _type == 'bool':
        _color = (RED, GREEN)[int(value)]
        _value = ('OFF','ON')[int(value)]
    
    elif _type == 'Object' and hasattr(value, 'name'):
        # retrieve name of blender object
        _value = getattr(value, 'name')

    return _fmt % _value, _color


def draw_actions(self, context, title_data, actions):
    prefs = self.prefs
    font_id = 0

    line_height = (blf.dimensions(font_id, "M")[1] * 0.7)
    overlap = context.preferences.system.use_region_overlap
    t_panel_width = 0
    if overlap:
        for region in bpy.context.area.regions:
            if region.type == 'TOOLS':
                t_panel_width = region.width

    width = context.region.width
    height = context.region.height
    span, h1, h2, h3, h4 = self.get_font_size()
    text_color, color_1, color_2 = self.get_font_colors()
    line_color = text_color
    text_pos_x = prefs.text_pos_x
    text_pos_y = prefs.text_pos_y


    # Retrieve title from modal class name
    title = self.mod_type.upper()

    # subtitle from definition
    subtitle = title_data[0]
    show_in_red = len(title_data) > 1 and title_data[1]
    # display action + value as subtitle
    show_value = (len(title_data) > 2 and title_data[2]) or len(title_data)>3
    # print(title_data)
    # compute columns width for label and shortcut
    # Label      (key)|Value
    # Key are aligned on left of this column, values on right
    active_value = None
    blf.size(font_id, span, 72)
    parenthesis_x, parenthesis_y = blf.dimensions(font_id, "(")
    shortcut_col = 0
    values_col = 0

    if len(actions) < 1:
        return

    for enable, active, label, shortcut, value in actions:
        if enable:
            blf.size(font_id, span, 72)
            if shortcut is None:
                text_width, text_height = blf.dimensions(font_id, "{} :".format(label))
            else:
                text_width, text_height = blf.dimensions(font_id, "{} {} :".format(label, shortcut))
                text_width += 2 * parenthesis_x
            if text_width > shortcut_col:
                shortcut_col = text_width

            # compute max text width for values
            if value is not None:
                fmt_value, value_color = value_as_string(value, color_2)
                blf.size(font_id, h4, 72)
                text_width , text_height = blf.dimensions(font_id, fmt_value)
                if text_width > values_col:
                    values_col = text_width

                # store active_value for title line
                if active and show_value:
                    active_value = " : %s" % fmt_value



        # use action label as subtitle
        if active and show_value:

            if value is not None:
                fmt_value, value_color = value_as_string(value, color_2)
                active_value = " : %s" % fmt_value

            # TODO: detect pick actions, maybe using pick_actions to add "Pick xxx"
            subtitle = label

    # compute title size
    blf.size(font_id, h2, 72)
    title_width, title_height = blf.dimensions(font_id, "%s :" % title)

    # compute subtitle size
    blf.size(font_id, h2, 72)
    sub_width, sub_height = blf.dimensions(font_id, subtitle)

    # compute active value in title size
    act_width = 0
    if active_value is not None:
        blf.size(font_id, h1, 72)
        act_width, act_height = blf.dimensions(font_id, active_value)

    text_height = (line_height + span) * len(actions) + 0.75 * title_height + span + 20

    # TODO: ensure the panel does show on screen : check bottom right position
    x = min((width - t_panel_width - 250) * text_pos_x / 100 + t_panel_width, width - 250)
    y = min(((height - text_height) * text_pos_y / 100 + text_height), height - 20)

    text_shadow = enable_shadow(self)

    # Draw Title
    if show_in_red:
        draw_text(title, h2, RED, x, y, font_id)

    else:
        draw_text(title, h2, color_2, x, y, font_id)

    y -= 0.75 * title_height
    y -= 20

    # Draw subtitle
    draw_text(subtitle, h2, text_color, x, y, font_id)
    # draw_text(subtitle, h2, text_color, x + title_width, y, font_id)

    # Draw active action and value
    if active_value is not None:
        draw_text(active_value, h1, color_2, x + sub_width, y, font_id)

    y -= 0.75 * title_height
    y -= 20

    x_col = x + shortcut_col

    for enable, active, label, shortcut, value in actions:
        if enable:
            # Active state for label of current action
            if active:
                color_0 = RED
            else:
                color_0 = text_color

            # Label
            draw_text(label, span, color_0, x, y, font_id)

            # Shortcut
            if shortcut is not None:
                short_text = " {} ".format(shortcut)
                blf.size(font_id, span, 72)
                text_width, text_height = blf.dimensions(font_id, short_text)

                draw_text("(",      span, color_0, (x_col - text_width - 2 * parenthesis_x), y, font_id)
                draw_text(short_text,    span, color_1, (x_col - text_width - parenthesis_x), y, font_id)
                draw_text(")",      span, color_0, (x_col - parenthesis_x), y, font_id)

            if value is not None:
                # Color and string conversion
                fmt_value, color = value_as_string(value, color_2)
                   
                # draw input values as red
                if active and self.input:
                    color = RED

                # dont draw empty string value
                if value != "":
                    # Semicolon between shortcut and value
                    draw_text(":", span, color_0, (x_col + parenthesis_x), y, font_id)
                    draw_text(fmt_value, h4, color, (x_col + 3 * parenthesis_x), y, font_id)

            y -= line_height + span

    if text_shadow:
        blf.disable(0, blf.SHADOW)


def draw_keymap(self, context, keymaps):
    prefs = self.prefs
    font_id = 0

    overlap = context.preferences.system.use_region_overlap
    t_panel_width = 0
    if overlap:
        for region in bpy.context.area.regions:
            if region.type == 'TOOLS':
                t_panel_width = region.width

    n_panel_width = 0
    if overlap:
        for region in bpy.context.area.regions:
            if region.type == 'UI':
                n_panel_width = region.width

    width = context.region.width
    height = context.region.height
    text_color, color_1, color_2 = self.get_font_colors()
    keys_text_pos_x = prefs.keys_text_pos_x
    keys_text_pos_y = prefs.keys_text_pos_y

    keys_size = prefs.keys_size
    line_count = len(keymaps)

    # line_count_1 = keys_text_names.count("Carriage return")
    blf.size(font_id, keys_size, 72)
    line_height = (blf.dimensions(font_id, "M")[1] * 2)

    # Position X/Y
    # x = keys_text_pos_x + t_panel_width
    y = line_height * line_count + keys_text_pos_y


    x = min((width - t_panel_width - 250) * keys_text_pos_x / 100 + t_panel_width, width - 250)


    # x = 80 + t_panel_width
    # y = line_height * line_count + 70


    # compute keys column width for alignment
    x_col = 0

    for line in keymaps:
        key, label = line[0:2]
        enabled = len(line) < 4 or line[3]
        if enabled:
            text_width, text_height = blf.dimensions(font_id, key)
            if text_width > x_col:
                x_col = text_width

    x_col += x

    # draw
    text_shadow = enable_shadow(self)

    for line in keymaps:
        key, label = line[0:2]
        value = None
        if len(line) > 2:
            value = line[2]
        enabled = len(line) < 4 or line[3]
        if enabled:
            # shortcut  : label
            draw_text(key, keys_size, color_1, x, y, font_id)
            draw_text(" : %s" % label, keys_size, text_color, x_col, y, font_id)
            if value is not None:
                fmt_value, color = value_as_string(value, color_2)
                # dont draw empty string value
                if fmt_value != "":
                    text_width, text_height = blf.dimensions(font_id, ":  %s" % label)
                    draw_text(" : %s" % fmt_value, keys_size, color, x_col + text_width, y, font_id)

            y -= line_height

    if text_shadow:
        blf.disable(0, blf.SHADOW)

#Array
def get_axis_offset(self, mod):
    attr = ('constant_offset_displace', 'relative_offset_displace')[int(self.relative_offset_enabled)]
    values = getattr(mod, attr)[0:3]
    mini = min(values)
    maxi = max(values)
    if abs(mini) > maxi:
        index = values.index(mini)
    else:
        index = values.index(maxi)
    return index


# TODO: remove following when implementation of other is done


def draw_text_array_mpm(self, context, text, key_text):
    # def draw_text_array_mpm(self, text, key_text, keys_text_names):
    addon_pref = self.prefs
    font_id = 0
    x_offset = 0
    y_offset = 0

    line_height = (blf.dimensions(font_id, "M")[1] * 0.7)
    first_line_width = 0
    overlap = context.preferences.system.use_region_overlap
    t_panel_width = 0
    if overlap:
        for region in bpy.context.area.regions:
            if region.type == 'TOOLS':
                t_panel_width = region.width

    width = context.region.width
    height = context.region.height
    line_color = addon_pref.text_color
    span = addon_pref.text_size
    text_shadow = addon_pref.text_shadow
    shadow_color = addon_pref.shadow_color
    shadow_x = addon_pref.offset_shadow_x
    shadow_y = addon_pref.offset_shadow_y
    text_pos_x = addon_pref.text_pos_x
    text_pos_y = addon_pref.text_pos_y

    # Find the length of the first line
    for command in text:
        if len(command) == 3:
            Text, Color, Size = command
            blf.size(font_id, Size, 72)
            text_width, text_height = blf.dimensions(font_id, Text)
            first_line_width += text_width
        elif command == "line":
            break

    text_line_count = text.count("Carriage return") + 1
    text_height = (line_height + span) * text_line_count

    x = min((width - t_panel_width - 250) * text_pos_x / 100 + t_panel_width, width - 250)
    y = min(((height - text_height) * text_pos_y / 100 + text_height), height - 20)
    for command in text:
        if len(command) == 3:
            Text, Color, Size = command
            blf.color(0, *self.color)
            blf.size(font_id, Size, 72)
            text_width, text_height = blf.dimensions(font_id, Text)
            if text_shadow:
                blf.enable(0, blf.SHADOW)
                blf.shadow_offset(0, shadow_x, shadow_y)
                blf.shadow(0, 3, shadow_color[0], shadow_color[1], shadow_color[2], shadow_color[3])
            blf.position(font_id, (x + x_offset), (y + y_offset), 0)
            blf.draw(font_id, Text)
            x_offset += text_width

        else:
            x_offset = 0
            if command == "line":
                y_offset -= 20

            else:
                y_offset -= line_height + span

    if addon_pref.display_keymaps:
        keys_size = addon_pref.keys_size
        line_count = key_text.count("Carriage return")
        # line_count_1 = keys_text_names.count("Carriage return")
        x_offset = 0
        y_offset = 0
        blf.size(font_id, keys_size, 72)
        line_height = (blf.dimensions(font_id, "M")[1] * 2)
        pos_x = 80 + t_panel_width
        pos_y = line_height*line_count + 70
        # pos_y_name = line_height*line_count_1 + 70

        for command in key_text:
            if len(command) == 2:
                Text, Color = command
                blf.color(0, *self.color)
                text_width, text_height = blf.dimensions(font_id, Text)
                if text_shadow:
                    blf.enable(0, blf.SHADOW)
                    blf.shadow_offset(0, shadow_x, shadow_y)
                    blf.shadow(0, 3, shadow_color[0], shadow_color[1], shadow_color[2], shadow_color[3])
                blf.position(font_id, (pos_x + x_offset), (pos_y + y_offset), 0)
                blf.draw(font_id, Text)
                x_offset += text_width

            else:
                x_offset = 0
                y_offset -= line_height

    if text_shadow:
        blf.disable(0, blf.SHADOW)

#Rotate


def rotate_text(self, context):
    rotate_text = []
    # Use a web like notation
    span, h1, h2, h3, h4 = self.get_font_size()
    text_color, color_1, color_2 = self.get_font_colors()

    action = self.modal_action
    act_obj = context.object
    
    if action == 'free':
        rotate_text.extend([CR, ("Free navigation ", text_color, h2), "line"])

    elif action == 'cursor':
        rotate_text.extend([CR, ("Place the cursor ", text_color, h2), "line"])
    else:
        if self.axis == 0 :
            rotate_text.extend([CR, ("Rotate : ", text_color, h2), ("X ", color_1, h1), (str(round(degrees(act_obj.rotation_euler[0]),1)).split(".")[0]+ "°", color_2, h1), "line"])

        elif self.axis == 1 :
            rotate_text.extend([CR, ("Rotate : ", text_color, h2), ("Y ", color_1, h1), (str(round(degrees(act_obj.rotation_euler[1]),1)).split(".")[0] + "°", color_2, h1), "line"])

        else :
            rotate_text.extend([CR, ("Rotate : ", text_color, h2), ("Z ", color_1, h1), (str(round(degrees(act_obj.rotation_euler[2]),1)).split(".")[0]+ "°", color_2, h1), "line"])
 
    rotate_text.extend([CR, ("Axis            (", text_color, span), (" X, Y, Z ", color_1, span), (")", text_color, span)])
    rotate_text.extend([CR, ("10               (", text_color, span), (" Shift ", color_1, span), (")", text_color, span)])
    rotate_text.extend([CR, ("5                 (", text_color, span), (" Ctrl ", color_1, span), (")", text_color, span)])
    rotate_text.extend([CR, ("Orientation (", text_color, span), (" D ", color_1, span), (")   : ", text_color, span), (bpy.context.space_data.transform_orientation.lower(), color_2, h4)])
    rotate_text.extend([CR, ("Pivot           (", text_color, span), ("< >", color_1, span),(") : ", text_color, span), (bpy.context.space_data.pivot_point.lower(), color_2, h4)])

    return rotate_text

#Symmetrize
def get_axis_symmetrize(self):
    axis = []
    if self.sym_axis[0]:
        axis.append("X" if self.sym_axis[0] == 1 else "-X")
    if self.sym_axis[1]:
        axis.append("Y" if self.sym_axis[1] == 1 else "-Y")
    if self.sym_axis[2]:
        axis.append("Z" if self.sym_axis[2] == 1 else "-Z")

    return axis


def symmetrize_text(self, context):
    symmetrize_text = []
    # Use a web like notation
    span, h1, h2, h3, h4 = self.get_font_size()
    text_color, color_1, color_2 = self.get_font_colors()

    axis = get_axis_symmetrize(self) if get_axis_symmetrize(self) else ""
    if self.axis_value:
        symmetrize_text.extend([CR, ("Symmetrize ", text_color, h2), (self.axis_value.upper(), color_1, span*3), "line"])
        symmetrize_text.extend([CR, ("Click to choose side of symmetrize", text_color, span)])
    else:
        symmetrize_text.extend([CR, ("Symmetrize ", text_color, h2), (' '.join(axis), color_1, span*3), "line"])
        symmetrize_text.extend([CR, ("Select axis      ( ", text_color, span), ("X", color_1, span), (", ", text_color, span), ("Y", color_1, span), (", ", color_1, span), ("Z ", color_1, span),(")", text_color, span)])
        symmetrize_text.extend([CR, ("Add Axis          ( ", text_color, span), ("Shift", color_1, span), (" + ", text_color, span), ("axis ", color_1, span), (")", text_color, span)])
        symmetrize_text.extend([CR, ("Remove axis   ( ", text_color, span), ("Alt", color_1, span), (" + ", text_color, span), ("axis ", color_1, span), (")", text_color, span)])
 
    return symmetrize_text

# #Tubify
# def tubify_text(self, context):
#     tubify_text = []
#     # Use a web like notation
#     span, h1, h2, h3, h4 = self.get_font_size()
#     text_color, color_1, color_2 = self.get_font_colors()
#
#     action = self.modal_action
#
#     act_obj = context.object
#
#     if self.choose_profile:
#         tubify_text.extend([CR, ("Select Profile" , text_color, h2), "line"])
#
#     elif self.choose_taper:
#         tubify_text.extend([CR, ("Select Taper" , text_color, h2), "line"])
#     else:
#         if self.input:
#             if action == 'depth':
#                 tubify_text.extend([CR, ("Tubify ", text_color, h2), ("Depth : ", text_color, h3), (self.input, color_1, h1), "line"])
#             elif action == 'reso_u':
#                 tubify_text.extend([CR, ("Resolution ", text_color, h2), ("U : ", text_color, h2), (self.input, color_1, h1), "line"])
#             elif action == 'reso_v':
#                 tubify_text.extend([CR, ("Resolution ", text_color, h2), ("V : ", text_color, h2), (self.input, color_1, h1), "line"])
#             elif action == 'extrude':
#                 tubify_text.extend([CR, ("Extrude ", text_color, h2), (": ", text_color, h2), (self.input, color_1, h1), "line"])
#             elif action == 'offset':
#                 tubify_text.extend([CR, ("Offset ", text_color, h2), (": ", text_color, h2),(self.input, color_1, h1), "line"])
#
#             tubify_text.extend([CR, ("Press ", text_color, span), ("ENTER", color_1, span), (" to assign the value", text_color, span)])
#
#         else:
#             if action == 'free':
#                 tubify_text.extend([CR, ("Free navigation ", text_color, h2), "line"])
#             elif action == 'depth':
#                 tubify_text.extend([CR, ("Tubify ", text_color, h2), ("Depth : ", text_color, h3), (str(round(act_obj.data.bevel_object.scale[0], 2)) if self.profile and act_obj.data.bevel_object else str(round(act_obj.data.bevel_depth, 2)), color_1, h1), "line"])
#             elif action == 'reso_u':
#                 tubify_text.extend([CR, ("Resolution ", text_color, h2), ("U : ", text_color, h2), (str(round(act_obj.data.resolution_u, 2)), color_1, h1), "line"])
#             elif action == 'reso_v':
#                 tubify_text.extend([CR, ("Resolution ", text_color, h2), ("V : ", text_color, h2), (str(round(act_obj.data.bevel_object.data.resolution_u, 2)) if self.profile else str(round(act_obj.data.bevel_resolution, 2)), color_1, h1), "line"])
#             elif action == 'extrude':
#                 tubify_text.extend([CR, ("Extrude ", text_color, h2), (str(round(act_obj.data.extrude, 2)), color_1, h1), "line"])
#             elif action == 'offset':
#                 tubify_text.extend([CR, ("Offset ", text_color, h2),(str(round(act_obj.data.offset, 2)), color_1, h1), "line"])
#
#         tubify_text.extend([CR, ("Tubify Depth        (", text_color, span), (" S ", color_1, span), (")      :    ", text_color, span), (str(round(act_obj.data.bevel_object.scale[0], 2)) if self.profile and act_obj.data.bevel_object else str(round(act_obj.data.bevel_depth, 2)), color_2, h4)])
#         tubify_text.extend([CR, ("Resolution U        (", text_color, span), (" D ", color_1, span), (")     :    ", text_color, span), (str(round(act_obj.data.resolution_u, 2)), color_2, h4)])
#         tubify_text.extend([CR, ("Resolution V        (", text_color, span), (" F ", color_1, span), (")      :    ", text_color, span), (str(round(act_obj.data.bevel_object.data.resolution_u, 2)) if self.profile else str(round(act_obj.data.bevel_resolution, 2)), color_2, h4)])
#         tubify_text.extend([CR, ("Extrude               (", text_color, span), (" H ", color_1, span),
#                             (")     :    ", text_color, span),
#                             (str(round(act_obj.data.extrude, 2)), color_2, h4)])
#         tubify_text.extend([CR, ("Offset                  (", text_color, span), (" J ", color_1, span),
#                             (")       :    ", text_color, span),
#                             (str(round(act_obj.data.offset, 2)), color_2, h4)])
#         if self.profile:
#             tubify_text.extend([CR, ("Profile                  (", text_color, span), (" Z ", color_1, span), (")     :    ", text_color, span), (str(act_obj.data.bevel_object.name), color_2, h4)])
#
#         tubify_text.extend([CR, ("Shape                  (", text_color, span), (" K ", color_1, span),
#                              (")     :    ", text_color, span),
#                              ("2D", color_2, h4) if act_obj.data.dimensions =='2D' else ("3D", color_2, h4)])
#     return tubify_text


# Displace
# def displace_text(self, context):
#     displace_text = []
#
#     # Use a web like notation
#     span, h1, h2, h3, h4 = self.get_font_size()
#     text_color, color_1, color_2 = self.get_font_colors()
#
#     action = self.modal_action
#     act_mod = self.act_mod
#
#     # if self.choose_profile:
#     #     displace_text.extend([CR, ("Select Profile", text_color, h2), "line"])
#
#     # else:
#     if self.input:
#         if action == 'strength':
#             displace_text.extend(
#                 [CR, ("Strength : ", text_color, h2), (self.input, color_1, h1),
#                  ("" if act_mod.show_viewport else " Hidden", RED, h2), "line"])
#             displace_text.extend([CR, ("Press ", text_color, span), ("ENTER", color_1, span),
#                                   (" to assign the value", text_color, span)])
#
#         elif action == 'mid_level':
#             displace_text.extend(
#                 [CR, ("Mid Level : ", text_color, h2), (self.input, color_1, h1),
#                  ("" if act_mod.show_viewport else " Hidden", RED, h2), "line"])
#             displace_text.extend([CR, ("Press ", text_color, span), ("ENTER", color_1, span),
#                                   (" to assign the value", text_color, span)])
#     else:
#         if action == 'free':
#             displace_text.extend([CR, ("Free navigation ", text_color, h2),
#                                   (str(act_mod.direction), color_2, h2),
#                                   ("" if act_mod.show_viewport else " Hidden", RED, h2),
#                                   "line"])
#
#         elif action == 'strength':
#             displace_text.extend([CR, ("Strength ", text_color, h2),
#                                   (str(act_mod.direction), color_2, h1),
#                                   (" : ", text_color, h2),
#                                   (str(round(act_mod.strength, 2)), color_1, h1),
#
#                                   ("" if act_mod.show_viewport else " Hidden", RED, h2),
#                                   "line"])
#
#         elif action == 'mid_level':
#             displace_text.extend([CR, ("Mid Level: ", text_color, h2),
#                                   (str(round(act_mod.mid_level, 2)), color_1, h1),
#                                   ("" if act_mod.show_viewport else " Hidden", RED, h2),
#                                   "line"])
#
#
#         displace_text.extend([CR, ("Displace Name        :   ", text_color, span),
#                              (act_mod.name, color_2, h4)])
#
#         displace_text.extend([CR, ("Direction  (", text_color, span), (" X, Y, Z ", color_1, span),
#                               (")  :   ", text_color, span),
#                               (str(act_mod.direction), color_2, h4)])
#
#         displace_text.extend([CR, ("strength           (", text_color, span), (" S ", color_1, span),
#                               (")  :   ", text_color, span),
#                               (str(round(act_mod.strength, 2)), color_2, h4)])
#
#         displace_text.extend([CR, ("Mid Level         (", text_color, span), (" D ", color_1, span),
#                               (")  :   ", text_color, span),
#                               (str(round(act_mod.mid_level, 2)), color_2, h4)])
#
#         displace_text.extend([CR, ("Space               (", text_color, span), (" F ", color_1, span),
#                            (")  :   ", text_color, span),
#                            (str(act_mod.space), color_2, h4)])
#
#         displace_text.extend([CR, ("Vertex Group    (", text_color, span), (" V ", color_1, span),
#                            (")  :  ", text_color, span), \
#                            (str(act_mod.vertex_group), color_2, h4)])
#
#     return displace_text


#Keys
def keys_text(self, context, modal):
    keys_text = []
    # Use a web like notation
    span, h1, h2, h3, h4 = self.get_font_size()
    text_color, color_1, color_2 = self.get_font_colors()

    act_mod = self.act_mod

    #Array
    if modal == 'ROTATE':
        keys_text.extend([CR, ("F         ", color_1), (": Place Cursor 3D", text_color)])
        keys_text.extend([CR, ("ESC", color_1), ("    : Exit", text_color)])

    #Symmetrize
    elif modal == 'SYMMETRIZE':
        keys_text.extend([CR, ("DEL     ", color_1), (": Remove Modifier", text_color)])
        keys_text.extend([CR, ("TAB     ", color_1), (": Change Modal", text_color)])
        keys_text.extend([CR, ("ESC     ", color_1), (": Exit", text_color)])

    # #Tubify
    # elif modal == 'TUBIFY':
    #     keys_text.extend([CR, ("A         ", color_1), (": Convert in Poly", text_color)])
    #     keys_text.extend([CR, ("DEL     ", color_1), (": Remove Modifier", text_color)])
    #
    #     subsurf = False
    #     for obj in context.selected_objects:
    #         subsurf |= any([mod for mod in obj.modifiers if mod.type == 'SUBSURF'])
    #         if subsurf:
    #             break
    #
    #     keys_text.extend([CR, ("1         ", color_1), (": Add Subsurf Modifier", text_color),
    #                       ("  ON", GREEN) if subsurf else
    #                       ("  OFF", RED)
    #                       ]
    #                      )
    #     keys_text.extend([CR, ("Q         ", color_1), (": Cyclic", text_color)])
    #     keys_text.extend([CR, ("G         ", color_1),
    #                       (": Keep Wire Visible", text_color),
    #                       ("  ON", GREEN) if self.wire else
    #                       ("  OFF", RED)
    #                       ]
    #                      )
    #
    #     keys_text.extend([CR, ("Z         ", color_1), (": Select Profil", text_color)])
    #     keys_text.extend([CR, ("X         ", color_1), (": Delete Profil", text_color)])
    #     keys_text.extend([CR, ("T         ", color_1), (": Select Taper", text_color)])
    #     keys_text.extend([CR, ("C         ", color_1), (": Delete Taper", text_color)])
    #     keys_text.extend([CR, ("E         ", color_1), (": Add Edge Split Modifier", text_color)])
    #     keys_text.extend([CR, ("R         ", color_1), (": Toggle Spline Poly/Bezier", text_color)])
    #     keys_text.extend([CR, ("TAB     ", color_1), (": Change Modal", text_color)])
    #     keys_text.extend([CR, ("ESC     ", color_1), (": Exit", text_color)])

    # # Displace
    # elif modal == 'DISPLACE':
    #
    #
    #     keys_text.extend([CR, ("A            ", color_1), (": Apply Modifier", text_color)])
    #     keys_text.extend([CR, ("H            ", color_1), (": Hide Modifier", text_color)])
    #     keys_text.extend([CR, ("DEL        ", color_1), (": Remove Modifier", text_color)])
    #     keys_text.extend([CR, ("SHIFT W ", color_1), (": Add new Modifier", text_color)])
    #     keys_text.extend([CR, ("LEFT/<   ", color_1), (": Next Modifier", text_color)])
    #     keys_text.extend([CR, ("RIGHT/> ", color_1), (": Previous Modifier", text_color)])
    #     keys_text.extend([CR, ("1             ", color_1),
    #                       (": Keep Wire Visible", text_color),
    #                       ("    ON", GREEN) if self.prefs.keep_wire else
    #                       ("    OFF", RED)
    #                       ]
    #                      )
    #     keys_text.extend([CR, ("2             ", color_1),
    #                       (": Show Wire In the Modal", text_color),
    #                       ("    ON", GREEN) if self.prefs.show_wire_in_modal else
    #                       ("    OFF", RED)
    #                       ]
    #                      )
    #     keys_text.extend([CR, ("UP          ", color_1), (": Move Up", text_color)])
    #     keys_text.extend([CR, ("DOWN    ", color_1), (": Move Down", text_color)])
    #     keys_text.extend([CR, ("TAB     ", color_1), (": Change Modal", text_color)])
    #     keys_text.extend([CR, ("ESC        ", color_1), (": Exit", text_color)])

    return keys_text
