import os
import ssl
import urllib.request
import urllib
import json
import re
import bpy

from datetime import datetime
from . import bl_info

class AddonUpdater:
    def __init__(self):
        self._engine = GithubEngine()
        self._user =  "pitiwazou"
        self._repo =  "sfc_update"
        self._tags = []
        self._tag_latest = None
        self._tag_names = []
        self._use_releases = True # True to get the release note
        self._latest_release = None
        self._include_branches = False
        self._include_branch_list = ['master']

        self._verbose = False # for debugging
        self._one_per_day = False # False only for developpment test
        self.skip_tag = None

        self._addon = __package__.split(".")[0].lower()
        self._addon_package = __package__.split(".")[0]
        self.addon_root = os.path.dirname(__file__)
        self._updater_path = os.path.join(self.addon_root, self._addon+"_updater")

        self._json = {}
        self._error = None
        self._error_msg = None
        self._prefiltered_tag_count = 0

    @property
    def api_url(self):
        return self._engine.api_url

    @api_url.setter
    def api_url(self, value):
        if self.check_is_url(value) == False:
            raise ValueError("Not a valid URL: " + value)
        self._engine.api_url = value

    @property
    def error(self):
        return self._error

    @property
    def error_msg(self):
        return self._error_msg

    @property
    def user(self):
        return self._user

    @user.setter
    def user(self, value):
        try:
            self._user = str(value)
        except:
            raise ValueError("User must be a string value")

    @property
    def repo(self):
        return self._repo

    @repo.setter
    def repo(self, value):
        try:
            self._repo = str(value)
        except:
            raise ValueError("User must be a string")

    @property
    def tag_latest(self):
        if self._tag_latest == None:
            return None
        if self._use_releases:
            return self._tag_latest["tag_name"]
        else:
            return self._tag_latest["name"]

    @property
    def tags(self):
        if self._tags == []:
            return []
        tag_names = []
        for tag in self._tags:
            tag_names.append(tag["name"])
        return tag_names

    def check_is_url(self, url):
        if not ("http://" in url or "https://" in url):
            return False
        if "." not in url:
            return False
        return True

    @property
    def use_releases(self):
        return self._use_releases

    @use_releases.setter
    def use_releases(self, value):
        try:
            self._use_releases = bool(value)
        except:
            raise ValueError("use_releases must be a boolean value")

    @property
    def release_note(self):
        return self._json.get("release_note")

    def get_simple_date(self, date):
        regex = "\d{4}-\d{2}-\d{2}"
        valid_date = re.search(regex, str(date))
        if valid_date:
            return valid_date.group()

        return None

    def async_check_update(self, now):
        """Perform update check, run as target of background thread"""

        self.set_updater_json()
        if self._one_per_day:
            last = self.get_simple_date(self._json["last_check"])
            if last:
                now = self.get_simple_date(datetime.now())
                if now == last:
                    print(f"{self._addon} already checked today")
                    return

        if self._verbose:
            print(f"{self._addon} BG thread: Checking for update now in "
                  "background")
        try:
            self.check_for_update(now=now)
        except Exception as exception:
            print("Checking for update error:")
            print(exception)
            if not self._error:
                self._error = "Error occurred"
                self._error_msg = "Encountered an error while checking for updates"

        if self._verbose:
            print(f"{self._addon} BG thread: Finished checking for update")


    def check_for_update(self, now=False):
        SFC = bpy.context.window_manager.SFC

        if self._verbose:
            print("Checking for update function")

        self._error = None
        self._error_msg = None

        if self._repo == None:
            raise ValueError("repo not yet defined")
        if self._user == None:
            raise ValueError("username not yet defined")

        # primary internet call
        self.get_tags()  # sets self._tags and self._tag_latest

        # can be () or ('master') in addition to branches, and version tag
        last_version = self.version_tuple_from_text(self.tag_latest)
        if last_version:
            if not self._json["last_check"]:
                self._json["last_check"] = str(datetime.now())
                if self._use_releases:
                    self._json["release_note"] = self.get_release_note()
                self.save_updater_json()

            else:
                if self.is_update_available(last_version):
                    print(f"A new version of {self._addon} is available")
                    SFC.update_available = True
                    if self._use_releases:
                        self._json["release_note"] = self.get_release_note()
                        print(self._json["release_note"])
                else:
                    print(f"{self._addon} is up to date")

    def is_update_available(self, last_version):
        current = bl_info['version']
        if self._verbose:
            print(f"current version: {current}, last version: {last_version}")
        return current < last_version


    def form_tags_url(self):
        return self._engine.form_tags_url(self)

    def get_tags(self):
        request = self.form_tags_url()
        if self._verbose:
            print("Getting tags from server")

        # get all tags, internet call
        all_tags = self._engine.parse_tags(self.get_api(request), self)

        if all_tags is not None:
            self._prefiltered_tag_count = len(all_tags)
        else:
            self._prefiltered_tag_count = 0
            all_tags = []

        # pre-process to skip tags
        if self.skip_tag != None:
            self._tags = [tg for tg in all_tags if
                          self.skip_tag(self, tg) == False]
        else:
            self._tags = all_tags

        # get additional branches too, if needed, and place in front
        # Does NO checking here whether branch is valid
        if self._include_branches == True:
            temp_branches = self._include_branch_list.copy()
            temp_branches.reverse()
            for branch in temp_branches:
                request = self.form_branch_url(branch)
                include = {
                    "name": branch.title(),
                    "zipball_url": request
                    }
                self._tags = [include] + self._tags  # append to front

        if self._tags == None:
            # some error occurred
            self._tag_latest = None
            self._tags = []
            return
        elif self._prefiltered_tag_count == 0 and self._include_branches == False:
            self._tag_latest = None
            if self._error == None:  # if not None, could have had no internet
                self._error = "No releases found"
                self._error_msg = "No releases or tags found on this repository"
            if self._verbose:
                print("No releases or tags found on this repository")
        elif self._prefiltered_tag_count == 0 and self._include_branches == True:
            if not self._error:
                self._tag_latest = self._tags[0]
            if self._verbose:
                branch = self._include_branch_list[0]
                print("{} branch found, no releases".format(branch),
                      self._tags[0])
        elif (len(self._tags) - len(
                self._include_branch_list) == 0 and self._include_branches == True) \
                or (len(self._tags) == 0 and self._include_branches == False) \
                and self._prefiltered_tag_count > 0:
            self._tag_latest = None
            self._error = "No releases available"
            self._error_msg = "No versions found within compatible version range"
            if self._verbose:
                print("No versions found within compatible version range")
        else:
            if self._include_branches == False:
                self._tag_latest = self._tags[0]
                if self._verbose:
                    print("Most recent tag found:", self._tags[0]['name'])
            else:
                # don't return branch if in list
                n = len(self._include_branch_list)
                self._tag_latest = self._tags[
                    n]  # guaranteed at least len()=n+1
                if self._verbose:
                    print("Most recent tag found:", self._tags[n]['name'])

    # all API calls to base url
    def get_raw(self, url):
        # print("Raw request:", url)
        request = urllib.request.Request(url)
        try:
            context = ssl._create_unverified_context()
        except:
            # some blender packaged python versions don't have this, largely
            # useful for local network setups otherwise minimal impact
            context = None

        # setup private request headers if appropriate
        if self._engine.token != None:
            if self._verbose:
                print("Tokens not setup for engine yet")

        # run the request
        try:
            if context:
                result = urllib.request.urlopen(request, context=context)
            else:
                result = urllib.request.urlopen(request)
        except urllib.error.HTTPError as e:
            if str(e.code) == "403":
                self._error = "HTTP error (access denied)"
                self._error_msg = str(e.code) + " - server error response"
                print(self._error, self._error_msg)
            else:
                self._error = "HTTP error"
                self._error_msg = str(e.code)
                print(self._error, self._error_msg)
            self._update_ready = None
        except urllib.error.URLError as e:
            reason = str(e.reason)
            if "TLSV1_ALERT" in reason or "SSL" in reason.upper():
                self._error = "Connection rejected, download manually"
                self._error_msg = reason
                print(self._error, self._error_msg)
            else:
                self._error = "URL error, check internet connection"
                self._error_msg = reason
                print(self._error, self._error_msg)
            self._update_ready = None
            return None
        else:
            result_string = result.read()
            result.close()
            return result_string.decode()

    # result of all api calls, decoded into json format
    def get_api(self, url):
        # return the json version
        get = None
        get = self.get_raw(url)
        if get != None:
            try:
                return json.JSONDecoder().decode(get)
            except Exception as e:
                self._error = "API response has invalid JSON format"
                self._error_msg = str(e.reason)
                self._update_ready = None
                print(self._error, self._error_msg)
                return None
        else:
            return None

    def get_release_note(self):
        return self._tags[0].get("body")

    def version_tuple_from_text(self, text):
        if text == None:
            return ()

        # should go through string and remove all non-integers,
        # and for any given break split into a different section
        segments = []
        tmp = ''
        for l in str(text):
            if l.isdigit() == False:
                if len(tmp) > 0:
                    segments.append(int(tmp))
                    tmp = ''
            else:
                tmp += l
        if len(tmp) > 0:
            segments.append(int(tmp))

        if len(segments) == 0:
            if self._verbose:
                print("No version strings found text: ", text)
            if self._include_branches == False:
                return ()
            else:
                return (text)
        return tuple(segments)

    def set_updater_json(self):
        """Load or initialize JSON dictionary data for updater state"""
        if self._updater_path == None:
            raise ValueError("updater_path is not defined")
        elif os.path.isdir(self._updater_path) == False:
            os.makedirs(self._updater_path)

        jpath = self.get_json_path()
        if os.path.isfile(jpath):
            with open(jpath) as data_file:
                self._json = json.load(data_file)
                if self._verbose:
                    print(f"{self._addon} Updater: Read in JSON settings from file")
        else:
            # set data structure
            self._json = {
                "last_check": "",
                "release_note": ""
                }
            self.save_updater_json()

    def save_updater_json(self):

        jpath = self.get_json_path()
        outf = open(jpath, 'w')
        data_out = json.dumps(self._json, indent=4)
        outf.write(data_out)
        outf.close()
        if self._verbose:
            print(
                self._addon + ": Wrote out updater JSON settings to file, with the contents:")
            print(self._json)

    def get_json_path(self):
        """Returns the full path to the JSON state file used by this updater.

        Will also rename old file paths to addon-specific path if found
        """
        json_path = os.path.join(self._updater_path,
                                 "{}_updater_status.json".format(
                                     self._addon_package))
        old_json_path = os.path.join(self._updater_path, "updater_status.json")

        # rename old file if it exists
        try:
            os.rename(old_json_path, json_path)
        except FileNotFoundError:
            pass
        except Exception as err:
            print("Other OS error occurred while trying to rename old JSON")
            print(err)
        return json_path


class GithubEngine(object):
	"""Integration to Github API"""

	def __init__(self):
		self.api_url = 'https://api.github.com'
		self.token = None
		self.name = "github"

	def form_repo_url(self, updater):
		return "{}{}{}{}{}".format(self.api_url,"/repos/",updater.user,
								"/",updater.repo)

	def form_tags_url(self, updater):
		if updater.use_releases:
			return "{}{}".format(self.form_repo_url(updater),"/releases")
		else:
			return "{}{}".format(self.form_repo_url(updater),"/tags")

	def form_branch_list_url(self, updater):
		return "{}{}".format(self.form_repo_url(updater),"/branches")

	def form_branch_url(self, branch, updater):
		return "{}{}{}".format(self.form_repo_url(updater),
							"/zipball/",branch)

	def parse_tags(self, response, updater):
		if response == None:
			return []
		return response