
------------------------------------------------
--- Config Section
--- If you have the latest RE Framework this will be saved to JSON in between runs
--- and can be edited in the UI
-----------------------------------------------

function generateDefaultConfig()
    return { 
        showDetailedWindow = false, -- Show or hide the detailed window with all the monster/status information and numbers
        showEnemyStatusBars = true, -- Show or hide the status bars
        showAllEnemyStatusBars = false, -- If true, will show status bars for enemies that have not been interacted with, if false will only show bars if the status bar has changed
        showText = true, -- If true will label the bars and print a timer
        onlyTimers = false, -- If true only the timers will pop up

        -- Per Status Configuration options
        -- name and barColor are not configurable in UI
        -- showPlayer = wether to show the status bar for player attacks
        -- showWirebug = wether to show the status bar for wirebug status effects (e.g. status damage from mounted monsters)
        -- show - Just for Flash damage, wether to show the bar or not
        paramDetails = {
            _ParalyzeParam = {
                name = "Paralyze",
                barColor = { r = 237, g = 192, b = 123 },
                showPlayer = true,
                showWirebug = false
            },
            _SleepParam = {
                name = "Sleep",
                barColor = { r = 180, g = 208, b = 246 },
                showPlayer = true,
                showWirebug = false
            },
            _StunParam = {
                name ="Stun",
                barColor = { r = 242, g = 237, b = 79 },
                showPlayer = true,
                showWirebug = false
            },
            _FlashParam = {
                name ="Flash",
                barColor ={ r = 255, g = 252, b = 133 },
                hasMax = true,
                show = true
            },
            _PoisonParam = {
                name = "Poison",
                barColor = { r = 165, g = 101, b = 222 },
                showPlayer = true,
                showWirebug = false
            },
            _BlastParam = {
                name = "Blast",
                barColor = { r = 216, g = 117, b = 52 },
                showPlayer = false,
                showWirebug = false
            },
            _StaminaParam = {
                name = "Exhaust",
                barColor = { r = 88, g = 130, b = 239 },
                showPlayer = true,
                showWirebug = false
            },
            _WaterParam = {
                name = "Water Blight",
                barColor = { r = 125, g = 173, b = 241 },
                showPlayer = false,
                showWirebug = false
            },
            _FireParam = {
                name = "Fire Blight",
                barColor = { r = 221, g = 117, b = 64 },
                showPlayer = false,
                showWirebug = false
            },
            _IceParam = {
                name = "Ice Blight",
                barColor = { r = 179, g = 207, b = 246 },
                showPlayer = false,
                showWirebug = false
            },
            _ThunderParam = {
                name = "Thunder Blight",
                barColor = { r = 254, g = 252, b = 147 },
                showPlayer = false,
                showWirebug = false
            }
        },

        -- Sort priority is the priority that the bars should show up in for any monster
        sortPriority = {
            name = -1,
            paralyze = 0,
            sleep = 1,
            stun = 2,
            flash = 3,
            poison = 4,
            blast = 5,
            stamina = 6,
            water = 7,
            fire = 8,
            ice = 9,
            thunder = 10,
        },

        -- The largest width a bar can get
        maxBarWidth = 450,
        -- Height of an individual status bar
        barHeight = 20,
        -- How many pixels the smallest segment of any of the status bars should be
        -- Exhaust and Blast are handled specially because their buildup bars are weird
        minSegmentWidth = 35,
        -- Height of the bar that shows up to represent the timer
        timerBarHeight = 10,
        -- How much padding to have between each bar
        barSpacing = 15,
        -- The font size, waiting on RE Framework to support this
        fontHeight = 16,
        -- Guestimate at font width
        fontWidth = 8,
        -- The starting vertical position of the widget until configured
        initialTopPos = 500,
        -- The starting horizontal position of the widget until configured
        initialLeftPos = 3300,
        -- Transparency value of text
        textAlpha = 1,
        -- Transparency value of the bars
        barAlpha = 1,
        -- Color to use for text shadows
        shadowColor = {r = 0, g = 0, b = 0},
        -- Color to use for text
        textColor = {r = 224, g = 224, b = 224 },
        -- Color of the outline around the bars
        outlineColor = { r = 0, g = 0, b = 0 },
        -- How much darker the unfilled bar should be than its foreground
        barShade = 0.5,
        -- How much extra padding to have between two monsters
        monsterSpacing = 10,
        -- How wide the "count" pips should be in "stock" value (not pixel)
        countWidth = 30,
        -- The spacing between count pips in pixels
        countSpacing = 10,
        -- How much to scale the UI down as the rank and players change
        scales = {
            l1 = 1,
            l2 = 0.5,
            l3 = 0.35,
            l4 = 0.35,
            h1 = 1,
            h2 = 0.5,
            h3 = 0.35,
            h4 = 0.35,
            x1 = 1,
            x2 = 0.5,
            x3 = 0.35,
            x4 = 0.35,
        },
    }

end
---------------------------------


-- Taken from stack overflow, so you know it just works!
function spairs(t, order)
    local keys = {}
    for k in pairs(t) do keys[#keys+1] = k end

    if order then
        table.sort(keys, function(a,b) return order(t, a, b) end)
    else
        table.sort(keys)
    end

    local i = 0
    return function()
        i = i + 1
        if keys[i] then
            return keys[i], t[keys[i]]
        end
    end
end

-- Convienence function that converts rgba values to the integers imgui expects
function rgba(r, g, b, a)
    r = math.floor(r)
    g = math.floor(g)
    b = math.floor(b)
    return r + (g << 8) + (b << 16) + (math.floor(a * 255) << 24)
end

local status = "OK"; -- Used mostly for debug to track status in Ref UI
local configFile = "status_bar_config.json"
local config = generateDefaultConfig()
local isDefaultConfig = true
if json ~= nil then
    data = json.load_file(configFile)
    -- If we are unable to read our config file, dump the default in there
    if data == nil then
        json.dump_file(configFile, config)
    else
        isDefaultConfig = false
        config = data
    end
end

-- All of the numbers above were done on my tv, so this just scales it down if you are not
function scaleDefaultConfig(xRes, yRes)
    local xScale = xRes/3840;
    local yScale = yRes/2160;

    config.maxBarWidth = config.maxBarWidth * xScale;
    config.barHeight = config.barHeight * yScale;
    config.minSegmentWidth = config.minSegmentWidth * xScale;
    config.timerBarHeight = config.timerBarHeight * yScale;
    config.barSpacing = config.barSpacing * yScale;
    config.initialTopPos = config.initialTopPos * yScale;
    config.initialLeftPos = config.initialLeftPos * xScale;
    config.monsterSpacing = config.monsterSpacing * yScale;
    config.countWidth = config.countWidth * xScale;
    config.countSpacing = config.countSpacing * xScale;
end

-- Helper function that converts the various parameter arrays in the damage field
-- They store the damage in arrays of two for most things where the first index is the players value
-- and the second index is the value for wirebug events.
-- e.g. Stock[0] is the current status damage done via player moves (attacks, endemic life, etc)
--      Stock[1] is the current status damage done via wirebug events (attacking with a mounted monster)
function getParamArray(param, func)
    local data = param:call(func);
    local datas = {};

    if data ~= nil then
        for i = 0, (data:get_size() - 1), 1 do
            local label = i == 0 and "player" or "wirebug"
            datas[label] = data:get_element(i):get_field("mValue");
        end
    end

    return datas;
end

-- Just a wrapper around call, was going to be something more eventually but ended up not needing, too lazy to refactor it
function getParamValue(param, func)
    local data = param:call(func);

    return data;
end

-- Retrieves all the relevant data for a given status damage field in one easy to get table
function getStatusData(damageField, statusName) 
    local param = damageField:get_field(statusName);

    if param == nil then
        return {}
    end
    
    local results = {
        name          = statusName, -- used for lookup later
        isEnable      = getParamValue(param, "get_IsEnable"),      -- If the monster can be affected by the status effect
        isActive      = getParamValue(param, "get_IsActive"),      -- Doesnt seem to get set, grabbed just in case
        stocks        = getParamArray(param, "get_Stock"),         -- Stock is the current status damage to the target
        limits        = getParamArray(param, "get_Limit"),         -- Limit is the current threshold at which the status will trigger
        maxStocks     = getParamArray(param, "get_MaxStock"),      -- Never figured out how this one is used
        addLimits     = getParamArray(param, "get_AddLimit"),      -- This is the amount that will be added every time you trigger the status
        maxLimits     = getParamArray(param, "get_MaxLimit"),      -- The maximum value the limit can get to
        subs          = getParamArray(param, "get_Sub"),           -- This seems to be how fast the stock drains
        activeTime    = getParamValue(param, "get_ActiveTime"),    -- How long the status will be active
        activeTimer   = getParamValue(param, "get_ActiveTimer"),   -- If the status is active, contains a countdown until inactive
        subActiveTime = getParamValue(param, "get_SubActiveTime"), -- Every proc this will be subtracted from the active time
        minActiveTime = getParamValue(param, "get_MinActiveTime"), -- The minimum time the proc can last
        type          = param:get_field("<Type>k__BackingField")
    };

    -- Even though Flash is implemented using the same object type as the others, it operates entirely differently
    -- Pretty much all the above variables (aside from activeTimer) are meaningless for Flash, instead flash bombs just
    -- automatically trigger the effect if close enough (and in vision), up until some predetermined limit of how many you can use
    if statusName == "_FlashParam" and results.isEnable then
        -- Number of times a monster has been flash bombed
        results.activeCount = getParamValue(param, "get_ActivateCnt")

        -- As far as I can tell, the maximum number of times something can be flashed is a sum of these elements in this array
        -- It appears that there are stages of how long the flash lasts here as well (e.g. First 4 last 20s, last 3 last 10)
        local damageLvInfo = param:get_field("<DamageLvInfo>k__BackingField")
        local maxFlashes = 0
        for i = 0, (damageLvInfo:get_size() - 1), 1 do
            maxFlashes = maxFlashes + damageLvInfo:get_element(i):get_field("_ActivateCnt");
        end

        results.maxActiveCount = maxFlashes
    elseif statusName == "_FlashParam" then
        results.activeCount = 0
        results.maxActiveCount = 0
    end

    return results
end


-- Retrieves a list of all the monsters on the map and stores their status information in the various tables
-- Some of this was taken from the MHR_Overlay
function monsterStatuses()
    local monsterInfo = {}
    local monsters = {};
    local questInfo = {
        active = false,
        totalJoinNum = 1,
        questRank = 0
    };

    local enemyManager = sdk.get_managed_singleton("snow.enemy.EnemyManager");
    if not enemyManager then
        status = "No enemy manager";
        return monsters, questInfo;
    end

    local enemyCount = enemyManager:call("getBossEnemyCount");
    if enemyCount == nil then
        status = "No enemy count";
        return monsters, questInfo;
    end

    local questManager = sdk.get_managed_singleton("snow.QuestManager")
    local questData = questManager:call("getActiveQuestData")
    local questId = 0;
    questInfo.active = questManager:call("isActiveQuest") or false
    questInfo.totalJoinNum = questManager:call("getTotalJoinNum") or 1
    if questData ~= nil then
        questInfo.questId = questData:call("getQuestNo") or 0
        questInfo.questRank = questData:call("getQuestRank_Lv") or 0
    end

    for i = 0, enemyCount - 1 do
        local enemy = enemyManager:call("getBossEnemy", i);
        if enemy == nil then
            status = "No enemy";
            break;
        end

        local damageField = enemy:get_field("<DamageParam>k__BackingField");
        if damageField == nil then
            status = "No damage field";
            break;
        end
        
        local messageManager = sdk.get_managed_singleton("snow.gui.MessageManager");
        if not messageManager then
            status = "No message manager";
            break;
        end

        local enemyType = enemy:get_field("<EnemyType>k__BackingField");
        if not enemyType then
            status = "No enemy type";
            break;
        end

        local enemyName = messageManager:call("getEnemyNameMessage", enemyType);

        local uniqueId = enemy:get_field("<UniqueId>k__BackingField");
        local monsterKey = tostring(questId) .. "_" .. tostring(uniqueId)
        monsters[monsterKey] = {
            name = enemyName,
            paralyze = getStatusData(damageField, "_ParalyzeParam"),
            sleep = getStatusData(damageField, "_SleepParam"),
            stun = getStatusData(damageField, "_StunParam"),
            flash = getStatusData(damageField, "_FlashParam"),
            poison = getStatusData(damageField, "_PoisonParam"),
            blast = getStatusData(damageField, "_BlastParam"),
            stamina = getStatusData(damageField, "_StaminaParam"),
            water = getStatusData(damageField, "_WaterParam"),
            fire = getStatusData(damageField, "_FireParam"),
            ice = getStatusData(damageField, "_IceParam"),
            thunder = getStatusData(damageField, "_ThunderParam"),
        };

        for key, value in pairs(monsters[monsterKey]) do
            if key ~= 'name' then
                value.minLimits = {}
                local conditionData = enemy:call("getUseConditionDamageData", value.type, 0)
                local playerData = conditionData:get_field("_DefaultStockData")
                
                if playerData == nil then
                    value.minLimits.player = 0
                else
                    value.minLimits.player = playerData:get_field("_DefaultLimit")
                end

                local wirebugData = conditionData:get_field("_AttackByMarionetteStockData")
                if wirebugData == nil then
                    value.minLimits.wirebug = 0
                else
                    value.minLimits.wirebug = wirebugData:get_field("_DefaultLimit")
                end
            end
        end
    end

    return monsters, questInfo
end

-- This prints detailed information for a given paramter to imgui
function printMonsterStatusParam(param)
    local paramName = config.paramDetails[param.name].name
    imgui.text(paramName .. " Enabled: " .. tostring(param.isEnable))
    imgui.text(paramName .. " Active: " .. tostring(param.isActive))
    imgui.text(paramName .. " (Player): " .. param.stocks.player .. "/" .. param.limits.player)
    imgui.text(paramName .. " (Wirebug): " .. param.stocks.wirebug .. "/" .. param.limits.wirebug)
    imgui.text(paramName .. " Timer (Player): " .. param.activeTimer)
    if config.paramDetails[param.name].hasMax then
        imgui.text(paramName .. " Activations: " .. param.activeCount .. "/" .. param.maxActiveCount)
    end
    imgui.text("----------")
end

-- This prints all the information for a given monster to imgui
function printMonsterStatus(monster)
    imgui.text("==========")
    imgui.text(monster.name)
    imgui.text("==========")
    printMonsterStatusParam(monster.sleep)
    printMonsterStatusParam(monster.stun)
    printMonsterStatusParam(monster.flash)
    printMonsterStatusParam(monster.poison)
    printMonsterStatusParam(monster.blast)
    printMonsterStatusParam(monster.stamina)
    printMonsterStatusParam(monster.water)
    printMonsterStatusParam(monster.fire)
    printMonsterStatusParam(monster.ice)
    printMonsterStatusParam(monster.thunder)
end

function drawThiccBox(left, top, width, height, color, shade, alpha, outer)
    local barShade = shade or config.barShade
    local barAlpha = alpha or config.barAlpha
    local outlineColor = config.outlineColor

    if outer then
        draw.filled_rect(left, top, width, height, rgba(color.r * barShade, color.g * barShade, color.b * barShade, barAlpha))
        draw.outline_rect(left, top, width, height, rgba(outlineColor.r, outlineColor.g, outlineColor.b, barAlpha))
        draw.outline_rect(left - 1, top - 1, width + 2, height + 2, rgba(outlineColor.r, outlineColor.g, outlineColor.b, barAlpha))
        draw.outline_rect(left - 2, top - 1, width + 4, height + 4, rgba(outlineColor.r, outlineColor.g, outlineColor.b, barAlpha))
    else
        draw.filled_rect(left, top, width, height, rgba(color.r * barShade, color.g * barShade, color.b * barShade, barAlpha))
        draw.outline_rect(left, top, width, height, rgba(outlineColor.r, outlineColor.g, outlineColor.b, barAlpha))
        draw.outline_rect(left + 1, top + 1 , width, height - 2, rgba(outlineColor.r, outlineColor.g, outlineColor.b, barAlpha))
        draw.outline_rect(left + 2 , top + 2, width, height - 4, rgba(outlineColor.r, outlineColor.g, outlineColor.b, barAlpha))
    end

end

function getScaleFactor(questData)
    local active = questData.active
    local rankLookup = {
        p0 = "l",
        p1 = "h",
        p2 = "x"
    }
    local rank = questData.questRank > 2 and "x" or  rankLookup["p"..tostring(math.floor(questData.questRank))]
    local players = questData.totalJoinNum > 4 and "4" or tostring(math.floor(questData.totalJoinNum))
    -- local scaleInfo = {
    --     players = { p1 = config.playerScales.p1, p2 = config.playerScales.p2, p3 = config.playerScales.p3, p4 = config.playerScales.p4 },
    --     rank = { p0 = config.rankScales.p1, p1 = config.rankScales.p2, p2 = config.rankScales.p2}
    -- }

    local playerScale = 1
    local wirebugScale = 0.7

    if active == false then
        return {
            player = playerScale,
            wirebug = wirebugScale
        }
    end

    playerScale = config.scales[rank..players] * playerScale
    wirebugScale = config.scales[rank..players] * playerScale

    -- playerScale = playerScale * scaleInfo.players[players]
    -- playerScale = playerScale * scaleInfo.rank[rank]
 
    -- wirebugScale = wirebugScale * scaleInfo.players[players]
    -- wirebugScale = wirebugScale * scaleInfo.rank[rank]

    return {
        player = playerScale, -- Good For HR Solo
        wirebug = wirebugScale 
    }
end

-- This handles drawing a single status bar for a status
-- type can be either 'player' or 'wirebug'
function drawStatusBar(param, left, top, type, pixelPerStock)
    local paramDetails = config.paramDetails[param.name]

    -- No reason to show bars for effects that a monster can not be effected by
    if param.isEnable == false or param.limits[type] == 0 then
        return false, 0
    end
    
    if type == "player" and paramDetails.showPlayer == false then
        return false, 0
    end

    if type == "wirebug" and paramDetails.showWirebug == false then
        return false, 0
    end

    if config.onlyTimers and param.activeTimer == 0 then
        return false, 0
    end

    local isInactive = true; --param.activeTimer == 0;
    local hasNoDamage = param.stocks[type] == 0;
    local hasNotBrokenLimit = param.limits[type] == param.minLimits[type] and param.addLimits[type] > 0;
    if config.showAllEnemyStatusBars == false and isInactive and hasNoDamage and hasNotBrokenLimit then 
        return false, 0
    end

    local filledWidth = pixelPerStock * param.stocks[type];
    local name = type == "player" and paramDetails.name or paramDetails.name .. " (WB)"
    local barHeight = config.barHeight;
    local timerBarHeight = config.timerBarHeight;
    local fontHeight = config.fontHeight;
    local fontWidth = config.fontWidth;
    local barAlpha = config.barAlpha;
    local shadowColor = rgba(config.shadowColor.r, config.shadowColor.g, config.shadowColor.b, config.textAlpha);
    local textColor = rgba(config.textColor.r, config.textColor.g, config.textColor.b, config.textAlpha);
    local outlineColor = config.outlineColor;
    local barShade = config.barShade;
    local numLines = 1
    local numSegments = 1;

    if param.addLimits[type] == 0 or param.maxLimits[type] == 0 or param.maxLimits[type] < param.minLimits[type] then 
        numSegments = 1
    else 
        -- Add 1 here for the intial segment
        numSegments = math.ceil((param.maxLimits[type] - param.minLimits[type]) / param.addLimits[type]) + 1;
    end

    local barTop = top
    local barLeft = left
    local consumedWidth = 0
    local actualWidth = 0
    local totalStock = 0;

    for i = 0, numSegments, 1 do
        if config.onlyTimers then
            break
        end
        local segmentWidth = (i == 0) and param.minLimits[type] or param.addLimits[type]
        totalStock = totalStock + segmentWidth
        segmentWidth = segmentWidth * pixelPerStock

        local activeSegment = totalStock <= param.limits[type]
        local alpha = activeSegment and 1 or 0.5

        -- If it goes too far left, start a new line
        if ((barLeft + segmentWidth) - left) > config.maxBarWidth and i ~= 0 then
            barLeft = left
            barTop = barTop + barHeight
            numLines = numLines + 1
        end

        if ((barLeft + segmentWidth) - left) > actualWidth then
            actualWidth = (barLeft + segmentWidth) - left
        end

        drawThiccBox(barLeft, barTop, segmentWidth, barHeight, paramDetails.barColor, nil, barAlpha * alpha, false)
        if filledWidth - consumedWidth > 0 then
            local actualFilledWidth = filledWidth - consumedWidth > segmentWidth and segmentWidth or filledWidth - consumedWidth
            draw.filled_rect(barLeft + 3, barTop + 3, actualFilledWidth, barHeight - 6, rgba(paramDetails.barColor.r, paramDetails.barColor.g, paramDetails.barColor.b, barAlpha))
        end

        consumedWidth = consumedWidth + segmentWidth

        barLeft = barLeft + segmentWidth
    end

    if config.onlyTimers then
        actualWidth = config.maxBarWidth
    end

    local timerWidth =  actualWidth * (param.activeTimer / param.activeTime);
    local timerTop = barTop + barHeight + 5

    if config.onlyTimers then
        timerTop = barTop + 5
    end

    -- Draw the timer, is a bar that drains below the status bar
    if param.activeTimer > 0 and param.isActive then 
        drawThiccBox(left, timerTop, timerWidth, timerBarHeight, paramDetails.barColor, 1, nil, false)
    end
    
    -- Shadow text idea from MHR Overlay for easier reading
    -- Draw the status name
    if config.showText then 
        draw.text(name, left - (string.len(name) * fontWidth) + 1, top + 1 +  barHeight/2 - fontHeight/2, shadowColor)
        draw.text(name, left - (string.len(name) * fontWidth), top + barHeight/2 - fontHeight/2, textColor)
    end

    -- Draw the timer text to the right of the bars
    if config.showText then
        local timer = param.activeTimer
        if param.isActive == false then
            timer = 0
        end
        draw.text(string.format("%.1f", timer), left + actualWidth + 11, top + 1 +  barHeight/2 - fontHeight/2, shadowColor)
        draw.text(string.format("%.1f", timer), left + actualWidth + 10, top +  barHeight/2 - fontHeight/2, textColor)
    end

    return true, numLines
end

-- This draws a "count" bar for things that just have a count and a limit like the Flash status efect
function drawCountBar(param, left, top)
    local paramDetails = config.paramDetails[param.name]

    -- If not enabled on the monster no reason to draw it
    if param.isEnable == false or param.maxActiveCount == 0 then
        return false, 0
    end

    if config.onlyTimers then
        return false, 0
    end
    
    local filledWidth = config.countWidth;
    local width = (filledWidth * param.maxActiveCount) + (config.countSpacing * (param.maxActiveCount - 1));
    local originalLeft = left
    local activeTimer = param.activeTimer > 0 and param.activeTimer or 0
    local timerWidth = width * (activeTimer/ param.activeTime);
    local name = paramDetails.name;
    local barHeight = config.barHeight;
    local timerBarHeight = config.timerBarHeight
    local fontHeight = config.fontHeight;
    local fontWidth = config.fontWidth;
    local barAlpha = config.barAlpha;
    local shadowColor = rgba(config.shadowColor.r, config.shadowColor.g, config.shadowColor.b, config.textAlpha);
    local textColor = rgba(config.textColor.r, config.textColor.g, config.textColor.b, config.textAlpha);
    local outlineColor = config.outlineColor;
    local barShade = config.barShade;

    if config.showAllEnemyStatusBars == false and param.activeCount == 0 then 
        return false, 1
    end

    -- Draw the monster name
    if config.showText then
        draw.text(name, left - (string.len(name) * fontWidth) + 1, top + 1 +  barHeight/2 - fontHeight/2, shadowColor)
        draw.text(name, left - (string.len(name) * fontWidth), top +  barHeight/2 - fontHeight/2, textColor)
    end

    -- For each count that you can apply to the monster, draw a little section to represent it being used or not
    for i = 0, param.maxActiveCount - 1, 1 do
        shade = i < param.activeCount and 1 or barShade
        alpha = i <= param.activeCount and barAlpha or (0.5 * barAlpha)
        draw.filled_rect(left, top, filledWidth, barHeight, rgba(paramDetails.barColor.r * shade, paramDetails.barColor.g * shade, paramDetails.barColor.b * shade, alpha))
        draw.outline_rect(left, top, filledWidth, barHeight, rgba(outlineColor.r, outlineColor.g, outlineColor.b, alpha))
        draw.outline_rect(left + 1, top + 1, filledWidth, barHeight - 2, rgba(outlineColor.r, outlineColor.g, outlineColor.b, alpha))
        draw.outline_rect(left + 2, top + 2, filledWidth, barHeight - 4, rgba(outlineColor.r, outlineColor.g, outlineColor.b, alpha))
        left = left + filledWidth + config.countSpacing
    end

    -- Draw the timer
    if activeTimer > 0 then
        draw.filled_rect(originalLeft, top + barHeight + 5, timerWidth, timerBarHeight, rgba(paramDetails.barColor.r, paramDetails.barColor.g, paramDetails.barColor.b, barAlpha))
        draw.outline_rect(originalLeft, top + barHeight + 5, timerWidth, timerBarHeight, rgba(outlineColor.r, outlineColor.g, outlineColor.b, barAlpha))
        draw.outline_rect(originalLeft - 1, top + barHeight + 4, timerWidth + 2, timerBarHeight + 2, rgba(outlineColor.r, outlineColor.g, outlineColor.b, barAlpha))
        draw.outline_rect(originalLeft - 2, top + barHeight + 3, timerWidth + 4, timerBarHeight + 4, rgba(outlineColor.r, outlineColor.g, outlineColor.b, barAlpha))
    end

    -- Draw the timer text
    if config.showText then
        draw.text(string.format("%.1f", activeTimer), originalLeft + width + 11, top + 1 +  barHeight/2 - fontHeight/2, shadowColor);
        draw.text(string.format("%.1f", activeTimer), originalLeft + width + 10, top +  barHeight/2 - fontHeight/2, textColor)
    end

    return true, 1
end

-- Iterate over all the monsters/status effects and draw the relevant bars for them
function drawMonsterStatus(monsterObj, monster, lastTop, questData)
    local topPos = lastTop;
    local leftPos = config.initialLeftPos;
    local wasDrawn = false;
    local anyDrawn = false;
    local originalTop = lastTop;
    local barHeight = config.barHeight
    local timerBarHeight = config.timerBarHeight
    local barSpacing = config.barSpacing
    local fontHeight = config.fontHeight;
    local shadowColor = rgba(config.shadowColor.r, config.shadowColor.g, config.shadowColor.b, config.textAlpha);
    local textColor = rgba(config.textColor.r, config.textColor.g, config.textColor.b, config.textAlpha);
    local scaleFactor = getScaleFactor(questData)

    -- Spacing for the monster name (which will be drawn later if needed)
    topPos = topPos + fontHeight * 2

    local minPlayerLimit = -1;
    local minWirebugLimit = -1;
    -- Calculate a decent pixelPerStock, we aim for the configured smallest segment
    for key, value in pairs(monster) do
        if key ~= 'name' then
            if (config.paramDetails[value.name].showPlayer and ((minPlayerLimit == -1 and value.minLimits.player ~= 0) or (value.minLimits.player ~= 0 and (value.minLimits.player) < minPlayerLimit))) then
                minPlayerLimit = value.minLimits.player
            end
            if (config.paramDetails[value.name].showWirebug and ((minWirebugLimit == -1 and value.minLimits.player ~= 0) or (value.minLimits.wirebug ~= 0 and (value.minLimits.wirebug) < minWirebugLimit))) then
                minWirebugLimit = value.minLimits.wirebug
            end
        end
    end

    local pixelPerStock = {
        player = 1,
        wirebug = 1
    }
    if minPlayerLimit > 0 then
        pixelPerStock.player = config.minSegmentWidth/minPlayerLimit
    end
    if minWirebugLimit > 0 then
        pixelPerStock.wirebug = config.minSegmentWidth/minWirebugLimit
    end


    -- For each status effect draw the bar
    for key, value in spairs(monster, function(t, a, b) return config.sortPriority[a] < config.sortPriority[b] end) do
        if key ~= "name" and key ~= "flash" then
            -- We use wasDrawn to know if we need to move down a line basically
            wasDrawn, numLines = drawStatusBar(value, leftPos, topPos, "player", pixelPerStock.player * scaleFactor.player)
            anyDrawn = anyDrawn or wasDrawn
            topPos = wasDrawn and topPos + (barHeight * numLines) + timerBarHeight + barSpacing or topPos

            wasDrawn, numLines = drawStatusBar(value, leftPos, topPos, "wirebug", pixelPerStock.wirebug * scaleFactor.wirebug)
            anyDrawn = anyDrawn or wasDrawn
            topPos = wasDrawn and topPos + (barHeight * numLines) + timerBarHeight + barSpacing or topPos
        end

        if key == "flash" then
            wasDrawn, numLines = drawCountBar(value, leftPos, topPos)
            anyDrawn = anyDrawn or wasDrawn
            topPos = wasDrawn and topPos + (barHeight * numLines) + timerBarHeight + barSpacing or topPos
        end
    end

    -- If we did draw any bars, we write the monsters name above them
    if anyDrawn then 
        draw.text(monster.name, leftPos + 1, originalTop + 1, shadowColor);
        draw.text(monster.name, leftPos, originalTop, textColor)
    else -- Otherwise we undo the spacing we did up above as we didnt actually draw anything
        topPos = topPos - fontHeight * 2
    end

    return topPos
end

log.info("Status Bars Loaded");

--- Taken from MHR Overlay to get screen width/height
local screenWidth = 0;
local screenHeight = 0;

local sceneManager = sdk.get_native_singleton("via.SceneManager");
if not sceneManager then 
    log.error("[status_bars.lua] No scene manager");
    return
end

local sceneView = sdk.call_native_func(scene_manager, sdk.find_type_definition("via.SceneManager"), "get_MainView");
if not sceneView then
    log.error("[status_bars.lua] No main view");
    return
end

function getWindowSize()
    local size = sceneView:call("get_WindowSize");
    if not size then
        log.error("[status_bars.lua] No scene view size");
        return
    end

    screenWidth = size:get_field("w");
    if not screenWidth then
        log.error("[status_bars.lua] No screen width");
        return
    end

    screenHeight = size:get_field("h");
    if not screenHeight then
        log.error("[status_bars.lua] No screen height");
        return
    end
end

function draw_status_config(paramName, smallName)
    local anyChange = false
    local name = config.paramDetails[paramName].name
    imgui.text(name)
    if smallName == "flash" then
        changed, config.paramDetails[paramName].show = imgui.checkbox("Show##"..name, config.paramDetails[paramName].show)
        anyChange = changed and true or anyChange
        imgui.same_line()
    else
        changed, config.paramDetails[paramName].showPlayer = imgui.checkbox("Show Player##"..name, config.paramDetails[paramName].showPlayer)
        anyChange = changed and true or anyChange
        imgui.same_line()
        changed, config.paramDetails[paramName].showWirebug = imgui.checkbox("Show Wirebug##"..name, config.paramDetails[paramName].showWirebug)
        anyChange = changed and true or anyChange
        imgui.same_line()
    end
    changed, config.sortPriority[smallName] = imgui.slider_int("Order##"..name, config.sortPriority[smallName], 0, 10)
    anyChange = changed and true or anyChange

    return anyChange
end

local monsters = {};
local questData = {};
local savedMonsters = {};
local savedQuestData = {};
local useSavedBars = false;
local testBars = {
    v = { p1 = {}, p2 = {}, p3 = {}, p4 = {}},
    l = { p1 = {}, p2 = {}, p3 = {}, p4 = {}},
    h = { p1 = {}, p2 = {}, p3 = {}, p4 = {}}
}

re.on_draw_ui(function() 
    getWindowSize()

    if imgui.tree_node("Status Effect Bars") then
        if json == nil then
            imgui.text("===============");
            imgui.text("Please update to the latest REFramework to save settings between game sessions");
            imgui.text("===============");
        end
        local anyChange = false
        imgui.text("Status: " .. status);
        changed, config.showDetailedWindow = imgui.checkbox("Show Detailed Window", config.showDetailedWindow);
        anyChange = changed and true or anyChange
        changed, config.showEnemyStatusBars = imgui.checkbox("Show Status Bars", config.showEnemyStatusBars);
        anyChange = changed and true or anyChange
        changed, config.showAllEnemyStatusBars = imgui.checkbox("Show Empty Status Bars", config.showAllEnemyStatusBars);
        anyChange = changed and true or anyChange
        changed, config.showText = imgui.checkbox("Show All Text", config.showText);
        anyChange = changed and true or anyChange
        changed, config.onlyTimers = imgui.checkbox("Only Show Timers", config.onlyTimers);
        anyChange = changed and true or anyChange

        if imgui.tree_node("Per Status Effect Configuration") then
            changed = draw_status_config("_ParalyzeParam", "paralyze")
            anyChange = changed and true or anyChange
            changed = draw_status_config("_SleepParam", "sleep")
            anyChange = changed and true or anyChange
            changed = draw_status_config("_StunParam", "stun")
            anyChange = changed and true or anyChange
            changed = draw_status_config("_FlashParam", "flash")
            anyChange = changed and true or anyChange
            changed = draw_status_config("_PoisonParam", "poison")
            anyChange = changed and true or anyChange
            changed = draw_status_config("_BlastParam", "blast")
            anyChange = changed and true or anyChange
            changed = draw_status_config("_StaminaParam", "stamina")
            anyChange = changed and true or anyChange
            changed = draw_status_config("_WaterParam", "water")
            anyChange = changed and true or anyChange
            changed = draw_status_config("_FireParam", "fire")
            anyChange = changed and true or anyChange
            changed = draw_status_config("_IceParam", "ice")
            anyChange = changed and true or anyChange
            changed = draw_status_config("_ThunderParam", "thunder")
            anyChange = changed and true or anyChange
            imgui.tree_pop()
        end

        if imgui.tree_node("Status Bar Look Configuration") then
            changed, config.initialLeftPos = imgui.slider_int("Left Position", config.initialLeftPos, 0, screenWidth)
            anyChange = changed and true or anyChange
            changed, config.initialTopPos = imgui.slider_int("Top Position", config.initialTopPos, 0, screenHeight)
            anyChange = changed and true or anyChange
            changed, config.maxBarWidth = imgui.slider_int("Max Bar Width", config.maxBarWidth, 0, 1000)
            anyChange = changed and true or anyChange
            changed, config.barHeight = imgui.slider_int("Bar Height", config.barHeight, 0, 100)
            anyChange = changed and true or anyChange
            changed, config.minSegmentWidth = imgui.slider_int("Min Segment Width", config.minSegmentWidth, 0, 200)
            anyChange = changed and true or anyChange            
            changed, config.timerBarHeight = imgui.slider_int("Timer Bar Height", config.timerBarHeight, 0, 100)
            anyChange = changed and true or anyChange
            changed, config.barSpacing = imgui.slider_int("Space Between Bars", config.barSpacing, 0, 100)
            anyChange = changed and true or anyChange
            changed, config.textAlpha = imgui.slider_float("Text Transparency", config.textAlpha, 0, 1)
            anyChange = changed and true or anyChange
            changed, config.barAlpha = imgui.slider_float("Bar Transparency", config.barAlpha, 0, 1)
            anyChange = changed and true or anyChange
            changed, config.monsterSpacing = imgui.slider_int("Space Between Monsters", config.monsterSpacing, 0, 100)
            anyChange = changed and true or anyChange
            changed, config.countWidth = imgui.slider_int("Size Of Flash Pips", config.countWidth, 0, 300)
            anyChange = changed and true or anyChange
            changed, config.countSpacing = imgui.slider_int("Space Between Flash Pips", config.countSpacing, 0, 100)
            anyChange = changed and true or anyChange
            if imgui.tree_node("UI Player Scaling") then
                changed, config.scales.l1 = imgui.slider_float("Low Rank 1P/Village", config.scales.l1, 0, 2)
                anyChange = changed and true or anyChange
                changed, config.scales.l2 = imgui.slider_float("Low Rank 2P", config.scales.l2, 0, 2)
                anyChange = changed and true or anyChange
                changed, config.scales.l3 = imgui.slider_float("Low Rank 3P", config.scales.l3, 0, 2)
                anyChange = changed and true or anyChange
                changed, config.scales.l4 = imgui.slider_float("Low Rank 4P", config.scales.l4, 0, 2)
                anyChange = changed and true or anyChange
                changed, config.scales.h1 = imgui.slider_float("High Rank 1P", config.scales.h1, 0, 2)
                anyChange = changed and true or anyChange
                changed, config.scales.h2 = imgui.slider_float("High Rank 2P", config.scales.h2, 0, 2)
                anyChange = changed and true or anyChange
                changed, config.scales.h3 = imgui.slider_float("High Rank 3P", config.scales.h3, 0, 2)
                anyChange = changed and true or anyChange
                changed, config.scales.h4 = imgui.slider_float("High Rank 4P", config.scales.h4, 0, 2)
                anyChange = changed and true or anyChange
                changed, config.scales.x1 = imgui.slider_float("High Rank EX 1P", config.scales.x1, 0, 2)
                anyChange = changed and true or anyChange
                changed, config.scales.x2 = imgui.slider_float("High Rank EX 2P", config.scales.x2, 0, 2)
                anyChange = changed and true or anyChange
                changed, config.scales.x3 = imgui.slider_float("High Rank EX 3P", config.scales.x3, 0, 2)
                anyChange = changed and true or anyChange
                changed, config.scales.x4 = imgui.slider_float("High Rank EX 4P", config.scales.x4, 0, 2)
                anyChange = changed and true or anyChange
                imgui.tree_pop()
            end

            imgui.tree_pop()
        end
        
        if imgui.tree_node("Debug Tools") then
            pressed = imgui.button("Village Bars")
            if pressed then
                savedMonsters = testBars.v.p1.monsters
                savedQuestData = testBars.v.p1.quests
            end
            
            pressed = imgui.button("LR 1P")
            if pressed then
                savedMonsters = testBars.l.p1.monsters
                savedQuestData = testBars.l.p1.quests
            end
            imgui.same_line()
            pressed = imgui.button("LR 2P")
            if pressed then
                savedMonsters = testBars.l.p2.monsters
                savedQuestData = testBars.l.p2.quests
            end
            imgui.same_line()
            pressed = imgui.button("LR 3P")
            if pressed then
                savedMonsters = testBars.l.p3.monsters
                savedQuestData = testBars.l.p3.quests
            end
            imgui.same_line()
            pressed = imgui.button("LR 4P")
            if pressed then
                savedMonsters = testBars.l.p4.monsters
                savedQuestData = testBars.l.p4.quests
            end
            
            pressed = imgui.button("HR 1P")
            if pressed then
                savedMonsters = testBars.h.p1.monsters
                savedQuestData = testBars.h.p1.quests
            end
            imgui.same_line()
            pressed = imgui.button("HR 2P")
            if pressed then
                savedMonsters = testBars.h.p2.monsters
                savedQuestData = testBars.h.p2.quests
            end
            imgui.same_line()
            pressed = imgui.button("HR 3P")
            if pressed then
                savedMonsters = testBars.h.p3.monsters
                savedQuestData = testBars.h.p3.quests
            end
            imgui.same_line()
            pressed = imgui.button("HR 4P")
            if pressed then
                savedMonsters = testBars.h.p4.monsters
                savedQuestData = testBars.h.p4.quests
            end
            
            pressed = imgui.button("Save Bars For Testing")
            if pressed then
                savedMonsters = monsters
                savedQuestData = questData
            end
            
            changed, useSavedBars = imgui.checkbox("Use Saved Bars", useSavedBars);
        end
        
        local pressed = imgui.button("Reset To Default Settings")
        if pressed then
            config = generateDefaultConfig()
            getWindowSize()
            scaleDefaultConfig(screenWidth, screenHeight)
            anyChange = true
        end

        imgui.tree_pop()

        if anyChange and json ~= nil then
            json.dump_file(configFile, config)
        end
    end
end);
    
re.on_frame(function()
    monsters, questData = monsterStatuses()

    if isDefaultConfig then
        getWindowSize()
        scaleDefaultConfig(screenWidth, screenHeight)
        json.dump_file(configFile, config)
        isDefaultConfig = false
    end

    if useSavedBars then
        monsters = savedMonsters
        questData = savedQuestData
    end

    if config.showDetailedWindow then 
        imgui.begin_window("Detailed Status Window")

        for key, value in spairs(monsters, function(t, a, b) return t[a].name < t[b].name end) do
            printMonsterStatus(value)
        end

        imgui.end_window()
    end


    if config.showEnemyStatusBars then
        local topPos = config.initialTopPos;

        for key, value in spairs(monsters, function(t, a, b) return t[a].name < t[b].name end) do
            topPos = drawMonsterStatus(key, value, topPos, questData) + config.monsterSpacing
        end
    end
end);

-- This is test data to help adjust display, jammed it here in the bottom so you can just ignore it
testBars.v.p1.monsters = { firstMon = { blast = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 60.0, wirebug = 720.0 }, isActive = false, isEnable = true, limits = { player = 120.0, wirebug = 180.0 }, maxLimits = { player = 720.0, wirebug = 900.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 120.0, wirebug = 180.0 }, name = "_BlastParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 5.0 }, fire = { activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 130.0, wirebug = 720.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 570.0, wirebug = 900.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_FireParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 9.0 }, flash = { activeCount = 0.0, activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 0.0, wirebug = 0.0 }, isActive = false, isEnable = true, limits = { player = 0.0, wirebug = 0.0 }, maxActiveCount = 7.0, maxLimits = { player = 0.0, wirebug = 0.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 0.0, wirebug = 0.0 }, name = "_FlashParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.0, wirebug = 0.0 }, type = 3.0 }, ice = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 130.0, wirebug = 720.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 570.0, wirebug = 900.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_IceParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 10.0 }, name = "Anjanath", paralyze = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 130.0, wirebug = 720.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 570.0, wirebug = 900.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ParalyzeParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 0.0 }, poison = { activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 100.0, wirebug = 900.0 }, isActive = false, isEnable = true, limits = { player = 150.0, wirebug = 180.0 }, maxLimits = { player = 550.0, wirebug = 1080.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_PoisonParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 4.0 }, sleep = { activeTime = 40.0, activeTimer = 0.0, addLimits = { player = 100.0, wirebug = 720.0 }, isActive = false, isEnable = true, limits = { player = 150.0, wirebug = 180.0 }, maxLimits = { player = 550.0, wirebug = 900.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_SleepParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 1.0 }, stamina = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 75.0, wirebug = 720.0 }, isActive = false, isEnable = true, limits = { player = 225.0, wirebug = 180.0 }, maxLimits = { player = 900.0, wirebug = 900.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 225.0, wirebug = 180.0 }, name = "_StaminaParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 6.0 }, stun = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 120.0, wirebug = 720.0 }, isActive = false, isEnable = true, limits = { player = 150.0, wirebug = 180.0 }, maxLimits = { player = 630.0, wirebug = 900.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_StunParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 2.0 }, thunder = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 130.0, wirebug = 720.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 570.0, wirebug = 900.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ThunderParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 11.0 }, water = { activeTime = 40.0, activeTimer = 0.0, addLimits = { player = 130.0, wirebug = 720.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 570.0, wirebug = 900.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_WaterParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 8.0 } }, secondMon = { blast = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 60.0, wirebug = 720.0 }, isActive = false, isEnable = true, limits = { player = 120.0, wirebug = 180.0 }, maxLimits = { player = 720.0, wirebug = 900.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 120.0, wirebug = 180.0 }, name = "_BlastParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 5.0 }, fire = { activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 130.0, wirebug = 720.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 570.0, wirebug = 900.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_FireParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 9.0 }, flash = { activeCount = 0.0, activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 0.0, wirebug = 0.0 }, isActive = false, isEnable = true, limits = { player = 0.0, wirebug = 0.0 }, maxActiveCount = 7.0, maxLimits = { player = 0.0, wirebug = 0.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 0.0, wirebug = 0.0 }, name = "_FlashParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.0, wirebug = 0.0 }, type = 3.0 }, ice = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 130.0, wirebug = 720.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 570.0, wirebug = 900.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_IceParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 10.0 }, name = "Rathian", paralyze = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 130.0, wirebug = 720.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 570.0, wirebug = 900.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ParalyzeParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 0.0 }, poison = { activeTime = 15.0, activeTimer = 0.0, addLimits = { player = 150.0, wirebug = 900.0 }, isActive = false, isEnable = true, limits = { player = 250.0, wirebug = 180.0 }, maxLimits = { player = 700.0, wirebug = 1080.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 250.0, wirebug = 180.0 }, name = "_PoisonParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 4.0 }, sleep = { activeTime = 40.0, activeTimer = 0.0, addLimits = { player = 100.0, wirebug = 720.0 }, isActive = false, isEnable = true, limits = { player = 150.0, wirebug = 180.0 }, maxLimits = { player = 550.0, wirebug = 900.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_SleepParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 1.0 }, stamina = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 75.0, wirebug = 720.0 }, isActive = false, isEnable = true, limits = { player = 225.0, wirebug = 180.0 }, maxLimits = { player = 900.0, wirebug = 900.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 225.0, wirebug = 180.0 }, name = "_StaminaParam", stocks = { player = 200.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 6.0 }, stun = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 120.0, wirebug = 120.0 }, isActive = false, isEnable = true, limits = { player = 110.0, wirebug = 150.0 }, maxLimits = { player = 630.0, wirebug = 630.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 110.0, wirebug = 150.0 }, name = "_StunParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 2.0 }, thunder = { activeTime = 40.0, activeTimer = 0.0, addLimits = { player = 130.0, wirebug = 720.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 570.0, wirebug = 900.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ThunderParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 11.0 }, water = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 130.0, wirebug = 720.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 570.0, wirebug = 900.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_WaterParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 8.0 } }, thirdMon = { blast = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 30.0, wirebug = 720.0 }, isActive = false, isEnable = true, limits = { player = 70.0, wirebug = 180.0 }, maxLimits = { player = 670.0, wirebug = 900.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 70.0, wirebug = 180.0 }, name = "_BlastParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 5.0 }, fire = { activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 130.0, wirebug = 720.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 570.0, wirebug = 900.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_FireParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 9.0 }, flash = { activeCount = 2.0, activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 0.0, wirebug = 0.0 }, isActive = false, isEnable = true, limits = { player = 0.0, wirebug = 0.0 }, maxActiveCount = 7.0, maxLimits = { player = 0.0, wirebug = 0.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 0.0, wirebug = 0.0 }, name = "_FlashParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.0, wirebug = 0.0 }, type = 3.0 }, ice = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 130.0, wirebug = 720.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 570.0, wirebug = 900.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_IceParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 10.0 }, name = "Kulu-Ya-Ku", paralyze = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 130.0, wirebug = 720.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 570.0, wirebug = 900.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ParalyzeParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 0.0 }, poison = { activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 100.0, wirebug = 900.0 }, isActive = false, isEnable = true, limits = { player = 250.0, wirebug = 180.0 }, maxLimits = { player = 550.0, wirebug = 1080.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_PoisonParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 4.0 }, sleep = { activeTime = 40.0, activeTimer = 32.6, addLimits = { player = 100.0, wirebug = 720.0 }, isActive = true, isEnable = true, limits = { player = 250.0, wirebug = 180.0 }, maxLimits = { player = 550.0, wirebug = 900.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_SleepParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 1.0 }, stamina = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 75.0, wirebug = 720.0 }, isActive = false, isEnable = true, limits = { player = 225.0, wirebug = 180.0 }, maxLimits = { player = 900.0, wirebug = 900.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 225.0, wirebug = 180.0 }, name = "_StaminaParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 6.0 }, stun = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 75.0, wirebug = 720.0 }, isActive = false, isEnable = true, limits = { player = 250.0, wirebug = 180.0 }, maxLimits = { player = 400.0, wirebug = 900.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 100.0, wirebug = 180.0 }, name = "_StunParam", stocks = { player = 230.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 2.0 }, thunder = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 130.0, wirebug = 720.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 570.0, wirebug = 900.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ThunderParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 11.0 }, water = { activeTime = 40.0, activeTimer = 0.0, addLimits = { player = 130.0, wirebug = 720.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 570.0, wirebug = 900.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_WaterParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 8.0 } }}
testBars.v.p1.quests = { active = true, questId = 505.0, questRank = 0.0, totalJoinNum = 1.0 }
testBars.l.p1.monsters = {   firstMon = {   blast = {   activeTime = 0.0,   activeTimer = 0.0,   addLimits = {   player = 63.0,   wirebug = 756.0   },   isActive = false,   isEnable = true,   limits = {   player = 120.0,   wirebug = 180.0   },   maxLimits = {   player = 756.0,   wirebug = 945.0   },   maxStocks = {   player = 0.0,   wirebug = 0.0   },   minActiveTime = 0.0,   minLimits = {   player = 120.0,   wirebug = 180.0   },   name = "_BlastParam",   stocks = {   player = 0.0,   wirebug = 0.0   },   subActiveTime = 0.0,   subs = {   player = 0.5,   wirebug = 0.5   },   type = 5.0   },   fire = {   activeTime = 20.0,   activeTimer = 0.0,   addLimits = {   player = 137.0,   wirebug = 756.0   },   isActive = false,   isEnable = true,   limits = {   player = 180.0,   wirebug = 180.0   },   maxLimits = {   player = 599.0,   wirebug = 945.0   },   maxStocks = {   player = 0.0,   wirebug = 0.0   },   minActiveTime = 0.0,   minLimits = {   player = 180.0,   wirebug = 180.0   },   name = "_FireParam",   stocks = {   player = 0.0,   wirebug = 0.0   },   subActiveTime = 0.0,   subs = {   player = 0.5,   wirebug = 0.5   },   type = 9.0   },   flash = {   activeCount = 2,   activeTime = 20.0,   activeTimer = 0.0,   addLimits = {   player = 0.0,   wirebug = 0.0   },   isActive = false,   isEnable = true,   limits = {   player = 0.0,   wirebug = 0.0   },   maxActiveCount = 7.0,   maxLimits = {   player = 0.0,   wirebug = 0.0   },   maxStocks = {   player = 0.0,   wirebug = 0.0   },   minActiveTime = 0.0,   minLimits = {   player = 0.0,   wirebug = 0.0   },   name = "_FlashParam",   stocks = {   player = 0.0,   wirebug = 0.0   },   subActiveTime = 0.0,   subs = {   player = 0.0,   wirebug = 0.0   },   type = 3.0   },   ice = {   activeTime = 30.0,   activeTimer = 0.0,   addLimits = {   player = 137.0,   wirebug = 756.0   },   isActive = false,   isEnable = true,   limits = {   player = 180.0,   wirebug = 180.0   },   maxLimits = {   player = 599.0,   wirebug = 945.0   },   maxStocks = {   player = 0.0,   wirebug = 0.0   },   minActiveTime = 0.0,   minLimits = {   player = 180.0,   wirebug = 180.0   },   name = "_IceParam",   stocks = {   player = 0.0,   wirebug = 0.0   },   subActiveTime = 0.0,   subs = {   player = 0.5,   wirebug = 0.5   },   type = 10.0   },   name = "Rathalos",   paralyze = {   activeTime = 10.0,   activeTimer = 0.0,   addLimits = {   player = 137.0,   wirebug = 756.0   },   isActive = false,   isEnable = true,   limits = {   player = 180.0,   wirebug = 180.0   },   maxLimits = {   player = 599.0,   wirebug = 945.0   },   maxStocks = {   player = 0.0,   wirebug = 0.0   },   minActiveTime = 0.0,   minLimits = {   player = 180.0,   wirebug = 180.0   },   name = "_ParalyzeParam",   stocks = {   player = 0.0,   wirebug = 0.0   },   subActiveTime = 0.0,   subs = {   player = 0.5,   wirebug = 0.5   },   type = 0.0   },   poison = {   activeTime = 15.0,   activeTimer = 0.0,   addLimits = {   player = 158.0,   wirebug = 945.0   },   isActive = false,   isEnable = true,   limits = {   player = 408.0,   wirebug = 180.0   },   maxLimits = {   player = 735.0,   wirebug = 1134.0   },   maxStocks = {   player = 0.0,   wirebug = 0.0   },   minActiveTime = 0.0,   minLimits = {   player = 250.0,   wirebug = 180.0   },   name = "_PoisonParam",   stocks = {   player = 0.0,   wirebug = 0.0   },   subActiveTime = 0.0,   subs = {   player = 0.5,   wirebug = 0.5   },   type = 4.0   },   sleep = {   activeTime = 40.0,   activeTimer = 32.6,   addLimits = {   player = 105.0,   wirebug = 756.0   },   isActive = true,   isEnable = true,   limits = {   player = 255.0,   wirebug = 180.0   },   maxLimits = {   player = 578.0,   wirebug = 945.0   },   maxStocks = {   player = 0.0,   wirebug = 0.0   },   minActiveTime = 0.0,   minLimits = {   player = 150.0,   wirebug = 180.0   },   name = "_SleepParam",   stocks = {   player = 0.0,   wirebug = 0.0   },   subActiveTime = 0.0,   subs = {   player = 0.5,   wirebug = 0.5   },   type = 1.0   },   stamina = {   activeTime = 0.0,   activeTimer = 0.0,   addLimits = {   player = 94.0,   wirebug = 900.0   },   isActive = false,   isEnable = true,   limits = {   player = 225.0,   wirebug = 180.0   },   maxLimits = {   player = 1125.0,   wirebug = 1125.0   },   maxStocks = {   player = 0.0,   wirebug = 0.0   },   minActiveTime = 0.0,   minLimits = {   player = 225.0,   wirebug = 180.0   },   name = "_StaminaParam",   stocks = {   player = 0.0,   wirebug = 0.0   },   subActiveTime = 0.0,   subs = {   player = 0.5,   wirebug = 0.5   },   type = 6.0   },   stun = {   activeTime = 10.0,   activeTimer = 0.0,   addLimits = {   player = 144.0,   wirebug = 864.0   },   isActive = false,   isEnable = true,   limits = {   player = 443.0,   wirebug = 180.0   },   maxLimits = {   player = 756.0,   wirebug = 1080.0   },   maxStocks = {   player = 0.0,   wirebug = 0.0   },   minActiveTime = 0.0,   minLimits = {   player = 150.0,   wirebug = 180.0   },   name = "_StunParam",   stocks = {   player = 410.0,   wirebug = 0.0   },   subActiveTime = 0.0,   subs = {   player = 0.5,   wirebug = 0.5   },   type = 2.0   },   thunder = {   activeTime = 40.0,   activeTimer = 0.0,   addLimits = {   player = 137.0,   wirebug = 756.0   },   isActive = false,   isEnable = true,   limits = {   player = 180.0,   wirebug = 180.0   },   maxLimits = {   player = 599.0,   wirebug = 945.0   },   maxStocks = {   player = 0.0,   wirebug = 0.0   },   minActiveTime = 0.0,   minLimits = {   player = 180.0,   wirebug = 180.0   },   name = "_ThunderParam",   stocks = {   player = 0.0,   wirebug = 0.0   },   subActiveTime = 0.0,   subs = {   player = 0.5,   wirebug = 0.5   },   type = 11.0   },   water = {   activeTime = 30.0,   activeTimer = 0.0,   addLimits = {   player = 137.0,   wirebug = 756.0   },   isActive = false,   isEnable = true,   limits = {   player = 180.0,   wirebug = 180.0   },   maxLimits = {   player = 599.0,   wirebug = 945.0   },   maxStocks = {   player = 0.0,   wirebug = 0.0   },   minActiveTime = 0.0,   minLimits = {   player = 180.0,   wirebug = 180.0   },   name = "_WaterParam",   stocks = {   player = 0.0,   wirebug = 0.0   },   subActiveTime = 0.0,   subs = {   player = 0.5,   wirebug = 0.5   },   type = 8.0   }   },   secondMon = {   blast = {   activeTime = 0.0,   activeTimer = 0.0,   addLimits = {   player = 94.0,   wirebug = 756.0   },   isActive = false,   isEnable = true,   limits = {   player = 120.0,   wirebug = 180.0   },   maxLimits = {   player = 2016.0,   wirebug = 945.0   },   maxStocks = {   player = 0.0,   wirebug = 0.0   },   minActiveTime = 0.0,   minLimits = {   player = 120.0,   wirebug = 180.0   },   name = "_BlastParam",   stocks = {   player = 0.0,   wirebug = 0.0   },   subActiveTime = 0.0,   subs = {   player = 0.5,   wirebug = 0.5   },   type = 5.0   },   fire = {   activeTime = 20.0,   activeTimer = 0.0,   addLimits = {   player = 137.0,   wirebug = 756.0   },   isActive = false,   isEnable = true,   limits = {   player = 180.0,   wirebug = 180.0   },   maxLimits = {   player = 599.0,   wirebug = 945.0   },   maxStocks = {   player = 0.0,   wirebug = 0.0   },   minActiveTime = 0.0,   minLimits = {   player = 180.0,   wirebug = 180.0   },   name = "_FireParam",   stocks = {   player = 0.0,   wirebug = 0.0   },   subActiveTime = 0.0,   subs = {   player = 0.5,   wirebug = 0.5   },   type = 9.0   },   flash = {   activeCount = 0.0,   activeTime = 20.0,   activeTimer = 0.0,   addLimits = {   player = 0.0,   wirebug = 0.0   },   isActive = false,   isEnable = true,   limits = {   player = 0.0,   wirebug = 0.0   },   maxActiveCount = 7.0,   maxLimits = {   player = 0.0,   wirebug = 0.0   },   maxStocks = {   player = 0.0,   wirebug = 0.0   },   minActiveTime = 0.0,   minLimits = {   player = 0.0,   wirebug = 0.0   },   name = "_FlashParam",   stocks = {   player = 0.0,   wirebug = 0.0   },   subActiveTime = 0.0,   subs = {   player = 0.0,   wirebug = 0.0   },   type = 3.0   },   ice = {   activeTime = 30.0,   activeTimer = 0.0,   addLimits = {   player = 137.0,   wirebug = 756.0   },   isActive = false,   isEnable = true,   limits = {   player = 180.0,   wirebug = 180.0   },   maxLimits = {   player = 599.0,   wirebug = 945.0   },   maxStocks = {   player = 0.0,   wirebug = 0.0   },   minActiveTime = 0.0,   minLimits = {   player = 180.0,   wirebug = 180.0   },   name = "_IceParam",   stocks = {   player = 0.0,   wirebug = 0.0   },   subActiveTime = 0.0,   subs = {   player = 0.5,   wirebug = 0.5   },   type = 10.0   },   name = "Magnamalo",   paralyze = {   activeTime = 10.0,   activeTimer = 0.0,   addLimits = {   player = 137.0,   wirebug = 756.0   },   isActive = false,   isEnable = true,   limits = {   player = 180.0,   wirebug = 180.0   },   maxLimits = {   player = 599.0,   wirebug = 945.0   },   maxStocks = {   player = 0.0,   wirebug = 0.0   },   minActiveTime = 0.0,   minLimits = {   player = 180.0,   wirebug = 180.0   },   name = "_ParalyzeParam",   stocks = {   player = 0.0,   wirebug = 0.0   },   subActiveTime = 0.0,   subs = {   player = 0.5,   wirebug = 0.5   },   type = 0.0   },   poison = {   activeTime = 20.0,   activeTimer = 0.0,   addLimits = {   player = 105.0,   wirebug = 945.0   },   isActive = false,   isEnable = true,   limits = {   player = 150.0,   wirebug = 180.0   },   maxLimits = {   player = 578.0,   wirebug = 1134.0   },   maxStocks = {   player = 0.0,   wirebug = 0.0   },   minActiveTime = 0.0,   minLimits = {   player = 150.0,   wirebug = 180.0   },   name = "_PoisonParam",   stocks = {   player = 0.0,   wirebug = 0.0   },   subActiveTime = 0.0,   subs = {   player = 0.5,   wirebug = 0.5   },   type = 4.0   },   sleep = {   activeTime = 40.0,   activeTimer = 0.0,   addLimits = {   player = 105.0,   wirebug = 756.0   },   isActive = false,   isEnable = true,   limits = {   player = 150.0,   wirebug = 180.0   },   maxLimits = {   player = 578.0,   wirebug = 945.0   },   maxStocks = {   player = 0.0,   wirebug = 0.0   },   minActiveTime = 0.0,   minLimits = {   player = 150.0,   wirebug = 180.0   },   name = "_SleepParam",   stocks = {   player = 0.0,   wirebug = 0.0   },   subActiveTime = 0.0,   subs = {   player = 0.5,   wirebug = 0.5   },   type = 1.0   },   stamina = {   activeTime = 0.0,   activeTimer = 0.0,   addLimits = {   player = 94.0,   wirebug = 900.0   },   isActive = false,   isEnable = true,   limits = {   player = 225.0,   wirebug = 180.0   },   maxLimits = {   player = 1125.0,   wirebug = 1125.0   },   maxStocks = {   player = 0.0,   wirebug = 0.0   },   minActiveTime = 0.0,   minLimits = {   player = 225.0,   wirebug = 180.0   },   name = "_StaminaParam",   stocks = {   player = 200.0,   wirebug = 0.0   },   subActiveTime = 0.0,   subs = {   player = 0.5,   wirebug = 0.5   },   type = 6.0   },   stun = {   activeTime = 10.0,   activeTimer = 0.0,   addLimits = {   player = 144.0,   wirebug = 864.0   },   isActive = false,   isEnable = true,   limits = {   player = 150.0,   wirebug = 180.0   },   maxLimits = {   player = 756.0,   wirebug = 1080.0   },   maxStocks = {   player = 0.0,   wirebug = 0.0   },   minActiveTime = 0.0,   minLimits = {   player = 150.0,   wirebug = 180.0   },   name = "_StunParam",   stocks = {   player = 0.0,   wirebug = 0.0   },   subActiveTime = 0.0,   subs = {   player = 0.5,   wirebug = 0.5   },   type = 2.0   },   thunder = {   activeTime = 30.0,   activeTimer = 0.0,   addLimits = {   player = 137.0,   wirebug = 756.0   },   isActive = false,   isEnable = true,   limits = {   player = 180.0,   wirebug = 180.0   },   maxLimits = {   player = 599.0,   wirebug = 945.0   },   maxStocks = {   player = 0.0,   wirebug = 0.0   },   minActiveTime = 0.0,   minLimits = {   player = 180.0,   wirebug = 180.0   },   name = "_ThunderParam",   stocks = {   player = 0.0,   wirebug = 0.0   },   subActiveTime = 0.0,   subs = {   player = 0.5,   wirebug = 0.5   },   type = 11.0   },   water = {   activeTime = 40.0,   activeTimer = 0.0,   addLimits = {   player = 137.0,   wirebug = 756.0   },   isActive = false,   isEnable = true,   limits = {   player = 180.0,   wirebug = 180.0   },   maxLimits = {   player = 599.0,   wirebug = 945.0   },   maxStocks = {   player = 0.0,   wirebug = 0.0   },   minActiveTime = 0.0,   minLimits = {   player = 180.0,   wirebug = 180.0   },   name = "_WaterParam",   stocks = {   player = 0.0,   wirebug = 0.0   },   subActiveTime = 0.0,   subs = {   player = 0.5,   wirebug = 0.5   },   type = 8.0   }   },   thirdMon = {   blast = {   activeTime = 0.0,   activeTimer = 0.0,   addLimits = {   player = 31.0,   wirebug = 756.0   },   isActive = false,   isEnable = true,   limits = {   player = 70.0,   wirebug = 180.0   },   maxLimits = {   player = 703.0,   wirebug = 945.0   },   maxStocks = {   player = 0.0,   wirebug = 0.0   },   minActiveTime = 0.0,   minLimits = {   player = 70.0,   wirebug = 180.0   },   name = "_BlastParam",   stocks = {   player = 0.0,   wirebug = 0.0   },   subActiveTime = 0.0,   subs = {   player = 0.5,   wirebug = 0.5   },   type = 5.0   },   fire = {   activeTime = 20.0,   activeTimer = 0.0,   addLimits = {   player = 137.0,   wirebug = 756.0   },   isActive = false,   isEnable = true,   limits = {   player = 180.0,   wirebug = 180.0   },   maxLimits = {   player = 599.0,   wirebug = 945.0   },   maxStocks = {   player = 0.0,   wirebug = 0.0   },   minActiveTime = 0.0,   minLimits = {   player = 180.0,   wirebug = 180.0   },   name = "_FireParam",   stocks = {   player = 0.0,   wirebug = 0.0   },   subActiveTime = 0.0,   subs = {   player = 0.5,   wirebug = 0.5   },   type = 9.0   },   flash = {   activeCount = 0.0,   activeTime = 20.0,   activeTimer = 0.0,   addLimits = {   player = 0.0,   wirebug = 0.0   },   isActive = false,   isEnable = true,   limits = {   player = 0.0,   wirebug = 0.0   },   maxActiveCount = 7.0,   maxLimits = {   player = 0.0,   wirebug = 0.0   },   maxStocks = {   player = 0.0,   wirebug = 0.0   },   minActiveTime = 0.0,   minLimits = {   player = 0.0,   wirebug = 0.0   },   name = "_FlashParam",   stocks = {   player = 0.0,   wirebug = 0.0   },   subActiveTime = 0.0,   subs = {   player = 0.0,   wirebug = 0.0   },   type = 3.0   },   ice = {   activeTime = 40.0,   activeTimer = 0.0,   addLimits = {   player = 137.0,   wirebug = 756.0   },   isActive = false,   isEnable = true,   limits = {   player = 180.0,   wirebug = 180.0   },   maxLimits = {   player = 599.0,   wirebug = 945.0   },   maxStocks = {   player = 0.0,   wirebug = 0.0   },   minActiveTime = 0.0,   minLimits = {   player = 180.0,   wirebug = 180.0   },   name = "_IceParam",   stocks = {   player = 0.0,   wirebug = 0.0   },   subActiveTime = 0.0,   subs = {   player = 0.5,   wirebug = 0.5   },   type = 10.0   },   name = "Great Wroggi",   paralyze = {   activeTime = 12.0,   activeTimer = 0.0,   addLimits = {   player = 79.0,   wirebug = 756.0   },   isActive = false,   isEnable = true,   limits = {   player = 130.0,   wirebug = 180.0   },   maxLimits = {   player = 451.0,   wirebug = 945.0   },   maxStocks = {   player = 0.0,   wirebug = 0.0   },   minActiveTime = 0.0,   minLimits = {   player = 130.0,   wirebug = 180.0   },   name = "_ParalyzeParam",   stocks = {   player = 0.0,   wirebug = 0.0   },   subActiveTime = 0.0,   subs = {   player = 0.5,   wirebug = 0.5   },   type = 0.0   },   poison = {   activeTime = 15.0,   activeTimer = 0.0,   addLimits = {   player = 158.0,   wirebug = 945.0   },   isActive = false,   isEnable = true,   limits = {   player = 250.0,   wirebug = 180.0   },   maxLimits = {   player = 735.0,   wirebug = 1134.0   },   maxStocks = {   player = 0.0,   wirebug = 0.0   },   minActiveTime = 0.0,   minLimits = {   player = 250.0,   wirebug = 180.0   },   name = "_PoisonParam",   stocks = {   player = 0.0,   wirebug = 0.0   },   subActiveTime = 0.0,   subs = {   player = 0.5,   wirebug = 0.5   },   type = 4.0   },   sleep = {   activeTime = 40.0,   activeTimer = 0.0,   addLimits = {   player = 94.0,   wirebug = 756.0   },   isActive = false,   isEnable = true,   limits = {   player = 100.0,   wirebug = 180.0   },   maxLimits = {   player = 388.0,   wirebug = 945.0   },   maxStocks = {   player = 0.0,   wirebug = 0.0   },   minActiveTime = 0.0,   minLimits = {   player = 100.0,   wirebug = 180.0   },   name = "_SleepParam",   stocks = {   player = 0.0,   wirebug = 0.0   },   subActiveTime = 0.0,   subs = {   player = 0.5,   wirebug = 0.5   },   type = 1.0   },   stamina = {   activeTime = 0.0,   activeTimer = 0.0,   addLimits = {   player = 94.0,   wirebug = 900.0   },   isActive = false,   isEnable = true,   limits = {   player = 225.0,   wirebug = 180.0   },   maxLimits = {   player = 1125.0,   wirebug = 1125.0   },   maxStocks = {   player = 0.0,   wirebug = 0.0   },   minActiveTime = 0.0,   minLimits = {   player = 225.0,   wirebug = 180.0   },   name = "_StaminaParam",   stocks = {   player = 0.0,   wirebug = 0.0   },   subActiveTime = 0.0,   subs = {   player = 0.5,   wirebug = 0.5   },   type = 6.0   },   stun = {   activeTime = 10.0,   activeTimer = 0.0,   addLimits = {   player = 120.0,   wirebug = 864.0   },   isActive = false,   isEnable = true,   limits = {   player = 120.0,   wirebug = 180.0   },   maxLimits = {   player = 624.0,   wirebug = 1080.0   },   maxStocks = {   player = 0.0,   wirebug = 0.0   },   minActiveTime = 0.0,   minLimits = {   player = 120.0,   wirebug = 180.0   },   name = "_StunParam",   stocks = {   player = 0.0,   wirebug = 0.0   },   subActiveTime = 0.0,   subs = {   player = 0.5,   wirebug = 0.5   },   type = 2.0   },   thunder = {   activeTime = 30.0,   activeTimer = 0.0,   addLimits = {   player = 137.0,   wirebug = 756.0   },   isActive = false,   isEnable = true,   limits = {   player = 180.0,   wirebug = 180.0   },   maxLimits = {   player = 599.0,   wirebug = 945.0   },   maxStocks = {   player = 0.0,   wirebug = 0.0   },   minActiveTime = 0.0,   minLimits = {   player = 180.0,   wirebug = 180.0   },   name = "_ThunderParam",   stocks = {   player = 0.0,   wirebug = 0.0   },   subActiveTime = 0.0,   subs = {   player = 0.5,   wirebug = 0.5   },   type = 11.0   },   water = {   activeTime = 30.0,   activeTimer = 0.0,   addLimits = {   player = 137.0,   wirebug = 756.0   },   isActive = false,   isEnable = true,   limits = {   player = 180.0,   wirebug = 180.0   },   maxLimits = {   player = 599.0,   wirebug = 945.0   },   maxStocks = {   player = 0.0,   wirebug = 0.0   },   minActiveTime = 0.0,   minLimits = {   player = 180.0,   wirebug = 180.0   },   name = "_WaterParam",   stocks = {   player = 0.0,   wirebug = 0.0   },   subActiveTime = 0.0,   subs = {   player = 0.5,   wirebug = 0.5   },   type = 8.0   }   }  }
testBars.l.p1.quests  = { active = true, questId = 10310.0, questRank = 0.0, totalJoinNum = 1.0 }
testBars.l.p2.monsters =  { firstMon = { blast = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 60.0, wirebug = 1440.0 }, isActive = false, isEnable = true, limits = { player = 70.0, wirebug = 180.0 }, maxLimits = { player = 1340.0, wirebug = 1800.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 70.0, wirebug = 180.0 }, name = "_BlastParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 5.0 }, fire = { activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 208.0, wirebug = 1152.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 912.0, wirebug = 1440.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_FireParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 9.0 }, flash = { activeCount = 2.0, activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 0.0, wirebug = 0.0 }, isActive = false, isEnable = true, limits = { player = 0.0, wirebug = 0.0 }, maxActiveCount = 7.0, maxLimits = { player = 0.0, wirebug = 0.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 0.0, wirebug = 0.0 }, name = "_FlashParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.0, wirebug = 0.0 }, type = 3.0 }, ice = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 208.0, wirebug = 1152.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 912.0, wirebug = 1440.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_IceParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 10.0 }, name = "Basarios", paralyze = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 260.0, wirebug = 1440.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1140.0, wirebug = 1800.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ParalyzeParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 0.0 }, poison = { activeTime = 35.0, activeTimer = 0.0, addLimits = { player = 260.0, wirebug = 1800.0 }, isActive = false, isEnable = true, limits = { player = 700.0, wirebug = 180.0 }, maxLimits = { player = 1140.0, wirebug = 2160.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_PoisonParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 4.0 }, sleep = { activeTime = 40.0, activeTimer = 32.6, addLimits = { player = 180.0, wirebug = 1440.0 }, isActive = true, isEnable = true, limits = { player = 460.0, wirebug = 180.0 }, maxLimits = { player = 740.0, wirebug = 1800.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 100.0, wirebug = 180.0 }, name = "_SleepParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 1.0 }, stamina = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 180.0, wirebug = 1728.0 }, isActive = false, isEnable = true, limits = { player = 225.0, wirebug = 180.0 }, maxLimits = { player = 2160.0, wirebug = 2160.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 225.0, wirebug = 180.0 }, name = "_StaminaParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 6.0 }, stun = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 264.0, wirebug = 1584.0 }, isActive = false, isEnable = true, limits = { player = 678.0, wirebug = 180.0 }, maxLimits = { player = 1386.0, wirebug = 1980.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_StunParam", stocks = { player = 620.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 2.0 }, thunder = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 208.0, wirebug = 1152.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 912.0, wirebug = 1440.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ThunderParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 11.0 }, water = { activeTime = 40.0, activeTimer = 0.0, addLimits = { player = 208.0, wirebug = 1152.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 912.0, wirebug = 1440.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_WaterParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 8.0 } }, secondMon = { blast = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 40.0, wirebug = 1440.0 }, isActive = false, isEnable = true, limits = { player = 40.0, wirebug = 180.0 }, maxLimits = { player = 880.0, wirebug = 1800.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 40.0, wirebug = 180.0 }, name = "_BlastParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 5.0 }, fire = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 208.0, wirebug = 1152.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 912.0, wirebug = 1440.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_FireParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 9.0 }, flash = { activeCount = 0.0, activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 0.0, wirebug = 0.0 }, isActive = false, isEnable = true, limits = { player = 0.0, wirebug = 0.0 }, maxActiveCount = 7.0, maxLimits = { player = 0.0, wirebug = 0.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 0.0, wirebug = 0.0 }, name = "_FlashParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.0, wirebug = 0.0 }, type = 3.0 }, ice = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 208.0, wirebug = 1152.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 912.0, wirebug = 1440.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_IceParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 10.0 }, name = "Barroth", paralyze = { activeTime = 12.0, activeTimer = 0.0, addLimits = { player = 150.0, wirebug = 1440.0 }, isActive = false, isEnable = true, limits = { player = 130.0, wirebug = 180.0 }, maxLimits = { player = 860.0, wirebug = 1800.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 130.0, wirebug = 180.0 }, name = "_ParalyzeParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 0.0 }, poison = { activeTime = 25.0, activeTimer = 0.0, addLimits = { player = 180.0, wirebug = 1800.0 }, isActive = false, isEnable = true, limits = { player = 100.0, wirebug = 180.0 }, maxLimits = { player = 740.0, wirebug = 2160.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 100.0, wirebug = 180.0 }, name = "_PoisonParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 4.0 }, sleep = { activeTime = 40.0, activeTimer = 0.0, addLimits = { player = 200.0, wirebug = 1440.0 }, isActive = false, isEnable = true, limits = { player = 150.0, wirebug = 180.0 }, maxLimits = { player = 1100.0, wirebug = 1800.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_SleepParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 1.0 }, stamina = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 180.0, wirebug = 1728.0 }, isActive = false, isEnable = true, limits = { player = 225.0, wirebug = 180.0 }, maxLimits = { player = 2160.0, wirebug = 2160.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 225.0, wirebug = 180.0 }, name = "_StaminaParam", stocks = { player = 200.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 6.0 }, stun = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 330.0, wirebug = 1584.0 }, isActive = false, isEnable = true, limits = { player = 200.0, wirebug = 180.0 }, maxLimits = { player = 1760.0, wirebug = 1980.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 200.0, wirebug = 180.0 }, name = "_StunParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 2.0 }, thunder = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 208.0, wirebug = 1152.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 912.0, wirebug = 1440.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ThunderParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 11.0 }, water = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 208.0, wirebug = 1152.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 912.0, wirebug = 1440.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_WaterParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 8.0 } }, thirdMon = { blast = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 60.0, wirebug = 1440.0 }, isActive = false, isEnable = true, limits = { player = 70.0, wirebug = 180.0 }, maxLimits = { player = 1340.0, wirebug = 1800.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 70.0, wirebug = 180.0 }, name = "_BlastParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 5.0 }, fire = { activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 208.0, wirebug = 1152.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 912.0, wirebug = 1440.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_FireParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 9.0 }, flash = { activeCount = 0.0, activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 0.0, wirebug = 0.0 }, isActive = false, isEnable = true, limits = { player = 0.0, wirebug = 0.0 }, maxActiveCount = 7.0, maxLimits = { player = 0.0, wirebug = 0.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 0.0, wirebug = 0.0 }, name = "_FlashParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.0, wirebug = 0.0 }, type = 3.0 }, ice = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 208.0, wirebug = 1152.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 912.0, wirebug = 1440.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_IceParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 10.0 }, name = "Kulu-Ya-Ku", paralyze = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 260.0, wirebug = 1440.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1140.0, wirebug = 1800.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ParalyzeParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 0.0 }, poison = { activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 200.0, wirebug = 1800.0 }, isActive = false, isEnable = true, limits = { player = 150.0, wirebug = 180.0 }, maxLimits = { player = 1100.0, wirebug = 2160.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_PoisonParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 4.0 }, sleep = { activeTime = 40.0, activeTimer = 0.0, addLimits = { player = 200.0, wirebug = 1440.0 }, isActive = false, isEnable = true, limits = { player = 150.0, wirebug = 180.0 }, maxLimits = { player = 1100.0, wirebug = 1800.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_SleepParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 1.0 }, stamina = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 180.0, wirebug = 1728.0 }, isActive = false, isEnable = true, limits = { player = 225.0, wirebug = 180.0 }, maxLimits = { player = 2160.0, wirebug = 2160.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 225.0, wirebug = 180.0 }, name = "_StaminaParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 6.0 }, stun = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 165.0, wirebug = 1584.0 }, isActive = false, isEnable = true, limits = { player = 100.0, wirebug = 180.0 }, maxLimits = { player = 880.0, wirebug = 1980.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 100.0, wirebug = 180.0 }, name = "_StunParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 2.0 }, thunder = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 208.0, wirebug = 1152.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 912.0, wirebug = 1440.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ThunderParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 11.0 }, water = { activeTime = 40.0, activeTimer = 0.0, addLimits = { player = 208.0, wirebug = 1152.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 912.0, wirebug = 1440.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_WaterParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 8.0 } }}
testBars.l.p2.quests = { active = true, questId = 10210.0, questRank = 0.0, totalJoinNum = 2.0 }
testBars.l.p3.monsters = { firstMon = { blast = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 189.0, wirebug = 2268.0 }, isActive = false, isEnable = true, limits = { player = 120.0, wirebug = 180.0 }, maxLimits = { player = 2268.0, wirebug = 2835.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 120.0, wirebug = 180.0 }, name = "_BlastParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 5.0 }, fire = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 287.0, wirebug = 1588.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1257.0, wirebug = 1984.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_FireParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 9.0 }, flash = { activeCount = 2.0, activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 0.0, wirebug = 0.0 }, isActive = false, isEnable = true, limits = { player = 0.0, wirebug = 0.0 }, maxActiveCount = 7.0, maxLimits = { player = 0.0, wirebug = 0.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 0.0, wirebug = 0.0 }, name = "_FlashParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.0, wirebug = 0.0 }, type = 3.0 }, ice = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 287.0, wirebug = 1588.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1257.0, wirebug = 1984.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_IceParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 10.0 }, name = "Almudron", paralyze = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 409.0, wirebug = 2268.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1795.0, wirebug = 2835.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ParalyzeParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 0.0 }, poison = { activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 315.0, wirebug = 2835.0 }, isActive = false, isEnable = true, limits = { player = 780.0, wirebug = 180.0 }, maxLimits = { player = 1732.0, wirebug = 3402.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_PoisonParam", stocks = { player = 49.39905548095703, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 4.0 }, sleep = { activeTime = 40.0, activeTimer = 32.6, addLimits = { player = 315.0, wirebug = 2268.0 }, isActive = true, isEnable = true, limits = { player = 465.0, wirebug = 180.0 }, maxLimits = { player = 1732.0, wirebug = 2835.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_SleepParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 1.0 }, stamina = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 281.0, wirebug = 2700.0 }, isActive = false, isEnable = true, limits = { player = 225.0, wirebug = 180.0 }, maxLimits = { player = 3375.0, wirebug = 3375.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 225.0, wirebug = 180.0 }, name = "_StaminaParam", stocks = { player = 9.399070739746094, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 6.0 }, stun = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 432.0, wirebug = 2592.0 }, isActive = false, isEnable = true, limits = { player = 1014.0, wirebug = 180.0 }, maxLimits = { player = 2268.0, wirebug = 3240.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_StunParam", stocks = { player = 970.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 2.0 }, thunder = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 287.0, wirebug = 1588.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1257.0, wirebug = 1984.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ThunderParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 11.0 }, water = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 287.0, wirebug = 1588.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1257.0, wirebug = 1984.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_WaterParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 8.0 } }, secondMon = { blast = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 94.0, wirebug = 2268.0 }, isActive = false, isEnable = true, limits = { player = 70.0, wirebug = 180.0 }, maxLimits = { player = 2111.0, wirebug = 2835.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 70.0, wirebug = 180.0 }, name = "_BlastParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 5.0 }, fire = { activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 287.0, wirebug = 1588.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1257.0, wirebug = 1984.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_FireParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 9.0 }, flash = { activeCount = 0.0, activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 0.0, wirebug = 0.0 }, isActive = false, isEnable = true, limits = { player = 0.0, wirebug = 0.0 }, maxActiveCount = 7.0, maxLimits = { player = 0.0, wirebug = 0.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 0.0, wirebug = 0.0 }, name = "_FlashParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.0, wirebug = 0.0 }, type = 3.0 }, ice = { activeTime = 40.0, activeTimer = 0.0, addLimits = { player = 287.0, wirebug = 1588.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1257.0, wirebug = 1984.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_IceParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 10.0 }, name = "Diablos", paralyze = { activeTime = 7.0, activeTimer = 0.0, addLimits = { player = 158.0, wirebug = 2268.0 }, isActive = false, isEnable = true, limits = { player = 100.0, wirebug = 180.0 }, maxLimits = { player = 945.0, wirebug = 2835.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 100.0, wirebug = 180.0 }, name = "_ParalyzeParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 0.0 }, poison = { activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 315.0, wirebug = 2835.0 }, isActive = false, isEnable = true, limits = { player = 150.0, wirebug = 180.0 }, maxLimits = { player = 1732.0, wirebug = 3402.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_PoisonParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 4.0 }, sleep = { activeTime = 40.0, activeTimer = 0.0, addLimits = { player = 315.0, wirebug = 2268.0 }, isActive = false, isEnable = true, limits = { player = 150.0, wirebug = 180.0 }, maxLimits = { player = 1732.0, wirebug = 2835.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_SleepParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 1.0 }, stamina = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 281.0, wirebug = 2700.0 }, isActive = false, isEnable = true, limits = { player = 225.0, wirebug = 180.0 }, maxLimits = { player = 3375.0, wirebug = 3375.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 225.0, wirebug = 180.0 }, name = "_StaminaParam", stocks = { player = 200.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 6.0 }, stun = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 540.0, wirebug = 2592.0 }, isActive = false, isEnable = true, limits = { player = 200.0, wirebug = 180.0 }, maxLimits = { player = 2880.0, wirebug = 3240.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 200.0, wirebug = 180.0 }, name = "_StunParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 2.0 }, thunder = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 287.0, wirebug = 1588.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1257.0, wirebug = 1984.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ThunderParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 11.0 }, water = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 287.0, wirebug = 1588.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1257.0, wirebug = 1984.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_WaterParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 8.0 } }, thirdMon = { blast = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 189.0, wirebug = 2268.0 }, isActive = false, isEnable = true, limits = { player = 120.0, wirebug = 180.0 }, maxLimits = { player = 2268.0, wirebug = 2835.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 120.0, wirebug = 180.0 }, name = "_BlastParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 5.0 }, fire = { activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 287.0, wirebug = 1588.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1257.0, wirebug = 1984.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_FireParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 9.0 }, flash = { activeCount = 0.0, activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 0.0, wirebug = 0.0 }, isActive = false, isEnable = true, limits = { player = 0.0, wirebug = 0.0 }, maxActiveCount = 7.0, maxLimits = { player = 0.0, wirebug = 0.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 0.0, wirebug = 0.0 }, name = "_FlashParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.0, wirebug = 0.0 }, type = 3.0 }, ice = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 287.0, wirebug = 1588.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1257.0, wirebug = 1984.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_IceParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 10.0 }, name = "Anjanath", paralyze = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 409.0, wirebug = 2268.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1795.0, wirebug = 2835.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ParalyzeParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 0.0 }, poison = { activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 315.0, wirebug = 2835.0 }, isActive = false, isEnable = true, limits = { player = 150.0, wirebug = 180.0 }, maxLimits = { player = 1732.0, wirebug = 3402.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_PoisonParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 4.0 }, sleep = { activeTime = 40.0, activeTimer = 0.0, addLimits = { player = 315.0, wirebug = 2268.0 }, isActive = false, isEnable = true, limits = { player = 150.0, wirebug = 180.0 }, maxLimits = { player = 1732.0, wirebug = 2835.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_SleepParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 1.0 }, stamina = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 281.0, wirebug = 2700.0 }, isActive = false, isEnable = true, limits = { player = 225.0, wirebug = 180.0 }, maxLimits = { player = 3375.0, wirebug = 3375.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 225.0, wirebug = 180.0 }, name = "_StaminaParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 6.0 }, stun = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 432.0, wirebug = 2592.0 }, isActive = false, isEnable = true, limits = { player = 150.0, wirebug = 180.0 }, maxLimits = { player = 2268.0, wirebug = 3240.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_StunParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 2.0 }, thunder = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 287.0, wirebug = 1588.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1257.0, wirebug = 1984.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ThunderParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 11.0 }, water = { activeTime = 40.0, activeTimer = 0.0, addLimits = { player = 287.0, wirebug = 1588.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1257.0, wirebug = 1984.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_WaterParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 8.0 } }}
testBars.l.p3.quests = { active = true, questId = 10306.0, questRank = 0.0, totalJoinNum = 3.0 }

testBars.l.p4.monsters = { firstMon = { blast = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 252.0, wirebug = 3024.0 }, isActive = false, isEnable = true, limits = { player = 372.0, wirebug = 180.0 }, maxLimits = { player = 3024.0, wirebug = 3780.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 120.0, wirebug = 180.0 }, name = "_BlastParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 5.0 }, fire = { activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 328.0, wirebug = 1814.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1436.0, wirebug = 2268.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_FireParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 9.0 }, flash = { activeCount = 2.0, activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 0.0, wirebug = 0.0 }, isActive = false, isEnable = true, limits = { player = 0.0, wirebug = 0.0 }, maxActiveCount = 7.0, maxLimits = { player = 0.0, wirebug = 0.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 0.0, wirebug = 0.0 }, name = "_FlashParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.0, wirebug = 0.0 }, type = 3.0 }, ice = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 328.0, wirebug = 1814.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1436.0, wirebug = 2268.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_IceParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 10.0 }, name = "Anjanath", paralyze = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 546.0, wirebug = 3024.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 2394.0, wirebug = 3780.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ParalyzeParam", stocks = { player = 114.628173828125, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 0.0 }, poison = { activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 420.0, wirebug = 3780.0 }, isActive = false, isEnable = true, limits = { player = 990.0, wirebug = 180.0 }, maxLimits = { player = 2310.0, wirebug = 4536.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_PoisonParam", stocks = { player = 14.628154754638672, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 4.0 }, sleep = { activeTime = 40.0, activeTimer = 32.6, addLimits = { player = 420.0, wirebug = 3024.0 }, isActive = true, isEnable = true, limits = { player = 570.0, wirebug = 180.0 }, maxLimits = { player = 2310.0, wirebug = 3780.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_SleepParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 1.0 }, stamina = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 375.0, wirebug = 3600.0 }, isActive = false, isEnable = true, limits = { player = 225.0, wirebug = 180.0 }, maxLimits = { player = 4500.0, wirebug = 4500.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 225.0, wirebug = 180.0 }, name = "_StaminaParam", stocks = { player = 137.62818908691406, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 6.0 }, stun = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 576.0, wirebug = 3456.0 }, isActive = false, isEnable = true, limits = { player = 1302.0, wirebug = 180.0 }, maxLimits = { player = 3024.0, wirebug = 4320.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_StunParam", stocks = { player = 1200.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 2.0 }, thunder = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 328.0, wirebug = 1814.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1436.0, wirebug = 2268.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ThunderParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 11.0 }, water = { activeTime = 40.0, activeTimer = 0.0, addLimits = { player = 328.0, wirebug = 1814.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1436.0, wirebug = 2268.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_WaterParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 8.0 } }, secondMon = { blast = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 126.0, wirebug = 3024.0 }, isActive = false, isEnable = true, limits = { player = 70.0, wirebug = 180.0 }, maxLimits = { player = 2814.0, wirebug = 3780.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 70.0, wirebug = 180.0 }, name = "_BlastParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 5.0 }, fire = { activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 328.0, wirebug = 1814.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1436.0, wirebug = 2268.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_FireParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 9.0 }, flash = { activeCount = 0.0, activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 0.0, wirebug = 0.0 }, isActive = false, isEnable = true, limits = { player = 0.0, wirebug = 0.0 }, maxActiveCount = 7.0, maxLimits = { player = 0.0, wirebug = 0.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 0.0, wirebug = 0.0 }, name = "_FlashParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.0, wirebug = 0.0 }, type = 3.0 }, ice = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 328.0, wirebug = 1814.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1436.0, wirebug = 2268.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_IceParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 10.0 }, name = "Tigrex", paralyze = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 546.0, wirebug = 3024.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 2394.0, wirebug = 3780.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ParalyzeParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 0.0 }, poison = { activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 420.0, wirebug = 3780.0 }, isActive = false, isEnable = true, limits = { player = 150.0, wirebug = 180.0 }, maxLimits = { player = 2310.0, wirebug = 4536.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_PoisonParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 4.0 }, sleep = { activeTime = 40.0, activeTimer = 0.0, addLimits = { player = 420.0, wirebug = 3024.0 }, isActive = false, isEnable = true, limits = { player = 150.0, wirebug = 180.0 }, maxLimits = { player = 2310.0, wirebug = 3780.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_SleepParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 1.0 }, stamina = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 375.0, wirebug = 3600.0 }, isActive = false, isEnable = true, limits = { player = 225.0, wirebug = 180.0 }, maxLimits = { player = 4500.0, wirebug = 4500.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 225.0, wirebug = 180.0 }, name = "_StaminaParam", stocks = { player = 180.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 6.0 }, stun = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 576.0, wirebug = 3456.0 }, isActive = false, isEnable = true, limits = { player = 150.0, wirebug = 180.0 }, maxLimits = { player = 3024.0, wirebug = 4320.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_StunParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 2.0 }, thunder = { activeTime = 40.0, activeTimer = 0.0, addLimits = { player = 328.0, wirebug = 1814.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1436.0, wirebug = 2268.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ThunderParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 11.0 }, water = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 328.0, wirebug = 1814.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1436.0, wirebug = 2268.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_WaterParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 8.0 } }, thirdMon = { blast = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 126.0, wirebug = 3024.0 }, isActive = false, isEnable = true, limits = { player = 70.0, wirebug = 180.0 }, maxLimits = { player = 2814.0, wirebug = 3780.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 70.0, wirebug = 180.0 }, name = "_BlastParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 5.0 }, fire = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 328.0, wirebug = 1814.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1436.0, wirebug = 2268.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_FireParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 9.0 }, flash = { activeCount = 0.0, activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 0.0, wirebug = 0.0 }, isActive = false, isEnable = true, limits = { player = 0.0, wirebug = 0.0 }, maxActiveCount = 7.0, maxLimits = { player = 0.0, wirebug = 0.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 0.0, wirebug = 0.0 }, name = "_FlashParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.0, wirebug = 0.0 }, type = 3.0 }, ice = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 328.0, wirebug = 1814.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1436.0, wirebug = 2268.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_IceParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 10.0 }, name = "Royal Ludroth", paralyze = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 546.0, wirebug = 3024.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 2394.0, wirebug = 3780.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ParalyzeParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 0.0 }, poison = { activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 420.0, wirebug = 3780.0 }, isActive = false, isEnable = true, limits = { player = 150.0, wirebug = 180.0 }, maxLimits = { player = 2310.0, wirebug = 4536.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_PoisonParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 4.0 }, sleep = { activeTime = 40.0, activeTimer = 0.0, addLimits = { player = 420.0, wirebug = 3024.0 }, isActive = false, isEnable = true, limits = { player = 150.0, wirebug = 180.0 }, maxLimits = { player = 2310.0, wirebug = 3780.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_SleepParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 1.0 }, stamina = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 375.0, wirebug = 3600.0 }, isActive = false, isEnable = true, limits = { player = 225.0, wirebug = 180.0 }, maxLimits = { player = 4500.0, wirebug = 4500.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 225.0, wirebug = 180.0 }, name = "_StaminaParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 6.0 }, stun = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 576.0, wirebug = 3456.0 }, isActive = false, isEnable = true, limits = { player = 150.0, wirebug = 180.0 }, maxLimits = { player = 3024.0, wirebug = 4320.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_StunParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 2.0 }, thunder = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 328.0, wirebug = 1814.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1436.0, wirebug = 2268.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ThunderParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 11.0 }, water = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 328.0, wirebug = 1814.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1436.0, wirebug = 2268.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_WaterParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 8.0 } }}
testBars.l.p4.quests  = { active = true, questId = 10307.0, questRank = 0.0, totalJoinNum = 4.0}

testBars.h.p1.monsters = { firstMon = { blast = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 57.0, wirebug = 1368.0 }, isActive = false, isEnable = true, limits = { player = 70.0, wirebug = 180.0 }, maxLimits = { player = 1273.0, wirebug = 1710.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 70.0, wirebug = 180.0 }, name = "_BlastParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 5.0 }, fire = { activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 247.0, wirebug = 1368.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1083.0, wirebug = 1710.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_FireParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 9.0 }, flash = { activeCount = 2.0, activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 0.0, wirebug = 0.0 }, isActive = false, isEnable = true, limits = { player = 0.0, wirebug = 0.0 }, maxActiveCount = 7.0, maxLimits = { player = 0.0, wirebug = 0.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 0.0, wirebug = 0.0 }, name = "_FlashParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.0, wirebug = 0.0 }, type = 3.0 }, ice = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 247.0, wirebug = 1368.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1083.0, wirebug = 1710.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_IceParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 10.0 }, name = "Nargacuga", paralyze = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 247.0, wirebug = 1368.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1083.0, wirebug = 1710.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ParalyzeParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 0.0 }, poison = { activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 190.0, wirebug = 1710.0 }, isActive = false, isEnable = true, limits = { player = 340.0, wirebug = 180.0 }, maxLimits = { player = 1045.0, wirebug = 2052.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_PoisonParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 4.0 }, sleep = { activeTime = 40.0, activeTimer = 32.6, addLimits = { player = 190.0, wirebug = 1368.0 }, isActive = true, isEnable = true, limits = { player = 340.0, wirebug = 180.0 }, maxLimits = { player = 1045.0, wirebug = 1710.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_SleepParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 1.0 }, stamina = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 135.0, wirebug = 1296.0 }, isActive = false, isEnable = true, limits = { player = 225.0, wirebug = 180.0 }, maxLimits = { player = 1620.0, wirebug = 1620.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 225.0, wirebug = 180.0 }, name = "_StaminaParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 6.0 }, stun = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 168.0, wirebug = 1008.0 }, isActive = false, isEnable = true, limits = { player = 486.0, wirebug = 180.0 }, maxLimits = { player = 882.0, wirebug = 1260.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_StunParam", stocks = { player = 440.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 2.0 }, thunder = { activeTime = 40.0, activeTimer = 0.0, addLimits = { player = 247.0, wirebug = 1368.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1083.0, wirebug = 1710.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ThunderParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 11.0 }, water = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 247.0, wirebug = 1368.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1083.0, wirebug = 1710.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_WaterParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 8.0 } }, secondMon = { blast = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 57.0, wirebug = 1368.0 }, isActive = false, isEnable = true, limits = { player = 70.0, wirebug = 180.0 }, maxLimits = { player = 1273.0, wirebug = 1710.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 70.0, wirebug = 180.0 }, name = "_BlastParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 5.0 }, fire = { activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 247.0, wirebug = 1368.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1083.0, wirebug = 1710.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_FireParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 9.0 }, flash = { activeCount = 0.0, activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 0.0, wirebug = 0.0 }, isActive = false, isEnable = true, limits = { player = 0.0, wirebug = 0.0 }, maxActiveCount = 7.0, maxLimits = { player = 0.0, wirebug = 0.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 0.0, wirebug = 0.0 }, name = "_FlashParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.0, wirebug = 0.0 }, type = 3.0 }, ice = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 247.0, wirebug = 1368.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1083.0, wirebug = 1710.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_IceParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 10.0 }, name = "Tobi-Kadachi", paralyze = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 247.0, wirebug = 1368.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1083.0, wirebug = 1710.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ParalyzeParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 0.0 }, poison = { activeTime = 25.0, activeTimer = 0.0, addLimits = { player = 171.0, wirebug = 1710.0 }, isActive = false, isEnable = true, limits = { player = 100.0, wirebug = 180.0 }, maxLimits = { player = 703.0, wirebug = 2052.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 100.0, wirebug = 180.0 }, name = "_PoisonParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 4.0 }, sleep = { activeTime = 40.0, activeTimer = 0.0, addLimits = { player = 190.0, wirebug = 1368.0 }, isActive = false, isEnable = true, limits = { player = 150.0, wirebug = 180.0 }, maxLimits = { player = 1045.0, wirebug = 1710.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_SleepParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 1.0 }, stamina = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 135.0, wirebug = 1296.0 }, isActive = false, isEnable = true, limits = { player = 225.0, wirebug = 180.0 }, maxLimits = { player = 1620.0, wirebug = 1620.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 225.0, wirebug = 180.0 }, name = "_StaminaParam", stocks = { player = 200.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 6.0 }, stun = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 168.0, wirebug = 1008.0 }, isActive = false, isEnable = true, limits = { player = 150.0, wirebug = 180.0 }, maxLimits = { player = 882.0, wirebug = 1260.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_StunParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 2.0 }, thunder = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 247.0, wirebug = 1368.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1083.0, wirebug = 1710.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ThunderParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 11.0 }, water = { activeTime = 40.0, activeTimer = 0.0, addLimits = { player = 247.0, wirebug = 1368.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1083.0, wirebug = 1710.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_WaterParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 8.0 } }, thirdMon = { blast = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 57.0, wirebug = 1368.0 }, isActive = false, isEnable = true, limits = { player = 70.0, wirebug = 180.0 }, maxLimits = { player = 1273.0, wirebug = 1710.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 70.0, wirebug = 180.0 }, name = "_BlastParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 5.0 }, fire = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 247.0, wirebug = 1368.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1083.0, wirebug = 1710.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_FireParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 9.0 }, flash = { activeCount = 0.0, activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 0.0, wirebug = 0.0 }, isActive = false, isEnable = true, limits = { player = 0.0, wirebug = 0.0 }, maxActiveCount = 7.0, maxLimits = { player = 0.0, wirebug = 0.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 0.0, wirebug = 0.0 }, name = "_FlashParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.0, wirebug = 0.0 }, type = 3.0 }, ice = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 247.0, wirebug = 1368.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1083.0, wirebug = 1710.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_IceParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 10.0 }, name = "Royal Ludroth", paralyze = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 247.0, wirebug = 1368.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1083.0, wirebug = 1710.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ParalyzeParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 0.0 }, poison = { activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 190.0, wirebug = 1710.0 }, isActive = false, isEnable = true, limits = { player = 150.0, wirebug = 180.0 }, maxLimits = { player = 1045.0, wirebug = 2052.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_PoisonParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 4.0 }, sleep = { activeTime = 40.0, activeTimer = 0.0, addLimits = { player = 190.0, wirebug = 1368.0 }, isActive = false, isEnable = true, limits = { player = 150.0, wirebug = 180.0 }, maxLimits = { player = 1045.0, wirebug = 1710.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_SleepParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 1.0 }, stamina = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 135.0, wirebug = 1296.0 }, isActive = false, isEnable = true, limits = { player = 225.0, wirebug = 180.0 }, maxLimits = { player = 1620.0, wirebug = 1620.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 225.0, wirebug = 180.0 }, name = "_StaminaParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 6.0 }, stun = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 168.0, wirebug = 1008.0 }, isActive = false, isEnable = true, limits = { player = 150.0, wirebug = 180.0 }, maxLimits = { player = 882.0, wirebug = 1260.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_StunParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 2.0 }, thunder = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 247.0, wirebug = 1368.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1083.0, wirebug = 1710.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ThunderParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 11.0 }, water = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 247.0, wirebug = 1368.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1083.0, wirebug = 1710.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_WaterParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 8.0 } }}
testBars.h.p1.quests = { active = true, questId = 60152.0, questRank = 1.0, totalJoinNum = 1.0}
testBars.h.p2.monsters = { firstMon = { blast = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 90.0, wirebug = 2160.0 }, isActive = false, isEnable = true, limits = { player = 70.0, wirebug = 180.0 }, maxLimits = { player = 2010.0, wirebug = 2700.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 70.0, wirebug = 180.0 }, name = "_BlastParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 5.0 }, fire = { activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 312.0, wirebug = 1728.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1368.0, wirebug = 2160.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_FireParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 9.0 }, flash = { activeCount = 2, activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 0.0, wirebug = 0.0 }, isActive = false, isEnable = true, limits = { player = 0.0, wirebug = 0.0 }, maxActiveCount = 7.0, maxLimits = { player = 0.0, wirebug = 0.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 0.0, wirebug = 0.0 }, name = "_FlashParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.0, wirebug = 0.0 }, type = 3.0 }, ice = { activeTime = 40.0, activeTimer = 0.0, addLimits = { player = 312.0, wirebug = 1728.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1368.0, wirebug = 2160.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_IceParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 10.0 }, name = "Diablos", paralyze = { activeTime = 7.0, activeTimer = 0.0, addLimits = { player = 150.0, wirebug = 2160.0 }, isActive = false, isEnable = true, limits = { player = 100.0, wirebug = 180.0 }, maxLimits = { player = 900.0, wirebug = 2700.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 100.0, wirebug = 180.0 }, name = "_ParalyzeParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 0.0 }, poison = { activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 300.0, wirebug = 2700.0 }, isActive = false, isEnable = true, limits = { player = 450.0, wirebug = 180.0 }, maxLimits = { player = 1650.0, wirebug = 3240.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_PoisonParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 4.0 }, sleep = { activeTime = 40.0, activeTimer = 32.6, addLimits = { player = 300.0, wirebug = 2160.0 }, isActive = true, isEnable = true, limits = { player = 450.0, wirebug = 180.0 }, maxLimits = { player = 1650.0, wirebug = 2700.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_SleepParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 1.0 }, stamina = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 240.0, wirebug = 2304.0 }, isActive = false, isEnable = true, limits = { player = 225.0, wirebug = 180.0 }, maxLimits = { player = 2880.0, wirebug = 2880.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 225.0, wirebug = 180.0 }, name = "_StaminaParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 6.0 }, stun = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 390.0, wirebug = 1872.0 }, isActive = false, isEnable = true, limits = { player = 980.0, wirebug = 180.0 }, maxLimits = { player = 2080.0, wirebug = 2340.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 200.0, wirebug = 180.0 }, name = "_StunParam", stocks = { player = 900.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 2.0 }, thunder = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 312.0, wirebug = 1728.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1368.0, wirebug = 2160.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ThunderParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 11.0 }, water = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 312.0, wirebug = 1728.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1368.0, wirebug = 2160.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_WaterParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 8.0 } }, secondMon = { blast = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 180.0, wirebug = 2160.0 }, isActive = false, isEnable = true, limits = { player = 120.0, wirebug = 180.0 }, maxLimits = { player = 2160.0, wirebug = 2700.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 120.0, wirebug = 180.0 }, name = "_BlastParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 5.0 }, fire = { activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 312.0, wirebug = 1728.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1368.0, wirebug = 2160.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_FireParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 9.0 }, flash = { activeCount = 0.0, activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 0.0, wirebug = 0.0 }, isActive = false, isEnable = true, limits = { player = 0.0, wirebug = 0.0 }, maxActiveCount = 7.0, maxLimits = { player = 0.0, wirebug = 0.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 0.0, wirebug = 0.0 }, name = "_FlashParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.0, wirebug = 0.0 }, type = 3.0 }, ice = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 312.0, wirebug = 1728.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1368.0, wirebug = 2160.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_IceParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 10.0 }, name = "Anjanath", paralyze = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 390.0, wirebug = 2160.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1710.0, wirebug = 2700.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ParalyzeParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 0.0 }, poison = { activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 300.0, wirebug = 2700.0 }, isActive = false, isEnable = true, limits = { player = 150.0, wirebug = 180.0 }, maxLimits = { player = 1650.0, wirebug = 3240.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_PoisonParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 4.0 }, sleep = { activeTime = 40.0, activeTimer = 0.0, addLimits = { player = 300.0, wirebug = 2160.0 }, isActive = false, isEnable = true, limits = { player = 150.0, wirebug = 180.0 }, maxLimits = { player = 1650.0, wirebug = 2700.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_SleepParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 1.0 }, stamina = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 240.0, wirebug = 2304.0 }, isActive = false, isEnable = true, limits = { player = 465.0, wirebug = 180.0 }, maxLimits = { player = 2880.0, wirebug = 2880.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 225.0, wirebug = 180.0 }, name = "_StaminaParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 6.0 }, stun = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 312.0, wirebug = 1872.0 }, isActive = false, isEnable = true, limits = { player = 150.0, wirebug = 180.0 }, maxLimits = { player = 1638.0, wirebug = 2340.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_StunParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 2.0 }, thunder = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 312.0, wirebug = 1728.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1368.0, wirebug = 2160.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ThunderParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 11.0 }, water = { activeTime = 40.0, activeTimer = 0.0, addLimits = { player = 312.0, wirebug = 1728.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1368.0, wirebug = 2160.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_WaterParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 8.0 } }, thirdMon = { blast = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 180.0, wirebug = 2160.0 }, isActive = false, isEnable = true, limits = { player = 120.0, wirebug = 180.0 }, maxLimits = { player = 2160.0, wirebug = 2700.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 120.0, wirebug = 180.0 }, name = "_BlastParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 5.0 }, fire = { activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 312.0, wirebug = 1728.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1368.0, wirebug = 2160.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_FireParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 9.0 }, flash = { activeCount = 0.0, activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 0.0, wirebug = 0.0 }, isActive = false, isEnable = true, limits = { player = 0.0, wirebug = 0.0 }, maxActiveCount = 7.0, maxLimits = { player = 0.0, wirebug = 0.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 0.0, wirebug = 0.0 }, name = "_FlashParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.0, wirebug = 0.0 }, type = 3.0 }, ice = { activeTime = 40.0, activeTimer = 0.0, addLimits = { player = 312.0, wirebug = 1728.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1368.0, wirebug = 2160.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_IceParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 10.0 }, name = "Rakna-Kadaki", paralyze = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 390.0, wirebug = 2160.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1710.0, wirebug = 2700.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ParalyzeParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 0.0 }, poison = { activeTime = 15.0, activeTimer = 0.0, addLimits = { player = 450.0, wirebug = 2700.0 }, isActive = false, isEnable = true, limits = { player = 250.0, wirebug = 180.0 }, maxLimits = { player = 2100.0, wirebug = 3240.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 250.0, wirebug = 180.0 }, name = "_PoisonParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 4.0 }, sleep = { activeTime = 40.0, activeTimer = 0.0, addLimits = { player = 300.0, wirebug = 2160.0 }, isActive = false, isEnable = true, limits = { player = 150.0, wirebug = 180.0 }, maxLimits = { player = 1650.0, wirebug = 2700.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_SleepParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 1.0 }, stamina = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 240.0, wirebug = 2304.0 }, isActive = false, isEnable = true, limits = { player = 225.0, wirebug = 180.0 }, maxLimits = { player = 2880.0, wirebug = 2880.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 225.0, wirebug = 180.0 }, name = "_StaminaParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 6.0 }, stun = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 312.0, wirebug = 1872.0 }, isActive = false, isEnable = true, limits = { player = 150.0, wirebug = 180.0 }, maxLimits = { player = 1638.0, wirebug = 2340.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_StunParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 2.0 }, thunder = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 312.0, wirebug = 1728.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1368.0, wirebug = 2160.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ThunderParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 11.0 }, water = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 312.0, wirebug = 1728.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 1368.0, wirebug = 2160.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_WaterParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 8.0 } }}
testBars.h.p2.quests = { active = true, questId = 10711.0, questRank = 1.0, totalJoinNum = 2.0}

testBars.h.p3.monsters = { firstMon = { blast = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 342.0, wirebug = 4104.0 }, isActive = false, isEnable = true, limits = { player = 120.0, wirebug = 180.0 }, maxLimits = { player = 4104.0, wirebug = 5130.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 120.0, wirebug = 180.0 }, name = "_BlastParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 5.0 }, fire = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 519.0, wirebug = 2873.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 2274.0, wirebug = 3591.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_FireParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 9.0 }, flash = { activeCount = 2.0, activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 0.0, wirebug = 0.0 }, isActive = false, isEnable = true, limits = { player = 0.0, wirebug = 0.0 }, maxActiveCount = 7.0, maxLimits = { player = 0.0, wirebug = 0.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 0.0, wirebug = 0.0 }, name = "_FlashParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.0, wirebug = 0.0 }, type = 3.0 }, ice = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 519.0, wirebug = 2873.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 2274.0, wirebug = 3591.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_IceParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 10.0 }, name = "Almudron", paralyze = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 741.0, wirebug = 4104.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 3249.0, wirebug = 5130.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ParalyzeParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 0.0 }, poison = { activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 570.0, wirebug = 5130.0 }, isActive = false, isEnable = true, limits = { player = 720.0, wirebug = 180.0 }, maxLimits = { player = 3135.0, wirebug = 6156.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_PoisonParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 4.0 }, sleep = { activeTime = 40.0, activeTimer = 32.6, addLimits = { player = 570.0, wirebug = 4104.0 }, isActive = true, isEnable = true, limits = { player = 720.0, wirebug = 180.0 }, maxLimits = { player = 3135.0, wirebug = 5130.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_SleepParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 1.0 }, stamina = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 405.0, wirebug = 3888.0 }, isActive = false, isEnable = true, limits = { player = 225.0, wirebug = 180.0 }, maxLimits = { player = 4860.0, wirebug = 4860.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 225.0, wirebug = 180.0 }, name = "_StaminaParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 6.0 }, stun = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 504.0, wirebug = 3024.0 }, isActive = false, isEnable = true, limits = { player = 1158.0, wirebug = 180.0 }, maxLimits = { player = 2646.0, wirebug = 3780.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_StunParam", stocks = { player = 1100.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 2.0 }, thunder = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 519.0, wirebug = 2873.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 2274.0, wirebug = 3591.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ThunderParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 11.0 }, water = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 519.0, wirebug = 2873.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 2274.0, wirebug = 3591.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_WaterParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 8.0 } }, secondMon = { blast = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 171.0, wirebug = 4104.0 }, isActive = false, isEnable = true, limits = { player = 70.0, wirebug = 180.0 }, maxLimits = { player = 3819.0, wirebug = 5130.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 70.0, wirebug = 180.0 }, name = "_BlastParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 5.0 }, fire = { activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 519.0, wirebug = 2873.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 2274.0, wirebug = 3591.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_FireParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 9.0 }, flash = { activeCount = 0.0, activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 0.0, wirebug = 0.0 }, isActive = false, isEnable = true, limits = { player = 0.0, wirebug = 0.0 }, maxActiveCount = 7.0, maxLimits = { player = 0.0, wirebug = 0.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 0.0, wirebug = 0.0 }, name = "_FlashParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.0, wirebug = 0.0 }, type = 3.0 }, ice = { activeTime = 40.0, activeTimer = 0.0, addLimits = { player = 519.0, wirebug = 2873.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 2274.0, wirebug = 3591.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_IceParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 10.0 }, name = "Zinogre", paralyze = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 1425.0, wirebug = 1425.0 }, isActive = false, isEnable = true, limits = { player = 300.0, wirebug = 300.0 }, maxLimits = { player = 6840.0, wirebug = 6840.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 300.0, wirebug = 300.0 }, name = "_ParalyzeParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 0.0 }, poison = { activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 570.0, wirebug = 5130.0 }, isActive = false, isEnable = true, limits = { player = 150.0, wirebug = 180.0 }, maxLimits = { player = 3135.0, wirebug = 6156.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_PoisonParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 4.0 }, sleep = { activeTime = 40.0, activeTimer = 0.0, addLimits = { player = 570.0, wirebug = 4104.0 }, isActive = false, isEnable = true, limits = { player = 150.0, wirebug = 180.0 }, maxLimits = { player = 3135.0, wirebug = 5130.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_SleepParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 1.0 }, stamina = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 405.0, wirebug = 3888.0 }, isActive = false, isEnable = true, limits = { player = 225.0, wirebug = 180.0 }, maxLimits = { player = 4860.0, wirebug = 4860.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 225.0, wirebug = 180.0 }, name = "_StaminaParam", stocks = { player = 200.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 6.0 }, stun = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 504.0, wirebug = 3024.0 }, isActive = false, isEnable = true, limits = { player = 150.0, wirebug = 180.0 }, maxLimits = { player = 2646.0, wirebug = 3780.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_StunParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 2.0 }, thunder = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 519.0, wirebug = 2873.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 2274.0, wirebug = 3591.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ThunderParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 11.0 }, water = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 519.0, wirebug = 2873.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 2274.0, wirebug = 3591.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_WaterParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 8.0 } }}
testBars.h.p3.quests = { active = true, questId = 10748.0, questRank = 1.0, totalJoinNum = 3.0}

testBars.h.p4.monsters = { firstMon = { blast = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 456.0, wirebug = 5472.0 }, isActive = false, isEnable = true, limits = { player = 120.0, wirebug = 180.0 }, maxLimits = { player = 5472.0, wirebug = 6840.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 120.0, wirebug = 180.0 }, name = "_BlastParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 5.0 }, fire = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 593.0, wirebug = 3283.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 2599.0, wirebug = 4104.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_FireParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 9.0 }, flash = { activeCount = 2.0, activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 0.0, wirebug = 0.0 }, isActive = false, isEnable = true, limits = { player = 0.0, wirebug = 0.0 }, maxActiveCount = 7.0, maxLimits = { player = 0.0, wirebug = 0.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 0.0, wirebug = 0.0 }, name = "_FlashParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.0, wirebug = 0.0 }, type = 3.0 }, ice = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 593.0, wirebug = 3283.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 2599.0, wirebug = 4104.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_IceParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 10.0 }, name = "Almudron", paralyze = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 988.0, wirebug = 5472.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 4332.0, wirebug = 6840.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ParalyzeParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 0.0 }, poison = { activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 760.0, wirebug = 6840.0 }, isActive = false, isEnable = true, limits = { player = 910.0, wirebug = 180.0 }, maxLimits = { player = 4180.0, wirebug = 8208.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_PoisonParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 4.0 }, sleep = { activeTime = 40.0, activeTimer = 32.6, addLimits = { player = 760.0, wirebug = 5472.0 }, isActive = true, isEnable = true, limits = { player = 910.0, wirebug = 180.0 }, maxLimits = { player = 4180.0, wirebug = 6840.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_SleepParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 1.0 }, stamina = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 540.0, wirebug = 5184.0 }, isActive = false, isEnable = true, limits = { player = 225.0, wirebug = 180.0 }, maxLimits = { player = 6480.0, wirebug = 6480.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 225.0, wirebug = 180.0 }, name = "_StaminaParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 6.0 }, stun = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 672.0, wirebug = 4032.0 }, isActive = false, isEnable = true, limits = { player = 1494.0, wirebug = 180.0 }, maxLimits = { player = 3528.0, wirebug = 5040.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_StunParam", stocks = { player = 1300.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 2.0 }, thunder = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 593.0, wirebug = 3283.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 2599.0, wirebug = 4104.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ThunderParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 11.0 }, water = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 593.0, wirebug = 3283.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 2599.0, wirebug = 4104.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_WaterParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 8.0 } }, secondMon = { blast = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 228.0, wirebug = 5472.0 }, isActive = false, isEnable = true, limits = { player = 70.0, wirebug = 180.0 }, maxLimits = { player = 5092.0, wirebug = 6840.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 70.0, wirebug = 180.0 }, name = "_BlastParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 5.0 }, fire = { activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 593.0, wirebug = 3283.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 2599.0, wirebug = 4104.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_FireParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 9.0 }, flash = { activeCount = 0.0, activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 0.0, wirebug = 0.0 }, isActive = false, isEnable = true, limits = { player = 0.0, wirebug = 0.0 }, maxActiveCount = 7.0, maxLimits = { player = 0.0, wirebug = 0.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 0.0, wirebug = 0.0 }, name = "_FlashParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.0, wirebug = 0.0 }, type = 3.0 }, ice = { activeTime = 40.0, activeTimer = 0.0, addLimits = { player = 593.0, wirebug = 3283.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 2599.0, wirebug = 4104.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_IceParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 10.0 }, name = "Zinogre", paralyze = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 1900.0, wirebug = 1900.0 }, isActive = false, isEnable = true, limits = { player = 300.0, wirebug = 300.0 }, maxLimits = { player = 9120.0, wirebug = 9120.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 300.0, wirebug = 300.0 }, name = "_ParalyzeParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 0.0 }, poison = { activeTime = 20.0, activeTimer = 0.0, addLimits = { player = 760.0, wirebug = 6840.0 }, isActive = false, isEnable = true, limits = { player = 150.0, wirebug = 180.0 }, maxLimits = { player = 4180.0, wirebug = 8208.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_PoisonParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 4.0 }, sleep = { activeTime = 40.0, activeTimer = 0.0, addLimits = { player = 760.0, wirebug = 5472.0 }, isActive = false, isEnable = true, limits = { player = 150.0, wirebug = 180.0 }, maxLimits = { player = 4180.0, wirebug = 6840.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_SleepParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 1.0 }, stamina = { activeTime = 0.0, activeTimer = 0.0, addLimits = { player = 540.0, wirebug = 5184.0 }, isActive = false, isEnable = true, limits = { player = 225.0, wirebug = 180.0 }, maxLimits = { player = 6480.0, wirebug = 6480.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 225.0, wirebug = 180.0 }, name = "_StaminaParam", stocks = { player = 200.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 6.0 }, stun = { activeTime = 10.0, activeTimer = 0.0, addLimits = { player = 672.0, wirebug = 4032.0 }, isActive = false, isEnable = true, limits = { player = 150.0, wirebug = 180.0 }, maxLimits = { player = 3528.0, wirebug = 5040.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 150.0, wirebug = 180.0 }, name = "_StunParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 2.0 }, thunder = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 593.0, wirebug = 3283.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 2599.0, wirebug = 4104.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_ThunderParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 11.0 }, water = { activeTime = 30.0, activeTimer = 0.0, addLimits = { player = 593.0, wirebug = 3283.0 }, isActive = false, isEnable = true, limits = { player = 180.0, wirebug = 180.0 }, maxLimits = { player = 2599.0, wirebug = 4104.0 }, maxStocks = { player = 0.0, wirebug = 0.0 }, minActiveTime = 0.0, minLimits = { player = 180.0, wirebug = 180.0 }, name = "_WaterParam", stocks = { player = 0.0, wirebug = 0.0 }, subActiveTime = 0.0, subs = { player = 0.5, wirebug = 0.5 }, type = 8.0 } }}
testBars.h.p4.quests = { active = true, questId = 10748.0, questRank = 1.0, totalJoinNum = 4.0}