Scriptname GTB_ControlScript extends Quest
{Mod: Go To Bed
Desc: control script}

;---CONSTANTS---

int Property ActivationMethodNotSet = -1 AutoReadOnly ; Activation method not set
int Property ActivationMethodAbility = 0 AutoReadOnly ; Activation by using ability
int Property ActivationMethodKey = 1 AutoReadOnly ; Activation by key
int Property ActivationMethodBed = 2 AutoReadOnly ; Activation by using bed

int Property ActivationMethodDefault = 2 AutoReadOnly ; Default activation method
int Property ActivationKeyDefault = -1 AutoReadOnly ; Default activation key
bool Property ClosedEyesFixEnabledDefault = false AutoReadOnly ; Default state of Closed Eyes Fix
bool Property RotatedAnimationFixEnabledDefault = false AutoReadOnly; Default state of Turned Animation Fix
bool Property LogEnabledDefault = false AutoReadOnly ; Default state for Enable Log
bool Property AutoShowSleepMenuDefault = false AutoReadOnly ; Default state for Auto Show Sleep Menu
float Property AutoShowSleepMenuDelayDefault = 0.0 AutoReadOnly ; Default value for Auto Show Sleep Menu delay

;---PROPERTIES---

int Property ActivationMethod Hidden
	Function Set( int value )
		SetActivationMethod( value )
	EndFunction
	int Function Get()
		return _ActivationMethod
	EndFunction
EndProperty

int Property ActivationKey Hidden
	Function Set( int value )
		SetActivationKey( value )
	EndFunction
	int Function Get()
		return _ActivationKey
	EndFunction
EndProperty

bool Property ClosedEyesFixEnabled Hidden
	Function Set( bool value )
		SetClosedEyesFixEnabled( value )
	EndFunction
	bool Function Get()
		return _ClosedEyesFixEnabled
	EndFunction
EndProperty

bool Property LogEnabled = false Auto Hidden
bool Property AutoShowSleepMenu = false Auto Hidden
float Property AutoShowSleepMenuDelay = 0.0 Auto Hidden
bool Property RotatedAnimationFixEnabled = false Auto Hidden

Actor Property PlayerRef Auto
Quest Property GoToBedQuest Auto
Spell Property GoToBedSpell Auto
Perk Property BedActivationPerk Auto

;---PRIVATE VARS---

int scriptVersion = 1
int _ActivationMethod = -1
int _ActivationKey = -1
bool _ClosedEyesFixEnabled = false


;---MAIN STATE---
;---EVENTS AND FUNCTIONS---

int Function GetScriptVersion()
	return 2
EndFunction

Event OnInit()
	Trace( "[GoToBed] OnInit()" )

	scriptVersion = GetScriptVersion()
	
	ActivationMethod = ActivationMethodDefault
	ActivationKey = ActivationKeyDefault
	ClosedEyesFixEnabled = ClosedEyesFixEnabledDefault
	RotatedAnimationFixEnabled = RotatedAnimationFixEnabledDefault
	LogEnabled = LogEnabledDefault
	AutoShowSleepMenu = AutoShowSleepMenuDefault
	AutoShowSleepMenuDelay = AutoShowSleepMenuDelayDefault
EndEvent

Function OnPlayerLoadGame()
	Trace( "[GoToBed] OnPlayerLoadGame()" )

	Maintenance()
	
	if ( ActivationMethod == ActivationMethodBed && !InitActivationPerk() )
		Debug.MessageBox( "Go to bed\n\nFailed to initialize activation perk.\nActivation method will be set to \"Ability\"." )
		ActivationMethod = ActivationMethodAbility
	endif
EndFunction

Function Maintenance()
	if ( scriptVersion != GetScriptVersion() )
		OnVersionUpdate( scriptVersion, GetScriptVersion() )
		scriptVersion = GetScriptVersion()
	endif
EndFunction

Function OnVersionUpdate( int currentVersion, int newVersion )
	if ( currentVersion < 2 && newVersion >= 2 )
		Trace( "[GoToBed] Updating script " + self + " to version 2", true )
		
		if ( ActivationMethod == ActivationMethodBed )
			UnregisterForCrosshairRef()
		endif
	endif
EndFunction

Event OnAnimationEvent( ObjectReference source, string eventName )
	if ( source == PlayerRef )		
		if ( eventName == "idleBedGetUp" )
			if ( ClosedEyesFixEnabled )
				RefreshFace()
			endif
		endif
	endif
EndEvent

Event OnKeyDown( int keyCode )
	Trace( "[GoToBed] OnKeyDown() keyCode=" + keyCode )
	
	if ( Utility.IsInMenuMode() )
		return
	endif
	
	if ( ActivationMethod == ActivationMethodKey && ActivationKey != -1 && KeyCode == ActivationKey )
		Activate()
	endif	
EndEvent

Function OnGoToBedSpellEffectStart( Actor target, Actor caster )
	Trace( "[GoToBed] OnGoToBedSpellEffectStart()" )
	
	if ( target != PlayerRef || caster != PlayerRef )
		return
	endif
	
	if ( ActivationMethod == ActivationMethodAbility )
		Activate()	
	endif
EndFunction

Function OnUseBed( ObjectReference akTargetRef, Actor akActor )
	Trace( "[GoToBed] OnUseBed()" )

	if ( akTargetRef == none || akActor != PlayerRef )
		return
	endif
	
	if ( ActivationMethod == ActivationMethodBed )
		if ( GetState() == "" )
			; Let bed's scripts ( if any ) to handle "OnActivate" event. For example, dawnguard coffin's script will open the coffin if it is already occupied by someone.
			akTargetRef.OnActivate( PlayerRef )
			
			; Check if we can sleep here.
			if ( !GTB_PluginScript.CanSleepWaitHere( akTargetRef ) )
				return
			endif
		endif

		Activate()
	endif
EndFunction
	
Function OnGoToBedPackageBegin()
	Trace( "[GoToBed] OnGoToBedPackageBegin()" )
EndFunction

Function OnGoToBedPackageEnd()
	Trace( "[GoToBed] OnGoToBedPackageEnd()" )
EndFunction

Function OnPlayerSit( ObjectReference akFurniture )
	Trace( "[GoToBed] OnPlayerSit()" )
EndFunction

Function OnPlayerGetUp( ObjectReference akFurniture )
	Trace( "[GoToBed] OnPlayerGetUp()" )
EndFunction

Event OnControlDown( string control )
	Trace( "[GoToBed] OnControlDown()" )
EndEvent

Function Activate()
	Trace( "[GoToBed] Activate()" )
	
	GotoState( "GoingToBed" )
EndFunction

;---GOING TO BED STATE---

State GoingToBed

	Event OnBeginState()
		Trace( "[GoToBed] BeginState: GoingToBed" )
		
		if ( Game.GetCameraState() == 0 )
			Game.ForceThirdPerson()
		elseif ( Game.GetCameraState() == 9 && RotatedAnimationFixEnabled )
			; Reset camera to prevent animation to play at the wrong angle.
			GTB_PluginScript.ResetThirdPersonCamera()
		endif
		Game.DisablePlayerControls( true, true, true, false, true, false, true, false )
		Game.SetPlayerAIDriven( true )
		
		GoToBedQuest.Start()
	EndEvent
	
	Function OnGoToBedPackageBegin()
		Trace( "[GoToBed] OnGoToBedPackageBegin()" )
	EndFunction

	Function OnGoToBedPackageEnd()
		Trace( "[GoToBed] OnGoToBedPackageEnd()" )
		
		if ( PlayerRef.GetSleepState() == 3 )
			GotoState( "Sleeping" )
		else
			Debug.Notification( "$GTB CantFindBed" )
			GotoState( "" )
		endif
	EndFunction
	
	Function OnPlayerSit( ObjectReference akFurniture )
		Trace( "[GoToBed] OnPlayerSit()" )
	EndFunction	
	
	Function OnPlayerGetUp( ObjectReference akFurniture )
		Trace( "[GoToBed] OnPlayerGetUp()" )
	EndFunction	
	
	Function AfterSetClosedEyesFixEnabled( bool value )
	EndFunction
	
	Function Activate()
		Trace( "[GoToBed] Activate()" )
	EndFunction
	
	Event OnEndState()
		Trace( "[GoToBed] EndState: GoingToBed" )
		
		Game.EnablePlayerControls()
		Game.SetPlayerAIDriven( false )
		
		GoToBedQuest.Stop()
	EndEvent	
	
EndState

;---SLEEPING STATE---

State Sleeping
	
	Event OnBeginState()
		Trace( "[GoToBed] BeginState: Sleeping" )
		
		RegisterForMovementControls()
		RegisterForCameraState()
		
		if ( AutoShowSleepMenu )
			if ( AutoShowSleepMenuDelay > 0 )
				RegisterForSingleUpdate( AutoShowSleepMenuDelay )
			else
				GTB_PluginScript.ShowSleepWaitMenu( true )
			endif
		endif
	EndEvent

	Event OnControlDown( string control )
		Trace( "[GoToBed] OnControlDown()" )
		
		if ( control == "Forward" || control == "Back" || control == "Strafe Left" || control == "Strafe right" )
			GoToState( "GettingUp" )
		endif
	EndEvent
	
	Event OnUpdate()
		GTB_PluginScript.ShowSleepWaitMenu( true )
	EndEvent
	
	Event OnPlayerCameraState( int oldState, int newState )
		if ( oldState != 3 && newState == 3 )
			; Stop listening for movement keys when free camera is on ( tfc )
			; to avoid getting up when player is trying to move camera using movement keys.
			UnregisterForMovementControls()
		elseif ( oldState == 3 && newState != 3 )
			; Start listening for movement keys again when free camera is off.
			RegisterForMovementControls()
		endif
	EndEvent
	
	Function OnPlayerGetUp( ObjectReference akFurniture )
		Trace( "[GoToBed] OnPlayerGetUp()" )
		
		GotoState( "" )
	EndFunction
	
	Function AfterSetClosedEyesFixEnabled( bool value )
	EndFunction
	
	Function Activate()
		Trace( "[GoToBed] Activate()" )
		
		UnregisterForUpdate()
		GTB_PluginScript.ShowSleepWaitMenu( true )
	EndFunction
	
	Event OnEndState()
		Trace( "[GoToBed] EndState: Sleeping" )

		UnregisterForCameraState()
		UnregisterForMovementControls()
		UnregisterForUpdate()
	EndEvent
	
EndState

;---GETTING UP STATE---

State GettingUp

	Event OnBeginState()
		Trace( "[GoToBed] BeginState: GettingUp" )
		
		ObjectReference currentBed = GTB_PluginScript.GetCurrentFurnitureRef( PlayerRef )
		if ( currentBed != none )
			; Activation will cause player to get up from bed when he/she already is in bed.
			; Full processing is not necessary.		
			currentBed.Activate( PlayerRef, true )
		else
			Trace( "[GoToBed] Can't define current bed" )
		endif
		
	EndEvent
	
	Function OnPlayerGetUp( ObjectReference akFurniture )
		Trace( "[GoToBed] OnPlayerGetUp()" )
		
		GotoState( "" )
	EndFunction
	
	Function AfterSetClosedEyesFixEnabled( bool value )
	EndFunction
	
	Function Activate()
		Trace( "[GoToBed] Activate()" )
	EndFunction	
	
	Event OnEndState()
		Trace( "[GoToBed] EndState: GettingUp" )
	EndEvent
	
EndState

;---MISC FUNCTIONS---

Function SetActivationMethod( int value )
	if ( _ActivationMethod == value )
		return
	endif
	
	if ( !BeforeSetActivationMethod( value, _ActivationMethod ) )
		return
	endif
		
	if ( _ActivationMethod == ActivationMethodKey )
		if ( ActivationKey != -1 )
			UnregisterForKey( ActivationKey )
		endif
	elseif ( _ActivationMethod == ActivationMethodAbility )
		if ( PlayerRef.HasSpell( GoToBedSpell ) )
			PlayerRef.RemoveSpell( GoToBedSpell )
		endif
	elseif ( _ActivationMethod == ActivationMethodBed )
		if ( PlayerRef.HasPerk( BedActivationPerk ) )
			PlayerRef.RemovePerk( BedActivationPerk )
		endif
	endif

	_ActivationMethod = value
	
	if ( _ActivationMethod == ActivationMethodKey )
		if ( ActivationKey != -1 )
			RegisterForKey( ActivationKey )
		endif
	elseif ( _ActivationMethod == ActivationMethodAbility )
		if ( !PlayerRef.HasSpell( GoToBedSpell ) )
			PlayerRef.AddSpell( GoToBedSpell, false )
		endif
	elseif ( _ActivationMethod == ActivationMethodBed )
		if ( !PlayerRef.HasPerk( BedActivationPerk ) )
			PlayerRef.AddPerk( BedActivationPerk )
		endif
	endif
EndFunction

bool Function BeforeSetActivationMethod( int newValue, int oldValue )
	if ( newValue == ActivationMethodBed && !InitActivationPerk() )
		Debug.MessageBox( "Go to bed\n\nFailed to initialize activation perk.\nActivation method \"Bed\" can not be set." )
		return false
	endif
	
	return true
EndFunction

Function SetActivationKey( int value )
	if ( _ActivationKey == value )
		return
	endif
		
	if ( ActivationMethod == ActivationMethodKey && _ActivationKey != -1 )
		UnregisterForKey( _ActivationKey )
	endif
	
	_ActivationKey = value
	
	if ( ActivationMethod == ActivationMethodKey && _ActivationKey != -1 )
		RegisterForKey( _ActivationKey )
	endif
EndFunction

Function SetClosedEyesFixEnabled( bool value )
	if ( _ClosedEyesFixEnabled == value )
		return
	endif

	_ClosedEyesFixEnabled = value
		
	if ( _ClosedEyesFixEnabled )
		RegisterForAnimationEvent( PlayerRef, "idleBedGetUp" )		
	else
		UnregisterForAnimationEvent( PlayerRef, "idleBedGetUp" )
	endif
	
	AfterSetClosedEyesFixEnabled( _ClosedEyesFixEnabled )
EndFunction

Function AfterSetClosedEyesFixEnabled( bool value )
	if ( value )
		RefreshFace()
	endif
EndFunction

bool Function InitActivationPerk()
	bool result = true
	
	; Perk entry 0 ( not lying on the bed ).
	; Add Target.IsBed() condition.
	if ( result && !GTB_PluginScript.HasIsBedConditionPE( BedActivationPerk, 0, 1 ) )
		result = GTB_PluginScript.AddIsBedConditionPE( BedActivationPerk, 0, 1 )
	endif
	
	; Perk entry 1 ( lying on the bed ).
	; Add Target.IsBed() condition.
	if ( result && !GTB_PluginScript.HasIsBedConditionPE( BedActivationPerk, 1, 1 ) )
		result = GTB_PluginScript.AddIsBedConditionPE( BedActivationPerk, 1, 1 )
	endif
	; Add Target.IsCurrentFurnitureRef( Player ) condition.
	if ( result && !GTB_PluginScript.HasIsCurrentFurnitureRefConditionPE( BedActivationPerk, 1, 1 ) )
		result = GTB_PluginScript.AddIsCurrentFurnitureRefConditionPE( BedActivationPerk, 1, 1, PlayerRef )
	endif
	
	; Perk entry 2 ( in jail and not lying on the bed ).
	; Add Target.IsBed() condition.
	if ( result && !GTB_PluginScript.HasIsBedConditionPE( BedActivationPerk, 2, 1 ) )
		result = GTB_PluginScript.AddIsBedConditionPE( BedActivationPerk, 2, 1 )
	endif
	; Add IsPCInJail() condition.
	if ( result && !GTB_PluginScript.HasIsPCInJailConditionPE( BedActivationPerk, 2, 1 ) )
		result = GTB_PluginScript.AddIsPCInJailConditionPE( BedActivationPerk, 2, 1 )
	endif
	
	if ( !result )
		Trace( "[GoToBed] Failed to initialize activation perk, for more info look into Data\\SKSE\\Plugins\\gotobed_plugin.log" )
		GTB_PluginScript.DumpCommandsList()
	endif
	
	return result
EndFunction

Function RegisterForMovementControls()
	RegisterForControl( "Forward" )
	RegisterForControl( "Back" )
	RegisterForControl( "Strafe Left" )
	RegisterForControl( "Strafe Right" )
EndFunction

Function UnregisterForMovementControls()
	UnregisterForControl( "Forward" )
	UnregisterForControl( "Back" )
	UnregisterForControl( "Strafe Left" )
	UnregisterForControl( "Strafe Right" )
EndFunction

Function RefreshFace()	
	; Disabling facegen
	bool oldUseFaceGen = Utility.GetINIBool( "bUseFaceGenPreprocessedHeads:General" )
	if ( oldUseFaceGen )
		Utility.SetINIBool( "bUseFaceGenPreprocessedHeads:General", false )
	endif
	
	; Updating player
	PlayerRef.QueueNiNodeUpdate()
	
	; Restoring facegen
	if ( oldUseFaceGen )
		Utility.SetINIBool( "bUseFaceGenPreprocessedHeads:General", true )
	endif
EndFunction

Function Trace( string msg, bool forceTrace = false )
	Debug.TraceConditional( msg, LogEnabled || forceTrace )
EndFunction

Function ResetState()
	Trace( "[GoToBed] Resetting state" )

	GoToState( "" )
EndFunction