import { Observable } from 'rxjs';
import { ViewContainerRef, AfterViewInit, OnDestroy } from '@angular/core';
import { BaseTabComponent, BaseTabProcess, GetRecoveryTokenOptions } from './baseTab.component';
import { TabRecoveryProvider, RecoveryToken } from '../api/tabRecovery';
import { TabsService, NewTabParameters } from '../services/tabs.service';
import { HotkeysService } from '../services/hotkeys.service';
import { TabRecoveryService } from '../services/tabRecovery.service';
import { ConfigService } from '../api';
export declare type SplitOrientation = 'v' | 'h';
export declare type SplitDirection = 'r' | 't' | 'b' | 'l';
export declare type ResizeDirection = 'v' | 'h' | 'dv' | 'dh';
/**
 * Describes a horizontal or vertical split row or column
 */
export declare class SplitContainer {
    orientation: SplitOrientation;
    /**
     * Children could be tabs or other containers
     */
    children: (BaseTabComponent | SplitContainer)[];
    /**
     * Relative sizes of children, between 0 and 1. Total sum is 1
     */
    ratios: number[];
    x: number;
    y: number;
    w: number;
    h: number;
    /**
     * @return Flat list of all tabs inside this container
     */
    getAllTabs(): BaseTabComponent[];
    /**
     * Remove unnecessarily nested child containers and renormalizes [[ratios]]
     */
    normalize(): void;
    /**
     * Makes all tabs have the same size
     */
    equalize(): void;
    /**
     * Gets the left/top side offset for the given element index (between 0 and 1)
     */
    getOffsetRatio(index: number): number;
    serialize(tabsRecovery: TabRecoveryService, options?: GetRecoveryTokenOptions): Promise<RecoveryToken>;
}
/**
 * Represents a spanner (draggable border between two split areas)
 */
export interface SplitSpannerInfo {
    container: SplitContainer;
    /**
     * Number of the right/bottom split in the container
     */
    index: number;
}
/**
 * Represents a tab drop zone
 */
export declare type SplitDropZoneInfo = {
    x: number;
    y: number;
    w: number;
    h: number;
} & ({
    type: 'absolute';
    container: SplitContainer;
    position: number;
} | {
    type: 'relative';
    relativeTo?: BaseTabComponent | SplitContainer;
    side: SplitDirection;
});
/**
 * Split tab is a tab that contains other tabs and allows further splitting them
 * You'll mainly encounter it inside [[AppService]].tabs
 */
export declare class SplitTabComponent extends BaseTabComponent implements AfterViewInit, OnDestroy {
    private hotkeys;
    private tabsService;
    private tabRecovery;
    private config;
    static DIRECTIONS: SplitDirection[];
    /** @hidden */
    viewContainer: ViewContainerRef;
    /**
     * Top-level split container
     */
    root: SplitContainer;
    /** @hidden */
    _recoveredState: any;
    /** @hidden */
    _spanners: SplitSpannerInfo[];
    /** @hidden */
    _dropZones: SplitDropZoneInfo[];
    /** @hidden */
    _allFocusMode: boolean;
    /**
     * Disables display of dynamic window/tab title provided by the shell
     */
    disableDynamicTitle: boolean;
    /** @hidden */
    private focusedTab;
    private maximizedTab;
    private viewRefs;
    private tabAdded;
    private tabAdopted;
    private tabRemoved;
    private splitAdjusted;
    private focusChanged;
    private initialized;
    get tabAdded$(): Observable<BaseTabComponent>;
    /**
     * Fired when an existing top-level tab is dragged into this tab
     */
    get tabAdopted$(): Observable<BaseTabComponent>;
    get tabRemoved$(): Observable<BaseTabComponent>;
    /**
     * Fired when split ratio is changed for a given spanner
     */
    get splitAdjusted$(): Observable<SplitSpannerInfo>;
    /**
     * Fired when a different sub-tab gains focus
     */
    get focusChanged$(): Observable<BaseTabComponent>;
    /**
     * Fired once tab layout is created and child tabs can be added
     */
    get initialized$(): Observable<void>;
    /** @hidden */
    constructor(hotkeys: HotkeysService, tabsService: TabsService, tabRecovery: TabRecoveryService, config: ConfigService);
    /** @hidden */
    ngAfterViewInit(): Promise<void>;
    /** @hidden */
    ngOnDestroy(): void;
    /** @returns Flat list of all sub-tabs */
    getAllTabs(): BaseTabComponent[];
    getFocusedTab(): BaseTabComponent | null;
    getMaximizedTab(): BaseTabComponent | null;
    focus(tab: BaseTabComponent): void;
    maximize(tab: BaseTabComponent | null): void;
    /**
     * Focuses the first available tab inside the given [[SplitContainer]]
     */
    focusAnyIn(parent?: BaseTabComponent | SplitContainer): void;
    addTab(tab: BaseTabComponent, relative: BaseTabComponent | null, side: SplitDirection): Promise<void>;
    /**
     * Inserts a new `tab` to the `side` of the `relative` tab
     */
    add(thing: BaseTabComponent | SplitContainer, relative: BaseTabComponent | SplitContainer | null, side: SplitDirection): Promise<void>;
    removeTab(tab: BaseTabComponent): void;
    replaceTab(tab: BaseTabComponent, newTab: BaseTabComponent): void;
    /**
      * Changes the size of the focused pane in the given direction
      */
    resizePane(direction: ResizeDirection): void;
    /**
     * Moves focus in the given direction
     */
    navigate(dir: SplitDirection): void;
    navigateLinear(delta: number): void;
    navigateSpecific(target: number): void;
    splitTab(tab: BaseTabComponent, dir: SplitDirection): Promise<BaseTabComponent | null>;
    /**
     * @returns the immediate parent of `tab`
     */
    getParentOf(tab: BaseTabComponent | SplitContainer, root?: SplitContainer): SplitContainer | null;
    /** @hidden */
    canClose(): Promise<boolean>;
    /** @hidden */
    getRecoveryToken(options?: GetRecoveryTokenOptions): Promise<any>;
    /** @hidden */
    getCurrentProcess(): Promise<BaseTabProcess | null>;
    /** @hidden */
    onSpannerAdjusted(spanner: SplitSpannerInfo): void;
    /** @hidden */
    onTabDropped(tab: BaseTabComponent, zone: SplitDropZoneInfo): void;
    destroy(): void;
    layout(): void;
    clearActivity(): void;
    get icon(): string | null;
    set icon(icon: string | null);
    get color(): string | null;
    set color(color: string | null);
    equalize(): void;
    private updateTitle;
    private attachTabView;
    private onAfterTabAdded;
    private layoutInternal;
    private recoverContainer;
}
/** @hidden */
export declare class SplitTabRecoveryProvider extends TabRecoveryProvider<SplitTabComponent> {
    applicableTo(recoveryToken: RecoveryToken): Promise<boolean>;
    recover(recoveryToken: RecoveryToken): Promise<NewTabParameters<SplitTabComponent>>;
}
