Scriptname PEM_MCMScript extends SKI_ConfigBase

;
;	Changelog
;	---------
;
;	DSM200725	V3a
;				Add new master hotkey to select outfit from a wheel menu (from UIExtensions)
;				Outfit names in wheel are color coded as Base or Addon
;
;				V3b
;				Outfit type CYCLE, each selection cycle thru one item outfit from the outfit container
;				Double tap for next Base outfit or next item outfit in Cycle outfit
;				Triple tap for wheel menu for last applied outfit
;				Wheelmenu is filled counterclockwise starting with a <next wheel> option
;				Change nb of available outfit container to 28
;
;	DSM200804	V3c
;				Change CYCLE outfits to Base cycling. Add cycling effect to Addon as Addon cycling
;				Define user selectable behavior for single, double and triple tap master key:
;					SHOW - Show wheel menu
;					OPEN - Show wheel menu and open selected outfit container NOT USED (using shift key)
;					NEXT - Equip next outfit (in order) or next outfit item if in Base cycling
;					SKIP - Same as Next but skip Addon outfits 
;					BUMP - Equip next Addon outfit (in order) or next outfit item if in Addon cycling
;					LAST - Show wheel of last Base outfit (or first if none)
;					MENU - Show wheel pointing to all outfit selection wheels (one for 7 outfits)
;							entries are named with first outfit name in a wheel
;
;				Add long tap (at least 1 sec) a new master key tap mode
;				Regorganize MCM menu to have all data per outfit on same page.
;				Add multiple pages to support more than 21 outfits (6 options per outfits, 128 MCM limit)
;				Reregister keys on save load to avois lost mapping if ESL flag is set
;				Change nb of outfits supported to 56
;			
;	DSM200811	V3 RC
;				Add User Profile to export/import Outfits in Json files
;				Add a user selectable limit to nb of outfits (from 7 up to 56)
;
;
;
;------------------------------------------------------------------------------------------------

import UIExtensions

Actor Property PlayerRef  Auto
FormList Property PEM_ContainerList  Auto 
FormList Property PEM_CurrentOutfit  Auto 
FormList Property PEM_CurrentOutfitMultiple  Auto 
FormList Property PEM_CurrentOutfitAddon  Auto   
FormList Property PEM_CurrentOutfitAddonMultiple  Auto
Message Property  PEM_OutfitSelectMsg Auto


int[] OID_Container 
int[] OID_Names 
int[] OID_Apply 
int[] OID_Hotkey 
int[] OID_Mode 
int[] OID_UnequipSingle

int OID_unequip
int OID_hotkeyContainer
int OID_modifier
int OID_useLocations
int OID_currentLocation
int OID_MasterKey			; DSM200725 Master Key for outfit selection
int OID_TextColorBase		; DSM200725 Text color for Base
int OID_TextColorAddon		; DSM200725 Text color for Addon
int OID_TextColorBCycle		; DSM200804 Text color for Base cycling
int OID_TextColorACycle		; DSM200804	Text color for Addon cycling
int OID_SingleTapMode		; DSM200804	Mode behavior for single tap MK
int OID_DoubleTapMode		; DSM200804	Mode behavior for double tap MK
int OID_TripleTapMode		; DSM200804	Mode behavior for triple tap MK
int OID_LongTapMode			; DSM200804	Mode behavior for long tap MK
int OID_ProfileName			; DSM200811 Profile name for Json file
int OID_ProfileExport		; DSM200811 Request profile export
int OID_ProfileImport		; DSM200811 Request profile import
int OID_MaxIndex			; DSM200811 Request profile import

Form[] ContainerList
Location currentLocation
int max_index
int change_index
int default_hotkey = 1 ;ESC
int modifier_key

int master_key					; DSM200725	Master key for outfit selection
int tap_count = 0				;			Master Key pressed
int textcolor_base = 0xFFFFFF	;			Color for Base
int textcolor_addon = 0xFFFFFF	;			Color for Addon
int textcolor_bcycle = 0xFFFFFF	;			Color for Base cycling
int textcolor_acycle = 0xFFFFFF	;			Color for Addon cycling
int singletap_mode				; DSM200804 action for this mode
int doubletap_mode				;
int tripletap_mode				; 
int longtap_mode				; 
bool longtap					; 			in long tap			
string[] tap_actions			; 			action ids
string profile_name				; DSM200811 Profile name for Json file
int size_index					; 			size of container list


int current_version = 0
bool unequip_all
bool hotkey_container
bool use_locations

;StorageUtil Keys
String Key_CustomName = "PEM_CustomName"
String Key_UnequipAll = "PEM_UnequipAll"
String Key_HotkeyContainer = "PEM_HotkeyContainer"
String Key_UseLocations = "PEM_UseLocations"
String Key_LocationList = "PEM_LocationList"
String Key_Hotkey = "PEM_Hotkey"
String Key_CurrentOutfit = "PEM_CurrentOutfitIndex"
String Key_CurrentOutfitAddon = "PEM_CurrentOutfitAddonIndex"
String Key_Mode = "PEM_CustomMode"
String Key_UnequipSingle = "PEM_UnequipSingle"

String Key_MasterKey = "PEM_MasterKey"				; DSM200725	Master hotkey for outfit selection
String Key_TextColorBase = "PEM_TextColorBase"		; 			Text color for outfit name
String Key_TextColorAddon = "PEM_TextColorAddon"	; 
String Key_TextColorBCycle = "PEM_TextColorBCycle"	;
String Key_TextColorACycle = "PEM_TextColorACycle"	;
String Key_SingleTapMode = "PEM_SingleTapMode"		; DSM200804 Tap behavior
String Key_DoubleTapMode = "PEM_DoubleTapMode"		;
String Key_TripleTapMode = "PEM_TripleTapMode"		;  
String Key_LongTapMode = "PEM_LongTapMode"			;
String Key_ProfileName = "PEM_ProfileName"			; DSM200811 Current profile
String Key_MaxIndex = "PEM_MaxIndex"				; 
 

int function GetVersion()
	return 3				; DSM200725 Set new version (3) for reconfig
	
endFunction

event OnVersionUpdate(int version)
	if (current_version == 0)
		Debug.Notification("PEM: First time install at version " + version)
		OnConfigInit()
	Elseif (version > current_version)
		Debug.Notification("PEM: Updating script from version " + current_version + " to version " + version)
		OnConfigInit()
	endIf
endEvent

Event OnConfigInit()
	ContainerList = PEM_ContainerList.ToArray()
	size_index = PEM_ContainerList.GetSize()
	max_index = StorageUtil.GetIntValue(none,Key_MaxIndex, size_index)

	currentLocation = none
	
	OID_Container = Utility.CreateIntArray(max_index, 0)
	OID_Names = Utility.CreateIntArray(max_index, 0)
	OID_Apply = Utility.CreateIntArray(max_index, 0)
	OID_Hotkey = Utility.CreateIntArray(max_index, 0)
	OID_Mode = Utility.CreateIntArray(max_index, 0)
	OID_UnequipSingle = Utility.CreateIntArray(max_index, 0)
	
	
	
	pages = new string[6]
	pages[0] = "General Settings"
	pages[1] = "Import / Export"	
	
	pages[2] = "Outfits  1 - 14"		; wheel 1 & 2
	if max_index >= 14
		pages[3] = "Outfits 15 - 28"	; wheel 3 & 4
	EndIf
	if max_index >= 28
		pages[4] = "Outfits 29 - 42"	; wheel 5 & 6
	EndIf
	if max_index >= 43
		pages[5] = "Outfits 43 - 56"	; wheel 7 & 8	maximum nb of wheel for wheel MENU
	EndIf
	
	unequip_all = StorageUtil.GetIntValue(none,Key_UnequipAll,0)
	hotkey_container = StorageUtil.GetIntValue(none,Key_HotkeyContainer,0)
	use_locations = StorageUtil.GetIntValue(none,Key_UseLocations,0)
	modifier_key = 42 ;L-Shift
	
	master_key = StorageUtil.GetIntValue(none,Key_MasterKey,default_hotkey)		;
	textcolor_base = StorageUtil.GetIntValue(none,Key_TextColorBase,0)			;
	textcolor_addon = StorageUtil.GetIntValue(none,Key_TextColorAddon,0)		;
	textcolor_acycle = StorageUtil.GetIntValue(none,Key_TextColorACycle,0)		;
	textcolor_bcycle = StorageUtil.GetIntValue(none,Key_TextColorBCycle,0)		; DSM200725
	singletap_mode = StorageUtil.GetIntValue(none,Key_SingleTapMode,0)			; DSM200804	default is SHOW
	doubletap_mode = StorageUtil.GetIntValue(none,Key_DoubleTapMode,1)			; DSM200804	default is NEXT
	tripletap_mode = StorageUtil.GetIntValue(none,Key_TripleTapMode,4)			; DSM200804	default is LAST
	longtap_mode = StorageUtil.GetIntValue(none,Key_LongTapMode,5)				; DSM200804	default is MENU
	profile_name = 	StorageUtil.GetStringValue(none,Key_ProfileName, "Profile")	; DSM200811 Current profile
	
	tap_actions = new string[6]		; actions/behavior for master key tap
	tap_actions[0] = "SHOW "
	tap_actions[1] = "NEXT "
	tap_actions[2] = "SKIP "
	tap_actions[3] = "BUMP "
	tap_actions[4] = "LAST "
	tap_actions[5] = "MENU "
;	tap_actions[6] = "OPEN "		; not used (using shift key)
		
	current_version = GetVersion()
EndEvent


;	DSM200807	Reset mapped keys on game Reload (mapping lost if flagged as ESL)

Event OnGameReload()

	parent.OnGameReload()				; call skyui game reload handler
	
	if master_key != default_hotkey
		RegisterForKey(master_key)		; register master hotkey
	EndIf
	
	int index = 0
	while index < max_index				; loop and register all specific keys
		if getCustomHotkey(index) != default_hotkey
			RegisterForKey(getCustomHotkey(index))
		EndIf
		index += 1
	EndWhile

EndEvent



Event OnPageReset(string page)
	
	If (page == "" || page == pages[0])				; General Settings
		SetCursorFillMode(TOP_TO_BOTTOM)
		SetCursorPosition(0)
		
		AddHeaderOption("<font color='#A2A4F4'>General</font>")
			
		OID_unequip = AddToggleOption("Unequip all Items before applying Outfit",unequip_all)
		OID_hotkeyContainer = AddToggleOption("Toggle Hotkey Functionality",hotkey_container)
		OID_modifier = AddKeyMapOption("Modify toggle behavior",modifier_key)

		AddEmptyOption()
		AddHeaderOption("<font color='#A2A4F4'>Wheel Menu</font>")
		OID_MasterKey = AddKeyMapOption("Master Hotkey for outfit selection",master_key)			; DSM200725
		OID_SingleTapMode = AddMenuOption(" Single tap behavior",tap_actions[singletap_mode])		; DSM200804
		OID_DoubleTapMode = AddMenuOption(" Double tap behavior",tap_actions[doubletap_mode])		; DSM200804
		OID_TripleTapMode = AddMenuOption(" Triple tap behavior",tap_actions[tripletap_mode])		; DSM200804
		OID_LongTapMode = AddMenuOption(" Long tap behavior",tap_actions[longtap_mode])				; DSM200804
		
		SetCursorPosition(5)
		currentLocation = PlayerRef.getCurrentLocation()
		OID_useLocations = AddToggleOption("Only allow outfit changes in safe locations.",use_locations)
		OID_currentLocation = AddToggleOption("Set location " + currentLocation.GetName() + " as a safe location.",getCustomLocation())

		AddEmptyOption()
		AddHeaderOption("<font color='#A2A4F4'>Wheel Menu, Text Color Options</font>")
		OID_TextColorBase = AddColorOption(" Base outfits",textcolor_base)								; DSM200725
		OID_TextColorAddon = AddColorOption(" Addon outfits",textcolor_addon)							; DSM200725
		OID_TextColorBCycle = AddColorOption(" Base cycling outfits",textcolor_bcycle)					; DSM200804
		OID_TextColorACycle = AddColorOption(" Addon cycling outfits",textcolor_acycle)					; DSM200804

	ElseIf page == pages[1]
		SetCursorFillMode(TOP_TO_BOTTOM)
		SetCursorPosition(0)

		AddEmptyOption()		
		AddHeaderOption("<font color='#A2A4F4'>Import / Export outfits</font>")
		OID_ProfileName	= AddInputOption("Profile name : ", profile_name)
				
		AddEmptyOption()
		OID_ProfileExport = AddToggleOption("Export outfits and parameters to profile", 0)
		OID_ProfileImport = AddToggleOption("Import outfits and parameters from profile", 0)
		
		SetCursorPosition(3)
		AddHeaderOption("<font color='#A2A4F4'>Mod Version</font>")
		AddTextOption("This is version " + current_version,"", OPTION_FLAG_DISABLED)
		
		AddEmptyOption()
		AddHeaderOption("<font color='#A2A4F4'>Available outfit containers is " + size_index)
		OID_MaxIndex = AddSliderOption(" Limit to a maximum of ", max_index as float, "{0}")
		
	ElseIf page == pages[2]	||	page == pages[3] ||	page == pages[4] ||	page == pages[5]
		SetCursorFillMode(LEFT_TO_RIGHT)
		SetCursorPosition(0)

		
		int index = pages.find(page) - 1				; get our page number (page[0] is 0, starting at 2 for outfits)
		
		int index_limit = index * 14					; max index for this page
		index = (index - 1) * 14						; starting outfit index number for this page (from 0)
		
;		AddHeaderOption("Available Outfits")
;		AddHeaderOption("Apply Outfits")						; fill 2 slots
		
		while (index < max_index && index < index_limit)		; fill next 112 slots		
			AddHeaderOption("<font color='#A2A4F4'>Outfit "+(index+1))
			AddEmptyOption()
			OID_Names[index] = AddInputOption("Name: ",getCustomName(index))
			OID_Hotkey[index] = AddKeyMapOption("Set Hotkey ",getCustomHotkey(index))
			OID_Mode[index] = AddTextOption("Outfit Type: ",getCustomMode(index))
			OID_Container[index] = AddTextOption("Open Container ","Open ...")
			OID_UnequipSingle[index] = AddTextOption("Unequip all: ",getCustomUnequip(index))
			OID_Apply[index] = AddToggleOption("Currently Applied ",getIsCurrentOutfit(index))
			index += 1
		EndWhile
	EndIf
EndEvent 

Event OnOptionHighlight(Int option)
	
	If (option == OID_unequip)
		SetInfoText("When this is enabled, the mod will first unequip all items from the player before equipping the outfit. Otherwise, only the items in the outfit will be added/removed and any items that are not replaced will stay equipped.")
	ElseIf (option == OID_hotkeyContainer)
		SetInfoText("Normally, the hotkeys in this mod change your outfit to whatever you have stored in it. If this option is enabled, the hotkeys will instead open the corresponding containers (so that you can edit the outfit). This option is meant to make the setup of multiple outfits faster.")
	ElseIf (option == OID_useLocations)
		SetInfoText("When this option is enabled then you can only change your outfit while inside a safe location. You have to set all safe locations manually with the below option.")
	ElseIf (option == OID_currentLocation)
		SetInfoText("Toggle if the current location should be marked as a safe location.")
	ElseIf (option == OID_modifier)
		SetInfoText("When holding this key while pressing a hotkey, you will instead open the chest. This behavior will reverse if the above option is activated.")
	ElseIf (option == OID_MasterKey)			; DSM200725
		SetInfoText("Hotkey to select Outfit from a wheel menu.\nSelect action to perform on signle tap, double tap, triple tap and long tap.")
	ElseIf (option == OID_TextColorBase)		; DSM200725
		SetInfoText("Text color for Base outfits in wheel menu")
	ElseIf (option == OID_TextColorAddon)		; DSM200725
		SetInfoText("Text color for Addon outfits in wheel menu")
	ElseIf (option == OID_TextColorBCycle)		; DSM200804
		SetInfoText("Text color for Base cycling outfits in wheel menu")
	ElseIf (option == OID_TextColorACycle)		; DSM200804
		SetInfoText("Text color for Addon cycling outfits in wheel menu")
	ElseIf (option == OID_SingleTapMode || option == OID_DoubleTapMode || option == OID_TripleTapMode || option == OID_LongTapMode) 	; DSM200804
		SetInfoText("SHOW: Show wheel menu.\nNEXT: Equip next outfit or next Base cycling outfit item. SKIP: Same as NEXT but skip Addon outfit.\nBUMP: Equip next Addom outfit or next Addon cycling outfit item.\nLAST: Show wheel of last equip Base outfit.\nMENU: Show wheel with all first outfits shown in each wheel menus.")
	ElseIf (option == OID_ProfileExport)
		SetInfoText("Export all outfits to the profile")
	ElseIf (option == OID_ProfileImport)
		SetInfoText("Import all outfits from the profile")
	ElseIf (option == OID_ProfileName)
		SetInfoText("Select a name for the current profile. Profile saved as data/skse/plugins/PlayerEquipmentManager/<profile name>.json")
	ElseIf (option == OID_MaxIndex)
		SetInfoText("Limit the number of usable outfits to reduce wheel count")	
	
	Else
		int index = 0
		while index < max_index
			If (option == OID_Container[index])
				SetInfoText("Opens the container which contains the outfit (this exits the MCM). Everything inside the opened container will be equipped when the outfit is selected. To change the Outfit, simply add or remove items.")
			ElseIf (option == OID_Names[index])
				SetInfoText("Set a custom name for your outfit to organize them.")
			ElseIf (option == OID_Apply[index])
				SetInfoText("Apply this outfit now. This removes all items from the current outfit (if applicable) from your inventory and equips all items of this outfit. If this is the current outfit, it will only be removed.")
			ElseIf (option == OID_Hotkey[index])
				SetInfoText("Set a hotkey to apply the outfit whenever you press it.")
			ElseIf (option == OID_Mode[index])
				SetInfoText("Set the mode of this outfit.\nBase Outfits are applied first, Addon outfits after, overwriting Base items. You can have a maximum of two outfits active: one Base and one Addon. Selecting an outfit already applied will unequip it, except cycling outfits will cycle outfit items one at the time.")
			ElseIf (option == OID_UnequipSingle[index])
				SetInfoText("Here you can overwrite the global toggle on the Settings page for only this outfit, or leave it as is.")
			EndIf
			
			index += 1
		EndWhile
	EndIf
	
EndEvent 

Event OnOptionSelect(int option)
	
	If (option == OID_unequip)
		unequip_all = !unequip_all
		SetToggleOptionValue(option, unequip_all)
		StorageUtil.SetIntValue(none,Key_UnequipAll,unequip_all as int)
	ElseIf (option == OID_hotkeyContainer)
		hotkey_container = !hotkey_container
		SetToggleOptionValue(option, hotkey_container)
		StorageUtil.SetIntValue(none,Key_HotkeyContainer,hotkey_container as int)
	ElseIf (option == OID_useLocations)
		use_locations = !use_locations
		SetToggleOptionValue(option, use_locations)
		StorageUtil.SetIntValue(none,Key_UseLocations,use_locations as int)
	ElseIf (option == OID_currentLocation)
		setCustomLocation(!getCustomLocation())
		SetToggleOptionValue(option, getCustomLocation())
	ElseIf (option == OID_ProfileExport)
		SetToggleOptionValue(option, 1)
		ExportAllOutfit(profile_name)
		SetToggleOptionValue(option, 0)
	ElseIf (option == OID_ProfileImport)
		SetToggleOptionValue(option, 1)
		ImportAllOutfit(profile_name)
		SetToggleOptionValue(option, 0)	
	Else
		int index = 0
		while index < max_index
			If (option == OID_Container[index])
				openContainer(index)
			ElseIf (option == OID_Apply[index])
				applyOutfit(index)
				resetApplyToggles()
			ElseIf (option == OID_Mode[index])
				if (getCustomMode(index) == "Base ")
					setCustomMode(index,"Addon ")
					SetTextOptionValue(option,"Addon ")
				ElseIf (getCustomMode(index) == "Addon ")
					setCustomMode(index,"Base cycling ")
					SetTextOptionValue(option,"Base cycling ")
				ElseIf (getCustomMode(index) == "Base cycling ")	; DSM200804
					setCustomMode(index,"Addon cycling ")
					SetTextOptionValue(option,"Addon cycling ")
				Else
					setCustomMode(index,"Base ")
					SetTextOptionValue(option,"Base ")
				EndIf
				if (getIsCurrentOutfit(index))
					applyOutfit(index)
				endIf
				
			ElseIf (option == OID_UnequipSingle[index])
				if (getCustomUnequip(index) == "Global ")
					setCustomUnequip(index,"Yes ")
					SetTextOptionValue(option,"Yes ")
				Elseif (getCustomUnequip(index) == "Yes ")
					setCustomUnequip(index,"No ")
					SetTextOptionValue(option,"No ")
				Else
					setCustomUnequip(index,"Global ")
					SetTextOptionValue(option,"Global ")
				EndIf
			EndIf
			
			index += 1
		EndWhile
	EndIf
	
EndEvent

Event OnOptionInputOpen(int option)
	int index = 0
	if (option == OID_ProfileName)
		SetInputDialogStartText(profile_name)		
	Else
		while index < max_index
			If (option == OID_Names[index])
				SetInputDialogStartText(getCustomName(index))
			EndIf
			index += 1
		EndWhile
	EndIf
endEvent


Event OnOptionInputAccept(int option, string inputValue)
	int index = 0
	if (option == OID_ProfileName)
		profile_name = 	inputValue
		SetInputOptionValue(option, inputValue)
	Else
		while index < max_index
			If (option == OID_Names[index])
				setCustomName(index,inputValue)
				SetInputOptionValue(option, inputValue)
			EndIf
			index += 1
		EndWhile
	EndIf
endEvent


;	DSM200725 MCM Add color options

event OnOptionColorOpen(int a_option)
	{Called when a color option has been selected}
	if (a_option == OID_TextColorBase)
		SetColorDialogStartColor(textcolor_base)
		SetColorDialogDefaultColor(0xFFFFFF)		; default is white
	elseif (a_option == OID_TextColorAddon)
		SetColorDialogStartColor(textcolor_addon)
		SetColorDialogDefaultColor(0xFFFFFF)		; default is white
	elseif (a_option == OID_TextColorACycle)
		SetColorDialogStartColor(textcolor_acycle)
		SetColorDialogDefaultColor(0xFFFFFF)		; default is white
	elseif (a_option == OID_TextColorBCycle)
		SetColorDialogStartColor(textcolor_bcycle)
		SetColorDialogDefaultColor(0xFFFFFF)		; default is white
	EndIf
endEvent

event OnOptionColorAccept(int a_option, int a_color)
	{Called when a new color has been accepted}
	if (a_option == OID_TextColorBase)
		textcolor_base = a_color
		SetColorOptionValue(a_option, a_color)
	elseif (a_option == OID_TextColorAddon)
		textcolor_addon = a_color
		SetColorOptionValue(a_option, a_color)	
	elseif (a_option == OID_TextColorBCycle)
		textcolor_bcycle = a_color
		SetColorOptionValue(a_option, a_color)	
	elseif (a_option == OID_TextColorACycle)
		textcolor_acycle = a_color
		SetColorOptionValue(a_option, a_color)
	endIf
endEvent

;
;	DSM200808 Option menu
;
Event OnOptionMenuOpen(int a_option)
	{Called when the user selects a menu option}
	If 		(a_option == OID_SingleTapMode)
				SetMenuDialogStartIndex(singletap_mode)
				SetMenuDialogOptions(tap_actions)
	ElseIf	(a_option == OID_DoubleTapMode)
				SetMenuDialogStartIndex(doubletap_mode)
				SetMenuDialogOptions(tap_actions)
	ElseIf	(a_option == OID_TripleTapMode)
				SetMenuDialogStartIndex(tripletap_mode)
				SetMenuDialogOptions(tap_actions)
	ElseIf	(a_option == OID_LongTapMode)
				SetMenuDialogStartIndex(longtap_mode)
				SetMenuDialogOptions(tap_actions)	
	EndIf
EndEvent


event OnOptionMenuAccept(int a_option, int a_index)
	{Called when the user accepts a new menu entry}
	If 		(a_option == OID_SingleTapMode)
				singletap_mode = a_index
				SetMenuOptionValue(a_option, tap_actions[singletap_mode])
				StorageUtil.SetIntValue(none,Key_SingleTapMode,singletap_mode as int)
	ElseIf	(a_option == OID_DoubleTapMode)
				doubletap_mode = a_index
				SetMenuOptionValue(a_option, tap_actions[doubletap_mode])
				StorageUtil.SetIntValue(none,Key_DoubleTapMode,singletap_mode as int)
	ElseIf	(a_option == OID_TripleTapMode)
				tripletap_mode = a_index
				SetMenuOptionValue(a_option, tap_actions[tripletap_mode])
				StorageUtil.SetIntValue(none,Key_TripleTapMode,singletap_mode as int)
	ElseIf	(a_option == OID_LongTapMode)
				longtap_mode = a_index	
				SetMenuOptionValue(a_option, tap_actions[longtap_mode])
				StorageUtil.SetIntValue(none,Key_LongTapMode,singletap_mode as int)	
	EndIf
endEvent

Event OnOptionSliderOpen(int a_option)
	if (a_option == OID_MaxIndex)
		SetSliderDialogStartValue(7 as Float)
		SetSliderDialogDefaultValue(7 as Float)
		SetSliderDialogRange(7, size_index as Float)
		SetSliderDialogInterval(7 as Float)
	EndIf
EndEvent


Event OnOptionSliderAccept(int a_option, float a_value)
	if (a_option == OID_MaxIndex)
		max_index = a_value as int
		SetSliderOptionValue(OID_MaxIndex, max_index as Float)
		StorageUtil.SetIntValue(none, Key_MaxIndex, max_index as int)	
		ForcePageReset()
	EndIf
EndEvent

;
;	DSM200725 Key Map Changed, add Master Key support
;
event OnOptionKeyMapChange(int option, int keyCode, string conflictControl, string conflictName)
	If (option == OID_modifier)
		modifier_key = keyCode
		SetKeyMapOptionValue(option, keyCode)
	ElseIf (option == OID_MasterKey)		; DSm200725 Is this for the master key for outfit selection
				bool continue = true	
				if (keyCode != 1 && conflictControl != "")
					string msg
					if (conflictName != "")
						msg = "This key is already mapped to:\n\"" + conflictControl + "\"\n(" + conflictName + ")\n\nAre you sure you want to continue?"
					else
						msg = "This key is already mapped to:\n\"" + conflictControl + "\"\n\nAre you sure you want to continue?"
					endIf

					continue = ShowMessage(msg, true, "$Yes", "$No")
				endIf
				
				If continue
					UnregisterForKey(master_key)
					if keyCode != 1
						RegisterForKey(keyCode)
					EndIf
					master_key = keyCode
					SetKeyMapOptionValue(option, keyCode)
				EndIf
	Else
		int index = 0
		while index < max_index
			If (option == OID_Hotkey[index])
				bool continue = true
				if (keyCode != 1 && conflictControl != "")
					string msg
					if (conflictName != "")
						msg = "This key is already mapped to:\n\"" + conflictControl + "\"\n(" + conflictName + ")\n\nAre you sure you want to continue?"
					else
						msg = "This key is already mapped to:\n\"" + conflictControl + "\"\n\nAre you sure you want to continue?"
					endIf

					continue = ShowMessage(msg, true, "$Yes", "$No")
				endIf
				
				If continue
					UnregisterForKey(getCustomHotkey(index))
					if keyCode != 1
						RegisterForKey(keyCode)
					EndIf
					setCustomHotkey(index,keyCode)
					SetKeyMapOptionValue(option, keyCode)
				EndIf
			EndIf
			index += 1
		EndWhile
	EndIf
endEvent

Event OnOptionDefault(int option)
	If (option == OID_modifier)
		modifier_key = 42
		SetKeyMapOptionValue(option, 42)
	Else
		int index = 0
		while index < max_index
			If (option == OID_Hotkey[index])
				UnregisterForKey(getCustomHotkey(index))
				setCustomHotkey(index,1)
				SetKeyMapOptionValue(option, 1)
			EndIf
			index += 1
		EndWhile
	EndIf
endEvent



;	DSM200725 Add wheel menu called with Master Key


Event OnKeyDown(Int keyCode)
	;Debug.Notification("Keypress!!!")
	if (keyCode != 1 && !Utility.IsInMenuMode() && Game.IsMenuControlsEnabled() && !UI.IsTextInputEnabled())
		int index = 0
		bool modified = Input.IsKeyPressed(modifier_key)
		
		If (keyCode == master_key)				; Is this the master key for outfit selection
		
			tap_count = 1						; start tap counter
			longtap = True						; assume will start a long tap
			GoToState("MasterKey")
			
			while longtap											
				Utility.wait (0.5)				; wait for taps
			EndWhile

;			Debug.Notification("Master Keypress, tap count is " + tap_count)

			if tap_count == 0 						; long tap
				index = DoTapAction(longtap_mode)
			elseif tap_count == 1 					; single tap
				index = DoTapAction(singletap_mode)
			elseif tap_count == 2					; double tap
				index = DoTapAction(doubletap_mode)
			else 									; triple tap (or more)
				index = DoTapAction(tripletap_mode)
			endif 
			
			tap_count = 0			; Reset
			GoToState("")

		Else		; Check for dedicated hotkeys
			
			bool found = false

			while (index < max_index && !found)
				If (keyCode == getCustomHotkey(index))
					found = true
				Else
					index += 1
				endif
			endWhile
			
		EndIf

;		now we have an index, check if valid. Index is -1 when wheel selection aborted or is outofbound (at max_index) when not a valid hotkey

;		Debug.Notification( "PEM: Index is " + index )	; DEBUG	

		if index >= 0 && index < max_index
		
;			we have a valid index to proceed	
			If (!hotkey_container && !modified) || (hotkey_container && modified)	; check for update outfit container

				if getisModeCycle(index)					; Is this a cycling mode ?
					Debug.Notification("PEM: Cycling an Outfit item from \"" + getCustomName(index) + "\"")
				Else
					Debug.Notification("PEM: Changing Outfit to \"" + getCustomName(index) + "\"")
				EndIf
				applyOutfit(index)
			Else
				openContainer(index)
			EndIf
		Else
;			Debug.Notification( "PEM: Invalid Index " + index )	; DEBUG
		EndIf
	EndIf
EndEvent

;
;	DSM200725 State MasterKey for tap count
;

state MasterKey

Event OnKeyDown(Int keyCode)

	if (keyCode != 1 && !Utility.IsInMenuMode() && Game.IsMenuControlsEnabled() && !UI.IsTextInputEnabled())
		If (keyCode == master_key)		; Is this the master key for outfit selection ?
			tap_count += 1				; 	bump tap counter for multi tap
			longtap = false				;	 and set not in long tap	
		endif
	endif
endevent

endstate

;	DSM200804 Long tap

Event OnKeyUp(Int keyCode, float time)

	if (keyCode != 1 && !Utility.IsInMenuMode() && Game.IsMenuControlsEnabled() && !UI.IsTextInputEnabled())
		If (keyCode == master_key)			; Is this the master key for outfit selection
			longtap = false					; set not in long tap
			if time > 1.0
				tap_count = 0				; set to nul indicates long tap
			EndIf
		EndIf
	EndIf
EndEvent



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;Storage Functions
;Getter
String Function getCustomName(int index)
	Form current = (ContainerList[index] as ObjectReference).GetBaseObject()
	return StorageUtil.GetStringValue(none,Key_CustomName+index,current.GetName())
EndFunction

Int Function getCustomHotkey(int index)
	return StorageUtil.GetIntValue(none,Key_Hotkey+index,default_hotkey)
EndFunction

Int Function getCurrentOutfit(bool addon = false)
	if addon
		return StorageUtil.GetIntValue(none,Key_CurrentOutfitAddon,-1)
	endIf
	return StorageUtil.GetIntValue(none,Key_CurrentOutfit,-1)
EndFunction

Bool Function getIsCurrentOutfit(int index)
	return (getCurrentOutfit() == index) || (getCurrentOutfit(true) == index)
EndFunction

String Function getCustomMode(int index)
	return StorageUtil.GetStringValue(none,Key_Mode+index,"Base ")
EndFunction

String Function getCustomUnequip(int index)
	return StorageUtil.GetStringValue(none,Key_UnequipSingle+index,"Global ")
EndFunction

bool Function getCustomLocation()
	return StorageUtil.FormListHas(none,Key_LocationList,currentLocation)
EndFunction


;Setter
Function setCustomName(int index, String name)
	StorageUtil.SetStringValue(none,Key_CustomName+index,name)
EndFunction

Function setCustomHotkey(int index, int keycode)
	StorageUtil.SetIntValue(none,Key_Hotkey+index,keycode)
EndFunction

Function setCurrentOutfit(int index, bool addon = false)
	if addon
		StorageUtil.SetIntValue(none,Key_CurrentOutfitAddon,index)
	else
		StorageUtil.SetIntValue(none,Key_CurrentOutfit,index)
	endIf
EndFunction

Function setCustomMode(int index, String option)
	StorageUtil.SetStringValue(none,Key_Mode+index,option)
EndFunction

Function setCustomUnequip(int index, String option)
	StorageUtil.SetStringValue(none,Key_UnequipSingle+index,option)
EndFunction

Function setCustomLocation(bool safe)
	if (safe)
		StorageUtil.FormListAdd(none,Key_LocationList,currentLocation,false)
	else
		StorageUtil.FormListRemove(none,Key_LocationList,currentLocation,true)
	endIf
EndFunction


;	DSM200725 New functions from V3



Function SetWheelName(String MenuName)
;	OutfitNameRef   .GetBaseObject().SetName(MenuName)
EndFunction

Function SetWheelOption(int index, string option)
		UIExtensions.SetMenuPropertyIndexString("UIWheelMenu", "optionText", index , option)
		UIExtensions.SetMenuPropertyIndexString("UIWheelMenu", "optionLabelText", index , option)
		UIExtensions.SetMenuPropertyIndexBool("UIWheelMenu", "optionEnabled", index , true )
EndFunction

Function SetWheelOptionColor(int index, int color)
		UIExtensions.SetMenuPropertyIndexInt("UIWheelMenu", "optionTextColor", index , color)
EndFunction

Bool Function GetIsModeBase(int a_index)
	return getCustomMode(a_index) == "Base " || getCustomMode(a_index) == "Base cycling "
EndFunction

Bool Function GetIsModeAddon(int a_index)
	return getCustomMode(a_index) == "Addon " || getCustomMode(a_index) == "Addon cycling "
EndFunction

Bool Function GetIsModeCycle(int a_index)
	return getCustomMode(a_index) == "Base cycling " || getCustomMode(a_index) == "Addon cycling "
EndFunction

Int Function DoTapAction(int a_tapmode)		; perform long, single, double & triple tap configured actions

	int index		; returned value, index of next outfit to load

	If a_tapmode == 0					; SHOW - Show wheel menu
		index = GetIndexFromWheel(0)		; use wheel menu to obtain outfit index (starting at first outfit)
	
	ElseIf a_tapmode == 1					; NEXT - load next outfit (in order) or next outfit item if in Base cycling 

		index = getCurrentOutfit()			; get current Base outfit applied
;											; same outfit index to load next item outfit in cycling
;											; or get next Base
		if index == -1						; no current Base ?
			index = 0						;	start at first
		EndIf	
		if !getisModeCycle(index)			; Base outfit not cycling ?
			index += 1						; get next outfit in list
			If index >= max_index 			;	at end of list ?
				index = 0					;		then start at first
			EndIf
		endif								; same outfit index to load next item outfit

	ElseIf 	a_tapmode == 2				; SKIP - same as Next but skip Addon outfits 

		index = getCurrentOutfit()			; get current Base outfit applied
;											; same outfit index to load next item outfit in cycling
;											; or get next Base
		if index == -1						; no current Base ?
			index = 0						;	start at first
		EndIf
		if !getisModeCycle(index)			; Base outfit not cycling ?
			index += 1						; get next outfit in list
			while getisModeAddon(index) && index < max_index	; 	but skip Addon outfit
				index += 1
			endwhile
			If index >= max_index 			; at end of list ?
				index = 0					;	then start at first
				while getisModeAddon(index) && index < max_index	; one more pass from the beginning
					index += 1
				endwhile					; if none found index will be outofbound
			EndIf
		endif

	ElseIf 	a_tapmode == 3				; BUMP - load next Addom outfit (in order) or next outfit item if in Addon cycling
	
		index = getCurrentOutfit(true)		; get current Addon outfit applied
;											; same outfit index to load next item outfit in cycling
;											; or get next Addon
		if index == -1						; no current Addon ?
			index = 0						;	start at first
		EndIf
		if !getisModeCycle(index)			; Addon outfit not cycling ?
			index += 1						; get next outfit in list
			while getisModeBase(index) && index < max_index	; 	but skip Base outfit
				index += 1
			endwhile
			If index >= max_index 			; at end of list ?
				index = 0					;	then start at first
				while getisModeBase(index) && index < max_index	; one more pass from the beginning
					index += 1
				endwhile					; if none found index will be outofbound
			EndIf
		endif
	
	ElseIf 	a_tapmode == 4				; LAST - show wheel of last Base outfit (or first if none)

		index = getCurrentOutfit()			; get current Base outfit applied
		if index == -1						; no current Base ?
			index = 0						;	start at first
		Else
			index = ( index / 7 ) * 7		; start from last applied  (in exact wheel position)
		EndIf
		index = GetIndexFromWheel(index)	; use wheel menu to obtain outfit index 

	ElseIf 	a_tapmode == 5				; MENU - show wheel pointing to all outfit selection wheels (one for 7 outfits)
;										;		entries are named with first outfit name in a wheel

		index = GetIndexFromWheel(0, true)	; use wheel menu (menu mode) to obtain an index to one of the wheel menus
		index = GetIndexFromWheel(index)	; use wheel menu to obtain final outfit index 		

	EndIf 

	return index
EndFunction



int function GetIndexFromWheel(int index, bool menu = false)

			UIExtensions.InitMenu("UIWheelMenu")
			
			SetWheelName("Outfits")
			
			bool continue = true
			int starting_index = index	; initial index

			While continue		; Proceed wheels until one outfit is selected


				int wheel_index = 0			; index in wheel

					if !menu
						SetWheelOption(0, "-- NEXT --")	; First entry if for Next Wheel option (when more than 8 defined outfits)
						wheel_index = 1
					EndIf
				
				while (index < max_index &&	wheel_index < 8)	; max 8 entries per wheel
				
						int fill_index
						if wheel_index > 3							; doing right side of the wheel ?
							fill_index = 11 - wheel_index			;	then start filling from the bottom
						Else										; 		in perfect counterclockwise	rotation
;																	; 1 oclock is 4, 2 oclock is 5, 4 oclock is 6 and 5 oclock is 7
							fill_index = wheel_index				; left side follows wheel index		
						EndIf
						
						SetWheelOption(fill_index, getCustomName(index))
						
						if GetIsModeAddon(index) && !GetIsModeCycle(index)		; change color as per mode
							SetWheelOptionColor(fill_index, textcolor_addon)
						ElseIf GetIsModeAddon(index) && GetIsModeCycle(index)
							SetWheelOptionColor(fill_index, textcolor_acycle)
						ElseIf GetIsModeBase(index) && GetIsModeCycle(index)
							SetWheelOptionColor(fill_index, textcolor_bcycle)
						Else
							SetWheelOptionColor(fill_index, textcolor_base)
						EndIf
						index = index + 1 + (menu as int * 6)		; next Outfit index (or by 7 for menu )
						wheel_index += 1							; next wheel entry
				endWhile
				
				wheel_index = UIExtensions.OpenMenu("UIWheelMenu")	; get selected outfit index
;																	; wheel_index starting at 0, -1 if cancelled
				if wheel_index != -1
				
					if !menu && wheel_index == 0			; Next Wheel selected (first item)
						if index >= max_index				; last wheel ?
							index = 0						;  then start at first outfit
						EndIf
						starting_index = index				; remember this wheel starting index
					Else

						if wheel_index > 3								; from right side of the wheel ?
;																		; in menu mode, all entries are in multiple of 7
							index = starting_index + (10 - wheel_index) + ((6 * (11 - wheel_index) + 1) * menu as int)
;																		;	then start from the bottom
						else
							index = starting_index + (wheel_index - 1) +  (((6 * wheel_index) + 1) * menu as int)
;																		; Get the outfit index
						EndIf
						continue = False								;	and return
;						Debug.Notification( "PEM: Index is " + index + " at wheel index " + wheel_index)	; DEBUG	
					EndIf
				Else
					continue = False	; wheel selection aborted
					index = -1			;  no index selected
				EndIf
				
			EndWhile
			
		return index
EndFunction


; DSM200811 Import/Export outfits to Json file

Function ExportAllOutfit(string a_profile)
	String FilePath = "../PlayerEquipmentManager/" + profile_name
	Bool result
	bool continue = true
	int index
	int i
	int j	

	continue = ShowMessage("Exporting will replace all outfit definitions and general parameters in the profile. \nAre you sure you want to continue?", true, "$Yes", "$No")
	
	if continue
	
		; I - save  all general parameters

		JsonUtil.SetIntValue(FilePath, "MasterKey", Master_Key)
		JsonUtil.SetIntValue(FilePath, "TextColorBase", TextColor_Base)
		JsonUtil.SetIntValue(FilePath, "TextColorAddon", TextColor_Addon)
		JsonUtil.SetIntValue(FilePath, "TextColorBaseCycling", TextColor_Bcycle)
		JsonUtil.SetIntValue(FilePath, "TextColorAddonCycling", TextColor_Acycle)
		JsonUtil.SetIntValue(FilePath, "SingleTap", singletap_mode)
		JsonUtil.SetIntValue(FilePath, "DoubleTap", doubletap_mode)
		JsonUtil.SetIntValue(FilePath, "TripleTap", tripletap_mode)
		JsonUtil.SetIntValue(FilePath, "LongTap", longtap_mode)
		JsonUtil.SetIntValue(FilePath, "ModifierKey", modifier_key)
		JsonUtil.SetIntValue(FilePath, "UnEquipAll", unequip_all as int)
		JsonUtil.SetIntValue(FilePath, "HotkeyContainer", hotkey_container as int)
		JsonUtil.SetIntValue(FilePath, "UseLocations", use_locations as int)
		

		; II - save all outfit names
	
		JsonUtil.StringListClear(FilePath, "Outfit Names")
		index = 0
		While index < max_index
			JsonUtil.StringListAdd(FilePath, "Outfit Names", getCustomName(index))
			index += 1
		EndWhile
		
		; III - save all specific hot keys

		JsonUtil.IntListClear(FilePath, "Outfit HotKeys")
		index = 0
		While index < max_index
			JsonUtil.IntListAdd(FilePath, "Outfit HotKeys", getCustomHotkey(index))
		index += 1
		EndWhile
		
		; IV - save all outfit modes

		JsonUtil.StringListClear(FilePath, "Outfit Modes")
		index = 0
		While index < max_index
			JsonUtil.StringListAdd(FilePath, "Outfit Modes", getCustomMode(index))
		index += 1
		EndWhile

		; V - save all equip mode
	
		JsonUtil.StringListClear(FilePath, "Outfit Equip Modes")
		index = 0
		While index < max_index
			JsonUtil.StringListAdd(FilePath, "Outfit Equip Modes", getCustomUnequip(index))
		index += 1
		EndWhile
		
		; VI - save all locations
		
		JsonUtil.FormListClear(FilePath, "Locations")
		j = StorageUtil.FormListCount(none,Key_LocationList)
		i = 0
		While  i < j
			JsonUtil.FormListAdd(FilePath, "Locations", StorageUtil.FormListGet(none,Key_LocationList,i))
			i += 1
		EndWhile
		
		; VII - save all outfit definitions

		ObjectReference currentContainer = none
		Form current = none
		index = 0
		While index < max_index

			currentContainer = ContainerList[index] as ObjectReference
			string name
			if index < 10
				name = "Outfit 0" + index as string
			Else
				name = "Outfit " + index as string
			EndIf

			JsonUtil.FormListClear(FilePath, name)
			j = currentContainer.GetNumItems()				; get size of container
			i = 0
			while i < j
				current = currentContainer.GetNthForm(i)
				JsonUtil.FormListAdd(FilePath, name, current)		
			i += 1
			EndWhile
		index += 1
		EndWhile
		
		; force save
		JsonUtil.Save(filePath)
		ShowMessage("Profile saved.", False, "Ok")
		
	EndIf
	
EndFunction

Function ImportAllOutfit(string a_profile)
	String FilePath = "../PlayerEquipmentManager/" + profile_name
	Bool result
	bool continue = true
	int index
	int i
	int j
	
	If JsonUtil.JsonExists(FilePath) == true

		continue = ShowMessage("Importing will replace all outfit definitions and general parameters.\nAre you sure you want to continue?", true, "$Yes", "$No")
	Else
		ShowMessage("Missing Profile file : " + FilePath, False, "Ok")
		continue = false
	EndIf

	if continue
	
			; I - get all general parameters


		UnregisterForKey(master_key)
		Master_Key = JsonUtil.GetIntValue(FilePath, "MasterKey")
		if Master_Key != default_hotkey
			RegisterForKey(Master_Key)
		EndIf
		TextColor_Base = JsonUtil.GetIntValue(FilePath, "TextColorBase")
		TextColor_Addon = JsonUtil.GetIntValue(FilePath, "TextColorAddon")
		TextColor_Bcycle = JsonUtil.GetIntValue(FilePath, "TextColorBaseCycling")
		TextColor_Acycle = JsonUtil.GetIntValue(FilePath, "TextColorAddonCycling")
		singletap_mode = JsonUtil.GetIntValue(FilePath, "SingleTap")
		doubletap_mode = JsonUtil.GetIntValue(FilePath, "DoubleTap")
		tripletap_mode = JsonUtil.GetIntValue(FilePath, "TripleTap")
		longtap_mode = JsonUtil.GetIntValue(FilePath, "LongTap")

		modifier_key = JsonUtil.GetIntValue(FilePath, "ModifierKey")
		unequip_all = JsonUtil.GetIntValue(FilePath, "UnEquipAll") as bool
		hotkey_container  = JsonUtil.GetIntValue(FilePath, "HotkeyContainer") as bool
		use_locations = JsonUtil.GetIntValue(FilePath, "UseLocations") as bool
		

		; II - get all outfit names
	
		j = JsonUtil.StringListCount(FilePath, "Outfit Names")
		index = 0
		While index < j
			string name
			setCustomName(index, JsonUtil.StringListGet(FilePath, "Outfit Names", index))
			index += 1
		EndWhile
		
		; III - get all specific hot keys
	
		j = JsonUtil.StringListCount(FilePath, "Outfit HotKeys")
		index = 0
		While index < j
			i = JsonUtil.IntListGet(FilePath, "Outfit HotKeys", index)
			UnregisterForKey(i)
			setCustomHotkey(index,i)
			if i != default_hotkey
				RegisterForKey(i)
			EndIf
		index += 1
		EndWhile
		
		; IV - get all outfit modes
	
		j = JsonUtil.StringListCount(FilePath, "Outfit Modes")
		index = 0
		While index < j
			setCustomMode(index, JsonUtil.StringListGet(FilePath, "Outfit Modes", index))
		index += 1
		EndWhile

		; V - get all equip mode
	
		j = JsonUtil.StringListCount(FilePath, "Outfit Equip Modes")
		index = 0
		While index < j
			setCustomUnequip(index, JsonUtil.StringListGet(FilePath, "Outfit Equip Modes", index))
		index += 1
		EndWhile
		
		; VI - get all locations
		
		StorageUtil.FormListClear(none,Key_LocationList)
		j = JsonUtil.FormListCount(FilePath, "Locations" )
		index = 0
		While  index < j
			StorageUtil.FormListAdd(none,Key_LocationList, JsonUtil.FormListGet(FilePath, "Locations", index))
			index += 1
		EndWhile
		
		; VII - get all outfit definitions

		ObjectReference currentContainer = none
		Form current = none
		index = 0
		While index < max_index

			currentContainer = ContainerList[index] as ObjectReference
			currentContainer.RemoveAllItems()
			string name
			if index < 10
				name = "Outfit 0" + index as string
			Else
				name = "Outfit " + index as string
			EndIf
			
			j = JsonUtil.FormListCount(FilePath, name)
			i = 0
			while i < j
				current = JsonUtil.FormListGet(FilePath, name, i)
				currentContainer.AddItem(current, 1, true)				
			i += 1
			EndWhile
		index += 1
		EndWhile
		
		; Flush everything			
		PEM_CurrentOutfit.Revert()
		PEM_CurrentOutfitAddon.Revert()
		PEM_CurrentOutfitMultiple.Revert()
		PEM_CurrentOutfitAddonMultiple.Revert()
		setCurrentOutfit(-1)		
		setCurrentOutfit(-1,true)
		
		ShowMessage("Profile restored. You have no current outfits.", False, "Ok")		
		ForcePageReset()
	EndIf
EndFunction

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;General Functions

Function openContainer(int index)
	ObjectReference current = ContainerList[index] as ObjectReference
	while Utility.IsInMenuMode()
		Input.TapKey(1)
		Utility.waitmenumode(0.1)
	endWhile
	current.Activate(PlayerRef)
EndFunction

Function applyOutfit(int index)

	if (use_locations && !checkCurrentLocation())
		Debug.Notification("PEM: You cannot change outfits in unsafe locations.")
		return
	endif
	
	gotostate("ApplyLocked")
	
	change_index = index
	int old_outfit = getCurrentOutfit()
	int old_addon = getCurrentOutfit(true)
	int count = 1
	bool weapon_slot = false
	bool spell_slot = false
	String custom_unequip = getCustomUnequip(index)
	bool base_outfit = getisModeBase(index) 			; DSM200725 set mode flags for current outfit
	bool addon_outfit = getisModeAddon(index) 			; 
	bool cycle_outfit = getisModeCycle(index) 			; DSM200804 mode is cycling ?
	Form current = none
	ObjectReference currentContainer = none
	Spell currentSpell = none
	int item_index = 0									; index into the container (first or last)	
	
	;Remove old Base
	if (old_outfit != -1)
		currentContainer = ContainerList[old_outfit] as ObjectReference

		while PEM_CurrentOutfit.GetSize() > 0		; remove all items
			current = PEM_CurrentOutfit.GetAt(0)
			PlayerRef.removeItem(current, 1, true, currentContainer)
			PEM_CurrentOutfit.RemoveAddedForm(current)
		endWhile
		
		while PEM_CurrentOutfitMultiple.GetSize() > 0
			current = PEM_CurrentOutfitMultiple.GetAt(0)
			count = PlayerRef.GetItemCount(current)
			PlayerRef.removeItem(current, count, true, currentContainer)
			PEM_CurrentOutfitMultiple.RemoveAddedForm(current)
		endWhile

	endif
	
	;Remove old Addon
	if (old_addon != -1)
		currentContainer = ContainerList[old_addon] as ObjectReference

		while PEM_CurrentOutfitAddon.GetSize() > 0
			current = PEM_CurrentOutfitAddon.GetAt(0)
			PlayerRef.removeItem(current, 1, true, currentContainer)
			PEM_CurrentOutfitAddon.RemoveAddedForm(current)
		endWhile

		while PEM_CurrentOutfitAddonMultiple.GetSize() > 0
			current = PEM_CurrentOutfitAddonMultiple.GetAt(0)
			count = PlayerRef.GetItemCount(current)
			PlayerRef.removeItem(current, count, true, currentContainer)
			PEM_CurrentOutfitAddonMultiple.RemoveAddedForm(current)
		endWhile
		
	EndIf
	
	if (custom_unequip == "Global " && unequip_all) || (custom_unequip == "Yes ")
		PlayerRef.UnequipAll()
	endIf
	
	PEM_CurrentOutfit.Revert()
	PEM_CurrentOutfitAddon.Revert()
	PEM_CurrentOutfitMultiple.Revert()
	PEM_CurrentOutfitAddonMultiple.Revert()
	

	
	;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
	
	;Add Base Outfit
	
	if (base_outfit && old_outfit != index) || (addon_outfit && old_outfit != -1) || (base_outfit && cycle_outfit)
	; proceed if new outfit is a Base and previously applied outfit was different
	;	or new outfit is an Addon and there was a Base outfit previously applied
	;		or new outfit is a Base cycling outfit
	;
	; index is the outfit to apply,
	;

		int temp_index = index			; install this new outfit as Base
		if (addon_outfit)				; but if new is an Addon,
			temp_index = old_outfit		;	reinstall old Base outfit instead
		endif

		setCurrentOutfit(temp_index)
		currentContainer = ContainerList[temp_index] as ObjectReference

		int cycle_maxindex = currentContainer.GetNumItems()	; get size of container
		item_index = 0										; start at first

		if getisModeCycle(temp_index) && cycle_maxindex > 0		; are we applying a cycling Base (container not empty) ?
			if addon_outfit 									;  and are we reinstalling because of a new Addon ?
				item_index = cycle_maxindex - 1					; 	get last entry (was just removed in previous section)
			Endif
			cycle_maxindex = 1									; get one item only
		endif

;		Debug.Notification("PEM: Processing Base index: " + temp_index + ", subindex: " + item_index + ", total items: " + cycle_maxindex) ; DEBUG

		while cycle_maxindex > 0	; process all items in container (removing one by one)

			current = currentContainer.GetNthForm(item_index)		; get first or last item

			count = currentContainer.GetItemCount(current)
			currentContainer.removeItem(current, count, true, PlayerRef)
			if count > 1
				PEM_CurrentOutfitMultiple.AddForm(current)
			else
				PEM_CurrentOutfit.AddForm(current)
			endIf
			
			if (current as Book)
				currentSpell = (current as Book).GetSpell()
				if (currentSpell)
					PlayerRef.EquipSpell(currentSpell,spell_slot as int)
					spell_slot = !spell_slot
				EndIf
			elseIf (current as Weapon)
				PlayerRef.EquipItemEx(current,(weapon_slot as int) + 1)
				weapon_slot = !weapon_slot
			else
				PlayerRef.EquipItemEx(current)
			endIf
			
			cycle_maxindex -= 1
		endWhile
		
	else
		setCurrentOutfit(-1)
	endIf


	;Add addon Outfit
	if ( (addon_outfit && old_addon != index) || (base_outfit && old_addon != -1) || (addon_outfit && cycle_outfit) )
	; 	proceed if new outfit is an Addon and previously applied Addon outfit was different
	;		or new outfit is an Base and there was an Addon outfit previously applied
	;			or new outfit is a Addon cycling outfit
	
		int temp_index = index			; install this new outfit as Addon
		if (base_outfit)				; but if new is a Base,
			temp_index = old_addon		;	reinstall old Addon outfit instead
		endif
		
		setCurrentOutfit(temp_index,true)
		currentContainer = ContainerList[temp_index] as ObjectReference

		int cycle_maxindex = currentContainer.GetNumItems()	; get size of container
		item_index = 0										; start at first
		
		if getisModeCycle(temp_index) && cycle_maxindex > 0		; are we applying a cycling Addon (container not empty) ?
			if base_outfit 										; are we reinstalling because of a new Base ?
				item_index = cycle_maxindex - 1					; 	get last entry (was just removed in previous section)
			EndIf
			cycle_maxindex = 1									; 	get one item only
		endif

;		Debug.Notification("PEM: Processing Addon index: " + temp_index + ", subindex: " + item_index + ", total items: " + cycle_maxindex) ; DEBUG
		
		while cycle_maxindex > 0									; process all items in container (removing one by one)
			current = currentContainer.GetNthForm(item_index)		; get first or last item
			count = currentContainer.GetItemCount(current)
			currentContainer.removeItem(current, count, true, PlayerRef)
			if count > 1
				PEM_CurrentOutfitAddonMultiple.AddForm(current)
			else
				PEM_CurrentOutfitAddon.AddForm(current)
			endIf
			
			if (current as Book)
				currentSpell = (current as Book).GetSpell()
				if (currentSpell)
					PlayerRef.EquipSpell(currentSpell,spell_slot as int)
					spell_slot = !spell_slot
				EndIf
			elseIf (current as Weapon)
				PlayerRef.EquipItemEx(current,(weapon_slot as int) + 1)
				weapon_slot = !weapon_slot
			else
				PlayerRef.EquipItemEx(current)
			endIf
			
			cycle_maxindex -= 1
		endWhile
	else
		setCurrentOutfit(-1,true)
	endIf
	gotostate("")
EndFunction

state ApplyLocked
	Function applyOutfit(int index)
		Debug.Notification("PEM: Busy changing to \"" + getCustomName(change_index) + "\"")
	EndFunction
endstate

Function resetApplyToggles()
	int index = 0
	int option = -1
	while index < max_index
		option = OID_Apply[index]
		SetToggleOptionValue(option, getIsCurrentOutfit(index),(index < (max_index - 1)))	
		index += 1
	EndWhile
EndFunction

String Function GetCustomControl(int keyCode)
	int index = 0
	while index < max_index
		if (getCustomHotkey(index) == keyCode)
			return "Player Equipment Manager: " + getCustomName(index)
		EndIf
		index += 1
	EndWhile
	return ""
EndFunction

bool Function checkCurrentLocation()
	currentLocation = PlayerRef.getCurrentLocation()
	return getCustomLocation()
EndFunction
