import * as GameTest from "mojang-gametest";
import { BlockLocation, MinecraftBlockTypes, MinecraftItemTypes, world } from "mojang-minecraft";
import GameTestExtensions from "./GameTestExtensions.js";

function placeDripstoneTip(test, pos, hanging, waterlogged = false) {
  const pointedDripstonePermutation = MinecraftBlockTypes.pointedDripstone.createDefaultBlockPermutation();
  pointedDripstonePermutation.getProperty("hanging").value = hanging;
  pointedDripstonePermutation.getProperty("dripstone_thickness").value = "tip";

  const pointedDripstoneBlock = test.getDimension().getBlock(test.worldBlockLocation(pos));
  pointedDripstoneBlock.setPermutation(pointedDripstonePermutation);
  pointedDripstoneBlock.isWaterlogged = waterlogged;
}

function assertDripstone(test, pos, hanging, thickness, waterlogged = false) {
  const testEx = new GameTestExtensions(test);
  test.assertBlockPresent(MinecraftBlockTypes.pointedDripstone, pos, true);
  test.assertIsWaterlogged(pos, waterlogged);
  testEx.assertBlockProperty("hanging", hanging, pos);
  testEx.assertBlockProperty("dripstone_thickness", thickness, pos);
}

function assertColumnBaseToTip(test, basePos, hanging, ...thicknesses) {
  let checkPos = basePos;
  for (const thickness of thicknesses) {
    assertDripstone(test, checkPos, hanging, thickness);
    if (hanging == true) {
      checkPos = checkPos.offset(0, -1, 0);
    } else {
      checkPos = checkPos.offset(0, 1, 0);
    }
  }
}

///
// Concrete Tests
///
GameTest.register("DripstoneTests", "thickness_update", (test) => {
  // Check that each stalactite got loaded correctly
  assertColumnBaseToTip(test, new BlockLocation(0, 12, 0), true, "base", "middle", "frustum", "tip");
  assertColumnBaseToTip(test, new BlockLocation(1, 12, 0), true, "base", "frustum", "tip");
  assertColumnBaseToTip(test, new BlockLocation(2, 12, 0), true, "frustum", "tip");
  assertColumnBaseToTip(test, new BlockLocation(3, 12, 0), true, "tip");

  // Check that each stalagmite got loaded correctly
  assertColumnBaseToTip(test, new BlockLocation(0, 2, 0), false, "base", "middle", "frustum", "tip");
  assertColumnBaseToTip(test, new BlockLocation(1, 2, 0), false, "base", "frustum", "tip");
  assertColumnBaseToTip(test, new BlockLocation(2, 2, 0), false, "frustum", "tip");
  assertColumnBaseToTip(test, new BlockLocation(3, 2, 0), false, "tip");

  // Extend each stalactite
  placeDripstoneTip(test, new BlockLocation(0, 8, 0), true);
  placeDripstoneTip(test, new BlockLocation(1, 9, 0), true);
  placeDripstoneTip(test, new BlockLocation(2, 10, 0), true);
  placeDripstoneTip(test, new BlockLocation(3, 11, 0), true);

  // Extend each stalagmite
  placeDripstoneTip(test, new BlockLocation(0, 6, 0), false);
  placeDripstoneTip(test, new BlockLocation(1, 5, 0), false);
  placeDripstoneTip(test, new BlockLocation(2, 4, 0), false);
  placeDripstoneTip(test, new BlockLocation(3, 3, 0), false);

  test.succeedIf(() => {
    // Check the shape of each stalactite
    assertColumnBaseToTip(test, new BlockLocation(0, 12, 0), true, "base", "middle", "middle", "frustum", "tip");
    assertColumnBaseToTip(test, new BlockLocation(1, 12, 0), true, "base", "middle", "frustum", "tip");
    assertColumnBaseToTip(test, new BlockLocation(2, 12, 0), true, "base", "frustum", "tip");
    assertColumnBaseToTip(test, new BlockLocation(3, 12, 0), true, "frustum", "tip");

    // Check the shape of each stalagmite
    assertColumnBaseToTip(test, new BlockLocation(0, 2, 0), false, "base", "middle", "middle", "frustum", "tip");
    assertColumnBaseToTip(test, new BlockLocation(1, 2, 0), false, "base", "middle", "frustum", "tip");
    assertColumnBaseToTip(test, new BlockLocation(2, 2, 0), false, "base", "frustum", "tip");
    assertColumnBaseToTip(test, new BlockLocation(3, 2, 0), false, "frustum", "tip");
  });
}).tag(GameTest.Tags.suiteDefault);

GameTest.register("DripstoneTests", "stalactite_fall", (test) => {
  const landingPos = new BlockLocation(1, 2, 1);
  test.assertEntityPresent("minecraft:item", landingPos, false);

  test.pressButton(new BlockLocation(0, 3, 0));
  test.succeedWhenEntityPresent("minecraft:item", landingPos, true);
})
  .tag("suite:java_parity")
  .tag(GameTest.Tags.suiteDisabled); //Test failed occasionally due to bug 587521: Collision box of pointed dripstone becomes larger and offsets when falling.

GameTest.register("DripstoneTests", "stalactite_fall_bedrock", (test) => {
  const landingPos = new BlockLocation(1, 2, 1);
  test.assertEntityPresent("minecraft:item", landingPos, false);

  test.pressButton(new BlockLocation(0, 3, 0));
  test.succeedWhenEntityPresent("minecraft:item", landingPos, true);
}).tag(GameTest.Tags.suiteDefault);

GameTest.register("DripstoneTests", "stalactite_hurt", (test) => {
  const poorInnocentVictimPos = new BlockLocation(1, 2, 1);
  const poorInnocentVictim = test.spawnWithoutBehaviors("minecraft:pig", poorInnocentVictimPos);

  test.pressButton(new BlockLocation(0, 6, 0));

  const healthComponent = poorInnocentVictim.getComponent("minecraft:health");

  test.succeedWhen(() => {
    test.assert(healthComponent.current < healthComponent.value, "Mob should be hurt!");
  });
}).tag(GameTest.Tags.suiteDefault);

GameTest.register("DripstoneTests", "stalagmite_break", (test) => {
  test.assertBlockPresent(MinecraftBlockTypes.pointedDripstone, new BlockLocation(1, 2, 1), true);
  test.assertBlockPresent(MinecraftBlockTypes.pointedDripstone, new BlockLocation(1, 3, 1), true);
  test.assertBlockPresent(MinecraftBlockTypes.pointedDripstone, new BlockLocation(1, 4, 1), true);

  test.pressButton(new BlockLocation(0, 3, 0));

  test.succeedWhen(() => {
    test.assertBlockPresent(MinecraftBlockTypes.pointedDripstone, new BlockLocation(1, 2, 1), false);
    test.assertBlockPresent(MinecraftBlockTypes.pointedDripstone, new BlockLocation(1, 3, 1), false);
    test.assertBlockPresent(MinecraftBlockTypes.pointedDripstone, new BlockLocation(1, 4, 1), false);
  });
}).tag(GameTest.Tags.suiteDefault);

GameTest.register("DripstoneTests", "stalagmite_stalactite_separation", (test) => {
  assertColumnBaseToTip(test, new BlockLocation(1, 2, 1), false, "frustum", "merge");
  assertColumnBaseToTip(test, new BlockLocation(1, 5, 1), true, "frustum", "merge");
  assertColumnBaseToTip(test, new BlockLocation(2, 2, 1), false, "frustum", "merge");
  assertColumnBaseToTip(test, new BlockLocation(2, 5, 1), true, "frustum", "merge");

  test.pressButton(new BlockLocation(0, 3, 0));
  test.pressButton(new BlockLocation(3, 4, 0));

  test.succeedWhen(() => {
    // the right-hand stalagmite should be gone
    test.assertBlockPresent(MinecraftBlockTypes.pointedDripstone, new BlockLocation(1, 2, 1), false);
    test.assertBlockPresent(MinecraftBlockTypes.pointedDripstone, new BlockLocation(1, 3, 1), false);

    // the right-hand stalactite should be intact, but the tip should no longer be a merged tip
    assertColumnBaseToTip(test, new BlockLocation(1, 5, 1), true, "frustum", "tip");

    // the left-hand stalagmite should be intact, but the tip should no longer be a merged tip
    assertColumnBaseToTip(test, new BlockLocation(2, 2, 1), false, "frustum", "tip");

    // the left-hand stalactite should be gone
    test.assertBlockPresent(MinecraftBlockTypes.pointedDripstone, new BlockLocation(2, 5, 1), false);
    test.assertBlockPresent(MinecraftBlockTypes.pointedDripstone, new BlockLocation(2, 4, 1), false);
  });
}).tag(GameTest.Tags.suiteDefault);

GameTest.register("DripstoneTests", "stalagmite_hurt", (test) => {
  const unluckyPig = test.spawn("minecraft:pig", new BlockLocation(1, 4, 1));
  const luckyPig = test.spawn("minecraft:pig", new BlockLocation(3, 4, 1));

  const unluckyPigHealthComponent = unluckyPig.getComponent("minecraft:health");
  const luckyPigHealthComponent = luckyPig.getComponent("minecraft:health");

  test.succeedWhen(() => {
    test.assert(unluckyPigHealthComponent.current < unluckyPigHealthComponent.value, "This pig should be hurt!");
    test.assert(luckyPigHealthComponent.current == luckyPigHealthComponent.value, "This pig shouldn't be hurt!");
  });
}).tag(GameTest.Tags.suiteDefault);

GameTest.register("DripstoneTests", "stalactite_fall_no_dupe", (test) => {
  test.pressButton(new BlockLocation(4, 9, 0));
  test.pressButton(new BlockLocation(8, 8, 0));
  test.pressButton(new BlockLocation(12, 6, 0));
  test.pressButton(new BlockLocation(16, 5, 0));

  test
    .startSequence()
    .thenExecuteAfter(60, () => {
      test.assertItemEntityCountIs(MinecraftItemTypes.pointedDripstone, new BlockLocation(2, 2, 2), 1, 5);
      test.assertItemEntityCountIs(MinecraftItemTypes.pointedDripstone, new BlockLocation(6, 2, 2), 1, 5);
      test.assertItemEntityCountIs(MinecraftItemTypes.pointedDripstone, new BlockLocation(10, 2, 2), 1, 2);
      test.assertItemEntityCountIs(MinecraftItemTypes.pointedDripstone, new BlockLocation(14, 2, 2), 1, 2);
    })
    .thenSucceed();
}).tag(GameTest.Tags.suiteDefault);

// SIG // Begin signature block
// SIG // MIIjogYJKoZIhvcNAQcCoIIjkzCCI48CAQExDzANBglg
// SIG // hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
// SIG // BgEEAYI3AgEeMCQCAQEEEBDgyQbOONQRoqMAEEvTUJAC
// SIG // AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
// SIG // sCbzMZtzUKl67t2pDMLUFQGYZJGYXkr7YkQdUegnWZWg
// SIG // gg2BMIIF/zCCA+egAwIBAgITMwAAAlKLM6r4lfM52wAA
// SIG // AAACUjANBgkqhkiG9w0BAQsFADB+MQswCQYDVQQGEwJV
// SIG // UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
// SIG // UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBv
// SIG // cmF0aW9uMSgwJgYDVQQDEx9NaWNyb3NvZnQgQ29kZSBT
// SIG // aWduaW5nIFBDQSAyMDExMB4XDTIxMDkwMjE4MzI1OVoX
// SIG // DTIyMDkwMTE4MzI1OVowdDELMAkGA1UEBhMCVVMxEzAR
// SIG // BgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1v
// SIG // bmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlv
// SIG // bjEeMBwGA1UEAxMVTWljcm9zb2Z0IENvcnBvcmF0aW9u
// SIG // MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEA
// SIG // 0OTPj7P1+wTbr+Qf9COrqA8I9DSTqNSq1UKju4IEV3HJ
// SIG // Jck61i+MTEoYyKLtiLG2Jxeu8F81QKuTpuKHvi380gzs
// SIG // 43G+prNNIAaNDkGqsENQYo8iezbw3/NCNX1vTi++irdF
// SIG // qXNs6xoc3B3W+7qT678b0jTVL8St7IMO2E7d9eNdL6RK
// SIG // fMnwRJf4XfGcwL+OwwoCeY9c5tvebNUVWRzaejKIkBVT
// SIG // hApuAMCtpdvIvmBEdSTuCKZUx+OLr81/aEZyR2jL1s2R
// SIG // KaMz8uIzTtgw6m3DbOM4ewFjIRNT1hVQPghyPxJ+ZwEr
// SIG // wry5rkf7fKuG3PF0fECGSUEqftlOptpXTQIDAQABo4IB
// SIG // fjCCAXowHwYDVR0lBBgwFgYKKwYBBAGCN0wIAQYIKwYB
// SIG // BQUHAwMwHQYDVR0OBBYEFDWSWhFBi9hrsLe2TgLuHnxG
// SIG // F3nRMFAGA1UdEQRJMEekRTBDMSkwJwYDVQQLEyBNaWNy
// SIG // b3NvZnQgT3BlcmF0aW9ucyBQdWVydG8gUmljbzEWMBQG
// SIG // A1UEBRMNMjMwMDEyKzQ2NzU5NzAfBgNVHSMEGDAWgBRI
// SIG // bmTlUAXTgqoXNzcitW2oynUClTBUBgNVHR8ETTBLMEmg
// SIG // R6BFhkNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtp
// SIG // b3BzL2NybC9NaWNDb2RTaWdQQ0EyMDExXzIwMTEtMDct
// SIG // MDguY3JsMGEGCCsGAQUFBwEBBFUwUzBRBggrBgEFBQcw
// SIG // AoZFaHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraW9w
// SIG // cy9jZXJ0cy9NaWNDb2RTaWdQQ0EyMDExXzIwMTEtMDct
// SIG // MDguY3J0MAwGA1UdEwEB/wQCMAAwDQYJKoZIhvcNAQEL
// SIG // BQADggIBABZJN7ksZExAYdTbQJewYryBLAFnYF9amfhH
// SIG // WTGG0CmrGOiIUi10TMRdQdzinUfSv5HHKZLzXBpfA+2M
// SIG // mEuJoQlDAUflS64N3/D1I9/APVeWomNvyaJO1mRTgJoz
// SIG // 0TTRp8noO5dJU4k4RahPtmjrOvoXnoKgHXpRoDSSkRy1
// SIG // kboRiriyMOZZIMfSsvkL2a5/w3YvLkyIFiqfjBhvMWOj
// SIG // wb744LfY0EoZZz62d1GPAb8Muq8p4VwWldFdE0y9IBMe
// SIG // 3ofytaPDImq7urP+xcqji3lEuL0x4fU4AS+Q7cQmLq12
// SIG // 0gVbS9RY+OPjnf+nJgvZpr67Yshu9PWN0Xd2HSY9n9xi
// SIG // au2OynVqtEGIWrSoQXoOH8Y4YNMrrdoOmjNZsYzT6xOP
// SIG // M+h1gjRrvYDCuWbnZXUcOGuOWdOgKJLaH9AqjskxK76t
// SIG // GI6BOF6WtPvO0/z1VFzan+2PqklO/vS7S0LjGEeMN3Ej
// SIG // 47jbrLy3/YAZ3IeUajO5Gg7WFg4C8geNhH7MXjKsClsA
// SIG // Pk1YtB61kan0sdqJWxOeoSXBJDIzkis97EbrqRQl91K6
// SIG // MmH+di/tolU63WvF1nrDxutjJ590/ALi383iRbgG3zkh
// SIG // EceyBWTvdlD6FxNbhIy+bJJdck2QdzLm4DgOBfCqETYb
// SIG // 4hQBEk/pxvHPLiLG2Xm9PEnmEDKo1RJpMIIHejCCBWKg
// SIG // AwIBAgIKYQ6Q0gAAAAAAAzANBgkqhkiG9w0BAQsFADCB
// SIG // iDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0
// SIG // b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1p
// SIG // Y3Jvc29mdCBDb3Jwb3JhdGlvbjEyMDAGA1UEAxMpTWlj
// SIG // cm9zb2Z0IFJvb3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5
// SIG // IDIwMTEwHhcNMTEwNzA4MjA1OTA5WhcNMjYwNzA4MjEw
// SIG // OTA5WjB+MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2Fz
// SIG // aGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UE
// SIG // ChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSgwJgYDVQQD
// SIG // Ex9NaWNyb3NvZnQgQ29kZSBTaWduaW5nIFBDQSAyMDEx
// SIG // MIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEA
// SIG // q/D6chAcLq3YbqqCEE00uvK2WCGfQhsqa+laUKq4Bjga
// SIG // BEm6f8MMHt03a8YS2AvwOMKZBrDIOdUBFDFC04kNeWSH
// SIG // fpRgJGyvnkmc6Whe0t+bU7IKLMOv2akrrnoJr9eWWcpg
// SIG // GgXpZnboMlImEi/nqwhQz7NEt13YxC4Ddato88tt8zpc
// SIG // oRb0RrrgOGSsbmQ1eKagYw8t00CT+OPeBw3VXHmlSSnn
// SIG // Db6gE3e+lD3v++MrWhAfTVYoonpy4BI6t0le2O3tQ5GD
// SIG // 2Xuye4Yb2T6xjF3oiU+EGvKhL1nkkDstrjNYxbc+/jLT
// SIG // swM9sbKvkjh+0p2ALPVOVpEhNSXDOW5kf1O6nA+tGSOE
// SIG // y/S6A4aN91/w0FK/jJSHvMAhdCVfGCi2zCcoOCWYOUo2
// SIG // z3yxkq4cI6epZuxhH2rhKEmdX4jiJV3TIUs+UsS1Vz8k
// SIG // A/DRelsv1SPjcF0PUUZ3s/gA4bysAoJf28AVs70b1FVL
// SIG // 5zmhD+kjSbwYuER8ReTBw3J64HLnJN+/RpnF78IcV9uD
// SIG // jexNSTCnq47f7Fufr/zdsGbiwZeBe+3W7UvnSSmnEyim
// SIG // p31ngOaKYnhfsi+E11ecXL93KCjx7W3DKI8sj0A3T8Hh
// SIG // hUSJxAlMxdSlQy90lfdu+HggWCwTXWCVmj5PM4TasIgX
// SIG // 3p5O9JawvEagbJjS4NaIjAsCAwEAAaOCAe0wggHpMBAG
// SIG // CSsGAQQBgjcVAQQDAgEAMB0GA1UdDgQWBBRIbmTlUAXT
// SIG // gqoXNzcitW2oynUClTAZBgkrBgEEAYI3FAIEDB4KAFMA
// SIG // dQBiAEMAQTALBgNVHQ8EBAMCAYYwDwYDVR0TAQH/BAUw
// SIG // AwEB/zAfBgNVHSMEGDAWgBRyLToCMZBDuRQFTuHqp8cx
// SIG // 0SOJNDBaBgNVHR8EUzBRME+gTaBLhklodHRwOi8vY3Js
// SIG // Lm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9N
// SIG // aWNSb29DZXJBdXQyMDExXzIwMTFfMDNfMjIuY3JsMF4G
// SIG // CCsGAQUFBwEBBFIwUDBOBggrBgEFBQcwAoZCaHR0cDov
// SIG // L3d3dy5taWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNS
// SIG // b29DZXJBdXQyMDExXzIwMTFfMDNfMjIuY3J0MIGfBgNV
// SIG // HSAEgZcwgZQwgZEGCSsGAQQBgjcuAzCBgzA/BggrBgEF
// SIG // BQcCARYzaHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3Br
// SIG // aW9wcy9kb2NzL3ByaW1hcnljcHMuaHRtMEAGCCsGAQUF
// SIG // BwICMDQeMiAdAEwAZQBnAGEAbABfAHAAbwBsAGkAYwB5
// SIG // AF8AcwB0AGEAdABlAG0AZQBuAHQALiAdMA0GCSqGSIb3
// SIG // DQEBCwUAA4ICAQBn8oalmOBUeRou09h0ZyKbC5YR4WOS
// SIG // mUKWfdJ5DJDBZV8uLD74w3LRbYP+vj/oCso7v0epo/Np
// SIG // 22O/IjWll11lhJB9i0ZQVdgMknzSGksc8zxCi1LQsP1r
// SIG // 4z4HLimb5j0bpdS1HXeUOeLpZMlEPXh6I/MTfaaQdION
// SIG // 9MsmAkYqwooQu6SpBQyb7Wj6aC6VoCo/KmtYSWMfCWlu
// SIG // WpiW5IP0wI/zRive/DvQvTXvbiWu5a8n7dDd8w6vmSiX
// SIG // mE0OPQvyCInWH8MyGOLwxS3OW560STkKxgrCxq2u5bLZ
// SIG // 2xWIUUVYODJxJxp/sfQn+N4sOiBpmLJZiWhub6e3dMNA
// SIG // BQamASooPoI/E01mC8CzTfXhj38cbxV9Rad25UAqZaPD
// SIG // XVJihsMdYzaXht/a8/jyFqGaJ+HNpZfQ7l1jQeNbB5yH
// SIG // PgZ3BtEGsXUfFL5hYbXw3MYbBL7fQccOKO7eZS/sl/ah
// SIG // XJbYANahRr1Z85elCUtIEJmAH9AAKcWxm6U/RXceNcbS
// SIG // oqKfenoi+kiVH6v7RyOA9Z74v2u3S5fi63V4GuzqN5l5
// SIG // GEv/1rMjaHXmr/r8i+sLgOppO6/8MO0ETI7f33VtY5E9
// SIG // 0Z1WTk+/gFcioXgRMiF670EKsT/7qMykXcGhiJtXcVZO
// SIG // SEXAQsmbdlsKgEhr/Xmfwb1tbWrJUnMTDXpQzTGCFXkw
// SIG // ghV1AgEBMIGVMH4xCzAJBgNVBAYTAlVTMRMwEQYDVQQI
// SIG // EwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4w
// SIG // HAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKDAm
// SIG // BgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25pbmcgUENB
// SIG // IDIwMTECEzMAAAJSizOq+JXzOdsAAAAAAlIwDQYJYIZI
// SIG // AWUDBAIBBQCggcAwGQYJKoZIhvcNAQkDMQwGCisGAQQB
// SIG // gjcCAQQwHAYKKwYBBAGCNwIBCzEOMAwGCisGAQQBgjcC
// SIG // ARUwLwYJKoZIhvcNAQkEMSIEIBrGs3NzJj0k5NZB6txE
// SIG // HX1BPrg/LMY9MP9DjXH9QbjfMFQGCisGAQQBgjcCAQwx
// SIG // RjBEoCSAIgBNAGkAbgBlAGMAcgBhAGYAdAAgAEIAZQBk
// SIG // AHIAbwBjAGuhHIAaaHR0cHM6Ly93d3cubWluZWNyYWZ0
// SIG // Lm5ldC8wDQYJKoZIhvcNAQEBBQAEggEAuTSDOzB8f2zh
// SIG // HbtdAJM1xCOAyoT2GYwhugoT3OLYL/nUsh98RhdNLJyF
// SIG // aLBtQi+Peb+3F5HfrC5wetbAt3vjZ/zUL7GQZbD7t1JB
// SIG // v9TmVxZ/KLyACb7sZUwfNM6MhyWn5GWxWn5sTd7uunOH
// SIG // TwdvL/fU+mRaOwvITgswhwo8hcwUVgPAm/C0TcpLl/Fw
// SIG // 0Z6jRoIA4fe0evr+m+zcbUZYseQBGzdKjnE3rDF7wehx
// SIG // ZENi2alVELESyBZrS18yKtX556nQQ9VZSC7O2ETpJu+V
// SIG // +1Nfgy12EMDoaTRkuKkd5WVQW8Jp95rwNklZUQz2o6rD
// SIG // MGhaU5CqwpcjTbVkFCrghaGCEvEwghLtBgorBgEEAYI3
// SIG // AwMBMYIS3TCCEtkGCSqGSIb3DQEHAqCCEsowghLGAgED
// SIG // MQ8wDQYJYIZIAWUDBAIBBQAwggFVBgsqhkiG9w0BCRAB
// SIG // BKCCAUQEggFAMIIBPAIBAQYKKwYBBAGEWQoDATAxMA0G
// SIG // CWCGSAFlAwQCAQUABCAMti1d+m+slqEJneD3WXpFfcDL
// SIG // 473F5i5WtiJz911tEwIGYfxEOidBGBMyMDIyMDIxMjAx
// SIG // NDMzNC4yMjVaMASAAgH0oIHUpIHRMIHOMQswCQYDVQQG
// SIG // EwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UE
// SIG // BxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENv
// SIG // cnBvcmF0aW9uMSkwJwYDVQQLEyBNaWNyb3NvZnQgT3Bl
// SIG // cmF0aW9ucyBQdWVydG8gUmljbzEmMCQGA1UECxMdVGhh
// SIG // bGVzIFRTUyBFU046NjBCQy1FMzgzLTI2MzUxJTAjBgNV
// SIG // BAMTHE1pY3Jvc29mdCBUaW1lLVN0YW1wIFNlcnZpY2Wg
// SIG // gg5EMIIE9TCCA92gAwIBAgITMwAAAVosuW5ENMtvKAAA
// SIG // AAABWjANBgkqhkiG9w0BAQsFADB8MQswCQYDVQQGEwJV
// SIG // UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
// SIG // UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBv
// SIG // cmF0aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1T
// SIG // dGFtcCBQQ0EgMjAxMDAeFw0yMTAxMTQxOTAyMTZaFw0y
// SIG // MjA0MTExOTAyMTZaMIHOMQswCQYDVQQGEwJVUzETMBEG
// SIG // A1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
// SIG // ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9u
// SIG // MSkwJwYDVQQLEyBNaWNyb3NvZnQgT3BlcmF0aW9ucyBQ
// SIG // dWVydG8gUmljbzEmMCQGA1UECxMdVGhhbGVzIFRTUyBF
// SIG // U046NjBCQy1FMzgzLTI2MzUxJTAjBgNVBAMTHE1pY3Jv
// SIG // c29mdCBUaW1lLVN0YW1wIFNlcnZpY2UwggEiMA0GCSqG
// SIG // SIb3DQEBAQUAA4IBDwAwggEKAoIBAQCwvVwcVw2tJwiu
// SIG // 9B3/hooejcUZdZgIcxSXaJPG7N8aSfO0zNvKxHh5GQGh
// SIG // Pim/RovVL06UN86xFslkqSEJMsb/1n9HCIQNZiTqsrtt
// SIG // Ezd5femb4M2K8zv60JT72ylu/aADuPmBbHpEAra8zPYU
// SIG // KEuQWBsD4fUM+ugbksR/UMHssDQULI5FaIpKQcZEvhF0
// SIG // iM6W2tmn750TF+yy3tnq8nPj5T15k0jnx9VjcpHNubvc
// SIG // 6OmV9IAHeI1KQc3TC5xwrROtmy9jztNHJgCbIraoGqL/
// SIG // t5Ra33dLDwAsExjvYsppSZW9rGr8I6kp4+UcxuEcRuq1
// SIG // Nou3rW50E6M0e9zhzBjlAgMBAAGjggEbMIIBFzAdBgNV
// SIG // HQ4EFgQUZHMlP/Ucj5J+0AaF2Tk0PMDeMWcwHwYDVR0j
// SIG // BBgwFoAU1WM6XIoxkPNDe3xGG8UzaFqFbVUwVgYDVR0f
// SIG // BE8wTTBLoEmgR4ZFaHR0cDovL2NybC5taWNyb3NvZnQu
// SIG // Y29tL3BraS9jcmwvcHJvZHVjdHMvTWljVGltU3RhUENB
// SIG // XzIwMTAtMDctMDEuY3JsMFoGCCsGAQUFBwEBBE4wTDBK
// SIG // BggrBgEFBQcwAoY+aHR0cDovL3d3dy5taWNyb3NvZnQu
// SIG // Y29tL3BraS9jZXJ0cy9NaWNUaW1TdGFQQ0FfMjAxMC0w
// SIG // Ny0wMS5jcnQwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAK
// SIG // BggrBgEFBQcDCDANBgkqhkiG9w0BAQsFAAOCAQEAAABt
// SIG // 7HxYG1IsOIFjFfFTsZWyBvwiDEonIg7SGZzy/ODjDeJS
// SIG // XleS3ia/h2AXKMea7lmZuPStvYgZib+lYlzLT/NaWGdR
// SIG // BTO99OmPsyyIf7Uue+D3lrB7zw9EDsUOBzjNhjT7xQHV
// SIG // L6PM+aYFf+RPpT7utblAiAUwxEohzvje62oKm9zBEz6k
// SIG // SjyzQCXCjw3GpSs55hj5Z82Oj9c6Kf+2vdFXAR0SikP4
// SIG // U75//AvGOm/jL5OZGCYeuyjW1VXLrgVshUjGUxxnpx57
// SIG // h1YPbLx+/pZv1f8CG2VCiiSTtB0JsqHckng/ySwuzTbA
// SIG // 2EaRTftvXS0bcdHlsmaLCZk/KimwSDCCBnEwggRZoAMC
// SIG // AQICCmEJgSoAAAAAAAIwDQYJKoZIhvcNAQELBQAwgYgx
// SIG // CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
// SIG // MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
// SIG // b3NvZnQgQ29ycG9yYXRpb24xMjAwBgNVBAMTKU1pY3Jv
// SIG // c29mdCBSb290IENlcnRpZmljYXRlIEF1dGhvcml0eSAy
// SIG // MDEwMB4XDTEwMDcwMTIxMzY1NVoXDTI1MDcwMTIxNDY1
// SIG // NVowfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hp
// SIG // bmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoT
// SIG // FU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMd
// SIG // TWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIwMTAwggEi
// SIG // MA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCpHQ28
// SIG // dxGKOiDs/BOX9fp/aZRrdFQQ1aUKAIKF++18aEssX8XD
// SIG // 5WHCdrc+Zitb8BVTJwQxH0EbGpUdzgkTjnxhMFmxMEQP
// SIG // 8WCIhFRDDNdNuDgIs0Ldk6zWczBXJoKjRQ3Q6vVHgc2/
// SIG // JGAyWGBG8lhHhjKEHnRhZ5FfgVSxz5NMksHEpl3RYRNu
// SIG // KMYa+YaAu99h/EbBJx0kZxJyGiGKr0tkiVBisV39dx89
// SIG // 8Fd1rL2KQk1AUdEPnAY+Z3/1ZsADlkR+79BL/W7lmsqx
// SIG // qPJ6Kgox8NpOBpG2iAg16HgcsOmZzTznL0S6p/TcZL2k
// SIG // AcEgCZN4zfy8wMlEXV4WnAEFTyJNAgMBAAGjggHmMIIB
// SIG // 4jAQBgkrBgEEAYI3FQEEAwIBADAdBgNVHQ4EFgQU1WM6
// SIG // XIoxkPNDe3xGG8UzaFqFbVUwGQYJKwYBBAGCNxQCBAwe
// SIG // CgBTAHUAYgBDAEEwCwYDVR0PBAQDAgGGMA8GA1UdEwEB
// SIG // /wQFMAMBAf8wHwYDVR0jBBgwFoAU1fZWy4/oolxiaNE9
// SIG // lJBb186aGMQwVgYDVR0fBE8wTTBLoEmgR4ZFaHR0cDov
// SIG // L2NybC5taWNyb3NvZnQuY29tL3BraS9jcmwvcHJvZHVj
// SIG // dHMvTWljUm9vQ2VyQXV0XzIwMTAtMDYtMjMuY3JsMFoG
// SIG // CCsGAQUFBwEBBE4wTDBKBggrBgEFBQcwAoY+aHR0cDov
// SIG // L3d3dy5taWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNS
// SIG // b29DZXJBdXRfMjAxMC0wNi0yMy5jcnQwgaAGA1UdIAEB
// SIG // /wSBlTCBkjCBjwYJKwYBBAGCNy4DMIGBMD0GCCsGAQUF
// SIG // BwIBFjFodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vUEtJ
// SIG // L2RvY3MvQ1BTL2RlZmF1bHQuaHRtMEAGCCsGAQUFBwIC
// SIG // MDQeMiAdAEwAZQBnAGEAbABfAFAAbwBsAGkAYwB5AF8A
// SIG // UwB0AGEAdABlAG0AZQBuAHQALiAdMA0GCSqGSIb3DQEB
// SIG // CwUAA4ICAQAH5ohRDeLG4Jg/gXEDPZ2joSFvs+umzPUx
// SIG // vs8F4qn++ldtGTCzwsVmyWrf9efweL3HqJ4l4/m87WtU
// SIG // VwgrUYJEEvu5U4zM9GASinbMQEBBm9xcF/9c+V4XNZgk
// SIG // Vkt070IQyK+/f8Z/8jd9Wj8c8pl5SpFSAK84Dxf1L3mB
// SIG // ZdmptWvkx872ynoAb0swRCQiPM/tA6WWj1kpvLb9BOFw
// SIG // nzJKJ/1Vry/+tuWOM7tiX5rbV0Dp8c6ZZpCM/2pif93F
// SIG // SguRJuI57BlKcWOdeyFtw5yjojz6f32WapB4pm3S4Zz5
// SIG // Hfw42JT0xqUKloakvZ4argRCg7i1gJsiOCC1JeVk7Pf0
// SIG // v35jWSUPei45V3aicaoGig+JFrphpxHLmtgOR5qAxdDN
// SIG // p9DvfYPw4TtxCd9ddJgiCGHasFAeb73x4QDf5zEHpJM6
// SIG // 92VHeOj4qEir995yfmFrb3epgcunCaw5u+zGy9iCtHLN
// SIG // HfS4hQEegPsbiSpUObJb2sgNVZl6h3M7COaYLeqN4DMu
// SIG // Ein1wC9UJyH3yKxO2ii4sanblrKnQqLJzxlBTeCG+Sqa
// SIG // oxFmMNO7dDJL32N79ZmKLxvHIa9Zta7cRDyXUHHXodLF
// SIG // VeNp3lfB0d4wwP3M5k37Db9dT+mdHhk4L7zPWAUu7w2g
// SIG // UDXa7wknHNWzfjUeCLraNtvTX4/edIhJEqGCAtIwggI7
// SIG // AgEBMIH8oYHUpIHRMIHOMQswCQYDVQQGEwJVUzETMBEG
// SIG // A1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
// SIG // ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9u
// SIG // MSkwJwYDVQQLEyBNaWNyb3NvZnQgT3BlcmF0aW9ucyBQ
// SIG // dWVydG8gUmljbzEmMCQGA1UECxMdVGhhbGVzIFRTUyBF
// SIG // U046NjBCQy1FMzgzLTI2MzUxJTAjBgNVBAMTHE1pY3Jv
// SIG // c29mdCBUaW1lLVN0YW1wIFNlcnZpY2WiIwoBATAHBgUr
// SIG // DgMCGgMVAMyABZi9xhx1nACWIt4WhsJaZRouoIGDMIGA
// SIG // pH4wfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hp
// SIG // bmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoT
// SIG // FU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMd
// SIG // TWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIwMTAwDQYJ
// SIG // KoZIhvcNAQEFBQACBQDlsU5nMCIYDzIwMjIwMjEyMDEw
// SIG // NjQ3WhgPMjAyMjAyMTMwMTA2NDdaMHcwPQYKKwYBBAGE
// SIG // WQoEATEvMC0wCgIFAOWxTmcCAQAwCgIBAAICIxoCAf8w
// SIG // BwIBAAICESkwCgIFAOWyn+cCAQAwNgYKKwYBBAGEWQoE
// SIG // AjEoMCYwDAYKKwYBBAGEWQoDAqAKMAgCAQACAwehIKEK
// SIG // MAgCAQACAwGGoDANBgkqhkiG9w0BAQUFAAOBgQBi0mld
// SIG // 9F2nVS685adFrZtxlXhHRCzJIEdwWZyFUi/35j7Yy+Sj
// SIG // hhaRSfq3ej2NQXGJENQPvKnsgPRgUaD2J0fqJ0Dijy6H
// SIG // GGK7RnEhW+5EmHA/te4TrM0XG9sMGpg9NeBCKB1CEFnT
// SIG // ozlZZcCpdUZmflfkA3j0TqstQz1lYo3yszGCAw0wggMJ
// SIG // AgEBMIGTMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpX
// SIG // YXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYD
// SIG // VQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAkBgNV
// SIG // BAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEw
// SIG // AhMzAAABWiy5bkQ0y28oAAAAAAFaMA0GCWCGSAFlAwQC
// SIG // AQUAoIIBSjAaBgkqhkiG9w0BCQMxDQYLKoZIhvcNAQkQ
// SIG // AQQwLwYJKoZIhvcNAQkEMSIEIDRPJNIac7fOv3UJsn7L
// SIG // k+G4ODx4z1jdtzg4fgnbjfF/MIH6BgsqhkiG9w0BCRAC
// SIG // LzGB6jCB5zCB5DCBvQQgk/yoJnTEsuRdMMOCGq/XJn+S
// SIG // LN6TNIVLiCJy+W2xf7AwgZgwgYCkfjB8MQswCQYDVQQG
// SIG // EwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UE
// SIG // BxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENv
// SIG // cnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGlt
// SIG // ZS1TdGFtcCBQQ0EgMjAxMAITMwAAAVosuW5ENMtvKAAA
// SIG // AAABWjAiBCBS7iqcVoUbsezGTRTYCfuvBcWlRVE2QaQJ
// SIG // 2TTlJ7DZUjANBgkqhkiG9w0BAQsFAASCAQAQ87aQI7e3
// SIG // a6hfz3T30jqV9KEg24QIhYYB3ZdG4zpLZE76KoI1o1+S
// SIG // Q0HeKq1CO+shpz1/c7FNtkVMaHp6EoH9JpaX55g+LmYd
// SIG // lCPsdpCoRozX56Wf2AzqA21My4m5yUd1GdgL2p+lhnOr
// SIG // u7HguPpsb38Y2xshRdeaigf9ZnGSeRfOUFsILReXY3wW
// SIG // LMVOSrwYRZ77Orant4NYdNzT67aWjZscDzW73VtpU3Mz
// SIG // oHCxZ1yrchpxHNCQ+bUFVEx0NiuFqs6xuXYdl18UQQZM
// SIG // 0CVpqBOiLLiwsfbUiHkyRTTeQIcWD/McePKcvmqw+7ys
// SIG // aKtEYllCDvhl8u6SMgdyJPAB
// SIG // End signature block
