<?php

namespace WPHealth\Actions\Admin;

if (!defined('ABSPATH')) {
    exit;
}

use WPHealth\Core\Hooks\ExecuteHooksBackend;
use WPHealth\Helpers\HealthcheckCategory;
use WPHealth\Helpers\Pages;
use WPHealth\Services\Api\Owner;

class Enqueue implements ExecuteHooksBackend
{
    public function __construct(Owner $getOwner)
    {
        $this->getOwnerService = $getOwner;
    }

    public function hooks()
    {
        add_action('admin_enqueue_scripts', [$this, 'adminEnqueueScripts']);
        add_action('admin_enqueue_scripts', [$this, 'adminEnqueueCSS']);
        add_action('admin_enqueue_scripts', [$this, 'pluginPage']);
    }

    /**
     * @param [type] $page
     *
     * @return void
     */
    public function pluginPage($page)
    {
        if ('plugins.php' !== $page) {
            return;
        }

        wp_enqueue_script('wp-umbrella-deactivate', WP_UMBRELLA_URL_DIST . '/deactivate-intent.js', ['jquery'], WP_UMBRELLA_VERSION, true);
        wp_localize_script('wp-umbrella-deactivate', 'WP_UMBRELLA_DATA', [
            'WP_UMBRELLA_URL_DIST'     => WP_UMBRELLA_URL_DIST,
            'ADMIN_AJAX_URL'           => wp_nonce_url(admin_url('admin-ajax.php'), 'wp_umbrella_deactivate_plugin'),
        ]);
        wp_localize_script('wp-umbrella-deactivate', 'wp_umbrella_i18n', [
            'modal_title' => __("Can we get some info on why you're disabling?", 'wp-health'),
            'reasons'     => [
                'deactivate_temporary'        => __("It's a temporary deactivation. I'm just debugging a problem.", 'wp-health'),
                'bad_support'                 => __('Support / Customer service was unsatisfactory', 'wp-health'),
                'bad_support_helper'          => __("We're sorry about that. Is there anything we could do to improve your experience?", 'wp-health'),
                'plugin_complicated'          => __('The plugin is too complicated to configure.', 'wp-health'),
                'plugin_complicated_helper'   => __('Our goal is to keep the plugin as simple as possible. If you need help or support, we can help you: support@wp-umbrella.com', 'wp-health'),
                'lack_feature'                => __('Lack of feature or functionnality', 'wp-health'),
                'lack_feature_helper'         => __("We're sorry about that. Is there anything we could do to improve your experience?", 'wp-health'),
            ],
            'button_submit' => __('Send & Deactivate', 'wp-health'),
            'cancel'        => __('Cancel', 'wp-health'),
            'skip'          => __('Skip & Deactivate', 'wp-health'),
        ]);
    }

    public function adminEnqueueCSS($page)
    {
        wp_enqueue_style('wp-umbrella-admin-global-css', WP_UMBRELLA_URL_DIST . '/css/admin-global-css.css', [], WP_UMBRELLA_VERSION);

        if (!in_array($page, ['settings_page_wp-umbrella-settings'], true) && false === strpos($page, 'wp-umbrella')) {
            return;
        }

        wp_enqueue_style('wp-umbrella-admin-css', WP_UMBRELLA_URL_DIST . '/css/admin-css.css', [], WP_UMBRELLA_VERSION);
    }

    /**
     * @see admin_enqueue_scripts
     *
     * @param string $page
     */
    public function adminEnqueueScripts($page)
    {
        if (!in_array($page, ['settings_page_wp-umbrella-settings'], true) && false === strpos($page, 'wp-umbrella')) {
            return;
        }

        $owner = null;
        if (wp_umbrella_allowed()) {
            $owner = $this->getOwnerService->getOwnerImplicitApiKey();
        }

        $allowTracking = get_option('wp_health_allow_tracking');

        $tips = wp_umbrella_get_service('Healthcheck')->getTips();

        $data = [
            'API_KEY'                   => wp_umbrella_get_option('api_key'),
            'API_URL'                   => WP_UMBRELLA_API_URL,
            'APP_URL'                   => WP_UMBRELLA_API_URL,
            'ADMIN_URL_PAGE_MONITORS'   => admin_url('options-general.php?page=' . Pages::SETTINGS),
            'WP_UMBRELLA_URL_DIST'      => WP_UMBRELLA_URL_DIST,
            'ADMIN_AJAX'                => admin_url('admin-ajax.php'),
            'USER'                      => $owner,
            'ALLOW_TRACKING'            => $allowTracking && !empty($allowTracking) ? 'true' : 'false',
            'HEALTHCHECK'               => wp_umbrella_get_service('Healthcheck')->getData(),
            'HEALTHCHECK_TIPS'          => wp_umbrella_get_service('Healthcheck')->getTips(),
            'HEALTHCHECK_CATEGORIES'    => HealthcheckCategory::getCategories(),
            'LIMIT_EXCEEDED'            => [
                'email_notifications' => false,
            ],
        ];

        if ($owner) {
            try {
                if (count($owner['project']['ProjectEmailNotification']) >= $owner['plan']['limit_email_notifications']) {
                    $data['LIMIT_EXCEEDED']['email_notifications'] = true;
                }
            } catch (\Exception $e) {
                $data['LIMIT_EXCEEDED']['email_notifications'] = false;
            }
        }

        wp_register_script('wp-umbrella-app', WP_UMBRELLA_URL_DIST . '/settings.js', ['wp-i18n'], WP_UMBRELLA_VERSION, true);
        wp_enqueue_script('wp-umbrella-app');

        if (function_exists('wp_set_script_translations')) {
            wp_set_script_translations('wp-umbrella-app', 'wp-health', WP_UMBRELLA_LANGUAGES);
        }
        wp_localize_script('wp-umbrella-app', 'WP_UMBRELLA_DATA', $data);
    }
}
