<?php

namespace WPHealth\Healthcheck;

if (!defined('ABSPATH')) {
    exit;
}

use WPHealth\Helpers\HealthcheckCategory;
use WPHealth\Helpers\HealthcheckState;

class WordPress
{
    /**
     * WordPress is up to date ?
     *
     * @return array
     */
    public function isUpToDate()
    {
        $core_update = false;

        $from_api = get_site_option('_site_transient_update_core');
        if (isset($from_api->updates) && is_array($from_api->updates)) {
            $core_update = $from_api->updates;
        }

        $isUpToDate = $core_update && (!isset($core_update[0]->response) || 'latest' == $core_update[0]->response);
        $data = [
            'key'      => 'is_up_to_date',
            'label'    => __('Is your WordPress up to date?', 'wp-health'),
            'category' => [HealthcheckCategory::SECURITY],
        ];

        if ($isUpToDate) {
            return  array_merge($data, [
                'message' => __('Your WordPress is up to date. Great job!', 'wp-health'),
                'code'    => HealthcheckState::GOOD,
            ]);
        } elseif (!$core_update) {
            return  array_merge($data, [
                'message' => __('Encountered an error. WordPress version not checked. Please check again later.', 'wp-health'),
                'code'    => HealthcheckState::UNKNOWN,
            ]);
        }

        return  array_merge($data, [
            'message' => __('Your WordPress is not up to date. Your site has not received the latest security fixes and is less secure from hackers. Please consider updating.', 'wp-health'),
            'code'    => HealthcheckState::BAD,
        ]);
    }

    /**
     * WordPress is indexable by robot (eg. Google) ?
     *
     * @return array
     */
    public function isIndexable()
    {
        $data = [
            'key'      => 'is_indexable',
            'label'    => __('Can your WordPress be crawled by SEO robots? (Google, Bing,...)', 'wp-health'),
            'category' => [HealthcheckCategory::SEO],
        ];

        if (1 === (int) get_option('blog_public')) {
            return array_merge($data, [
                'message' => sprintf(
                    __('Your WordPress can be indexed by Search Engines.  Great! We advice you install <a href="%s" target="_blank">SEOPress Plugin.</a> if you want to rank better!', 'wp-health'),
                    WP_UMBRELLA_APP_URL . '/go/seopress'
                ),
                'code'    => HealthcheckState::GOOD,
            ]);
        }

        return array_merge($data, [
            'message' => __('Your WordPress site is currently discouraging search engines from indexing your site. If you want search engines to index your site, you can disable this option by unchecking the "Search Engine Visibility" on the "Reading" page of the "Settings" menu.', 'wp-health'),
            'code'    => HealthcheckState::BAD,
        ]);
    }

    /**
     * @return array
     */
    public function getTestPHPVersion()
    {
        $data = [
            'key'      => 'is_last_php_version',
            'label'    => __('PHP Version', 'wp-health'),
            'category' => [HealthcheckCategory::PERFORMANCE, HealthcheckCategory::SECURITY],
        ];

        $response = wp_check_php_version();

        if (!$response || version_compare(PHP_VERSION, $response['recommended_version'], '>=')) {
            return array_merge($data, [
                /* translators: %s: The server PHP version. */
                'message' => sprintf(__('Your WordPress uses the PHP version %s. Great, your version is in accordance with WordPress recommendations.', 'wp-health'), PHP_VERSION),
                'code'    => HealthcheckState::GOOD,
            ]);
        }

        if ($response['is_supported']) {
            return array_merge($data, [
                /* translators: %s: The server PHP version. */
                'message' => sprintf(
                    __("You are not using the last PHP version (%s).  It's not to bad, but we recommend you to move to %s.", 'wp-health'),
                    PHP_VERSION,
                    $response['recommended_version']
                ),
                'code'    => HealthcheckState::GOOD,
            ]);
        }

        if ($response['is_secure']) {
            return array_merge($data, [
                /* translators: %s: The server PHP version. */
                'message' => sprintf(
                    __('You are not using the last PHP version (%s). This version of PHP is secured, but we recommend you to move to %s.', 'wp-health'),
                    PHP_VERSION,
                    $response['recommended_version']
                ),
                'code'    => HealthcheckState::BAD,
            ]);
        }

        return array_merge($data, [
            /* translators: %s: The server PHP version. */
            'message' => sprintf(
                __('Your WordPress uses outdated version of PHP (%s). This is no longer maintained and there may be security breaches. We recommend that you update your version of PHP as soon as possible. Recommanded : %s', 'wp-health'),
                PHP_VERSION,
                $response['recommended_version']
            ),
            'code'    => HealthcheckState::BAD,
        ]);
    }

    public function isDebugModeActive()
    {
        $data = [
            'label'       => __('Is debug mode enabled?', 'wp-health'),
            'key'         => 'is_debug_mode_active',
            'category'    => [HealthcheckCategory::SECURITY],
        ];

        if (defined('WP_DEBUG') && WP_DEBUG) {
            if (defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
                return array_merge($data, [
                    'message' => sprintf(
                        '<p>%s</p><p>%s</p>',
                        __('Your site has debug mode enabled as well as writing logs to a potentially public file.', 'wp-health'),
                        __('In production, you are not supposed to need to enable these options. In addition, using WP Umbrella, we retrieve information about potential errors that are generated. You can disable the WP_DEBUG constant and WP_DEBUG_LOG with peace of mind.', 'wp-health')
                    ),
                    'code'    => HealthcheckState::BAD,
                ]);
            }

            if (defined('WP_DEBUG_DISPLAY') && WP_DEBUG_DISPLAY) {
                return array_merge($data, [
                    'message' => sprintf(
                        '<p>%s</p><p>%s</p>',
                        __('Warning! Your site makes publicly visible any errors that may exist.', 'wp-health'),
                        __('In production, you are not supposed to need to enable these options. In addition, using WP Umbrella, we retrieve information about potential errors that are generated. You can disable the WP_DEBUG constant and WP_DEBUG_DISPLAY with peace of mind.', 'wp-health')
                    ),
                    'code'    => HealthcheckState::BAD,
                ]);
            }

            return array_merge($data, [
                'message' => sprintf(
                    '<p>%s</p><p>%s</p>',
                    __('Debug mode is enabled, PHP errors are potentially visible to your visitors..', 'wp-health'),
                    __('In production, you are not supposed to need to enable these options. In addition, using WP Umbrella, we retrieve information about potential errors that are generated. You can to set "false" the WP_DEBUG constant with peace of mind.', 'wp-health')
                ),
                'code'    => HealthcheckState::BAD,
            ]);
        }

        return array_merge($data, [
            'message' => sprintf(
                '<p>%s</p>',
                __("Great, the debug mode is disabled. Errors will not be visible to your visitors and this doesn't prevent our plugin from offering you the error log if some errors occur.", 'wp-health'),
            ),
            'code'    => HealthcheckState::GOOD,
        ]);
    }
}
