const co = require('co');
const cheerio = require('cheerio');

const platformCheckUrlMap = {
    wxb: /http[s]?:\/\/www.wxb.com\/article/,
    weixin: /^http[s]?:\/\/mp\.weixin\.qq\.com/,
    toutiao: /^http[s]?:\/\/www\.toutiao\.com/,
    baijiahao: /^http[s]?:\/\/baijiahao\.baidu\.com/,
    omqq: /^http[s]?:\/\/kuaibao\.qq\.com/,
    weibo: /^http[s]?:\/\/[^\.]*[\.]?weibo\.com/,
    yidianzixun: /^http[s]?:\/\/www\.yidianzixun\.com/,
    uc: /^http[s]?:\/\/[^\.]*.mp\.uc\.cn/,
    zhihu: /^http[s]?:\/\/zhuanlan\.zhihu\.com/,
    jianshu: /^http[s]?:\/\/www\.jianshu\.com/,
    qq: /^http[s]?:\/\/post\.mp\.qq\.com/,
    sohu: /^http[s]?:\/\/mt\.sohu\.com/
};

function ArticleFetch() {}

ArticleFetch.prototype.escape2Html = function (str) {
    if (!str) {
        return null;
    }
    const arrEntities = { 'lt': '<', 'gt': '>', 'nbsp': ' ', 'amp': '&', 'quot': '"' };
    return str.replace(/&(lt|gt|nbsp|amp|quot);/ig, function (all, t) { return arrEntities[t]; });
};

ArticleFetch.prototype.clearSpace = function (str) {
    return str.replace(/\s\r\n/ig, '');
};

ArticleFetch.prototype.getUrlType = function (url) {
    const platforms = Object.keys(platformCheckUrlMap);
    let type = null;

    for (let i = 0; i < platforms.length; i++) {
        if (platformCheckUrlMap[platforms[i]].test(url)) {
            type = platforms[i];
            break;
        }
    }

    return type;
};

ArticleFetch.prototype.formatStr = function (str) {
    if (str) {
        str = this.escape2Html(this.clearSpace(str));
    }

    return str || '';
};

ArticleFetch.prototype.formatContent = function (content) {
    if (!content) {
        return '';
    }
    const self = this;
    const $ = cheerio.load(content, {
        decodeEntities: false
    });

    $('img').each(function () {
        const src = $(this).attr('src');
        if (/^\/\//.test(src)) {
            $(this).attr('src', `http:${self.escape2Html(decodeURIComponent(src))}`);
        }
    });

    return $.html();
};

const articleFetch = new ArticleFetch();

module.exports = {
    send: co.wrap(function*(ctx, next) {
        const { url } = ctx.message;
        const type = articleFetch.getUrlType(url);

        if (type) {
            ctx.body = yield ctx.app.sendMessage({
                name: `/article/fetch/${type}`,
                url
            }).then(res => ({
                title: articleFetch.formatStr(res.body.title),
                content: articleFetch.formatContent(res.body.content),
                headImg: articleFetch.escape2Html(res.body.headImg),
                copyright_info: res.body.copyright_info || null
            })).catch(e => {
                console.log(e);
                return {
                    content: null
                };
            });
        }

        yield next();
    }),

    wxb: co.wrap(function*(ctx, next) {
        ctx.body = yield ctx.app.WxbApi.fetchWxbArticle(ctx.message.url).then(res => ({
            title: res.title,
            content: ctx.app.PlatformApi.beautifyHtml(res.content),
            headImg: res.headImg
        }));

        yield next();
    }),

    weixin: co.wrap(function*(ctx, next) {
        ctx.body = yield ctx.app.PlatformApi.WeiXin.fetchWeiXinArticle(ctx.message.url);

        yield next();
    }),

    toutiao: co.wrap(function*(ctx, next) {
        const body = yield ctx.app.PlatformApi.get(ctx.message.url);
        const $ = cheerio.load(body, {
            decodeEntities: false
        });

        ctx.body = {
            title: $('.article-title').text(),
            content: $('.article-content').html()
        };

        yield next();
    }),

    baijiahao: co.wrap(function*(ctx, next) {
        const body = yield ctx.app.PlatformApi.get(ctx.message.url);
        const $ = cheerio.load(body, {
            decodeEntities: false
        });

        ctx.body = {
            title: $('.mth-header').text(),
            content: $('.mth-editor-content').html(),
            headImg: $('.mth-editor-content').data('coverimage')
        };

        yield next();
    }),

    omqq: co.wrap(function*(ctx, next) {
        const body = yield ctx.app.PlatformApi.get(ctx.message.url);
        const $ = cheerio.load(body, {
            decodeEntities: false
        });

        let title = null;
        let content = '';
        let headImg = null;
        $('#content').children().each(function (key) {
            if (key === 0) {
                title = $(this).text().replace(/[\s\n\r]/ig, '');
            }

            if (key > 1) {
                const attr = Object.keys(this.attribs).map(item => (`${item}=${this.attribs[item]}`));
                content += `<${this.tagName}${attr.length > 0 ? ` ${attr.join(' ')}` : ''}>
                    ${$(this).html()}</${this.tagName}>`;
            }
        });

        const match = body.match(/contentModel = ({[^}]*})/);
        if (match) {
            const info = this.eval(`(${articleFetch.clearSpace(match[1])})`);
            headImg = info.img_url;
        }

        ctx.body = {
            title,
            content,
            headImg
        };

        yield next();
    }),

    weibo: co.wrap(function*(ctx, next) {
        const body = yield ctx.app.PlatformApi.get(ctx.message.url, {
            follow_max: 10,
            headers: {
                Cookie: 'SUB=_2AkMvExydf8NhqwJRmP4XxWjhZYtxzAjEieLBAH7sJRMxHRl-yT83qkIltRBS0E4FSQh0Guvi6BGK_ENBJvQZZQ..;',
                // 过期时间2017-12-13 2:22:34
            }
        });

        const $ = cheerio.load(body, {
            decodeEntities: false
        });

        ctx.body = {
            title: $('div[node-type="articleTitle"]').text(),
            content: $('.WB_editor_iframe').html(),
            headImg: $('img[node-type="articleHeaderPic"]').attr('src')
        };

        yield next();
    }),

    yidianzixun: co.wrap(function*(ctx, next) {
        const body = yield ctx.app.PlatformApi.get(ctx.message.url);
        const $ = cheerio.load(body, {
            decodeEntities: false
        });

        let headImg = null;
        const match = body.match(/yidian.share_info = ({[^}]*})/);
        if (match) {
            const json = this.eval(`(${articleFetch.clearSpace(match[1])})`);
            headImg = json.share_image;
        }

        ctx.body = {
            title: $('.content-hd h2').text() || $('.content-hd-media h2').text(),
            content: $('.content-bd body').html(),
            headImg
        };

        yield next();
    }),

    uc: co.wrap(function*(ctx, next) {
        const body = yield ctx.app.PlatformApi.get(ctx.message.url);
        const $ = cheerio.load(body);
        const json = JSON.parse($('textarea#article_content_json').val()).data;

        ctx.body = {
            title: json.title,
            content: json.content,
            headImg: json.cover_url
        };

        yield next();
    }),

    zhihu: co.wrap(function*(ctx, next) {
        const match = ctx.message.url.match(/p\/([\d]*$)/);
        if (match) {
            const api = `https://zhuanlan.zhihu.com/api/posts/${match[1]}`;
            const body = yield ctx.app.PlatformApi.get(api);

            ctx.body = {
                title: body.title,
                content: body.content,
                headImg: body.titleImage
            };
        }
        yield next();
    }),

    jianshu: co.wrap(function*(ctx, next) {
        const body = yield ctx.app.PlatformApi.get(ctx.message.url);
        const $ = cheerio.load(body, {
            decodeEntities: false
        });

        ctx.body = {
            title: $('meta[property="og:title"]').attr('content'),
            content: $('.show-content').html(),
            headImg: $('.image-package img').attr('src')
        };

        yield next();
    }),

    qq: co.wrap(function*(ctx, next) {
        const body = yield ctx.app.PlatformApi.get(ctx.message.url);
        const $ = cheerio.load(body, {
            decodeEntities: false
        });

        $('#js_content').find('svg').each(function () {
            $(this).remove();
        });

        ctx.body = {
            title: $('#activity-name').text(),
            content: $('#js_content').html(),
            headImg: $('meta[itemprop="image"]').attr('content')
        };

        yield next();
    }),

    sohu: co.wrap(function*(ctx, next) {
        const body = yield ctx.app.PlatformApi.get(ctx.message.url);
        const $ = cheerio.load(body, {
            decodeEntities: false
        });

        ctx.body = {
            title: $('.news-title h1').text(),
            content: $('div[itemprop="articleBody"]').html(),
            headImg: $('#mpCover img').attr('src')
        };

        yield next();
    }),

    none: co.wrap(function*(ctx, next) {
        if (!ctx.body.content) {
            const { url } = ctx.message;
            ctx.body = yield ctx.app.PlatformApi.fetchPageArticle(url);
        }

        yield next();
    })
};
