const co = require('co');
const fsExtra = require('fs-extra');
const path = require('path');
const util = require('../modules/util');
const ImageModel = require('../model/Image');
const cheerio = require('cheerio');
const baidu = require('../modules/baidu-search');
const fs = require('fs');
const os = require('os');
const http = require('http');
const https = require('https');
const lodash = require('lodash');
const OnlineImgReg = /<img.*?src=["'](.*?)["'].*?>/g;
const backgroundReg = /background.*?url\s*\(["']?(.*?)["']?\)/g;
const uniq = lodash.uniq;
const lodashTemplate = lodash.template;
const moment = require('moment');
const fileType = require('file-type');

function _getAllImgUrl(html) {
    const l = [];
    let _match;
    while ((_match = OnlineImgReg.exec(html)) != null) {
        l.push(_match[1]);
    }
    while ((_match = backgroundReg.exec(html)) != null) {
        l.push(_match[1]);
    }
    return uniq(l);
}

function getImgSrc(html) {
    return _getAllImgUrl(html).map(url => (url.replace('file://', '')));
}

function createCanvas(image, pixelCrop, ratio) {
    const canvas = document.createElement('canvas');
    canvas.setAttribute('width', `${pixelCrop.width * ratio}px`);
    canvas.setAttribute('height', `${pixelCrop.height * ratio}px`);
    canvas.getContext('2d').drawImage(
        image,
        pixelCrop.x,
        pixelCrop.y,
        pixelCrop.width,
        pixelCrop.height,
        0,
        0,
        pixelCrop.width * ratio,
        pixelCrop.height * ratio
    );

    return canvas;
}

module.exports = {
    getImageInfo: co.wrap(function* (ctx, next) {
        ctx.body = yield util.Img.getImageInfo(ctx.message.url);

        yield next();
    }),

    addVoice: co.wrap(function*(ctx, next) {
        ctx.log('add voice.');
        const app = ctx.app;
        const sourceFile = ctx.message.sourceFile;
        const ext = path.extname(sourceFile);
        const targetFile = path.join(app.materialDir, Date.now() + ext);
        fsExtra.copySync(sourceFile, targetFile);
        let playLength = yield util.Media.meta(targetFile);
        console.log(playLength);
        playLength = playLength.duration * 1000;
        ctx.body = {
            sourceFile,
            targetFile,
            playLength,
            playLengthText: `${Math.floor(playLength / 60000)}:${(playLength % 60000) / 1000}`
        };
        yield next();
    }),

    saveLocalImage: co.wrap(function* (ctx, next) {
        // console.log(ctx.message.img)
        ctx.log('saveLocalImage');
        const app = ctx.app;
        const img = ctx.message.img;
        const pixelCrop = ctx.message.pixelCrop;
        let type = null;
        const hasHttp = img.match(/^http[s]?/);
        const hp = {
            http,
            https,
        };
        if (hasHttp) {
            type = yield new Promise(resolve => {
                hp[hasHttp[0]].get(img, res => {
                    res.once('data', chunk => {
                        res.destroy();
                        resolve(fileType(chunk));
                    });
                });
            });
        } else {
            let typeMatch = img.match(/\.([^/.]*)$/);
            typeMatch = typeMatch ? typeMatch[1] : 'png';
            type = {
                ext: typeMatch,
                mime: `image/${typeMatch}`
            };
        }
        if (!hasHttp && type.ext === 'gif') {
            ctx.body.localImgUrl = img.match(/file:\/\/(.*)$/)[1];
        } else {
            let data = null;
            if (type.ext !== 'gif') {
                const image = document.createElement('img');
                image.src = img;
                const imgData = yield new Promise(resolve => {
                    image.onload = () => {
                        resolve(createCanvas(image, pixelCrop, 1).toDataURL(type.mime, 0.8));
                    };
                });
                const base64Data = imgData.replace(/^data:image\/\w+;base64,/, '');
                data = new Buffer(base64Data, 'base64').toString('binary');
            } else {
                data = yield new Promise(resolve => {
                    hp[hasHttp[0]].get(img, res => {
                        let imgData = '';

                        res.setEncoding('binary');
                        res.once('data', chunk => {
                            imgData += chunk;
                        });

                        res.on('end', () => {
                            resolve(imgData);
                        });
                    });
                });
            }

            const currentDate = new Date();
            let localImgUrl = path.join(app.materialDir, `${currentDate.getTime()}.${type.ext}`);
            yield new Promise(resolve => {
                fs.writeFile(localImgUrl, data, 'binary', (err) => {
                    if (err) {
                        console.log(err);
                    } else {
                        if (os.platform() != 'darwin') {
                            localImgUrl = localImgUrl.replace(/\\/g, '\\\\');
                        }

                        resolve(localImgUrl);
                    }
                });
            }).then(() => {
                ctx.body = {
                    localImgUrl
                };
                return;
            });
        }
        yield next();
    }),

    fetchWeiXinArticle: co.wrap(function* (ctx, next) {
        try {
            const res = yield ctx.app.PlatformApi.WeiXin.fetchWeiXinArticle(ctx.message.url);
            ctx.body = res;
        } catch (e) {
            ctx.log(e);
        }
        yield next();
    }),

    fetchImages: co.wrap(function* (ctx, next) {
        ctx.log('fetchImages');
        const res = ctx.body;
        const result = {
            title: res.title,
            copyright_info: res.copyright_info
        };
        result.content = res.content.replace(/data-src=["'].*?["']/g, '');
        // 匹配所有的图片路径
        const imgList = getImgSrc(result.content);
        // 封面图片
        imgList.push(res.headImg);

        const imgListRes = yield ctx.app.WxbApi.fetchImg(imgList);

        result.headImg = imgListRes.pop();
        imgListRes.forEach(imgItem => {
            if (imgItem && imgItem.onlineImgUrl) {
                result.content = result.content.replace(imgItem.onlineImgUrl, `file://${imgItem.localImgUrl}`);
            }
        });

        ctx.body = result;
        yield next();
    }),

    convertOnlyVideo: co.wrap(function* (ctx, next) {
        const $ = cheerio.load(ctx.message.content, { decodeEntities: false });
        const frame = $('.video_iframe').eq(0);
        const videoTemplate = {
            type0: '<p><%= html%></iframe>' +
                '<strong style="width: 500px;height: 42px;margin-top: -42px;' +
                'color: rgb(255, 255, 255);display: block;text-align: center;">' +
                '关注我们查看更多视频</strong>' +
                '</p><section style="margin-top:-58px;"></section>',
            type1: '<p><%= html%></iframe></p>'
        };
        // 修改iframe高度和宽度
        frame.attr('height', 280).attr('width', 500);
        // 转换视频路径，去除其他参数
        const match = ctx.message.content.match(/<iframe[^>]*?vid=([\w]+)[^>]*>/im);
        if (!match) {
            ctx.body = {
                ret: 0,
                message: '视频地址不正确，缺少vid'
            };
            yield next();
        }
        const src = `https://v.qq.com/iframe/preview.html?vid=${match[1]}`;
        frame.attr('src', src).attr('data-src', src);
        // 获取首个视频iframe
        const iframes = $.html('.video_iframe');
        const html = iframes.match(/<iframe[^>]*>/im)[0];
        const template = lodashTemplate(videoTemplate[`type${ctx.message.type}`]);
        ctx.body = {
            ret: 1,
            template: template({
                html
            })
        };
        yield next();
    }),

    addImage: co.wrap(function* (ctx, next) {
        const token = ctx.message.token;
        const localImgUrl = ctx.message.localImgUrl;
        try {
            const res = yield ctx.app.WxbApi.uploadImage(token, localImgUrl);
            const file = res.full_url;
            ctx.body = {
                sourceFile: localImgUrl,
                file
            };
            const image = ImageModel.create({
                sourceFile: localImgUrl,
                file
            });
            image.save();
        } catch (e) {
            ctx.body = {
                err: 1,
                msg: typeof e === 'object' ? e.message : e
            };
        }

        yield next();
    }),

    searchImage: co.wrap(function* (ctx, next) {
        const { key, page, limit } = ctx.message;
        try {
            const list = yield baidu.getSearchImageList(key, (page - 1) * limit, limit);
            ctx.body = {
                page,
                list: list.data,
                hasNext: list.data.length > 0,
                prediction: list.listNum
            };
            console.log('baidu:', list);
            ctx.log('baidu::: searchImage:');
            ctx.log(list);
        } catch (e) {
            ctx.log(e);
        }
        yield next();
    }),

    saveNetworkImageToLocal: co.wrap(function* (ctx, next) {
        yield new Promise(resolve => {
            http.get(ctx.message.url, function (res) {
                res.setEncoding('binary');
                let imageData = '';
                res.on('data', function (data) {
                    imageData += data;
                }).on('end', function () {
                    console.log(imageData);
                    console.log(imageData.length);
                    ctx.message.img = 'data:' + res.headers['content-type'] + ';base64,' + new Buffer(imageData, 'binary').toString('base64');
                    resolve(true);
                });
            });
        });
        yield next();
    }),

    searchMusic: co.wrap(function*(ctx, next) {
        yield next();
    }),

    getMySpaceImages: co.wrap(function*(ctx, next) {
        const { token, page } = ctx.message;
        const res = yield ctx.app.WxbApi.getMySpaceImages(token, {
            page,
            pageSize: 10
        });
        ctx.body = {
            data: res.data,
            pager: {
                page: 1,
                total: res.totalCount,
                perpage: 10
            }
        };
        yield next();
    }),

    getPreviewUrl: co.wrap(function*(ctx, next) {
        const { message, app } = ctx;
        const { article, type, account, token } = message;

        if (!type) {
            yield next();
            return;
        }
        let tmpl = fs.readFileSync(path.join(app.manifest.source, `assest/template/${type}/index.html`), 'utf-8');
        const src = path.join(app.materialDir, `${type}.html`);

        let templateBefore = '';
        let templateAfter = '';
        if (typeof article[type] === 'object' && article[type].isTemplate && token && account.uid) {
            let templateInfo = yield ctx.app.WxbApi.getTemplate(token, account.uid, type);
            templateInfo = JSON.parse(templateInfo.data);

            if (article[type].templateChecked.includes('followGuide')) {
                templateBefore += `<section contenteditable="false"
                    class="wxb-editor-template before">${templateInfo.follow_guide}</section>`;
            }

            if (article[type].templateChecked.includes('signature')) {
                templateAfter += `<section contenteditable="false"
                    class="wxb-editor-template before">${templateInfo.signature}</section>`;
            }

            if (article[type].templateChecked.includes('readSourceGuide')) {
                templateAfter += `<section contenteditable="false"
                    class="wxb-editor-template before">${templateInfo.read_source_guide}</section>`;
            }
        }

        let authorName = '';

        if (article[type]) {
            switch (type) {
            case 'weixin':
                authorName = article[type].author || '';
                break;
            case 'qq':
                authorName = article[type].author || '';
                break;
            case 'weibo':
                authorName = article[type].writer || '';
                break;
            case 'uc':
                authorName = article[type].author || '';
                break;
            default:
            }
        }

        const replaceData = {
            title: {
                rep: /<%title%>/gi,
                value: article[type] && article[type].title || ''
            },
            templateBefore: {
                rep: /<%templateBefore%>/gi,
                value: templateBefore
            },
            content: {
                rep: /<%content%>/gi,
                value: article[type] && article[type].content || ''
            },
            templateAfter: {
                rep: /<%templateAfter%>/gi,
                value: templateAfter
            },
            userIconSrc: {
                rep: /<%userIconSrc%>/gi,
                value: 'http://download.weituibao.com/desktop/mmp/default_user.png'
            },
            userName: {
                rep: /<%userName%>/gi,
                value: account && account.username || ''
            },
            avatar: {
                rep: /<%avatar%>/gi,
                value: account && account.avatar || ''
            },
            authorName: {
                rep: /<%authorName%>/gi,
                value: authorName
            },
            userText: {
                rep: /<%userText%>/gi,
                value: '账号介绍'
            },
            time: {
                rep: /<%time%>/gi,
                value: moment().format('YYYY-MM-DD')
            },
            cover: {
                rep: /<%cover%>/gi,
                value: (article[type] && article[type].cover) ? article[type].cover.file : ''
            }
        };

        Object.keys(replaceData).forEach(k => {
            tmpl = tmpl.replace(replaceData[k].rep, replaceData[k].value);
        });
        fs.writeFileSync(src, tmpl);
        console.log(src);
        ctx.body[type] = `file:///${src}`;
        yield next();
    }),

    readPreviewUrl: co.wrap(function*(ctx, next) {
        const { src } = ctx.message;
        let body = fs.readFileSync(src.substr(8, src.length - 25), 'utf-8');

        body = body.replace(/[\t\r\n]/g, '');
        body = body.replace(/"/g, "'");
        ctx.body = body;
        yield next();
    }),

    previewSendPhone: co.wrap(function*(ctx, next) {
        const { user, article, preusername, type } = ctx.message;
        const body = yield ctx.app.PlatformApi[type].previewSendPhone(user, article, preusername);

        ctx.body = body;

        yield next();
    })
};
