const co = require('co');
const util = require('../modules/util');
const platformConfig = require('../config/platform');
const cheerio = require('cheerio');
const path = require('path');
const ImageModel = require('../model/Image');
const ArticleModel = require('../model/Article');
const MultiArticleModel = require('../model/MultiArticle');
const AccountModel = require('../model/Account');
const SyncModel = require('../model/Sync');
const AppConfigModel = require('../model/AppConfig');
const PLAT_FORM = require('../config/platform');

function _checkArticle(_article, checkService) {
    const errs = [];
    if (!_article) return errs;
    if (!_article.title && !_article.content) return errs; // 没有标题和内容的认为不需要同步
    const checkMap = _article.checkMap;
    // const checkService = ctx.app.PlatformApi[PLAT_FORM[key].apiName].ArticleCheckMap;
    Object.keys(checkService.required).forEach(valKey => {
        const _checker = checkService.required[valKey];
        const msg = _checker(_article[valKey]);
        if (!msg) return;
        errs.push(msg);
    });
    Object.keys(checkService.optional).forEach(valKey => {
        const _checker = checkService.optional[valKey];
        if (!checkMap || !checkMap[valKey]) {
            return;
        }
        const msg = _checker(_article[valKey]);
        if (!msg) return;
        errs.push(msg);
    });
    return errs;
}

module.exports = {
    // 获取需要同步的任务
    getNeedSyncRecords: co.wrap(function* (ctx, next) {
        const needSyncRecords = yield SyncModel.find({
            status: 0
        }).then(syncRecords0 => {
            return SyncModel.find({
                status: 1
            }).then(syncRecords1 => {
                return syncRecords1.concat(syncRecords0);
            });
        });
        ctx.body.needSyncRecords = needSyncRecords;
        next();
    }),
    // 预校验
    checkArticle: co.wrap(function* (ctx, next) {
        ctx.log('预检查');
        const errs = {};

        ctx.message.article.platform_list.forEach(key => {
            ctx.log(`预检查${key}平台文章`);
            const onePlatformArticle = ctx.message.article[key];
            const checkService = ctx.app.PlatformApi[PLAT_FORM[key].apiName].ArticleCheckMap;

            if (typeof onePlatformArticle === 'object') {
                if (typeof onePlatformArticle.length === 'undefined') {
                    const result = _checkArticle(onePlatformArticle, checkService);
                    if (result.length) {
                        errs[key] = [result];
                    }
                } else {
                    let isErr = false;
                    const results = onePlatformArticle.map(article => {
                        const articleErrs = _checkArticle(article, checkService);
                        if (articleErrs.length) {
                            isErr = true;
                        }
                        return articleErrs;
                    });
                    if (isErr) errs[key] = results;
                }
            }
        });
        ctx.body.errs = errs;
        yield next();
    }),
    saveSyncRecords: co.wrap(function* (ctx, next) {
        ctx.log(ctx.message.data);
        const syncRecords = yield Promise.all(ctx.message.data.map(syncRecord => SyncModel.saveSyncRecord(syncRecord)));
        ctx.body.syncRecords = syncRecords;
        yield next();
    }),
    // 获取一条数据 query要传article_id 和 account_uid
    getOneSyncRecord: co.wrap(function* (ctx, next) {
        const syncRecord = yield SyncModel.findOne(ctx.message.query);
        ctx.body.syncRecord = syncRecord;
        yield next();
    }),
    // 获取一组数据
    getSyncRecords: co.wrap(function* (ctx, next) {
        const syncRecords = yield SyncModel.find(ctx.message.query);
        ctx.body.syncRecords = syncRecords;
        yield next();
    }),

    // 预处理，读取账号，文章信等息，基本判断
    syncPrepare: co.wrap(function* (ctx, next) {
        const app = ctx.app;
        const { syncRecordId, resetOneMaterial } = ctx.message;
        ctx.log('开始同步文章到单个账号');
        try {
            ctx.syncRecord = yield SyncModel.findOne({
                _id: syncRecordId
            });
            ctx.log('同步记录', ctx.syncRecord);
            if (ctx.syncRecord.status === -2) {
                throw new Error('已取消该任务');
            }
            if (ctx.syncRecord.status !== 0 && ctx.syncRecord.status !== 1) {
                throw new Error('同步任务系统似乎有点问题');
            }
            // 同步开始--标记账户的状态
            ctx.syncRecord.status = 1;
            yield SyncModel.saveSyncRecord(ctx.syncRecord);

            // 重新出发UI获取当前文章数据
            // 未来考虑移除，暂时这样实现
            if (resetOneMaterial) resetOneMaterial();

            ctx.account = yield AccountModel.findOneAccountByQuery({
                uid: ctx.syncRecord.account_uid
            });
            // 埋点
            ctx.app.sendMessage({
                name: '/sync/startSync',
                data: {
                    platform: ctx.account.type
                }
            });
            if (!ctx.account) {
                throw new Error(`“${ctx.syncRecord.account_info.username}”已被删除`);
            }
            if (!ctx.account.cookies || !ctx.account.cookiesMap) {
                throw new Error(`“${ctx.syncRecord.account_info.username}”登录信息过期`);
            }
            ctx.platform = ctx.account.type;
            ctx.platApi = app.PlatformApi[platformConfig[ctx.platform].apiName];
            let ArticleTypeModels = null;
            if (ctx.syncRecord.type === 'Article') {
                ArticleTypeModels = ArticleModel;
            }
            if (ctx.syncRecord.type === 'MultiArticle') {
                ArticleTypeModels = MultiArticleModel;
            }
            if (!ArticleTypeModels) {
                throw new Error('没有匹配到正确的素材模式');
            }
            const article = yield ArticleTypeModels.findOne({
                _id: ctx.syncRecord.article_id
            });
            if (!article) {
                throw new Error('您要同步的素材已经被删除');
            }
            ctx.relation_data = article.relation_data || {};
            ctx.articles = article[ctx.platform];
            if (!ctx.articles) {
                throw new Error('您要同步的素材中，该平台文章已经被删除');
            }
            if (!(ctx.articles instanceof Array)) {
                if (!ctx.articles.title) throw new Error('您的文章没有标题');
                if (!ctx.articles.content) throw new Error('您的文章没有内容');
                ctx.articles = [ctx.articles];
            } else {
                // 没有内容和标题的不会同步，直接抛异常
                ctx.articles.forEach((item, i) => {
                    if (!item.title) throw new Error(`您的第${i + 1}篇文章没有标题`);
                    if (!item.content) throw new Error(`您的第${i + 1}篇文章没有内容`);
                });
            }
            yield next();
        } catch (e) {
            ctx.log(e);
            if (e.message === '已取消该任务') {
                throw e;
            } else {
                ctx.syncRecord.error = e.message;
                ctx.syncRecord.status = -1;
                yield SyncModel.saveSyncRecord(ctx.syncRecord);
                throw e;
            }
        }
    }),
    // 模板处理
    template: co.wrap(function* (ctx, next) {
        if (!['weixin', 'qq'].includes(ctx.platform)) {
            yield next();
            return;
        }
        try {
            const articlesData = ctx.articles;
            ctx.log(`开始处理${ctx.platform}平台模板`);
            const wxbToken = yield AppConfigModel.findOne({}).then(WxbAccount => WxbAccount.token);
            const template = yield ctx.app.WxbApi.getTemplate(
                wxbToken,
                ctx.account.uid,
                ctx.account.type
            )
            .then(res => {
                if (!res.data) throw new Error('获取模板数据错误');
                ctx.log('模板数据', res.data);
                return JSON.parse(res.data);
            })
            .catch(() => null); // 如果发生错误将模板置空
            articlesData.forEach(article => {
                if (article.isTemplate) {
                    if (!wxbToken) throw new Error('获取模板数据失败，请登录您的微小宝账号');
                    if (!template) throw new Error('获取模板数据错误');
                    if (article.templateChecked.includes('followGuide')) {
                        article.content = `${template.follow_guide}${article.content}`;
                    }
                    if (article.templateChecked.includes('signature')) {
                        article.content = `${article.content}${template.signature}`;
                    }
                    if (article.templateChecked.includes('readSourceGuide')) {
                        article.content = `${article.content}${template.read_source_guide}`;
                    }
                    if (article.templateChecked.includes('sourceUrl')) {
                        article.sourceurl = template.source_url;
                    }
                }
            });
        } catch (e) {
            ctx.log(e);
            ctx.syncRecord.error = e.error;
            ctx.syncRecord.status = -1;
            yield SyncModel.saveSyncRecord(ctx.syncRecord);
            throw e;
        }
        yield next();
    }),
    reflashWeiXinTicket: co.wrap(function* (ctx, next) {
        const { app } = ctx;
        if (ctx.account.type != 'weixin') {
            yield next();
            return;
        }
        ctx.log('更新微信tickt');
        try {
            const info = yield app.PlatformApi.WeiXin.getUserInfo(ctx.account.token, ctx.account.cookiesMap);
            const ticket = info.base_resp.media_ticket;
            const ticketId = info.user_info.user_name;
            ctx.account.ticket = ticket;
            ctx.account.ticket_id = ticketId;
            ctx.account = yield AccountModel.updateAccount(ctx.account.uid, ctx.account);
        } catch (e) {
            ctx.log(e);
            ctx.syncRecord.error = '微信更新ticket失败';
            ctx.syncRecord.status = -1;
            yield SyncModel.saveSyncRecord(ctx.syncRecord);
            throw e;
        }
        yield next();
    }),

    // 下载cdn图片到本地
    syncDownloadImages: co.wrap(function* (ctx, next) {
        const app = ctx.app;
        const articlesData = ctx.articles;
        ctx.log('下载cdn图片到本地');
        try {
            ctx.remoteImages = util.Article.getArticlesRemoteImages(articlesData);
            // if (ctx.platform === 'weixin') {
            //     ctx.remoteImages = ctx.remoteImages.filter(url => !url.includes('http://mmbiz.qpic.cn/') && !url.includes('http://mmbiz.qlogo.cn/'));
            // }
            // ctx.log({
            //     '文章中所有cdn图片包括封面': JSON.stringify(ctx.remoteImages)
            // });
            let needDownloadImages = ctx.remoteImages;
            yield Promise.all(needDownloadImages.map(file => ImageModel.findOne({
                file,
                type: ctx.platform
            }).then(res => {
                return {
                    remoteUrl: file,
                    needDownLoad: !res || !res.sourceFile
                };
            }))).then(res => {
                needDownloadImages = res.filter(item => item.needDownLoad).map(item => item.remoteUrl);
            });
            // 本地图片丢失的情况下，更新sourceFile
            if (needDownloadImages[0]) {
                ctx.log('下载CDN图片');
                let downloadRes = [];
                while (needDownloadImages.length) {
                    const currentImage = needDownloadImages.pop();
                    const oneDownloadRes = yield app.WxbApi.downloadImages(
                        [currentImage],
                        app.materialDir
                    );
                    ctx.log('循环下载图片', currentImage);
                    downloadRes = downloadRes.concat(oneDownloadRes);
                }

                const plist = downloadRes.map(item =>
                    ImageModel.findOneAndUpdate({
                        file: item.url,
                        type: ctx.platform
                    }, {
                        file: item.url,
                        sourceFile: item.file,
                        type: ctx.platform
                    }, {
                        upsert: true
                    })
                );
                yield Promise.all(plist);
            }
            yield next();
            return;
        } catch (e) {
            ctx.syncRecord.error = e.message;
            ctx.syncRecord.status = -1;
            yield SyncModel.saveSyncRecord(ctx.syncRecord);
            throw e;
        }
    }),

    // 将所有图片链接对应的本地图片上传，并在文章中替换对应平台的路径
    syncUploadImages: co.wrap(function* (ctx, next) {
        ctx.log('上传本地图片到平台');
        const articlesData = ctx.articles;
        let needUploadImages = [];
        yield Promise.all(ctx.remoteImages.map(file => ImageModel.findOne({
            file,
            type: ctx.platform
        }).then(res => {
            return {
                sourceFile: res.sourceFile,
                needUpLoad: !res || !res.platformUrl
            };
        }))).then(res => {
            needUploadImages = res.filter(item => item.needUpLoad).map(item => item.sourceFile);
        });
        // ctx.log({
        //     '需要上传到平台上的图片': JSON.stringify(needUploadImages)
        // });
        // 更新platformUrl，或保存本地图片路径
        if (needUploadImages[0]) {
            try {
                // 转换路基，避免windows下路径识别错误
                needUploadImages = needUploadImages.map(item => {
                    return item.replace(/%20/g, ' ');
                });
                // ctx.log(needUploadImages);
                let ret = [];
                while (needUploadImages.length) {
                    const currentImage = needUploadImages.pop();
                    ctx.log('循环上传图片', currentImage);
                    const oneUploadRes = yield ctx.platApi.uploadImages(
                        ctx.account,
                        [currentImage]
                    );
                    ret = ret.concat(oneUploadRes);
                }
                ctx.log(ret);
                console.warn(ret);
                const plist = ret.map(item =>
                    ImageModel.findOneAndUpdate({
                        sourceFile: item.file,
                        type: ctx.platform
                    }, {
                        // sourceFile: item.file,
                        platformUrl: item[ctx.platform],
                        platformInfo: item[`${ctx.platform}_info`]
                    })
                );
                const ressss = yield Promise.all(plist);
                console.warn('ressss:::', ressss);
            } catch (e) {
                ctx.log(e);
                ctx.syncRecord.error = e.message;
                console.warn(e.message);
                if (e && e.message == 'system error') {
                    ctx.syncRecord.error = '微信图片上传异常';
                }
                if (e && e.message == 'socket hang up') {
                    ctx.syncRecord.error = '您的文章中可能存在错误的图片格式';
                }
                ctx.syncRecord.status = -1;
                yield SyncModel.saveSyncRecord(ctx.syncRecord);
                throw e;
            }
        }

        // 图片替换对应Map
        ctx.replaceMap = {};
        // 所有图片的对应路径
        const allImg = yield Promise.all(ctx.remoteImages.map(file => ImageModel.findOne({
            file,
            type: ctx.platform
        }).then(res => {
            if (res && res.platformUrl) {
                ctx.replaceMap[res.file] = res.platformUrl;
            }
            return res;
        })));
        ctx.contentDBImages = allImg;

        // 替换文章里的所有content数据
        articlesData.forEach((article, index) => {
            ctx.log(`替换第${index}篇封内容`);
            article.content = util.Html.replaceHtmlImagesUrl(article.content, ctx.replaceMap);
        });

        // 头条，omqq，一点资讯的处理方式比较特殊，后续单独处理
        if (['toutiao', 'omqq', 'yidianzixun', 'sohu', 'qq', 'baijiahao'].includes(ctx.platform)) {
            yield next();
            return;
        }
        articlesData.forEach((article, index) => {
            ctx.log(`替换第${index}篇封面`);
            util.Article.replacePlatformCoverUrl(article, allImg);
        });
        yield next();
    }),

    syncBaijiahaoCover: co.wrap(function* (ctx, next) {
        if (ctx.platform !== 'baijiahao') {
            yield next();
            return;
        }
        ctx.log('上传百家号封面');
        try {
            const articlesData = ctx.articles;
            for (let i = 0; i < articlesData.length; i++) {
                const _article = articlesData[i];
                if (_article.cover && _article.cover.url && _article.cover.url.length) {
                    // cover: {
                    //     url: [{
                    //         url: 'string'
                    //     }]
                    // }
                    const localImgs = yield ctx.app.WxbApi.downloadImages(
                        _article.cover.url.filter(img => img).map(img => img.url),
                        ctx.app.materialDir
                    );
                    const uploadCover = [];
                    while (localImgs.length) {
                        const currentImg = localImgs.pop();
                        const platImg = yield ctx.platApi.uploadImage(ctx.account, currentImg, 'cover');
                        uploadCover.push(platImg.baijiahao);
                    }
                    articlesData[i].cover = uploadCover;
                } else {
                    articlesData[i].cover = [];
                }
            }
        } catch (e) {
            ctx.log('上传百家号封面失败', e);
            ctx.syncRecord.error = '上传百家号封面失败';
            ctx.syncRecord.status = -1;
            yield SyncModel.saveSyncRecord(ctx.syncRecord);
            throw e;
        }
        yield next();
    }),

    syncQQCover: co.wrap(function* (ctx, next) {
        if (ctx.platform !== 'qq') {
            yield next();
            return;
        }
        ctx.log('上传qq封面');
        try {
            const articlesData = ctx.articles;
            let i = 0;
            while (i < articlesData.length) {
                const coverCDNUrls = util.Article.getPlatformCover(articlesData[i]);
                if (coverCDNUrls && coverCDNUrls.length) {
                    const coverCDNUrl = coverCDNUrls[0];
                    const coverModel = yield ImageModel.findOne({
                        file: coverCDNUrl, // CDN地址
                        type: ctx.platform
                    });
                    const coverUpload = yield ctx.platApi.uploadImages(
                        ctx.account,
                        [coverModel.sourceFile]
                    );
                    const coverMap = [{
                        file: coverCDNUrl,
                        sourceFile: coverUpload[0].file,
                        platformUrl: coverUpload[0][ctx.platform],
                        platformInfo: coverUpload[0][`${ctx.platform}_info`]
                    }];
                    ctx.log(`第${i}篇封面,coverMap`);
                    ctx.log(coverMap);
                    util.Article.replacePlatformCoverUrl(articlesData[i], coverMap);
                }
                i++;
            }
            yield next();
        } catch (e) {
            ctx.log(e);
            ctx.syncRecord.error = e.message;
            ctx.syncRecord.status = -1;
            yield SyncModel.saveSyncRecord(ctx.syncRecord);
            throw e;
        }
    }),
    syncTouTiaoCover: co.wrap(function* (ctx, next) {
        if (ctx.platform !== 'toutiao') {
            yield next();
            return;
        }
        const { articles, replaceMap, platApi } = ctx;
        const articlesData = articles;
        ctx.log('头条封面处理');
        try {
            for (let i = 0; i < articlesData.length; i++) {
                const _article = articlesData[i];
                if (_article.cover && _article.cover.url.length) {
                    const uploadCover = yield platApi.uploadCovers(
                        _article.cover.url.filter(url => url),
                        replaceMap,
                        ctx.account
                    );
                    articlesData[i].cover = undefined;
                    articlesData[i].pgc_feed_covers = uploadCover;
                } else if (!_article.covers) {
                    articlesData[i].pgc_feed_covers = [];
                }
            }
        } catch (e) {
            ctx.log(e);
            ctx.syncRecord.error = e.message;
            ctx.syncRecord.status = -1;
            yield SyncModel.saveSyncRecord(ctx.syncRecord);
            throw e;
        }
        yield next();
    }),

    syncSoHuCover: co.wrap(function* (ctx, next) {
        if (ctx.platform !== 'sohu') {
            yield next();
            return;
        }
        const { articles, platApi } = ctx;
        const articlesData = articles;
        ctx.log('搜狐封面处理');
        try {
            for (let i = 0; i < articlesData.length; i++) {
                const _article = articlesData[i];
                if (_article.cover) {
                    const coverData = yield ImageModel.findOne({
                        file: _article.cover.url,
                        type: ctx.platform
                    });
                    const uploadCover = yield platApi.uploadCover({
                        file: coverData.sourceFile
                    }, ctx.account);
                    articlesData[i].cover = uploadCover;
                } else if (!_article.cover) {
                    articlesData[i].cover = '';
                }
            }
        } catch (e) {
            ctx.log(e);
            ctx.syncRecord.error = e.message;
            ctx.syncRecord.status = -1;
            yield SyncModel.saveSyncRecord(ctx.syncRecord);
            throw e;
        }
        yield next();
    }),

    syncYiDianCover: co.wrap(function* (ctx, next) {
        if (ctx.platform !== 'yidianzixun') {
            yield next();
            return;
        }
        const { articles, replaceMap, platApi, contentDBImages } = ctx;
        ctx.log('一点资讯封面处理');
        try {
            for (let i = 0; i < articles.length; i++) {
                const _article = articles[i];
                if (_article.cover && _article.cover.url && _article.cover.url.length) {
                    // 筛选出封面图对应的本地文件，
                    const files = [];
                    const urls = _article.cover.url.map(item => (item.source));
                    contentDBImages.forEach(item => {
                        if (urls.includes(item.file)) {
                            files.push({
                                file: item.sourceFile,
                                source: item
                            });
                        }
                    });
                    const uploadCover = yield platApi.uploadCovers(files, replaceMap, ctx.account);
                    articles[i].cover = uploadCover;
                } else {
                    articles[i].cover = [];
                }
            }
        } catch (e) {
            ctx.log(e);
            ctx.syncRecord.error = e.message;
            ctx.syncRecord.status = -1;
            yield SyncModel.saveSyncRecord(ctx.syncRecord);
            throw e;
        }
        yield next();
    }),

    syncOMQQCover: co.wrap(function* (ctx, next) {
        if (ctx.platform !== 'omqq') {
            yield next();
            return;
        }
        const { articles, platApi, replaceMap } = ctx;
        const articlesData = articles;
        ctx.log('omqq封面处理');
        try {
            for (let i = 0; i < articlesData.length; i++) {
                const _article = articlesData[i];
                const _articleImgs = util.Html.getRemoteImagesUrl(_article.content);
                const cover = [];
                if (!_articleImgs.length) {
                    throw new Error('文章中没有图片,无法设置封面');
                }
                if (_article.cover && _article.cover.url && _article.cover.url.length) {
                    for (let coverI = 0; coverI < _article.cover.url.length; coverI++) {
                        const coverItem = _article.cover.url[coverI];
                        if (!coverItem.source) throw new Error('请选择封面');
                        const upRes = yield platApi.cropupload(ctx.account, {
                            url: replaceMap[coverItem.source],
                            x: coverItem.left,
                            y: coverItem.top,
                            width: coverItem.w,
                            height: coverItem.h,
                            opCode: 155
                        });
                        upRes.index = 0;
                        cover.push(upRes);
                    }
                    if (cover.length == 3) {
                        articlesData[i].cover_type = 3;
                    }
                } else {
                    const cropRes = yield platApi.exactUpload(ctx.account, {
                        url: _articleImgs[0]
                    });
                    cover.push(cropRes);
                    articlesData[i].cover_type = -1;
                }
                articlesData[i].imgurl_ext = JSON.stringify(cover);
            }
        } catch (e) {
            ctx.log(e);
            if (e.message === 'Error: -7003:pure color image should not use here.') {
                ctx.syncRecord.error = '您不能使用单色图片作为封面';
            } else {
                ctx.syncRecord.error = e.message;
            }
            ctx.syncRecord.status = -1;
            yield SyncModel.saveSyncRecord(ctx.syncRecord);
            throw e;
        }
        yield next();
    }),

    // 处理微信的语音内容 rainey
    syncWeiXinVoice: co.wrap(function* (ctx, next) {
        if (ctx.platform !== 'weixin') {
            yield next();
            return;
        }
        ctx.log('上传语音');
        const app = ctx.app;
        const articlesData = ctx.articles;
        try {
            for (let i = 0; i < articlesData.length; i++) {
                const article = articlesData[i];
                const $ = cheerio.load(article.content, { decodeEntities: false });
                const mpvoice = $('mpvoice');
                if (!mpvoice.length) continue;
                for (let voiceI = 0; voiceI < mpvoice.length; voiceI++) {
                    const dom = $(mpvoice[voiceI]);
                    const mediaid = dom.attr('voice_encode_fileid');
                    let name = dom.attr('data-name');

                    let localFile;
                    if (!mediaid && dom.attr('src').indexOf('file:') == 0) {
                        ctx.log('本地语音');
                        ctx.log(localFile);
                        // 本地选取的音频
                        localFile = dom.attr('src');
                    } else {
                        ctx.log('下载语音');
                        const url = `http://res.wx.qq.com/voice/getvoice?mediaid=${mediaid}`;
                        ctx.log(url);
                        localFile = yield app.WxbApi.downloadFile(url,
                            path.join(app.materialDir, `${Date.now()}.mp3`));
                    }

                    // 上传到微信
                    localFile = util.Base.removeFileProtocol(localFile);
                    const voiceRes = yield ctx.platApi.createVoice(ctx.account, {
                        file: localFile,
                        title: name
                    });
                    voiceRes.play_length = yield util.Media.meta(localFile);
                    // voiceRes.play_length.duration * 1000;


                    voiceRes.play_length = dom.attr('play_length');
                    ctx.log({
                        voiceRes: voiceRes
                    });
                    const _attr = {};
                    _attr[ctx.account._id] = voiceRes;
                    // yield ImageModel.updateItem({
                    //     attr: _attr,
                    //     localFileRecordId
                    // });
                    // throw new Error('测试');
                    ctx.log('语音上传成功');
                    name = encodeURIComponent(name);
                    const len = voiceRes.play_length;
                    const voiceId = voiceRes.encode_file_id;
                    // const time = `${Math.floor(len / 60000)}:${(len % 60000) / 1000}`;
                    const time = dom.attr('data-length');
                    const frame = $('<mpvoice frameborder="0" class="res_iframe js_editor_audio audio_iframe" '
                        + `src="/cgi-bin/readtemplate?t=tmpl/audio_tmpl&amp;name=${name}&amp;play_length=${time}" `
                        + `name="${name}" play_length="${len}" voice_encode_fileid="${voiceId}"></mpvoice>`);
                    dom.replaceWith(frame);
                }
                article.content = $.html();
            }
        } catch (e) {
            ctx.log(e);
            ctx.syncRecord.error = e.message;
            ctx.syncRecord.status = -1;
            yield SyncModel.saveSyncRecord(ctx.syncRecord);
            throw e;
        }
        yield next();
    }),

    // 处理简书的视频
    syncJianShuVideo: co.wrap(function* (ctx, next) {
        if (ctx.platform !== 'jianshu') {
            yield next();
            return;
        }
        ctx.log('简书-转换视频');
        const articlesData = ctx.articles;
        try {
            for (let i = 0; i < articlesData.length; i++) {
                const article = articlesData[i];
                const $ = cheerio.load(article.content, { decodeEntities: false });
                const video = $('.video-iframe');
                $('.video-description').remove();
                if (!video.length) continue;
                for (let vI = 0; vI < video.length; vI++) {
                    const dom = $(video[vI]);
                    const url = dom.attr('data-originalurl');
                    const res = yield ctx.platApi.fetchVideo(ctx.account, url);
                    // 替换video标签
                    if (res) {
                        dom.replaceWith($(res));
                    }
                }
                article.content = $.html();
            }
        } catch (e) {
            ctx.log(e);
            ctx.syncRecord.error = e.message;
            ctx.syncRecord.status = -1;
            yield SyncModel.saveSyncRecord(ctx.syncRecord);
            throw e;
        }
        yield next();
    }),

    // 检测已同步的文章在该平台是否已经被删除
    checkPlatformExist: co.wrap(function*(ctx, next) {
        ctx.log('检测文章在该对应平台已经保存过的文章是否存在');
        try {
            if (ctx.relation_data[ctx.platform]) {
                yield ctx.platApi.checkMaterialExistById(
                    ctx.account,
                    ctx.relation_data[ctx.platform].platformId
                );
            }
        } catch (e) {
            // ctx.log('部分平台并未支持，可能会有getMaterialById不是函数的错误，可以忽略');
            ctx.log('此处所有的错误都可以被忽略，这里仅用于检测该文章是否存在于平台');
            ctx.log(e);
            ctx.relation_data[ctx.platform] = {};
        }
        yield next();
    }),

    // 同步文章内容到对应账号平台 rainey
    syncArticle: co.wrap(function* (ctx, next) {
        ctx.log('开始发送文章');
        const articlesData = ctx.articles;
        console.warn(ctx.articles);
        // ctx.log('文章发送前参数打印：', ctx.articles, ctx.account, ctx.relation_data);
        try {
            if (articlesData.length == 1) {
                // 发送单图文
                ctx.log('单图文');
                ctx.body.result = yield ctx.platApi.postArticle(
                    ctx.account,
                    articlesData[0],
                    ctx.relation_data[ctx.platform]
                );
                // 保存平台文章ID
                if (typeof ctx.body.result === 'object') {
                    ctx.relation_data[ctx.platform] = ctx.body.result;
                }
                console.warn(ctx.body.result);
                yield ArticleModel.findOneAndUpdate({
                    _id: ctx.syncRecord.article_id
                }, {
                    relation_data: ctx.relation_data
                });
            } else if (articlesData.length > 1) {
                // 发送多图文
                ctx.log('多图文');
                const _articles = articlesData.map(item => util.Article.removeUncheckedAttr(item));
                ctx.body.result = yield ctx.platApi.postArticles(
                    ctx.account,
                    _articles,
                    ctx.relation_data[ctx.platform]
                );
                // 保存平台文章ID
                ctx.relation_data[ctx.platform] = ctx.body.result;
                yield MultiArticleModel.findOneAndUpdate({
                    _id: ctx.syncRecord.article_id
                }, {
                    relation_data: ctx.relation_data
                });
            }
            ctx.log('同步后返回的参数', ctx.body.result);
        } catch (e) {
            ctx.log('上传文章异常', e);
            ctx.syncRecord.error = e.message;
            ctx.syncRecord.status = -1;
            yield SyncModel.saveSyncRecord(ctx.syncRecord);
            throw e;
        }
        // 同步成功之后标记账户的状态
        ctx.syncRecord.status = 2;
        ctx.body.syncRecord = yield SyncModel.saveSyncRecord(ctx.syncRecord);

        // 特殊日志:同步成功
        ctx.resetLog();
        ctx.log('success', 'syncdone');
        ctx.sendLog();
        // 埋点
        ctx.app.sendMessage({
            name: '/sync/saveSyncRecordsSuccess',
            data: {
                platform: ctx.platform
            }
        });
        yield next();
    }),

    // 强制取消所有任务
    cancelAll: co.wrap(function* (ctx, next) {
        const res = yield SyncModel.find({
            status: {
                $in: [0, 1]
            }
        });
        if (res && res.length) {
            for (let i = 0; i < res.length; i++) {
                res[i].status = -1;
                res[i].error = '客户端退出，强制取消';
                yield res[i].save();
            }
        }
        ctx.body = res;
        yield next();
    }),
};
