'use strict';

const Document = require('camo').Document;
const PLATFORM = require('../config/platform');
const PLANTFORM_LIST = Object.keys(PLATFORM);
// const _ = require('lodash');

class Article extends Document {
    constructor() {
        super();
        this.schema({
            user_id: {
                type: String,
                required: false
            },
            createtime: {
                type: Date,
                required: false
            },
            updatetime: {
                type: Date,
                required: false
            },
            cloud_id: {
                type: String,
                required: false
            },
            // 已被添加文章的平台列表
            platform_list: {
                type: Array,
                required: false
            },
            // 平台数据，单图文
            jianshu: {
                type: Object,
                required: false
            },
            omqq: {
                type: Object,
                required: false
            },
            qq: {
                type: Object,
                required: false
            },
            sohu: {
                type: Object,
                required: false
            },
            toutiao: {
                type: Object,
                required: false
            },
            uc: {
                type: Object,
                required: false
            },
            weibo: {
                type: Object,
                required: false
            },
            weixin: {
                type: Object,
                required: false
            },
            yidianzixun: {
                type: Object,
                required: false
            },
            zhihu: {
                type: Object,
                required: false
            },
            baijiahao: {
                type: Object,
                required: false
            },
            relation_data: {
                type: Object,
                required: false
            }
        });
    }

    static addArticle(source) {
        const data = Object.assign({}, source);
        delete data._id;
        data.createtime = data.createtime ? new Date(data.createtime) : new Date();
        data.updatetime = data.updatetime ? new Date(data.updatetime) : new Date();
        data.platform_list = PLANTFORM_LIST.filter(plantform => {
            const con = data[plantform];
            return con && (con.title || con.content);
        });
        const article = this.create(data);
        return article.save().catch(e => {
            return e;
        });
    }

    static saveArticle(query, article) {
        article.createtime = article.createtime || new Date();
        if (typeof(article.createtime) === 'string') {
            article.createtime = new Date(article.createtime);
        }
        article.updatetime = new Date();
        article.platform_list = PLANTFORM_LIST.filter(plantform => {
            const con = article[plantform];
            return con && (con.title || con.content);
        });
        // 删除其他平台数据
        PLANTFORM_LIST.forEach(key => {
            if (typeof article[key] === 'undefined') {
                article[key] = null;
            }
        });
        // 删除_schema否则会报错
        delete article._schema;
        // 过滤undefined，否则也会报错
        Object.keys(article).forEach(key => {
            if (article[key] === undefined) {
                delete article[key];
            }
        });
        if (!query) {
            const _article = this.create(article);
            return _article.save();
        }

        return this.findOne(query).then((l) => {
            if (l) {
                return this.findOneAndUpdate(query, article);
            }
            const _article = this.create(article);
            return _article.save();
        });
    }

    // 云端数据格式转换成本地数据格式
    static cloudToLocal(data) {
        const local = {
            cloud_id: data.id,
            createtime: new Date(),
            updatetime: new Date(),
            platform_list: data.platforms,
            user_id: data.user_id,
            relation_data: Object.assign({}, data.relation_data)
        };
        PLANTFORM_LIST.forEach(platform => {
            const platformData = data[platform];
            if (!platformData) return;
            local[platform] = this.toLoalItem(platformData, platform);
        });
        return local;
    }

    static toLoalItem(content, platform) {
        const con = Object.assign({}, content);
        const config = PLATFORM[platform];
        const cover = con.cover;
        const covers = con.covers;
        delete con.cover;
        delete con.covers;
        const localItemData = con;
        if (config.coverType == 'normal' && cover) {
            localItemData.cover = {
                url: cover,
                type: 'normal',
                isContent: con.show_cover_pic || con.is_show_in_content || 0
            };
        } else if (config.coverType == 'select' && covers && covers.length) {
            // 从云端covers第一个元素中取出纤细的封面设置值
            localItemData.cover = {
                url: covers,
                type: 'select',
                model: covers[0].model,
                isContent: 0
            };
        }
        return localItemData;
    }

    static toCloud(data) {
        const cloud = {
            // ctime: data.createtime,
            // utime: data.updatetime,
            ctime: new Date(),
            utime: new Date(),
            platform_list: data.platforms,
            user_id: data.user_id,
            relation_data: Object.assign({}, data.relation_data)
        };
        if (data.cloud_id) {
            cloud.id = data.cloud_id;
        }
        PLANTFORM_LIST.forEach(platform => {
            const platformData = data[platform];
            if (!platformData) return;
            cloud[platform] = this.toCloudItem(platformData, platform);
        });
        return cloud;
    }

    static toCloudItem(content, platform) {
        const con = Object.assign({}, content);
        const config = PLATFORM[platform];
        const cover = con.cover;
        delete con.cover;
        const cloudData = con;
        if (config.coverType == 'normal' && cover) {
            cloudData.cover = cover.url;
            if (cover && config.showCoverKey) {
                cloudData[config.showCoverKey] = cover.isContent;
            }
        } else if (config.coverType == 'select' && cover) {
            // 借用url数组的第一个值多保存一些信息来标识封面的设置
            let url = [];
            if (cover.url && cover.url.length) {
                url = cover.url.filter(item => (item && item.url));
            } else {
                url.push({});
            }
            url[0].model = cover.model;
            url[0].type = cover.type;
            cloudData.covers = url;
        }
        return cloudData;
    }
}

module.exports = Article;
