'use strict';

const Document = require('camo').Document;
const PLATFORM = require('../config/platform');
const PLANTFORM_LIST = Object.keys(PLATFORM);
const Article = require('./Article');
const ALLPLATFORMLIST = ['weixin', 'qq'];

class MultiArticle extends Document {
    constructor() {
        super();
        this.schema({
            user_id: {
                type: String,
                required: false
            },
            createtime: {
                type: Date,
                required: false
            },
            updatetime: {
                type: Date,
                required: false
            },
            cloud_id: {
                type: String,
                required: false
            },
            // 已被同步过的平台上文章的ID
            relation_data: {
                type: Object,
                required: false
            },
            // 已被添加文章的平台列表
            platform_list: {
                type: Array,
                required: false
            },
            // 平台数据，和先前一致，未合在一个Object中
            qq: {
                type: Array,
                required: false
            },
            weixin: {
                type: Array,
                required: false
            },
            weixin_relation: {
                type: Object,
                required: false
            },
            qq_relation: {
                type: Object,
                required: false
            }
        });
    }

    static saveArticle(query, article) {
        article.createtime = article.createtime || new Date();
        article.platform_list = PLANTFORM_LIST.filter(plantform => article[plantform]);
        if (typeof(article.createtime) === 'string') {
            article.createtime = new Date(article.createtime);
        }
        article.updatetime = new Date();
        delete article._schema;
        // 删除其他平台数据
        PLANTFORM_LIST.forEach(key => {
            if (ALLPLATFORMLIST.includes(key) && typeof article[key] === 'undefined') {
                article[key] = null;
            }
        });
        // 过滤undefined，否则会报错
        Object.keys(article).forEach(key => {
            if (article[key] === undefined) {
                delete article[key];
            }
        });
        if (query && Object.keys(query).length > 0) {
            return this.findOneAndUpdate(query, article, { upsert: true });
        }

        const articleModel = this.create(article);
        return articleModel.save().catch(e => {
            return e;
        });
    }

    // 云端数据格式转换成本地数据格式
    static cloudToLocal(data) {
        const local = {
            cloud_id: data.id,
            createtime: new Date(),
            updatetime: new Date(),
            platform_list: data.platforms,
            user_id: data.user_id
        };
        PLANTFORM_LIST.forEach(platform => {
            const platformData = data[platform];
            if (!platformData) return;
            local[platform] = platformData.map(item => {
                console.log(item);
                console.log(Article.toLoalItem(item, platform));
                return Article.toLoalItem(item, platform);
            });
        });
        return local;
    }

    static toCloud(data) {
        const cloud = {
            id: data.cloud_id,
            ctime: new Date(),
            utime: new Date()
        };
        PLANTFORM_LIST.forEach(platform => {
            const platformData = data[platform];
            if (!platformData) return;
            cloud[platform] = platformData.map(item => {
                return Article.toCloudItem(item, platform);
            });
        });
        return cloud;
    }
}

module.exports = MultiArticle;
