'use strict';

const moment = require('moment');
const CollectionStatus = require('../../model/Collection');

/**
 * Initial Collection Setting
 * @param {Object} PlatformApi 各平台抓取接口
 * @param {Object} WxbApi 服务端提交接口
 * @param {String} clientId 客户端ID
 * @param {String} token
 *
 */
function Collection(PlatformApi, WxbApi, clientId, token) {
    const pf = process.platform;
    const pfMap = {
        darwin: 'Mac OS',
        win32: 'Windows'
    };

    if (!PlatformApi || !WxbApi || !clientId) {
        throw new Error('缺少必要参数: PlatformApi, WxbApi, clientId');
    }

    this.PlatformApi = PlatformApi;
    this.WxbApi = WxbApi;
    this.token = token;
    this.clientId = clientId;
    this.os = pfMap[pf] || pf;
    this.today = moment().format('YYYY-MM-DD');
}

/**
 * Start the task of One Account
 * @param {Object} userInfo
 * @private
 */
Collection.prototype.start = function (userList) {
    const self = this;
    let list = [];
    userList.forEach(function (val) {
        list = self.getOne(val, list);
    });
    this.getList(list).then(function (res) {
        self.startTask(res);
    });
};

/**
 * Return the Tasks of One Account
 * @param {String} type
 * @param {Object} user
 * @return {Object}
 * @private
 *
 */
Collection.prototype.tasks = function (type, user) {
    const Task = {};
    const self = this;
    const args = {
        begin_date: moment().subtract(30, 'days').format('YYYY-MM-DD'),
        end_date: moment().subtract(1, 'days').format('YYYY-MM-DD')
    };
    const callback = {
        system: {
            error: {
                ret: 0,
                type: 'system',
                message: `${type}系统数据上报失败`
            },
            success: {
                ret: 1,
                type: 'system',
                message: `${type}系统数据上报成功`
            }
        },
        fans: {
            error: {
                ret: 0,
                type: 'fans',
                message: `${type}每日数据上报失败`
            },
            success: {
                ret: 1,
                type: 'fans',
                message: `${type}每日数据上报成功`
            }
        }
    };
    const Api = this.PlatformApi[type];

    function emptyPromise() {
        return new Promise(resolve => {
            resolve(null);
        });
    }

    // 上报用户信息
    Task.syncSystemData = () => {
        if (!Api.getSyncData) {
            return emptyPromise();
        }
        const data = {};
        return Api.getSyncData(user).then((response) => {
            data[type] = [response];
            return self.WxbApi.saveSyncAccountClient({
                client_id: self.clientId || null,
                device_type: 'pc',
                os: self.os,
                data
            }, self.token).then((res) => {
                if (res.errcode === 0) {
                    return callback.system.success;
                }
                callback.system.error.error = res.msg;
                return callback.system.error;
            });
        }).catch((err) => {
            callback.system.error.error = err.message || err;
            return callback.system.error;
        });
    };

    // 上报每日数据
    Task.syncFansSummaryData = () => {
        if (!Api.getSyncFansSummaryData) {
            return emptyPromise();
        }
        return Api.getSyncFansSummaryData(user, args)
        .then((response) => self.WxbApi[`saveSync${type}FansSummary`](
            { data: response }, self.token))
        .then((res) => {
            if (type == 'weibo') {
                return Api.getSyncVisitData(user, args)
                .then((response) => self.WxbApi[`saveSync${type}Visit`](
                    { data: response }, self.token)
                .then((res1) => {
                    if (res1.errcode === 0) {
                        return callback.fans.success;
                    }
                    callback.fans.error.error = res.msg;
                    return callback.fans.error;
                }));
            }

            if (res.errcode === 0) {
                return callback.fans.success;
            }
            callback.fans.error.error = res.msg;
            return callback.fans.error;
        })
        .catch((err) => {
            callback.fans.error.error = err.message;
            return callback.fans.error;
        });
    };

    return Task;
};

/**
 * Return the Promise of set DB
 * @param {Object} userInfo
 * @param {Object} status
 * @private
 */
Collection.prototype.setStatusDb = function (userInfo, status) {
    const id = userInfo.id;
    const data = {
        typeId: id,
        type: userInfo.type,
        date: this.today
    };

    const list = [];

    if (status.system) {
        list.push(
            function () {
                const sys = CollectionStatus.create(Object.assign(data, {
                    id: `${id}_system`,
                    reportType: 'system',
                    status: status.system.status
                }));
                return sys.save();
            }
        );
    }

    if (status.fans) {
        list.push(
            function () {
                const fans = CollectionStatus.create(Object.assign(data, {
                    id: `${id}_fans`,
                    reportType: 'fans',
                    status: status.fans.status
                }));
                return fans.save();
            }
        );
    }

    return Promise.all(list.map(function (item) {
        return item();
    }));
};

/**
 * Return the initial list of One Account.
 * @param {Object} userInfo
 * @return {Array}
 * @private
 *
 */
Collection.prototype.getOne = function (userInfo, list = []) {
    const id = `${userInfo.type}_${userInfo.uid}`;

    if (userInfo.cookiesMap) {
        list.push({
            id: id,
            type: userInfo.type,
            user: userInfo
        });
    }

    return list;
};

/**
 * Return the list of task for Collection data
 * @param {Array} list
 * @return {Array}
 * @private
 *
 */
Collection.prototype.getList = function (list) {
    const l = [];
    const promiseList = [];
    const self = this;

    list.forEach(val => {
        const tasks = this.tasks(val.type, val.user);
        promiseList.push(
           function () {
               return CollectionStatus.find({
                   typeId: val.id
               }).then((res) => {
                   if (res.length === 0) {
                       return self.setStatusDb(val, {
                           system: { status: 2 },
                           fans: { status: 2 }
                       }).then(function () {
                           l.push({
                               id: val.id,
                               type: val.type,
                               task: [tasks.syncSystemData, tasks.syncFansSummaryData]
                           });
                       });
                   }

                   const t = res.map(function (v) {
                       if (v.status === 0 || new Date(v.date).getTime() != new Date(self.today).getTime()) {
                           switch (v.reportType) {
                           case 'system':
                               return tasks.syncSystemData;

                           case 'fans':
                               return tasks.syncFansSummaryData;

                           default:
                           }
                       }
                       return null;
                   });

                   l.push({
                       id: val.id,
                       type: val.type,
                       task: t
                   });

                   return null;
               });
           }
        );
    });

    return Promise.all(promiseList.map(function (v) {
        return v();
    })).then(function () {
        return l;
    });
};

/**
 * Start the task of collection data
 * @param {Array} list
 * @private
 */
Collection.prototype.startTask = function (list) {
    const self = this;
    const _loop = function (i) {
        function next() {
            if (i < list.length - 1) {
                return _loop(i + 1);
            }
            return 'finish sync';
        }

        return Promise.all(list[i].task.map(function (item) {
            return item ? item() : null;
        }))
        .then((res) => {
            const s = {};
            res.forEach(function (val) {
                if (val) {
                    s[val.type] = {};
                    s[val.type].status = val.ret;
                }
            });
            if (Object.keys(s).length > 0) {
                self.setStatusDb(list[i], s);
            }

            console.warn('res:::', res);
            return next();
        }).catch((err) => {
            console.error('err:::', err);
            return next();
        });
    };

    if (list.length > 0) {
        _loop(0).then((response) => {
            console.log(response);
        });
    }
};

/**
 * Get CollectionStatus DB
 */
Collection.prototype.getDB = function () {
    CollectionStatus.find().then(function (l) {
        console.log('DB:::::', l);
    });
};

/**
 * Clean CollectionStatus DB
 */
Collection.prototype.cleanDB = function () {
    console.warn('clean');
    const self = this;
    CollectionStatus.deleteMany().then(function () {
        self.getDB();
    });
};

module.exports = Collection;
