'use strict';

const isGeneratorFunction = require('is-generator-function');
const convert = require('koa-convert');
const Emitter = require('events');
const compose = require('koa-compose');
const context = require('./context');

module.exports = class Application extends Emitter {

    constructor() {
        super();
        this.middleware = [];
        this.context = Object.create(context);
        this.taskOS = {};
        this.watch = {};
    }

    use(fn) {
        if (typeof fn !== 'function') throw new TypeError('middleware must be a function!');
        if (isGeneratorFunction(fn)) {
            fn = convert(fn);
        }
        this.middleware.push(fn);
        return this;
    }

    sendMessage(name, message) {
        let msg = {};
        if (typeof name === 'string') {
            msg.name = name;
            Object.assign(msg, message);
        } else {
            msg = Object.assign({}, name);
        }
        const fn = compose(this.middleware);
        const ctx = this.createContext(msg);
        return fn(ctx).then(() => {
            return ctx;
        }).catch(ctx.onerror);
    }

    createWatch(path, hooks) {
        const watch = {};
        const listener = {};
        listener.callback = (arg) => {
            if (arg) {
                arg.name = path;
                listener.arg = arg;
            } else {
                arg = listener.arg || { name: path };
            }

            if (typeof hooks.before === 'function') hooks.before();

            return this.sendMessage(arg).then(res => {
                if (typeof hooks.done === 'function') hooks.done(res.body);
            }).catch(e => {
                if (typeof watch.catch === 'function') watch.catch(e);
            });
        };

        if (this.watch[path]) {
            this.watch[path].push(listener);
        } else {
            this.watch[path] = [listener];
        }

        const index = this.watch[path].length - 1;

        watch.setParam = (arg) => {
            listener.callback(arg);
        };

        watch.destroy = () => {
            this.watch[path].splice(index, 1);
        };

        return watch;
    }

    createContext(message) {
        const ctx = Object.create(this.context);
        ctx.message = message;
        ctx.app = this;
        ctx.name = message.name;
        ctx.body = Object.assign({}, message.body); // 约定为需要处理的数据内容
        ctx.onerror = ctx.onerror.bind(ctx);
        return ctx;
    }
};
