'use strict';

const compose = require('koa-compose');
const Layer = require('./layer');


function Router(opts) {
    if (!(this instanceof Router)) {
        return new Router(opts);
    }

    this.opts = opts || {};
    this.params = {};
    this.stack = [];
}

Router.prototype.routes = Router.prototype.middleware = function () {
    const router = this;

    const dispatch = function dispatch(ctxOut, nextOut) {
        const path = ctxOut.name;
        const matched = router.match(path);
        if (ctxOut.matched) {
            ctxOut.matched.push.apply(ctxOut.matched, matched.path);
        } else {
            ctxOut.matched = matched.path;
        }

        if (!matched.route) return nextOut();

        const layerChain = matched.pathAndMethod.reduce(function (memo, layer) {
            memo.push(function (ctx, next) {
                ctx.captures = layer.captures(path, ctx.captures);
                ctx.params = layer.params(path, ctx.captures, ctx.params);
                return next();
            });
            return memo.concat(layer.stack);
        }, []);

        return compose(layerChain)(ctxOut, nextOut);
    };

    dispatch.router = this;

    return dispatch;
};

Router.prototype.on = Router.prototype.register = function (path, middleware, opts) {
    opts = opts || {};

    const router = this;
    const stack = this.stack;

  // support array of paths
    if (Array.isArray(path)) {
        path.forEach(function (p) {
            router.register.call(router, p, middleware, opts);
        });

        return this;
    }

  // create route
    const route = new Layer(path, middleware, {
        end: opts.end === false ? opts.end : true,
        name: opts.name,
        sensitive: opts.sensitive || this.opts.sensitive || false,
        strict: opts.strict || this.opts.strict || false,
        prefix: opts.prefix || this.opts.prefix || '',
    });

    if (this.opts.prefix) {
        route.setPrefix(this.opts.prefix);
    }

  // add parameter middleware
    Object.keys(this.params).forEach(function (param) {
        route.param(param, this.params[param]);
    }, this);

    stack.push(route);

    return route;
};

Router.prototype.match = function (path) {
    const layers = this.stack;

    let layer;
    const matched = {
        path: [],
        pathAndMethod: [],
        route: false
    };

    for (let len = layers.length, i = 0; i < len; i++) {
        layer = layers[i];
        if (layer.match(path)) {
            matched.path.push(layer);
            matched.pathAndMethod.push(layer);
            matched.route = true;
        }
    }

    return matched;
};


module.exports = Router;
