'use strict';

const pathToRegExp = require('path-to-regexp');

function safeDecodeURIComponent(text) {
    try {
        return decodeURIComponent(text);
    } catch (e) {
        return text;
    }
}


function Layer(path, middleware, opts) {
    this.opts = opts || {};
    this.name = this.opts.name || null;
    this.paramNames = [];
    this.stack = Array.isArray(middleware) ? middleware : [middleware];

    // ensure middleware is a function
    this.stack.forEach(function (fn) {
        const type = (typeof fn);
        if (type !== 'function') {
            throw new Error('router middleware must be a function.');
        }
    }, this);

    this.path = path;
    this.regexp = pathToRegExp(path, this.paramNames, this.opts);
}

/**
 * Returns whether request `path` matches route.
 *
 * @param {String} path
 * @returns {Boolean}
 * @private
 */

Layer.prototype.match = function (path) {
    return this.regexp.test(path);
};

/**
 * Returns map of URL parameters for given `path` and `paramNames`.
 *
 * @param {String} path
 * @param {Array.<String>} captures
 * @param {Object=} existingParams
 * @returns {Object}
 * @private
 */
Layer.prototype.params = function (path, captures, existingParams) {
    const params = existingParams || {};

    for (let len = captures.length, i = 0; i < len; i++) {
        if (this.paramNames[i]) {
            const c = captures[i];
            params[this.paramNames[i].name] = c ? safeDecodeURIComponent(c) : c;
        }
    }

    return params;
};

/**
 * Returns array of regexp url path captures.
 *
 * @param {String} path
 * @returns {Array.<String>}
 * @private
 */

Layer.prototype.captures = function (path) {
    return path.match(this.regexp).slice(1);
};

/**
 * Generate URL for route using given `params`.
 *
 * @example
 *
 * ```javascript
 * var route = new Layer(['GET'], '/users/:id', fn);
 *
 * route.url({ id: 123 }); // => "/users/123"
 * ```
 *
 * @param {Object} params url parameters
 * @returns {String}
 * @private
 */

Layer.prototype.url = function (params) {
    let args = params;
    const url = this.path;
    const toPath = pathToRegExp.compile(url);

    // argument is of form { key: val }
    if (typeof params != 'object') {
        args = Array.prototype.slice.call(arguments);
    }

    if (args instanceof Array) {
        const tokens = pathToRegExp.parse(url);
        const replace = {};
        for (let len = tokens.length, i = 0, j = 0; i < len; i++) {
            if (tokens[i].name) replace[tokens[i].name] = args[j++];
        }
        return toPath(replace);
    }
    return toPath(params);
};

/**
 * Run validations on route named parameters.
 *
 * @example
 *
 * ```javascript
 * router
 *   .param('user', function *(id, next) {
 *     this.user = users[id];
 *     if (!user) return this.status = 404;
 *     yield next;
 *   })
 *   .get('/users/:user', function *(next) {
 *     this.body = this.user;
 *   });
 * ```
 *
 * @param {String} param
 * @param {Function} middleware
 * @returns {Layer}
 * @private
 */

Layer.prototype.param = function (param, fn) {
    const stack = this.stack;
    const params = this.paramNames;
    const middleware = function *(next) {
        next = fn.call(this, this.params[param], next);
        if (typeof next.next === 'function') {
            yield* next;
        } else {
            yield Promise.resolve(next);
        }
    };
    middleware.param = param;

    params.forEach(function (p, i) {
        const prev = params[i - 1];

        if (param === p.name) {
      // insert param middleware in order params appear in path
            if (prev) {
                if (!stack.some(function (m, i1) {
                    if (m.param === prev.name) {
                        return stack.splice(i1, 0, middleware);
                    }
                    return null;
                })) {
                    stack.some(function (m, i1) {
                        if (!m.param) {
                            return stack.splice(i1, 0, middleware);
                        }
                        return null;
                    });
                }
            } else {
                stack.unshift(middleware);
            }
        }
    });

    return this;
};

/**
 * Prefix route path.
 *
 * @param {String} prefix
 * @returns {Layer}
 * @private
 */

Layer.prototype.setPrefix = function (prefix) {
    if (this.path) {
        this.path = prefix + this.path;
        this.paramNames = [];
        this.regexp = pathToRegExp(this.path, this.paramNames, this.opts);
    }

    return this;
};

module.exports = Layer;
