package com.fs.starfarer.api.impl.campaign.rulecmd;

import com.fs.starfarer.api.Global;
import com.fs.starfarer.api.campaign.CampaignFleetAPI;
import com.fs.starfarer.api.campaign.CargoAPI;
import com.fs.starfarer.api.campaign.CargoStackAPI;
import com.fs.starfarer.api.campaign.InteractionDialogAPI;
import com.fs.starfarer.api.campaign.econ.MarketAPI;
import com.fs.starfarer.api.campaign.rules.MemKeys;
import com.fs.starfarer.api.campaign.rules.MemoryAPI;
import com.fs.starfarer.api.fleet.FleetMemberAPI;
import com.fs.starfarer.api.util.Misc;
import com.fs.starfarer.api.util.Misc.Token;
import data.scripts.util.UW_Util;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import org.lazywizard.lazylib.MathUtils;

import static com.fs.starfarer.api.impl.campaign.rulecmd.CabalPickExtortionMethod.extortionAmount;

/**
 * CabalCargoCalc
 */
public class CabalCargoCalc extends BaseCommandPlugin {

    public static CargoAPI createCargoCopyWithCombinedCommodityStacks(CargoAPI cargo) {
        CargoAPI newCargo = Global.getFactory().createCargo(true);
        for (CargoStackAPI stack : cargo.getStacksCopy()) {
            newCargo.addFromStack(stack);
        }
        return newCargo;
    }

    public static float totalCargoValue(CargoAPI cargo) {
        float totalCreditsValue = 0f;
        for (CargoStackAPI stack : cargo.getStacksCopy()) {
            if (stack.isCommodityStack() && !stack.isCrewStack()) {
                float unadjustedValue = 0f;
                for (MarketAPI market : Global.getSector().getEconomy().getMarketsCopy()) {
                    float thisUnadjustedValue = market.getDemandPrice(stack.getCommodityId(), stack.getSize(), false);
                    unadjustedValue = Math.max(thisUnadjustedValue, unadjustedValue);
                }
                totalCreditsValue += unadjustedValue;
            } else if (stack.isModSpecStack()) {
                float value = stack.getHullModSpecIfHullMod().getBaseValue() * stack.getSize();
                totalCreditsValue += value;
            }
        }

        return totalCreditsValue;
    }

    public static float valueOfBiggestCommodityStack(CargoAPI cargo) {
        CargoAPI newCargo = createCargoCopyWithCombinedCommodityStacks(cargo);
        float supplyConsumptionPerDeploymentAndOneMonth = 0f;
        float fuelConsumptionIn20LY = 0f;
        for (FleetMemberAPI member : Global.getSector().getPlayerFleet().getFleetData().getMembersListCopy()) {
            supplyConsumptionPerDeploymentAndOneMonth += member.getDeploymentCostSupplies();
            supplyConsumptionPerDeploymentAndOneMonth += member.getStats().getSuppliesPerMonth().getModifiedValue();
            fuelConsumptionIn20LY += member.getFuelUse() * 20f;
        }

        float biggestCreditsValue = 0f;
        for (CargoStackAPI stack : newCargo.getStacksCopy()) {
            if (stack.isCommodityStack() && !stack.isCrewStack()) {
                float effectiveSize = stack.getSize();
                if (stack.isSupplyStack()) {
                    effectiveSize = Math.max(0f, effectiveSize - supplyConsumptionPerDeploymentAndOneMonth);
                }
                if (stack.isFuelStack()) {
                    effectiveSize = Math.max(0f, effectiveSize - fuelConsumptionIn20LY);
                }

                float value = 0f;
                for (MarketAPI market : Global.getSector().getEconomy().getMarketsCopy()) {
                    float thisValue = market.getDemandPrice(stack.getCommodityId(), effectiveSize, false);
                    value = Math.max(thisValue, value);
                }
                if (value > biggestCreditsValue) {
                    biggestCreditsValue = value;
                }
            } else if (stack.isModSpecStack()) {
                float value = stack.getHullModSpecIfHullMod().getBaseValue() * stack.getSize();
                if (value > biggestCreditsValue) {
                    biggestCreditsValue = value;
                }
            }
        }

        return biggestCreditsValue;
    }

    @Override
    public boolean execute(String ruleId, InteractionDialogAPI dialog, List<Token> params,
                           Map<String, MemoryAPI> memoryMap) {
        if (dialog == null) {
            return false;
        }

        CampaignFleetAPI fleet;
        if (dialog.getInteractionTarget() instanceof CampaignFleetAPI) {
            fleet = (CampaignFleetAPI) dialog.getInteractionTarget();
        } else {
            return false;
        }

        float supplyConsumptionPerDeploymentAndOneMonth = 0f;
        float fuelConsumptionIn20LY = 0f;
        for (FleetMemberAPI member : Global.getSector().getPlayerFleet().getFleetData().getMembersListCopy()) {
            supplyConsumptionPerDeploymentAndOneMonth += member.getDeploymentCostSupplies();
            supplyConsumptionPerDeploymentAndOneMonth += member.getStats().getSuppliesPerMonth().getModifiedValue();
            fuelConsumptionIn20LY += member.getFuelUse() * 20f;
        }

        CargoAPI cargo = createCargoCopyWithCombinedCommodityStacks(Global.getSector().getPlayerFleet().getCargo());
        float totalCreditsValue = totalCargoValue(cargo);
        double valueToTake = extortionAmount(totalCreditsValue);
        valueToTake = Math.min(valueToTake, fleet.getFleetPoints() * 5000.0);

        MarketAPI location = null;
        float biggestCreditsValue = 0f;
        float creditsPerBiggest = 0f;
        CargoStackAPI biggestStack = null;
        List<CargoExtortionInfo> multipleChoice = new ArrayList<>(2);
        for (CargoStackAPI stack : cargo.getStacksCopy()) {
            if (stack.isCommodityStack() && !stack.isCrewStack()) {
                float effectiveSize = stack.getSize();
                if (stack.isSupplyStack()) {
                    effectiveSize = Math.max(0f, effectiveSize - supplyConsumptionPerDeploymentAndOneMonth);
                }
                if (stack.isFuelStack()) {
                    effectiveSize = Math.max(0f, effectiveSize - fuelConsumptionIn20LY);
                }
                effectiveSize = (int) Math.min(effectiveSize, fleet.getCargo().getSpaceLeft());

                float value = 0f;
                MarketAPI bestPriceAt = null;
                for (MarketAPI market : Global.getSector().getEconomy().getMarketsCopy()) {
                    float thisValue = market.getDemandPrice(stack.getCommodityId(), effectiveSize, false);
                    if (thisValue > value) {
                        bestPriceAt = market;
                        value = thisValue;
                    }
                }
                if (value > biggestCreditsValue) {
                    biggestCreditsValue = value;
                    creditsPerBiggest = value / effectiveSize;
                    biggestStack = stack;
                    location = bestPriceAt;
                }
                if (value > valueToTake) {
                    multipleChoice.add(new CargoExtortionInfo(stack, value, value / effectiveSize, bestPriceAt));
                }
            } else if (stack.isModSpecStack()) {
                float value = stack.getSize();
                if (stack.isModSpecStack()) {
                    value *= stack.getHullModSpecIfHullMod().getBaseValue();
                }
                float stability = 0f;
                MarketAPI bestPriceAt = null;
                for (MarketAPI market : Global.getSector().getEconomy().getMarketsCopy()) {
                    float thisStability = market.getStabilityValue();
                    if (thisStability > stability) {
                        bestPriceAt = market;
                    }
                }
                if (value > biggestCreditsValue) {
                    biggestCreditsValue = value;
                    creditsPerBiggest = value / stack.getSize();
                    biggestStack = stack;
                    location = bestPriceAt;
                }
                if (value > valueToTake) {
                    multipleChoice.add(new CargoExtortionInfo(stack, value, value / stack.getSize(), bestPriceAt));
                }
            }
        }

        if (!multipleChoice.isEmpty()) {
            int index = MathUtils.getRandomNumberInRange(0, multipleChoice.size() - 1);
            biggestStack = multipleChoice.get(index).stack;
            biggestCreditsValue = multipleChoice.get(index).value;
            creditsPerBiggest = multipleChoice.get(index).valuePerUnit;
            location = multipleChoice.get(index).location;
        }
        if (biggestStack == null || location == null) {
            return false;
        }

        float effectiveSize = biggestStack.getSize();
        if (biggestStack.isSupplyStack()) {
            effectiveSize = Math.max(0f, effectiveSize - supplyConsumptionPerDeploymentAndOneMonth);
        }
        if (biggestStack.isFuelStack()) {
            effectiveSize = Math.max(0f, effectiveSize - fuelConsumptionIn20LY);
        }
        effectiveSize = (int) Math.min(effectiveSize, fleet.getCargo().getSpaceLeft());

        int amountToTake = (int) UW_Util.roundToSignificantFiguresLong(Math.min(effectiveSize, valueToTake /
                                                                                creditsPerBiggest), 2);
        int amountLeft = (int) (biggestStack.getSize() - amountToTake);
        if (amountToTake < 1) {
            return false;
        }

        float finalValue = amountToTake * (biggestCreditsValue / effectiveSize);
        float repImpact;
        if (finalValue <= 25000f) {
            repImpact = 0.01f;
        } else if (finalValue <= 50000f) {
            repImpact = 0.02f;
        } else if (finalValue <= 100000f) {
            repImpact = 0.03f;
        } else if (finalValue <= 200000f) {
            repImpact = 0.04f;
        } else {
            repImpact = 0.05f;
        }

        memoryMap.get(MemKeys.LOCAL).set("$Cabal_cargo_name", biggestStack.getDisplayName().toLowerCase(), 0);
        if (biggestStack.isCommodityStack()) {
            memoryMap.get(MemKeys.LOCAL).set("$Cabal_cargo_id", biggestStack.getCommodityId(), 0);
        } else if (biggestStack.isModSpecStack()) {
            memoryMap.get(MemKeys.LOCAL).set("$Cabal_cargo_id", biggestStack.getHullModSpecIfHullMod().getId(), 0);
        }
        memoryMap.get(MemKeys.LOCAL).set("$Cabal_cargo_amount", amountToTake, 0);
        memoryMap.get(MemKeys.LOCAL).set("$Cabal_cargo_amount_string", Misc.getWithDGS(amountToTake), 0);
        memoryMap.get(MemKeys.LOCAL).set("$Cabal_cargo_amount_left_string", Misc.getWithDGS(amountLeft), 0);
        memoryMap.get(MemKeys.LOCAL).set("$Cabal_cargo_best_location", location.getName(), 0);
        memoryMap.get(MemKeys.LOCAL).set("$Cabal_cargo_final_value", Misc.getWithDGS(
                                         UW_Util.roundToSignificantFiguresLong(finalValue, 2)), 0);
        memoryMap.get(MemKeys.LOCAL).set("$Cabal_repImpact", repImpact, 0);
        return true;
    }

    private static class CargoExtortionInfo {

        MarketAPI location;
        CargoStackAPI stack;
        float value;
        float valuePerUnit;

        CargoExtortionInfo(CargoStackAPI stack, float value, float valuePerUnit, MarketAPI location) {
            this.stack = stack;
            this.value = value;
            this.valuePerUnit = valuePerUnit;
            this.location = location;
        }
    }
}
