package data.scripts.campaign.fleets;

import com.fs.starfarer.api.EveryFrameScript;
import com.fs.starfarer.api.Global;
import com.fs.starfarer.api.campaign.CampaignFleetAPI;
import com.fs.starfarer.api.campaign.FleetAssignment;
import com.fs.starfarer.api.campaign.LocationAPI;
import com.fs.starfarer.api.campaign.SectorEntityToken;
import com.fs.starfarer.api.campaign.SectorEntityToken.VisibilityLevel;
import com.fs.starfarer.api.campaign.StarSystemAPI;
import com.fs.starfarer.api.campaign.econ.MarketAPI;
import com.fs.starfarer.api.campaign.rules.MemoryAPI;
import com.fs.starfarer.api.impl.campaign.fleets.BaseLimitedFleetManager;
import com.fs.starfarer.api.impl.campaign.fleets.FleetFactoryV2;
import com.fs.starfarer.api.impl.campaign.fleets.FleetParams;
import com.fs.starfarer.api.impl.campaign.ids.Factions;
import com.fs.starfarer.api.impl.campaign.ids.FleetTypes;
import com.fs.starfarer.api.impl.campaign.ids.MemFlags;
import com.fs.starfarer.api.impl.campaign.ids.Ranks;
import com.fs.starfarer.api.impl.campaign.rulecmd.CabalPickContributionMethod;
import com.fs.starfarer.api.loading.FleetCompositionDoctrineAPI;
import com.fs.starfarer.api.util.IntervalUtil;
import com.fs.starfarer.api.util.Misc;
import com.fs.starfarer.api.util.WeightedRandomPicker;
import data.scripts.UnderworldModPlugin;
import data.scripts.util.DS_Defs;
import java.io.IOException;
import java.util.List;
import org.json.JSONException;
import org.json.JSONObject;
import org.lazywizard.lazylib.MathUtils;
import org.lwjgl.util.vector.Vector2f;

public class UW_CabalFleetManager extends BaseLimitedFleetManager {

    public static int MAX_CABAL_FLEETS = 12;

    private static final String SETTINGS_FILE = "UNDERWORLD_OPTIONS.ini";

    public static void reloadSettings() throws IOException, JSONException {
        JSONObject settings = Global.getSettings().loadJSON(SETTINGS_FILE);

        MAX_CABAL_FLEETS = settings.getInt("maxCabalFleets");
    }

    private static int getMinPreferredMarketSize(float combat) {
        float fp = combat;
        if (fp <= 20) {
            return 1;
        } else if (fp <= 50) {
            return 3;
        } else if (fp <= 100) {
            return 5;
        } else if (fp <= 150) {
            return 7;
        }
        return 8;
    }

    @Override
    protected int getMaxFleets() {
        if (!UnderworldModPlugin.Module_StarlightCabal) {
            return 0;
        }

        return MAX_CABAL_FLEETS;
    }

    protected MarketAPI pickSourceMarket(Vector2f location, float combat) {
        List<MarketAPI> allMarkets = Global.getSector().getEconomy().getMarketsCopy();

        int size = getMinPreferredMarketSize(combat);
        WeightedRandomPicker<MarketAPI> closestPicker = new WeightedRandomPicker<>();
        WeightedRandomPicker<MarketAPI> closestMatchingSizePicker = new WeightedRandomPicker<>();

        for (MarketAPI market : allMarkets) {
            if (market.getPrimaryEntity() == null) {
                continue;
            }
            float weight = 0f;
            switch (market.getFactionId()) {
                case Factions.TRITACHYON:
                    weight += 10f;
                    break;
                case "cabal":
                    weight += 20f;
                    break;
                case "blackrock_driveyards":
                    weight += 5f;
                    break;
                case "exigency":
                    weight += 2f;
                    break;
            }
            if (market.hasCondition("cabal_influence")) {
                weight += 20f;
            }
            if (weight <= 0f) {
                continue;
            }

            float currDist = Misc.getDistance(market.getPrimaryEntity().getLocationInHyperspace(), location);
            closestPicker.add(market, weight * 5000f / Math.max(currDist, 2500f));

            if (market.getSize() >= size) {
                closestMatchingSizePicker.add(market, weight * 5000f / Math.max(currDist, 2500f));
            }
        }

        if (!closestMatchingSizePicker.isEmpty()) {
            return closestMatchingSizePicker.pick();
        }

        if (!closestPicker.isEmpty()) {
            return closestPicker.pick();
        }

        return null;
    }

    protected StarSystemAPI pickTargetSystem() {
        WeightedRandomPicker<StarSystemAPI> picker = new WeightedRandomPicker<>();
        for (StarSystemAPI system : Global.getSector().getStarSystems()) {
            float mult = Misc.getSpawnChanceMult(system.getLocation());

            float weight = 0f;
            for (MarketAPI market : Misc.getMarketsInLocation(system)) {
                if (market.getFactionId().equals(Factions.TRITACHYON)) {
                    continue;
                }
                if (market.getFactionId().equals("cabal")) {
                    continue;
                }
                if (market.getFactionId().equals("blackrock_driveyards")) {
                    continue;
                }
                if (market.getFactionId().equals("exigency")) {
                    continue;
                }
                float w = 1f + market.getStabilityValue() + market.getSize();
                if (w > weight) {
                    weight = w;
                }
            }

            weight *= mult;

            picker.add(system, weight);
        }
        return picker.pick();
    }

    @Override
    protected CampaignFleetAPI spawnFleet() {
        if (!UnderworldModPlugin.Module_StarlightCabal) {
            return null;
        }

        StarSystemAPI target = pickTargetSystem();
        if (target == null) {
            return null;
        }

        float combat = MathUtils.getRandomNumberInRange(1, MathUtils.getRandomNumberInRange(10,
                                                                                            MathUtils.getRandomNumberInRange(
                                                                                                    20, 30)));
        float freighter = Math.round(combat * MathUtils.getRandomNumberInRange(0.05f, 0.25f));
        float tanker = Math.round(combat * MathUtils.getRandomNumberInRange(0f, 0.2f));
        float utility = Math.round((freighter + tanker) * MathUtils.getRandomNumberInRange(0f, 0.5f));

        String fleetType;
        if (combat < 10) {
            fleetType = FleetTypes.PATROL_SMALL;
        } else if (combat < 20) {
            fleetType = FleetTypes.PATROL_MEDIUM;
        } else {
            fleetType = FleetTypes.PATROL_LARGE;
        }

        FleetCompositionDoctrineAPI doctrine = Global.getSector().getFaction("cabal").getCompositionDoctrine();
        float preSmall = doctrine.getSmall();
        float preFast = doctrine.getFast();
        float preMedium = doctrine.getMedium();
        float preLarge = doctrine.getLarge();
        float preCapital = doctrine.getCapital();
        float preEscortSmallFraction = doctrine.getEscortSmallFraction();
        float preEscortMediumFraction = doctrine.getEscortMediumFraction();
        float preCombatFreighterProbability = doctrine.getCombatFreighterProbability();
        float preMinPointsForCombatCapital = doctrine.getMinPointsForCombatCapital();
        float preMinPointsForLargeCarrier = doctrine.getMinPointsForLargeCarrier();
        float preSmallCarrierProbability = doctrine.getSmallCarrierProbability();
        float preMediumCarrierProbability = doctrine.getMediumCarrierProbability();
        float preLargeCarrierProbability = doctrine.getLargeCarrierProbability();
        float preOfficersPerPoint = doctrine.getOfficersPerPoint();
        float preOfficerLevelPerPoint = doctrine.getOfficerLevelPerPoint();
        float preOfficerLevelBase = doctrine.getOfficerLevelBase();
        float preOfficerLevelVariance = doctrine.getOfficerLevelVariance();

        CabalFleetType cabalFleetType;
        if (combat < 15 && Math.random() < 0.3) {
            cabalFleetType = CabalFleetType.FRIGATES;
            doctrine.setSmall(8f);
            doctrine.setFast(6f);
            doctrine.setMedium(4f);
            doctrine.setLarge(0.1f);
            doctrine.setCapital(0.1f);
            doctrine.setCombatFreighterProbability(0.25f);
            doctrine.setEscortSmallFraction(0.25f);
            doctrine.setEscortMediumFraction(0.25f);
            doctrine.setOfficersPerPoint(0.55f);
            doctrine.setOfficerLevelPerPoint(0.25f);
            doctrine.setSmallCarrierProbability(0f);
        } else if (combat >= 8 && combat < 25 && Math.random() < 0.3) {
            cabalFleetType = CabalFleetType.DESTROYERS;
            doctrine.setSmall(3f);
            doctrine.setFast(2f);
            doctrine.setMedium(15f);
            doctrine.setLarge(2f);
            doctrine.setCapital(0.1f);
            doctrine.setCombatFreighterProbability(0.25f);
            doctrine.setEscortSmallFraction(0.25f);
            doctrine.setEscortMediumFraction(0.25f);
            doctrine.setOfficersPerPoint(0.45f);
            doctrine.setOfficerLevelPerPoint(0.35f);
        } else if (combat >= 16 && Math.random() < 0.3) {
            cabalFleetType = CabalFleetType.CRUISERS;
            doctrine.setSmall(0.1f);
            doctrine.setFast(0.1f);
            doctrine.setMedium(4f);
            doctrine.setLarge(12f);
            doctrine.setCapital(1f);
            doctrine.setCombatFreighterProbability(0.25f);
            doctrine.setEscortSmallFraction(0.25f);
            doctrine.setEscortMediumFraction(0.25f);
            doctrine.setOfficersPerPoint(0.35f);
            doctrine.setOfficerLevelPerPoint(0.45f);
        } else if (combat >= 24 && Math.random() < 0.3) {
            cabalFleetType = CabalFleetType.CAPITALS;
            doctrine.setSmall(0.1f);
            doctrine.setFast(0.1f);
            doctrine.setMedium(0.1f);
            doctrine.setLarge(2f);
            doctrine.setCapital(8f);
            doctrine.setCombatFreighterProbability(0.25f);
            doctrine.setEscortSmallFraction(0.25f);
            doctrine.setEscortMediumFraction(0.25f);
            doctrine.setOfficersPerPoint(0.25f);
            doctrine.setOfficerLevelPerPoint(0.55f);
        } else if (combat >= 10 && Math.random() < 0.4) {
            cabalFleetType = CabalFleetType.CARRIERS;
            doctrine.setCombatFreighterProbability(0.5f);
            doctrine.setEscortSmallFraction(0.75f);
            doctrine.setEscortMediumFraction(0.75f);
            doctrine.setSmallCarrierProbability(0.75f);
            doctrine.setMediumCarrierProbability(0.75f);
            doctrine.setLargeCarrierProbability(0.75f);
        } else {
            cabalFleetType = CabalFleetType.BALANCED;
        }

        MarketAPI sourceMarket = pickSourceMarket(target.getLocation(), combat);
        if (sourceMarket == null) {
            return null;
        }

        CampaignFleetAPI fleet = FleetFactoryV2.createFleet(new FleetParams(
                         target.getLocation(), // location
                         sourceMarket, // market
                         "cabal", // fleet's faction, if different from above, which is also used for source market picking
                         null,
                         fleetType,
                         combat, // combatPts
                         freighter, // freighterPts
                         tanker, // tankerPts
                         0f, // transportPts
                         0f, // linerPts
                         0f, // civilianPts
                         utility, // utilityPts
                         0.25f, // qualityBonus
                         -1f, // qualityOverride
                         1f, // officer num mult
                         0 // officer level bonus
                 ));

        doctrine.setSmall(preSmall);
        doctrine.setFast(preFast);
        doctrine.setMedium(preMedium);
        doctrine.setLarge(preLarge);
        doctrine.setCapital(preCapital);
        doctrine.setEscortSmallFraction(preEscortSmallFraction);
        doctrine.setEscortMediumFraction(preEscortMediumFraction);
        doctrine.setCombatFreighterProbability(preCombatFreighterProbability);
        doctrine.setMinPointsForCombatCapital(preMinPointsForCombatCapital);
        doctrine.setMinPointsForLargeCarrier(preMinPointsForLargeCarrier);
        doctrine.setSmallCarrierProbability(preSmallCarrierProbability);
        doctrine.setMediumCarrierProbability(preMediumCarrierProbability);
        doctrine.setLargeCarrierProbability(preLargeCarrierProbability);
        doctrine.setOfficersPerPoint(preOfficersPerPoint);
        doctrine.setOfficerLevelPerPoint(preOfficerLevelPerPoint);
        doctrine.setOfficerLevelBase(preOfficerLevelBase);
        doctrine.setOfficerLevelVariance(preOfficerLevelVariance);

        if (fleet == null) {
            return null;
        }

        if (combat < 10) {
            fleet.getCommander().setRankId(Ranks.SPACE_COMMANDER);
        } else if (combat < 20) {
            fleet.getCommander().setRankId(Ranks.SPACE_CAPTAIN);
        } else {
            fleet.getCommander().setRankId(Ranks.SPACE_ADMIRAL);
        }

        switch (cabalFleetType) {
            case FRIGATES:
                fleet.setName("Frigate Pack");
                break;
            case DESTROYERS:
                fleet.setName("Destroyer Posse");
                break;
            case CRUISERS:
                fleet.setName("Cruiser Team");
                break;
            case CAPITALS:
                fleet.setName("Capital Group");
                break;
            case CARRIERS:
                if (combat < 20) {
                    fleet.setName("Carrier Fleet");
                } else {
                    fleet.setName("Death Swarm");
                }
                break;
            default:
            case BALANCED:
                if (combat < 10) {
                    fleet.setName("Prowlers");
                } else if (combat < 20) {
                    fleet.setName("Coterie");
                } else {
                    fleet.setName("Clan Fleet");
                }
                break;
        }

        fleet.getMemoryWithoutUpdate().set(MemFlags.MEMORY_KEY_FLEET_TYPE, "cabalFleet");
        fleet.getMemoryWithoutUpdate().set(MemFlags.MEMORY_KEY_PIRATE, true);
        fleet.getStats().getDynamic().getMod(DS_Defs.STAT_BATTLE_DEBRIS_CHANCE).modifyMult("uw_spawner", 0.5f);
        fleet.getStats().getDynamic().getMod(DS_Defs.STAT_BATTLE_DERELICTS_CHANCE).modifyMult("uw_spawner", 0.5f);
        fleet.getStats().getDynamic().getMod(DS_Defs.STAT_FLEET_DEBRIS_SCALE).modifyMult("uw_spawner", 0.75f);
        fleet.getStats().getDynamic().getMod(DS_Defs.STAT_FLEET_DERELICTS_SCALE).modifyMult("uw_spawner", 0.5f);

        MarketAPI source = Misc.getSourceMarket(fleet);
        if (source == null) {
            return null;
        }

        CampaignFleetAPI player = Global.getSector().getPlayerFleet();
        boolean spawnAtSource = true;
        if (player != null) {
            float sourceToPlayer = Misc.getDistance(player.getLocation(), source.getLocationInHyperspace());
            float targetToPlayer = Misc.getDistance(player.getLocation(), target.getLocation());
            spawnAtSource = sourceToPlayer < targetToPlayer && source.hasCondition("cabal_influence");
        }

        if (spawnAtSource) {
            source.getPrimaryEntity().getContainingLocation().addEntity(fleet);
            fleet.setLocation(source.getPrimaryEntity().getLocation().x, source.getPrimaryEntity().getLocation().y);

            fleet.addAssignment(FleetAssignment.ORBIT_AGGRESSIVE, source.getPrimaryEntity(), 2f +
                                (float) Math.random() * 2f, "orbiting " + source.getName());
        } else {
            Vector2f loc = Misc.pickHyperLocationNotNearPlayer(target.getLocation(),
                                                               Global.getSettings().getMaxSensorRange() + 500f);
            Global.getSector().getHyperspace().addEntity(fleet);
            fleet.setLocation(loc.x, loc.y);
            log.info("UW_CabalFleetManager: Spawned at " + loc);
        }
        fleet.addScript(new ScrewWithPlayer(fleet));

        Vector2f dest = Misc.getPointAtRadius(target.getLocation(), 1500);
        LocationAPI hyper = Global.getSector().getHyperspace();
        SectorEntityToken token = hyper.createToken(dest.x, dest.y);

        fleet.addAssignment(FleetAssignment.GO_TO_LOCATION, token, 1000, "travelling to the " + target.getBaseName() +
                            " star system");

        if ((float) Math.random() > 0.75f) {
            fleet.addAssignment(FleetAssignment.RAID_SYSTEM, target.getHyperspaceAnchor(), 20, "raiding around the " +
                                target.getBaseName() + " star system");
        } else {
            fleet.addAssignment(FleetAssignment.RAID_SYSTEM, target.getCenter(), 20,
                                "raiding the " + target.getBaseName() + " star system");
        }
        fleet.addAssignment(FleetAssignment.GO_TO_LOCATION, source.getPrimaryEntity(), 1000, "returning to " +
                            source.getName());
        fleet.addAssignment(FleetAssignment.ORBIT_PASSIVE, source.getPrimaryEntity(), 2f + 2f * (float) Math.random(),
                            "retiring");
        fleet.addAssignment(FleetAssignment.GO_TO_LOCATION_AND_DESPAWN, source.getPrimaryEntity(), 1000);

        return fleet;
    }

    public static class ScrewWithPlayer implements EveryFrameScript {

        private static long signalExtortionPaidGlobal = 0L;
        private static float timer = 0f;

        private final CampaignFleetAPI fleet;
        private long signalExtortionPaid = 0L;
        private final IntervalUtil tracker = new IntervalUtil(0.1f, 0.5f);

        ScrewWithPlayer(CampaignFleetAPI fleet) {
            this.fleet = fleet;
        }

        @Override
        public void advance(float amount) {
            float days = Global.getSector().getClock().convertToDays(amount);
            tracker.advance(days);
            if (timer > 0f) {
                timer -= days;
            }

            MemoryAPI mem = fleet.getMemoryWithoutUpdate();
            if (mem.getBoolean("$Cabal_extortionAskedFor")) {
                Misc.setFlagWithReason(mem, MemFlags.MEMORY_KEY_PURSUE_PLAYER, "cabalScrewWithPlayer", false, 0f);
            }

            float chaseDuration, tryAgainTimer;
            switch (Global.getSector().getFaction("cabal").getRelToPlayer().getLevel()) {
                default:
                case VENGEFUL:
                case HOSTILE:
                    return;
                case INHOSPITABLE:
                    tryAgainTimer = 30f;
                    chaseDuration = 6f;
                    break;
                case SUSPICIOUS:
                    tryAgainTimer = 45f;
                    chaseDuration = 5f;
                    break;
                case NEUTRAL:
                    tryAgainTimer = 60f;
                    chaseDuration = 4f;
                    break;
                case FAVORABLE:
                    tryAgainTimer = 75f;
                    chaseDuration = 3f;
                    break;
                case WELCOMING:
                    tryAgainTimer = 90f;
                    chaseDuration = 2f;
                    break;
                case FRIENDLY:
                case COOPERATIVE:
                    return;
            }

            if (signalExtortionPaidGlobal != signalExtortionPaid) {
                mem.set("$Cabal_extortionPaid", tryAgainTimer);
                timer = tryAgainTimer;
                signalExtortionPaid = signalExtortionPaidGlobal;
            }

            if (tracker.intervalElapsed() && timer <= 0f) {
                CampaignFleetAPI playerFleet = Global.getSector().getPlayerFleet();
                if (playerFleet == null) {
                    return;
                }

                if (mem.getBoolean("$Cabal_extortionPaid") && signalExtortionPaidGlobal == signalExtortionPaid) {
                    signalExtortionPaidGlobal = Global.getSector().getClock().getTimestamp();
                }

                if (playerFleet.getContainingLocation() != fleet.getContainingLocation()) {
                    return;
                }

                if (!CabalPickContributionMethod.playerHasAbilityToPayContribution(fleet)) {
                    return;
                }

                VisibilityLevel level = playerFleet.getVisibilityLevelTo(fleet);
                if (level == VisibilityLevel.COMPOSITION_AND_FACTION_DETAILS) {
                    Misc.setFlagWithReason(mem, MemFlags.MEMORY_KEY_PURSUE_PLAYER, "cabalScrewWithPlayer", true,
                                           chaseDuration);
                    timer = tryAgainTimer;
                }
            }
        }

        @Override
        public boolean isDone() {
            return !fleet.isAlive();
        }

        @Override
        public boolean runWhilePaused() {
            return false;
        }
    }

    private static enum CabalFleetType {

        FRIGATES, DESTROYERS, CRUISERS, CAPITALS, CARRIERS, BALANCED
    }
}
