package data.scripts.campaign.submarkets;

import com.fs.starfarer.api.Global;
import com.fs.starfarer.api.Script;
import com.fs.starfarer.api.campaign.CampaignFleetAPI;
import com.fs.starfarer.api.campaign.CargoAPI;
import com.fs.starfarer.api.campaign.CoreUIAPI;
import com.fs.starfarer.api.campaign.RepLevel;
import com.fs.starfarer.api.impl.campaign.ids.Factions;
import com.fs.starfarer.api.impl.campaign.ids.ShipRoles;
import com.fs.starfarer.api.impl.campaign.submarkets.BlackMarketPlugin;
import com.fs.starfarer.api.util.Highlights;
import com.fs.starfarer.api.util.Misc;
import com.fs.starfarer.api.util.WeightedRandomPicker;

public class UW_CabalMarketPlugin extends BlackMarketPlugin {

    private static final RepLevel MIN_STANDING = RepLevel.INHOSPITABLE;

    private boolean playerPaidToUnlock = false;
    private float sinceLastUnlock = 0f;

    public UW_CabalMarketPlugin() {
        minCargoUpdateInterval = 14;
    }

    @Override
    public void advance(float amount) {
        super.advance(amount);
        float days = Global.getSector().getClock().convertToDays(amount);
        sinceLastUnlock += days;
        if (sinceLastUnlock > 7f) {
            playerPaidToUnlock = false;
        }
    }

    @Override
    public DialogOption[] getDialogOptions(CoreUIAPI ui) {
        if (canPlayerAffordUnlock()) {
            return new DialogOption[]{
                new DialogOption("Pay", new Script() {
                    @Override
                    public void run() {
                        CampaignFleetAPI playerFleet = Global.getSector().getPlayerFleet();
                        playerFleet.getCargo().getCredits().subtract(getUnlockCost());
                        playerPaidToUnlock = true;
                        sinceLastUnlock = 0f;
                    }
                }),
                new DialogOption("Never mind", null)
            };
        } else {
            return new DialogOption[]{
                new DialogOption("Never mind", null)
            };
        }
    }

    @Override
    public String getDialogText(CoreUIAPI ui) {
        if (canPlayerAffordUnlock()) {
            return "\"We might consider letting you in today, so long as you pay the " +
                    Misc.getWithDGS(getUnlockCost()) + "-credit fee.\"";
        } else {
            return "\"You can't even pay the " + Misc.getWithDGS(getUnlockCost()) + "-credit fee? Get lost.\"";
        }
    }

    @Override
    public Highlights getDialogTextHighlights(CoreUIAPI ui) {
        Highlights h = new Highlights();
        h.setText("" + getUnlockCost());
        if (canPlayerAffordUnlock()) {
            h.setColors(Misc.getHighlightColor());
        } else {
            h.setColors(Misc.getNegativeHighlightColor());
        }
        return h;
    }

    @Override
    public OnClickAction getOnClickAction(CoreUIAPI ui) {
        if (playerPaidToUnlock || submarket.getFaction().getRelToPlayer().isAtWorst(RepLevel.FRIENDLY)) {
            return OnClickAction.OPEN_SUBMARKET;
        }
        return OnClickAction.SHOW_TEXT_DIALOG;
    }

    @Override
    public float getTariff() {
        return (playerPaidToUnlock || submarket.getFaction().getRelToPlayer().isAtWorst(RepLevel.FRIENDLY)) ? 0f : 10f;
    }

    @Override
    public String getTooltipAppendix(CoreUIAPI ui) {
        RepLevel level = submarket.getFaction().getRelationshipLevel(Global.getSector().getFaction(Factions.PLAYER));
        if (!level.isAtWorst(MIN_STANDING)) {
            return "Requires: " + submarket.getFaction().getDisplayName() + " - " +
                    MIN_STANDING.getDisplayName().toLowerCase();
        }
        if (Global.getSector().getPlayerFleet().isTransponderOn()) {
            return "Requires: a clandestine approach";
        }
        return super.getTooltipAppendix(ui);
    }

    @Override
    public boolean isEnabled(CoreUIAPI ui) {
        if (Global.getSector().getPlayerFleet().isTransponderOn()) {
            return false;
        }

        RepLevel level = submarket.getFaction().getRelationshipLevel(Global.getSector().getFaction(Factions.PLAYER));
        return level.isAtWorst(MIN_STANDING);
    }

    @Override
    public void updateCargoPrePlayerInteraction() {
        if (!okToUpdateCargo()) {
            return;
        }
        sinceLastCargoUpdate = 0f;

        CargoAPI cargo = getCargo();

        updateEconomicCommoditiesInCargo(true);

        pruneWeapons(0.5f);
        addWeaponsBasedOnMarketSize(5, 3, 4, null);

        addRandomWeapons(Math.max(1, market.getSize() - 2), 4);
        addRandomWings(Math.max(1, market.getSize() - 4), 3);

        addShips();
        addHullMods(4, 2 + itemGenRandom.nextInt(3));
        cargo.sort();
    }

    private void addShips() {
        int marketSize = market.getSize();

        pruneShips(0.75f);

        WeightedRandomPicker<String> rolePicker = new WeightedRandomPicker<>();
        rolePicker.add(ShipRoles.FREIGHTER_SMALL, 3f);
        rolePicker.add(ShipRoles.TANKER_SMALL, 3f);
        rolePicker.add(ShipRoles.PERSONNEL_SMALL, 1f);
        rolePicker.add(ShipRoles.COMBAT_SMALL, 15f);
        rolePicker.add(ShipRoles.ESCORT_SMALL, 10f);
        rolePicker.add(ShipRoles.COMBAT_MEDIUM, 15f);
        rolePicker.add(ShipRoles.CARRIER_SMALL, 3f);

        if (marketSize >= 4) {
            rolePicker.add(ShipRoles.UTILITY, 5f);
            rolePicker.add(ShipRoles.FREIGHTER_MEDIUM, 5f);
            rolePicker.add(ShipRoles.TANKER_MEDIUM, 5f);
            rolePicker.add(ShipRoles.PERSONNEL_MEDIUM, 2f);
            rolePicker.add(ShipRoles.COMBAT_MEDIUM, 10f);
            rolePicker.add(ShipRoles.ESCORT_MEDIUM, 10f);
            rolePicker.add(ShipRoles.CARRIER_SMALL, 10f);
            rolePicker.add(ShipRoles.COMBAT_LARGE, 10f);
            rolePicker.add(ShipRoles.CARRIER_MEDIUM, 5f);
            rolePicker.add(ShipRoles.COMBAT_CAPITAL, 3f);
        }

        if (marketSize >= 5) {
            rolePicker.add(ShipRoles.UTILITY, 3f);
            rolePicker.add(ShipRoles.FREIGHTER_LARGE, 3f);
            rolePicker.add(ShipRoles.TANKER_LARGE, 3f);
            rolePicker.add(ShipRoles.PERSONNEL_LARGE, 1f);
            rolePicker.add(ShipRoles.COMBAT_FREIGHTER_LARGE, 5f);
            rolePicker.add(ShipRoles.COMBAT_LARGE, 5f);
            rolePicker.add(ShipRoles.CARRIER_MEDIUM, 5f);
            rolePicker.add(ShipRoles.COMBAT_CAPITAL, 4f);
            rolePicker.add(ShipRoles.CARRIER_LARGE, 2f);
        }

        addShipsForRoles(Math.round(1.5f + marketSize * 1.5f), rolePicker, null);
    }

    private boolean canPlayerAffordUnlock() {
        CampaignFleetAPI playerFleet = Global.getSector().getPlayerFleet();
        int credits = (int) playerFleet.getCargo().getCredits().get();
        return credits >= getUnlockCost();
    }

    private int getUnlockCost() {
        float fudge;
        switch (submarket.getFaction().getRelToPlayer().getLevel()) {
            default:
            case VENGEFUL:
            case HOSTILE:
            case INHOSPITABLE:
                fudge = 2f;
                break;
            case SUSPICIOUS:
                fudge = 1.5f;
                break;
            case NEUTRAL:
                fudge = 1f;
                break;
            case FAVORABLE:
                fudge = 0.75f;
                break;
            case WELCOMING:
                fudge = 0.5f;
                break;
            case FRIENDLY:
            case COOPERATIVE:
                fudge = 0f;
                break;
        }
        return Math.round(market.getSize() * market.getStabilityValue() * fudge) * 1000;
    }
}
