package data.scripts.campaign.submarkets;

import com.fs.starfarer.api.Global;
import com.fs.starfarer.api.campaign.CargoAPI;
import com.fs.starfarer.api.campaign.CargoStackAPI;
import com.fs.starfarer.api.campaign.FactionAPI;
import com.fs.starfarer.api.campaign.econ.CommodityOnMarketAPI;
import com.fs.starfarer.api.campaign.econ.SubmarketAPI;
import com.fs.starfarer.api.combat.ShipVariantAPI;
import com.fs.starfarer.api.fleet.FleetMemberAPI;
import com.fs.starfarer.api.fleet.FleetMemberType;
import com.fs.starfarer.api.fleet.ShipRolePick;
import com.fs.starfarer.api.impl.campaign.DModManager;
import com.fs.starfarer.api.impl.campaign.ids.Commodities;
import com.fs.starfarer.api.impl.campaign.ids.Conditions;
import com.fs.starfarer.api.impl.campaign.ids.ShipRoles;
import com.fs.starfarer.api.impl.campaign.submarkets.BlackMarketPlugin;
import com.fs.starfarer.api.util.WeightedRandomPicker;
import data.scripts.util.UW_Defs;
import java.util.List;
import java.util.Random;

public class UW_ScrapyardMarketPlugin extends BlackMarketPlugin {

    @Override
    public float getDesiredAboveDemandCommodityQuantity(CommodityOnMarketAPI com) {
        switch (com.getId()) {
            case Commodities.FUEL:
            case Commodities.SUPPLIES:
            case Commodities.HAND_WEAPONS:
            case Commodities.HEAVY_MACHINERY:
            case Commodities.METALS:
            case Commodities.RARE_METALS:
            case Commodities.VOLATILES:
                break;
            default:
                return 0f;
        }

        boolean illegal = market.isIllegal(com.getId());
        if (illegal) {
            return com.getAboveDemandStockpile();
        }

        float blackMarketLegalFraction = 1f - 0.09f * market.getStabilityValue();
        return com.getAboveDemandStockpile() * blackMarketLegalFraction;
    }

    @Override
    public float getDesiredBelowDemandCommodityQuantity(CommodityOnMarketAPI com) {
        switch (com.getId()) {
            case Commodities.FUEL:
            case Commodities.SUPPLIES:
                break;
            default:
                return 0f;
        }

        boolean illegal = market.isIllegal(com.getId());

        float blackMarketLegalFraction = 0.5f - 0.02f * market.getStabilityValue();
        float blackMarketIllegalFraction = 0.5f - 0.025f * market.getStabilityValue();

        float fraction = blackMarketLegalFraction;
        if (illegal) {
            fraction = blackMarketIllegalFraction;
        }
        return Math.max(0, com.getStockpile() - com.getAboveDemandStockpile()) * fraction;
    }

    @Override
    public String getIllegalTransferText(CargoStackAPI stack, TransferAction action) {
        if (action == TransferAction.PLAYER_SELL) {
            return "No refunds!";
        }
        return "Illegal to trade on the " + submarket.getNameOneLine().toLowerCase() + " here";
    }

    @Override
    public String getIllegalTransferText(FleetMemberAPI member, TransferAction action) {
        if (action == TransferAction.PLAYER_SELL) {
            return "No refunds!";
        }
        return "Illegal to buy";
    }

    @Override
    public float getTariff() {
        return -0.3f;
    }

    @Override
    public void init(SubmarketAPI submarket) {
        super.init(submarket);
    }

    @Override
    public boolean isIllegalOnSubmarket(String commodityId, TransferAction action) {
        if (action == TransferAction.PLAYER_SELL) {
            return true;
        }
        if (market.hasCondition(Conditions.FREE_PORT)) {
            return false;
        }
        return submarket.getFaction().isIllegal(commodityId);
    }

    @Override
    public boolean isIllegalOnSubmarket(CargoStackAPI stack, TransferAction action) {
        if (action == TransferAction.PLAYER_SELL) {
            return true;
        }
        if (!stack.isCommodityStack()) {
            return false;
        }
        return isIllegalOnSubmarket((String) stack.getData(), action);
    }

    @Override
    public boolean isIllegalOnSubmarket(FleetMemberAPI member, TransferAction action) {
        if (action == TransferAction.PLAYER_SELL) {
            return true;
        }
        return false;
    }

    @Override
    public void updateCargoPrePlayerInteraction() {
        for (FleetMemberAPI member : getCargo().getMothballedShips().getMembersListCopy()) {
            /* f = 1 - dmods * 0.2
             dmods = 5 * (1 - f) */
            int dmods = Math.round(5f * (1f - member.getStatus().getHullFraction()));
            ShipVariantAPI variant = member.getVariant();
            int dModsAlready = DModManager.getNumDMods(variant);
            dmods = Math.max(0, dmods - dModsAlready);

            if (dmods > 0) {
                DModManager.setDHull(variant);
            }
            member.setVariant(variant, false, true);
            if (dmods > 0) {
                DModManager.addDMods(member, true, dmods, new Random());
            }
        }

        if (!okToUpdateCargo()) {
            return;
        }
        sinceLastCargoUpdate = 0f;

        CargoAPI cargo = getCargo();

        updateEconomicCommoditiesInCargo(false);

        pruneWeapons(0.5f);

        WeightedRandomPicker<FactionAPI> factionPicker = new WeightedRandomPicker<>();
        int index = 0;
        for (String item : UW_Defs.SCRAPYARD_FACTIONS.getItems()) {
            FactionAPI f;
            try {
                f = Global.getSector().getFaction(item);
            } catch (Exception e) {
                f = null;
            }
            if (f != null) {
                factionPicker.add(f, UW_Defs.SCRAPYARD_FACTIONS.getWeight(index));
            }
            index++;
        }

        addWeaponsBasedOnMarketSize(6, 3, 1, factionPicker);
        addRandomWeapons(Math.max(1, market.getSize()), 1);
        addRandomWings(Math.max(1, market.getSize() - 2), 1);

        addShips();
        addHullMods(1, 1 + itemGenRandom.nextInt(3));
        cargo.sort();
    }

    private void addShips() {
        int marketSize = market.getSize();

        pruneShips(0.75f);

        WeightedRandomPicker<String> rolePicker = new WeightedRandomPicker<>();
        rolePicker.add(ShipRoles.CIV_RANDOM, 5f);
        rolePicker.add(ShipRoles.FREIGHTER_SMALL, 3f);
        rolePicker.add(ShipRoles.TANKER_SMALL, 20f);
        rolePicker.add(ShipRoles.COMBAT_FREIGHTER_SMALL, 15f);
        rolePicker.add(ShipRoles.ESCORT_SMALL, 15f);
        rolePicker.add(ShipRoles.COMBAT_SMALL, 15f);
        rolePicker.add(ShipRoles.LINER_SMALL, 1f);

        rolePicker.add(ShipRoles.FREIGHTER_MEDIUM, 5f);
        rolePicker.add(ShipRoles.TANKER_MEDIUM, 10f);
        rolePicker.add(ShipRoles.COMBAT_FREIGHTER_MEDIUM, 10f);
        rolePicker.add(ShipRoles.COMBAT_MEDIUM, 10f);
        rolePicker.add(ShipRoles.ESCORT_MEDIUM, 10f);
        rolePicker.add(ShipRoles.CARRIER_SMALL, 10f);
        rolePicker.add(ShipRoles.LINER_MEDIUM, 1f);

        rolePicker.add(ShipRoles.FREIGHTER_LARGE, 3f);
        rolePicker.add(ShipRoles.TANKER_LARGE, 5f);
        rolePicker.add(ShipRoles.COMBAT_FREIGHTER_LARGE, 5f);
        rolePicker.add(ShipRoles.COMBAT_LARGE, 5f);
        rolePicker.add(ShipRoles.CARRIER_MEDIUM, 3f);
        rolePicker.add(ShipRoles.LINER_LARGE, 1f);

        rolePicker.add(ShipRoles.COMBAT_CAPITAL, 3f);
        rolePicker.add(ShipRoles.CARRIER_LARGE, 2f);

        WeightedRandomPicker<FactionAPI> factionPicker = new WeightedRandomPicker<>();
        int index = 0;
        for (String item : UW_Defs.SCRAPYARD_FACTIONS.getItems()) {
            FactionAPI f;
            try {
                f = Global.getSector().getFaction(item);
            } catch (Exception e) {
                f = null;
            }
            if (f != null) {
                factionPicker.add(f, UW_Defs.SCRAPYARD_FACTIONS.getWeight(index));
            }
            index++;
        }

        addShipsForRoles(2 + marketSize * 6, rolePicker, factionPicker);
    }

    @Override
    protected void addShipsForRole(String role, Random random, WeightedRandomPicker<FactionAPI> factionPicker) {
        float qf = -1f; /* qf always -1 */

        FactionAPI faction = submarket.getFaction();
        if (factionPicker != null && !factionPicker.isEmpty()) {
            faction = factionPicker.pick();
        }
        List<ShipRolePick> picks = faction.pickShip(role, qf, random);
        for (ShipRolePick pick : picks) {
            FleetMemberType type = FleetMemberType.SHIP;
            String variantId = pick.variantId;
            if (pick.isFighterWing()) {
                type = FleetMemberType.FIGHTER_WING;
            } else {
                FleetMemberAPI member = Global.getFactory().createFleetMember(type, pick.variantId);
                variantId = member.getHullId() + "_Hull";
            }
            FleetMemberAPI member = Global.getFactory().createFleetMember(type, variantId);
            member.getRepairTracker().setMothballed(true);

            int dmods = Math.round(random.nextFloat() * random.nextFloat() * 4f);
            ShipVariantAPI variant = member.getVariant();
            int dModsAlready = DModManager.getNumDMods(variant);
            int newDmods = Math.max(0, dmods - dModsAlready);

            if (newDmods > 0) {
                DModManager.setDHull(variant);
            }
            member.setVariant(variant, false, true);
            if (newDmods > 0) {
                DModManager.addDMods(member, true, newDmods, random);
            }

            member.getStatus().setHullFraction(1f - dmods * 0.1f);
            getCargo().getMothballedShips().addFleetMember(member);
        }
    }

    @Override
    protected void addShipsForRoles(int maxTotal, WeightedRandomPicker<String> rolePicker,
                                    WeightedRandomPicker<FactionAPI> factionPicker) {
        if (rolePicker.isEmpty()) {
            return;
        }

        rolePicker.setRandom(itemGenRandom);
        if (factionPicker != null) {
            factionPicker.setRandom(itemGenRandom);
        }

        int curr = getCargo().getMothballedShips().getMembersListCopy().size();
        int toAdd = maxTotal - curr;
        int added = 0;
        while (added < toAdd) {
            String role = rolePicker.pick();
            addShipsForRole(role, rolePicker.getRandom(), factionPicker);
            added++;
        }
    }
}
