﻿/*
 *  Konijima Engine Control
 *  Open-Source, you can edit & share this mod
 *  https://www.gta5-mods.com/scripts/net-konijima-engine-control-konijima
 */

using GTA;
using GTA.Math;
using GTA.Native;
using System;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Collections.Generic;

namespace KonijimaEngineControl
{
    public class Main : Script
    {
        #region SETTINGS

        // Can disable quick start engine (using acceleration/reverse)
        bool quickStart = true;

        // Turn On Engine KeyDown Delay (ticks)
        int turnOnEngineKeyDownDelay = 20;

        // Turn Off Engine KeyDown Delay (ticks)
        int turnOffEngineKeyDownDelay = 50;

        // Exit Vehicle KeyDown Delay (ticks)
        int exitCarKeyDownDelay = 20;

        // Turn Engine On Control
        Control turnOnEngineKey = Control.VehicleHandbrake;

        // Turn Engine Off Control
        Control turnOffEngineKey = Control.VehicleHandbrake;

        // Exit Vechile Control
        Control exitCarKey = Control.VehicleExit;

        #endregion

        // Variables
        private Player player;
        private Ped character;
        private Vehicle vehicle;
        private VehicleSeat vehicleSeat;
        private bool engineState = false;
        private int turnOnEngineKeyDownTime = 0;
        private bool turnOnEngineCompleted = false;
        private int turnOffEngineKeyDownTime = 0;
        private bool turnOffEngineCompleted = false;
        private int exitCarKeyDownTime = 0;
        private bool exitCarCompleted = false;

        // Constructor
        public Main()
        {
            player = Game.Player;
            character = Game.Player.Character;
            vehicle = (character.IsInVehicle()) ? character.CurrentVehicle : null;
            vehicleSeat = (vehicle != null) ? character.SeatIndex : VehicleSeat.None;

            Tick += Main_Tick;

            // If the mods are loading while the player is into a vehicle this will save the current engine state of that car
            if (vehicle != null && vehicleSeat == VehicleSeat.Driver)
                engineState = vehicle.EngineRunning;
            
        }

        // Tick
        void Main_Tick(object sender, EventArgs e)
        {
            // Update variable every tick
            player = Game.Player;
            character = Game.Player.Character;
            vehicle = (character.IsInVehicle()) ? character.CurrentVehicle : null;
            vehicleSeat = (vehicle != null) ? character.SeatIndex : VehicleSeat.None;

            // Get the vehicle the player is trying to enter
            // This will get/set the default engine state of the vehicle he enters.
            if (!character.IsInVehicle() && character.IsGettingIntoAVehicle)
            {
                Vehicle vehicleIsTryingToEnter = character.GetVehicleIsTryingToEnter();
                if (vehicleIsTryingToEnter != null)
                    engineState = vehicleIsTryingToEnter.EngineRunning;
            }

            // Release turnOffEngineKey
            if (Game.IsControlJustReleased(0, turnOffEngineKey))
            {
                turnOffEngineCompleted = false;
                turnOffEngineKeyDownTime = 0;
            }

            // Release turnOnEngineKey
            if ((Game.IsControlJustReleased(0, turnOnEngineKey)) || (quickStart && (Game.IsControlJustReleased(0, Control.VehicleAccelerate) || Game.IsControlJustReleased(0, Control.VehicleBrake))))
            {
                turnOnEngineCompleted = false;
                turnOnEngineKeyDownTime = 0;
            }

            // Release exitCarKey
            if (Game.IsControlJustReleased(0, exitCarKey))
            {
                exitCarCompleted = false;
                exitCarKeyDownTime = 0;
            }

            // Player is into a vehicle as driver
            if (vehicle != null && vehicleSeat == VehicleSeat.Driver)
            {
                // The engine is on
                if (engineState)
                {
                    // Force car engine ON
                    Function.Call(Hash.SET_VEHICLE_ENGINE_ON, vehicle, true, false, true);

                    /* Exit Vehicle */

                    // Cant exit the car while its moving (without jumping off)
                    if (vehicle.Speed < 1f)
                    {
                        // Cannot exit the car while the player is aiming
                        if (player.IsAiming) return;

                        // Disable the normal ExitCar function
                        Game.DisableControlThisFrame(0, exitCarKey);

                        // Update the exitCarKeyDownTime while the key is down
                        if (Game.IsControlPressed(0, exitCarKey) && !exitCarCompleted)
                            exitCarKeyDownTime++;

                        // Complete the action if you hold the key long enought (shut engine & exit)
                        if (exitCarKeyDownTime > exitCarKeyDownDelay && !exitCarCompleted)
                            exitCarCompleted = true;

                        // Complete the action if you release before the required time (do not shut the engine & exit)
                        if (Game.IsDisabledControlJustReleased(0, exitCarKey) && !exitCarCompleted)
                        {
                            character.Task.LeaveVehicle();
                            ForceTurnCarOn();
                        }

                        // The action may be executed
                        if (exitCarCompleted)
                        {
                            character.Task.LeaveVehicle();
                            engineState = false;
                        }
                    }

                    /* Turn off engine */

                    // Cannot stop the engine if the vehicle is moving
                    if (vehicle.Speed > 1f) return;

                    // Prevent repetition of this function until the button is released
                    if (turnOnEngineCompleted) return;

                    // Update the turnOffEngineKeyDownTime while the key is down
                    if (Game.IsControlPressed(0, turnOffEngineKey))
                        turnOffEngineKeyDownTime++;

                    // Complete the action if the button was pressed long enought
                    if (turnOffEngineKeyDownTime > turnOffEngineKeyDownDelay)
                        turnOffEngineCompleted = true;

                    // If the action is complete, turn on the engine
                    if (turnOffEngineCompleted)
                        engineState = false;
                }

                // The engine is off
                else
                {
                    // Force car engine OFF                    
                    Function.Call(Hash.SET_VEHICLE_ENGINE_ON, vehicle, false, true, true);

                    // Can't turn on the engine while aiming
                    if (player.IsAiming) return;

                    // Prevent repetition of this function until the button is released
                    if (turnOffEngineCompleted) return;

                    // Update the turnOnEngineKeyDownTime while the key is down
                    if (Game.IsControlPressed(0, turnOnEngineKey))
                        turnOnEngineKeyDownTime++;

                    // Complete the action if the button was pressed long enought
                    if (turnOnEngineKeyDownTime > turnOnEngineKeyDownDelay)
                        turnOnEngineCompleted = true;

                    // Complete the action if the quickstart buttons are pressed (no button holding necessary)
                    if (quickStart && (Game.IsControlPressed(0, Control.VehicleAccelerate) || Game.IsControlPressed(0, Control.VehicleBrake)))
                        turnOnEngineCompleted = true;

                    // If the action is complete, turn on the engine
                    if (turnOnEngineCompleted)
                        engineState = true;
                }
            }
        }

        // Small trick to force the engine to stay open while you get off
        void ForceTurnCarOn()
        {
            // So basicly the 200ms from the moment you get out of the vehicle this loop will force open the engine
            for (int x = 0; x < 20; x++)
            {
                Script.Wait(10);
                engineState = true;
                vehicle.EngineRunning = true;
            }
        }
    }
}
